///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// (c) Copyright OCP-IP 2008
// OCP-IP Confidential and Proprietary
//
//
//============================================================================
//      Project : OCP SLD WG
//       Author : James Aldis (TI France)
//                Robert Guenzel (from TU of Braunschweig) for Greensocs Ltd.
//
//          $Id:
//
//  Description :  Simple master for example.
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#include "simpleMaster.h"

#define NUM_TESTS         9

// -------------------------------------------------------------------
// constructor
// -------------------------------------------------------------------
template<unsigned int BUSWIDTH>
Master<BUSWIDTH>::Master(
    sc_core::sc_module_name name,
    int            id,
    std::ostream*       debug_os_ptr
) : sc_core::sc_module(name),
    ipP("ipPort", this, &Master::setOCPTL1SlaveTiming, ocpip::ocp_master_socket_tl1<BUSWIDTH>::mm_txn_with_data()),
    clk("clkPort"),
    m_ID(id),
    m_debug_os_ptr(debug_os_ptr),
    m_threads(1),
    m_addrspace(false),
    m_sthreadbusy(false),
    m_sthreadbusy_exact(false),
    m_mthreadbusy(false),
    m_mthreadbusy_exact(false),
    m_respaccept(true),
    m_datahandshake(false),
    m_writeresp_enable(false),
    m_writenonpost_enable(false),
    m_respaccept_fixeddelay(1),
    m_respaccept_delay(1),
    m_resp_txn(NULL),
    m_stb(0)
{
    // setup a SystemC thread process, which uses dynamic sensitive
    SC_THREAD(requestThreadProcess);
    sensitive<<clk.pos();

    // setup a SystemC thread process, which uses dynamic sensitive
    SC_THREAD(responseThreadProcess);
    sensitive<<clk.pos();
    
    //assuming default timing the master will expect sthreadbusy to be stable 1 ps after the clock edge    
    m_sthreadbusy_sample_time=sc_core::sc_time(1, sc_core::SC_PS);
}

template<unsigned int BUSWIDTH>
void Master<BUSWIDTH>::provideChannelConfiguration(ocpip::map_string_type& passedMap){
  m_parameters.set_ocp_configuration(name(), passedMap);
  parse_config();
}

template<unsigned int BUSWIDTH>
void Master<BUSWIDTH>::setModuleConfiguration(ocpip::map_string_type& passedMap){
    if (!(ocpip::ocp_parameters::getBoolOCPConfigValue("", "mrespaccept_fixeddelay", 
            m_respaccept_fixeddelay, passedMap)) ) {
        // Could not find the parameter so we must set it to a default
#ifdef DEBUG 
        cout << "Warning: paramter \"" << "mrespaccept_fixeddelay" 
               << "\" was not found in the module parameter map." << endl;
        cout << "         setting missing parameter to "<<m_respaccept_fixeddelay<<"." << endl;
#endif
    }

    if (!(ocpip::ocp_parameters::getIntOCPConfigValue("", "mrespaccept_delay", 
            m_respaccept_delay, passedMap)) ) {
        // Could not find the parameter so we must set it to a default
#ifdef DEBUG 
        cout << "Warning: paramter \"" << "mrespaccept_delay" 
               << "\" was not found in the module parameter map." << endl;
        cout << "         setting missing parameter to "<<m_respaccept_delay<<"." << endl;
#endif
    }
}

// -------------------------------------------------------------------
// SystemC Method Master::end_of_elaboration()
// -------------------------------------------------------------------
// 
//  At this point, everything has been built and connected.
//  We are now free to get our OCP parameters and to set up our
//  own variables that depend on them.
//
template<unsigned int BUSWIDTH>
void Master<BUSWIDTH>::parse_config()
{
    // Call the System C version of this function first
    //sc_module::end_of_elaboration();

    //-----------------------------------------
    //  OCP Parameters
    //-----------------------------------------

    // Get the number of threads
    m_threads=m_parameters.threads;

    // This Reference Master is single threaded.
    if (m_threads > 1) {
        std::cout << "ERROR: Single threaded Master \"" << name() 
                 << "\" connected to OCP with " << m_threads 
                 << " threads." << std::endl; 
    }

    // is the MAddrSpace field part of the OCP channel?
    m_addrspace=m_parameters.addrspace;

    // is SThreadBusy part of the channel?
    m_sthreadbusy=m_parameters.sthreadbusy;

    // Is SThreadBusy compliance required?
    m_sthreadbusy_exact=m_parameters.sthreadbusy_exact;

    // is MThreadBusy part of the channel?
    m_mthreadbusy=m_parameters.mthreadbusy;

    // Is MThreadBusy compliance required?
    m_mthreadbusy_exact=m_parameters.mthreadbusy_exact;

    // is MRespAccept part of the channel?
    m_respaccept=m_parameters.respaccept;

    // Just a double check here
    if (m_mthreadbusy_exact && m_respaccept) {
        std::cout << "ERROR: Master \"" << name() 
        << "\" connected to OCP with both MThreadBusy_Exact and MRespAccept active which are exclusive." << std::endl; 
    }

    // is Data Handshake part of the channel?
    m_datahandshake=m_parameters.datahandshake;
    // if so, quit. This core does not support it.
    assert(m_datahandshake == false);
    assert(m_parameters.reqdata_together);

    // is write response part of the channel?
    m_writeresp_enable=m_parameters.writeresp_enable;

    // is READ-EX part of the channel?
    m_readex_enable=m_parameters.readex_enable;

    // Are non-posted writes (write commands that receive responses) 
    //part of the channel?
    m_writenonpost_enable=m_parameters.writenonpost_enable;

    //-----------------------------------------
    //  Master Specific Parameters
    //-----------------------------------------

    // Retrieve any configuration parameters that were passed to this block
    // in the setConfiguration command.

#ifdef DEBUG
    cout << "I am configuring a Master!" << endl;
    cout << "Here is my configuration map for Master >" 
            << name() << "< that was passed to me." << endl;
    ocpip::map_string_type::iterator map_it;
    for (map_it = m_parameters.Map.begin(); map_it != m_parameters.Map.end(); ++map_it) {
        cout << "map[" << map_it->first << "] = " << map_it->second << endl;
    }
    cout << endl;
#endif
    ipP.set_ocp_config(m_parameters);


    //in case sthreadbusy is part of the channel, this master will not be
    //default timing anymore and gets timing sensitive, too
    if (m_sthreadbusy){
      ocpip::ocp_tl1_master_timing myTiming;
      //requests start after sthreadbusy is stable
      myTiming.RequestGrpStartTime=m_sthreadbusy_sample_time;
      //ipP->registerTimingSensitiveOCPTL1Master((OCP_TL1_Slave_TimingIF*) this);
      ipP.set_master_timing(myTiming);
    }
    
    if (m_mthreadbusy_exact | m_mthreadbusy){
      tb_txn=ipP.get_tb_transaction();
      tb=ipP.template get_extension<ocpip::thread_busy>(*tb_txn);
      tb->value.type=ocpip::M_THREAD;
      tb_ph=ocpip::THREAD_BUSY_CHANGE;
    }
    
    ipP.register_nb_transport_bw(this, &Master::nb_transport);
    ipP.activate_delta_cycle_protection();
}

template<unsigned int BUSWIDTH>
Master<BUSWIDTH>::~Master() {}

template<unsigned int BUSWIDTH>
void Master<BUSWIDTH>::setOCPTL1SlaveTiming(ocpip::ocp_tl1_slave_timing slave_timing){

  if (slave_timing.SThreadBusyStartTime+sc_core::sc_time(2,sc_core::SC_PS)>m_sthreadbusy_sample_time){
    m_sthreadbusy_sample_time=slave_timing.SThreadBusyStartTime+sc_core::sc_time(2,sc_core::SC_PS); //since we use DCP we add 1 for DCP and one for sampling
    ocpip::ocp_tl1_master_timing myTiming;
    myTiming.RequestGrpStartTime=m_sthreadbusy_sample_time;
    ipP.set_master_timing(myTiming);       
  }
}

template<unsigned int BUSWIDTH>
void Master<BUSWIDTH>::requestThreadProcess()
{
    sc_dt::uint64 Addr[] = {0x1784, 0x20, 0x20, 0x40};

    // start time of requests
    int NumWait[NUM_TESTS][4] = {
            {100,   3, 0xF, 0xF},
            {7,   1,   3, 0xF},
            {6, 0xF, 0xF, 0xF},
            {10,  2,   1, 0xF},
            {7,   1,   3, 0xF},
            {6,   1,   1,   1},
            {7,   2, 0xF, 0xF},
            {8,   2,   1, 0xF},// no data handshake
            {7,   2,   2,   2}
        };

    // specifies the command to use
    tlm::tlm_command Commands[NUM_TESTS][4]  = {
        {tlm::TLM_WRITE_COMMAND, tlm::TLM_READ_COMMAND, tlm::TLM_IGNORE_COMMAND, tlm::TLM_IGNORE_COMMAND},
        {tlm::TLM_WRITE_COMMAND, tlm::TLM_WRITE_COMMAND, tlm::TLM_WRITE_COMMAND, tlm::TLM_IGNORE_COMMAND},
        {tlm::TLM_READ_COMMAND, tlm::TLM_IGNORE_COMMAND, tlm::TLM_IGNORE_COMMAND, tlm::TLM_IGNORE_COMMAND},
        {tlm::TLM_READ_COMMAND, tlm::TLM_READ_COMMAND, tlm::TLM_READ_COMMAND, tlm::TLM_IGNORE_COMMAND},
        {tlm::TLM_READ_COMMAND, tlm::TLM_READ_COMMAND, tlm::TLM_READ_COMMAND, tlm::TLM_IGNORE_COMMAND},
        {tlm::TLM_READ_COMMAND, tlm::TLM_READ_COMMAND, tlm::TLM_READ_COMMAND, tlm::TLM_READ_COMMAND},
        {tlm::TLM_READ_COMMAND, tlm::TLM_READ_COMMAND, tlm::TLM_IGNORE_COMMAND, tlm::TLM_IGNORE_COMMAND},
        {tlm::TLM_WRITE_COMMAND, tlm::TLM_WRITE_COMMAND, tlm::TLM_WRITE_COMMAND, tlm::TLM_IGNORE_COMMAND},
        {tlm::TLM_READ_COMMAND, tlm::TLM_READ_COMMAND, tlm::TLM_READ_COMMAND, tlm::TLM_READ_COMMAND}
    };

    // number of specified transactions in a test
    int NumTr[] = {2, 3, 1, 3, 3, 4, 2, 3, 4};

    // -----------------------------------
    // (1) processing and preparation step
    // -----------------------------------

    // initialize data
    //OCPRequestGrp<Td,Ta> req;
    tlm::tlm_phase  phase;
    sc_core::sc_time time;
    int              Count = 0;
    int              Nr = 0;
    sc_core::sc_time          old_time;
    sc_core::sc_time          current_time;
    //bool             sthreadbusy;
    Td               my_data = 0;

    // calculate the new waiting time
    double  wait_for = NumWait[Nr][Count];

    // Do requests contain data (or will it be sent separately)
    // Always true as this core does not support data handshake
    //req.HasMData = true;        

    sc_core::wait();

    // main loop
    while (true) {
        // wait for the time to send the current request

        if (m_debug_os_ptr) {
            (*m_debug_os_ptr) << "DB (" << name() << "): "
                         << "master wait_for = " << wait_for << std::endl;
        }

        for (int i=0; i<wait_for; i++) sc_core::wait();

        // remember the time
        old_time = sc_core::sc_time_stamp();

        // ------------------------------------------------
        // (2) is SThreadBusy?
        // ------------------------------------------------
        
        // NOTE: we are single threaded so the thread busy signal
        // looks like a boolean (0 or 1). 
        //   Abritration based on thread busy will be needed for a
        //   multi-threaded model.
        if (m_sthreadbusy_exact) {
            sc_core::wait(m_sthreadbusy_sample_time); //wait until sthreadbusy is stable
            while (m_stb) {
                sc_core::wait();
                sc_core::wait(m_sthreadbusy_sample_time); //wait until sthreadbusy is stable
            }
        }

        // ------------------------------------------------
        // (3) send a request
        // ------------------------------------------------

        // NOTE: data handshake is not handled by this simple example.

        // Compute the next request
        tlm::tlm_generic_payload* txn=ipP.get_transaction();
        txn->set_command(Commands[Nr][Count]);

        // compute the address
        txn->set_address(Addr[Count] + m_ID*0x40);
        txn->set_byte_enable_ptr(0); //MByteEn = 0xf;
        if (m_addrspace) {
          ocpip::address_space* MAddrSpace;
          ipP.template get_extension<ocpip::address_space>(MAddrSpace, *txn);
          MAddrSpace->value = 0x1;
          ipP.template validate_extension<ocpip::address_space>(*txn);
        }
        // compute the data
        ipP.reserve_data_size(*txn, sizeof(Td)); //allocate enough data size for one word
        txn->set_data_length(sizeof(Td));
        
        txn->set_streaming_width(sizeof(Td)); //no streaming
        txn->set_response_status(tlm::TLM_INCOMPLETE_RESPONSE);
        
        switch (txn->get_command()) {
            case tlm::TLM_WRITE_COMMAND:
                // This is a write command - it has data
                my_data++;
                // put the data into the request
                *((Td*)txn->get_data_ptr()) = my_data + m_ID*0x40;
                break;
            case tlm::TLM_READ_COMMAND:
                // this is a read command - no data.
                *((Td*)txn->get_data_ptr()) = 0;
                break;
            default:
                std::cout << "ERROR: Master \"" << name() 
                     << "\" generates unknown command #"
                          << txn->get_command() << std::endl;
        }

        if (m_debug_os_ptr) {
            (*m_debug_os_ptr) << "DB (" << name() << "): "
                              << "send request." << std::endl;
            (*m_debug_os_ptr) << "DB (" << name() << "): "
                       << "    t = " << sc_core::sc_time_stamp() << std::endl;
            (*m_debug_os_ptr) << "DB (" << name() << "): "
                              << "    MCmd: " << ((txn->get_command()==tlm::TLM_WRITE_COMMAND)? ocpip_legacy::OCP_MCMD_WR 
                                                                                             : 
                                                                                             (txn->get_command()==tlm::TLM_READ_COMMAND)? ocpip_legacy::OCP_MCMD_RD
                                                                                             : 255 )<< std::endl;
            (*m_debug_os_ptr) << "DB (" << name() << "): "
                              << "    MData: " << (*((Td*)txn->get_data_ptr())) << std::endl;
            (*m_debug_os_ptr) << "DB (" << name() << "): "
                           << "    MByteEn: " << 0xf << std::endl;
        }

        // send the request
        //ipP->startOCPRequestBlocking(req);
        phase=tlm::BEGIN_REQ;
        time=sc_core::SC_ZERO_TIME;
        tlm::tlm_sync_enum returnCode=ipP->nb_transport_fw(*txn, phase, time);
        current_time = sc_core::sc_time_stamp();
        switch(returnCode){
          case tlm::TLM_UPDATED:
            if(phase==tlm::END_REQ)  break;
            else{
              std::cerr<<"Error unexpected phase "<<phase<<" received from fw return path in module "<<name()<<std::endl;
              exit(1);
            }
          case tlm::TLM_ACCEPTED: sc_core::wait(req_ack_event); break;
          case tlm::TLM_COMPLETED: ;
        }
        
        if (txn->get_command()==tlm::TLM_WRITE_COMMAND && !m_writeresp_enable){
          ipP.release_transaction(txn);
          if (m_debug_os_ptr) {
          //    (*m_debug_os_ptr) << "DB (" << name() << "): "
          //                      << "release txn (write)."<<txn << std::endl;
          }        
        }
        // -------------------------------
        // (1) processing and preparation step
        // -------------------------------
        
        //if (sc_core::sc_time_stamp()==current_time) 
        sc_core::wait(); //advance to next cycle to avoid sending multiple request in same cycle

        // compute the next pointer
        if (++Count >= NumTr[Nr]) {
            Count = 0;
            if (++Nr >= NUM_TESTS) Nr = 1;
        }

        // calculate the new waiting time
        wait_for = NumWait[Nr][Count];
        current_time = sc_core::sc_time_stamp();
        double delta_time = 
              (current_time.value() - old_time.value()) / 1000;
        if (delta_time >= wait_for) {
            wait_for = 0;
        } else {
            wait_for = wait_for - delta_time;
        }
    }
}

template<unsigned int BUSWIDTH>
tlm::tlm_sync_enum Master<BUSWIDTH>::nb_transport(tlm::tlm_generic_payload& gp, tlm::tlm_phase& ph, sc_core::sc_time& t){
  switch (ph){
    case tlm::BEGIN_RESP:
      m_resp_txn=&gp;
      resp_event.notify();
      if (m_mthreadbusy_exact){
        assert(tb->value.mask==0);
        ph=tlm::END_RESP;
        return tlm::TLM_UPDATED; //obey the rules!
      }
      break;
    case tlm::END_REQ: req_ack_event.notify(); 
      break;
    case tlm::BEGIN_REQ:
    case tlm::END_RESP:
      std::cerr<<"Wrong phase "<<ph<<" at "<<sc_core::sc_time_stamp()<<" in "<<name()<<std::endl;
      exit(1);
    default:
      if (ph==ocpip::THREAD_BUSY_CHANGE){
        assert((ipP.template get_extension<ocpip::thread_busy>(gp))->value.type==ocpip::S_THREAD);
        m_stb=ipP.template get_extension<ocpip::thread_busy>(gp)->value.mask;
      }
      else{
        std::cerr<<"Unexpected phase "<<ph<<" at "<<sc_core::sc_time_stamp()<<" in "<<name()<<std::endl;
        exit(1);
      }
  }
  return tlm::TLM_ACCEPTED;
}


template<unsigned int BUSWIDTH>
void Master<BUSWIDTH>::responseThreadProcess()
{

    // initialization
    double         wait_for;
    sc_core::sc_time time;
    tlm::tlm_phase phase=tlm::END_RESP;

    sc_core::wait();

    // main loop
    while (true) {
        // ------------------------------------------------
        // (1) wait for a response (blocking wait)
        // ------------------------------------------------

        // get the next response
        //ipP->getOCPResponseBlocking(resp);
        if(!m_resp_txn) sc_core::wait(resp_event);
        //wait(); //sync to clock after resp reception
        tlm::tlm_generic_payload* txn=m_resp_txn;
        m_resp_txn=NULL;

        // ------------------------
        // (2) process the response
        // ------------------------

        // compute the response acceptance time
        if (m_respaccept_fixeddelay) {
           wait_for = m_respaccept_delay;
        } else {
           // Go random up to max delay
           wait_for = 
            (int)((m_respaccept_delay+1) * rand() / (RAND_MAX + 1.0));
        }

        // --------------------------------------------------
        // (3) generate a one-cycle-pulse MRespAccept signal
        // --------------------------------------------------

        if (m_respaccept) {
            if (wait_for == 0) {
                // send an one-cycle-pulse MRespAccept signal
                //ipP->putMRespAccept();
                ipP->nb_transport_fw(*txn, phase, time);
                assert(phase==tlm::END_RESP); //may not be changed
            } else {
                // wait for the acceptance pulse cycle
                for (int i=0; i<wait_for; i++) sc_core::wait();
                // send an one-cycle-pulse MRespAccept signal
                ipP->nb_transport_fw(*txn, phase, time);
                assert(phase==tlm::END_RESP); //may not be changed
            }
        }

        if (m_mthreadbusy_exact) {
           // use the MThreadBusy signal instead of resp accept
            if (wait_for > 0) {
                // Set MThreadBusy
                sc_core::wait(); //wait until next cycle to set busy
                //ipP->putMThreadBusy(1);
                tb->value.mask=1;
                ipP->nb_transport_fw(*tb_txn, tb_ph, time);
                assert(tb_ph==ocpip::THREAD_BUSY_CHANGE);
                // keep MThreadBusy on 
                for (int i=0; i<wait_for; i++) sc_core::wait();
                // now release it
                //ipP->putMThreadBusy(0);
                tb->value.mask=0;
                ipP->nb_transport_fw(*tb_txn, tb_ph, time);
                assert(tb_ph==ocpip::THREAD_BUSY_CHANGE);
            }
        }
        ipP.release_transaction(txn);
        if (m_debug_os_ptr) {
        //    (*m_debug_os_ptr) << "DB (" << name() << "): "
        //                      << "release txn."<<txn << std::endl;
        }        
    }
}

template class Master< 32 >;
