///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// (c) Copyright OCP-IP 2008
// OCP-IP Confidential and Proprietary
//
//
//============================================================================
//      Project : OCP SLD WG
//       Author : James Aldis (TI France)
//                Robert Guenzel (from TU of Braunschweig) for Greensocs Ltd.
//
//          $Id:
//
//  Description :  Simple Main to read in Map data from files
//                 and then use that to configure and connect
//                 a master and slave.
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#include <map>
#include <set>
#include <string>
#include <algorithm>
#include <stdio.h>
#include <stdlib.h>
#include <iostream>

#include "simpleMaster.h"
#include "simpleSlave.h"

#define OCP_CLOCK_PERIOD 1
#define OCP_CLOCK_TIME_UNIT sc_core::SC_NS

void process_command_line(int   argc,
                          char* argv[],
                          std::string& ocp_params_file_name,
                          std::string& module_params_file_name,
                          double& simulation_end_time,
                          bool& debug_dump,
                          std::string& debug_file_name)
{
    // get the ocp parameters file name
    ocp_params_file_name = "";
    if (argc > 1) {
        std::string file_name(argv[1]);
        ocp_params_file_name = file_name;
    }

    // get the module parameters file name
    module_params_file_name = "";
    if (argc > 2) {
        std::string file_name(argv[2]);
        module_params_file_name = file_name;
    }

    // get the simulation end time
    simulation_end_time = 1000;
    if (argc > 3) {
        simulation_end_time = (double) atoll(argv[3]);
    }

    // do we dump out a log file?
    debug_dump= false;
    debug_file_name = "";
    if (argc > 4) {
        std::string file_name(argv[4]);
        debug_file_name = file_name;
        debug_dump = true;
    }
}

void readMapFromFile(const std::string &myFileName, map_string_type &myParamMap) 
{
    // read pairs of data from the passed file
    std::string leftside;
    std::string rightside;
    
    // (1) open the file
    std::ifstream inputfile(myFileName.c_str());
    assert( inputfile );

    // set the formatting
    inputfile.setf(std::ios::skipws);

    // Now read through all the pairs of values and add them to the passed map
    while ( inputfile ) {
        inputfile >> leftside;
        inputfile >> rightside;
        myParamMap.insert(std::make_pair(leftside,rightside));
    }

    // All done, close up
    inputfile.close();
}

int
sc_main(int argc, char* argv[])
{
  
  //OCP_TL1_Channel_Clocked< OCP_TL1_DataCl<OCPCHANNELBit32, OCPCHANNELBit32> >* pOCP;
    Master< 32 >* pMaster;
    Slave< 32 >*  pSlave;
    map_string_type  ocpParamMap;
    map_string_type  moduleParamMap;

    double         simulation_end_time;
    bool           debug_dump;
    std::string         ocpParamFileName;
    std::string         moduleParamFileName;
    std::string         dump_file_name;
    std::ofstream       debugFile;

    // --------------------------------
    // (1) process command line options
    //     and read my parameters
    // --------------------------------
    process_command_line(argc,argv,ocpParamFileName,moduleParamFileName,
           simulation_end_time,debug_dump,dump_file_name);

    if ( ! ocpParamFileName.empty() ) {
        readMapFromFile(ocpParamFileName, ocpParamMap);
    }

    if ( ! moduleParamFileName.empty() ) {
        readMapFromFile(moduleParamFileName, moduleParamMap);
    }

    // open a trace file
    if (debug_dump) {
        std::cout << "Debug dumpfilename: " << dump_file_name << std::endl;
        debugFile.open(dump_file_name.c_str());
    }

    // ----------------------------------------------------------
    // (2) Create the clocked OCP Channel
    // ----------------------------------------------------------
    sc_core::sc_clock clk("clk", OCP_CLOCK_PERIOD,OCP_CLOCK_TIME_UNIT) ;


    // ----------------------------------------------------------
    // (3) Create the Master and Slave
    // ----------------------------------------------------------

    pMaster = new Master< 32 >("master", 
                                                                              0, 
                                                                              &debugFile 
                                                                              );
    pSlave = new Slave< 32 >("slave", 
                                                                            0, 
                                                                            0x3FF,
                                                                            &debugFile 
                                                                            );
                                                                            
    pMaster->setModuleConfiguration(moduleParamMap);
    pSlave->setModuleConfiguration(moduleParamMap);

    // ----------------------------------------------------------
    // (4) connect channel, master, slave, and clock
    // ----------------------------------------------------------
#ifdef USE_OCP_MONITOR
    ocpip::ocp_connection_monitor<32> ocp_monitor_t_piece(pMaster->ipP, pSlave->tpP);
#else
    pMaster->ipP(pSlave->tpP);
#endif
    pMaster->clk(clk);
    pSlave->clk(clk);
    // ------------------------
    // (5) start the simulation
    // ------------------------
    pMaster->provideChannelConfiguration(ocpParamMap);
    sc_core::sc_start(simulation_end_time,sc_core::SC_NS);

    // -------------------
    // (6) post processing
    // -------------------

    std::cout << "main program finished at " 
         << sc_core::sc_time_stamp().to_double() << std::endl;

    sc_core::sc_simcontext* sc_curr_simcontext = sc_core::sc_get_curr_simcontext();
    std::cout << "delta_count: " << std::dec << sc_core::sc_delta_count() 
             << std::endl;
    std::cout << "next_proc_id: " << std::dec << sc_curr_simcontext->next_proc_id()
         << std::endl;
    delete pMaster;
    delete pSlave;
    return (0);
}
