// (c) Copyright OCP-IP 2004
// OCP-IP Confidential and Proprietary
//
// ============================================================================
//      Project : OCP SLD WG
//       Original Authors :
//                Anssi Haverinen, Nokia Inc.
//                Alan Kamas, for Sonics Inc.
//                Joe Chou, Sonics Inc.
//                James Aldis, Texas Instruments Inc.
//           $Id: ocp_mon.h,v 1.1.1.1 2004/10/16 20:18:51 Anssi Exp $
//
//  Description : Monitor component for OCP TL1 transaction API
//                Prints out a CoreCreator-compatible trace file.
//                Note: Instantiate after the channel monitored has been configured.
//   Old History:
//          04/09/2004 Original revision, based on Dec 2003 OCP TL1 channel
//          07/02/2004 Changed time to simulation time.
//
//          05/08/2004 Added a destructor so that
//                     the m_ocpMonPtr destructor can be called.
//
// ============================================================================


#ifndef _OCP_MON_H
#define _OCP_MON_H
#include "ocp_tl1_ocpmongen_cl.h"

// forward declaration needed for the class below:
template <class TdataCl> class OCP_TL1_Channel;

template<typename TdataCl>
class OCPMon : public sc_module
{
 public:
  typedef typename TdataCl::DataType Td;
  typedef typename TdataCl::AddrType Ta;

  SC_HAS_PROCESS(OCPMon);
  OCPMon(sc_module_name mn,
	 OCP_TL1_Channel<TdataCl> *myChannel,
	 std::string monFileName = "",
	 sc_clock * clk = NULL,
	 bool print_params = true)
    : sc_module(mn), m_channel(myChannel), m_print_params(print_params)
  {
    std::string n = (std::string)mn+"gen";
    m_ocpMonPtr = new OCP_TL1_OCPMonGenCl<TdataCl>((std::string)n, monFileName);
    m_clk_count = 0;
    SC_THREAD(AtClockEdge);
    sensitive_pos(*clk);
  }

  OCPMon(sc_module_name mn,
	 OCP_TL1_Channel<TdataCl> *myChannel,
	 std::string monFileName = "",
	 sc_in_clk * clk = NULL,
	 bool print_params = true)
    : sc_module(mn), m_channel(myChannel), m_print_params(print_params)
  {
    std::string n = (std::string)mn+"gen";
    m_ocpMonPtr = new OCP_TL1_OCPMonGenCl<TdataCl>((std::string)n, monFileName);

    m_clk_count = 0;
    SC_THREAD(AtClockEdge);
    sensitive_pos(*clk);
  }

  OCPMon(sc_module_name mn,
	 OCP_TL1_Channel<TdataCl> *myChannel,
	 std::string monFileName = "",
	 sc_signal<bool> * clk = NULL,
	 bool print_params = true)
    : sc_module(mn), m_channel(myChannel), m_print_params(print_params)
  {
    std::string n = (std::string)mn+"gen";
    m_ocpMonPtr = new OCP_TL1_OCPMonGenCl<TdataCl>((std::string)n, monFileName);
    m_clk_count = 0;
    SC_THREAD(AtClockEdge);
    sensitive_pos(*clk);
  }

  OCPMon(sc_module_name mn,
	 OCP_TL1_Channel<TdataCl> *myChannel,
	 double clock_period = 1,
	 sc_time_unit clock_time_unit = SC_NS,
	 std::string monFileName = "",
	 bool print_params = true)
    : sc_module(mn), m_channel(myChannel), m_print_params(print_params)
  {
    std::string n = (std::string)mn+"gen";
    m_ocpMonPtr = new OCP_TL1_OCPMonGenCl<TdataCl>((std::string)n, monFileName);
    m_clk_count = 0;
    SC_THREAD(SelfTimed);
  }

  // Destructor
  ~OCPMon()
  {
      // delete the OCP_TL1_OCPMonGenCl which contains a file to flush
      delete m_ocpMonPtr;
  }

  // This function is called every OCP clock tick
  void
  OCPClockTick(void)
  {
    // NOTE: would be more efficient to simply change the current MCmd/SResp/MDataValid
    //       and then change it back after the call.
    //       However, this is cleaner.
    OCPRequestGrp<Td,Ta> monRequest;
    OCPDataHSGrp<Td> monDataHS;
    OCPResponseGrp<Td> monResponse;


    bool monReqAccept =  (m_channel->m_CommCl->RequestEnd) || (m_channel->m_CommCl->RequestStart && m_channel->m_CommCl->RequestEarlyEnd);
    bool monRespAccept = m_channel->m_CommCl->ResponseEnd  || (m_channel->m_CommCl->ResponseStart && m_channel->m_CommCl->ResponseEarlyEnd);
    bool monDataAccept = m_channel->m_CommCl->DataRequestEnd  || (m_channel->m_CommCl->DataRequestStart &&m_channel->m_CommCl->DataRequestEarlyEnd);


    monRequest = m_channel->m_DataCl->SgetRequestGroup();
    if (m_channel->m_DataCl->MCmdSavedForOCPTrace != OCP_MCMD_IDLE) {
      monRequest.MCmd = m_channel->m_DataCl->MCmdSavedForOCPTrace;
      if (monReqAccept) {
	// Value has been used. We can now reset it.
	m_channel->m_DataCl->MCmdSavedForOCPTrace = OCP_MCMD_IDLE;
      }
    }
    monDataHS = m_channel->m_DataCl->SgetDataHSGroup();
    if (m_channel->m_DataCl->MDataValidSavedForOCPTrace != false) {
      monDataHS.MDataValid = m_channel->m_DataCl->MDataValidSavedForOCPTrace;
      if (monDataAccept) {
	// Value has been used. We can now reset it.
	m_channel->m_DataCl->MDataValidSavedForOCPTrace = false;
      }
    }
    monResponse = m_channel->m_DataCl->MgetResponseGroup();
    if (m_channel->m_DataCl->SRespSavedForOCPTrace != OCP_SRESP_NULL) {
      monResponse.SResp = m_channel->m_DataCl->SRespSavedForOCPTrace;
      if (monRespAccept) {
	// Value has been used. We can now reset it.
	  m_channel->m_DataCl->SRespSavedForOCPTrace = OCP_SRESP_NULL;
      }
    }

    //    m_clk_count++;

    // Print a line out
    m_ocpMonPtr-> monitorOut( m_channel->m_ParamCl,
			      //m_clk_count,
			      sc_time_stamp().to_default_time_units(),
			      monRequest,
			      monReqAccept,
			      monDataHS,
			      monDataAccept,
			      monResponse,
			      monRespAccept,
			      m_channel->m_DataCl->MgetLastSThreadBusy(),
			      m_channel->m_DataCl->MgetLastSDataThreadBusy(),
			      m_channel->m_DataCl->SgetLastMThreadBusy(),
			      m_channel->m_DataCl->m_MReset_n,
			      m_channel->m_DataCl->m_SReset_n,
			      m_channel->m_DataCl->m_MError,
			      m_channel->m_DataCl->m_MFlag,
			      m_channel->m_DataCl->m_SError,
			      m_channel->m_DataCl->m_SFlag,
			      m_channel->m_DataCl->m_SInterrupt,
			      m_channel->m_DataCl->m_Control,
			      m_channel->m_DataCl->m_ControlWr,
			      m_channel->m_DataCl->m_ControlBusy,
			      m_channel->m_DataCl->m_Status,
			      m_channel->m_DataCl->m_StatusRd,
			      m_channel->m_DataCl->m_StatusBusy );

    // Now that the trace as been dumped, exit if there were any
    // fatal errors during this OCP cycle
    if (m_channel->m_exitAfterOCPMon)
    {
      cout << name() << ": Exiting due to earlier errors." << endl;
      exit(1000);
    }
  }

  void startup(void)
  {
    m_ocpMonPtr->monitorHeader(m_channel->m_ParamCl);
  }

  void runcheck(void)
  {
    // for future
  }

  void AtClockEdge(void)
  {
    if (m_print_params)
      startup();
    while (true)
    {
      wait();
      OCPClockTick();
    }
  }

  void SelfTimed(void)
  {
    if (m_print_params)
      startup();
    // nothing to do before the simulation starts.
    m_channel->ocpWait();

    while (true)
    {
      OCPClockTick();
      m_channel->ocpWait();
    }
  }
 protected:

  OCP_TL1_Channel<TdataCl> *m_channel;

  OCP_TL1_OCPMonGenCl<TdataCl> *m_ocpMonPtr;

  bool m_print_params;

  double m_clk_count;
};

#endif
