#include "SimControl.h"
#include "IStl.h"
#include "MasterTL2.h"
#include "BigInt.h"
#include "ExpectQueue.h"
#include <algorithm>
#include "boost/bind.hpp"

using namespace std;
using namespace Sonics;
using namespace OcpIp;

// debug messages are sent through this macro, normally disabled
#ifndef NDEBUG
extern bool g_debug;
#define DBG_MSG( msg ) if ( g_debug ) cout << sc_time_stamp() \
        << "(" << sc_get_curr_simcontext()->delta_count()     \
        << "): " << msg << endl; // or something like that
#else
#define DBG_MSG( msg )
#endif

namespace {
// utility class for thread busy access
template <typename Td, typename Ta>
struct SThreadBusyTL2 : public Sonics::ThreadBusyProxy {
    SThreadBusyTL2( OCP_TL2_MasterPort<Td , Ta >& port )
        : m_port ( port ) {}
    virtual ~SThreadBusyTL2() {}
    OCP_TL2_MasterPort<Td, Ta >& m_port;
    virtual sc_event_finder& threadBusyEvent() {
        return m_port.SThreadBusyEvent();
    }
    virtual unsigned int     getThreadBusy  () const {
        if ( !m_port->GetParamCl()->sthreadbusy )
            return 0;
        return m_port->getSThreadBusy();
    }
};

// Type adaptation for 128-bit. STL uses BigInt and SystemC uses sc_biguint
typedef sc_biguint<128> Sc128BitData;
template <typename Td, typename Ta>
struct StlTypes {
    typedef Td Tdata;
    typedef Ta Taddr;
};

template <typename Ta>
struct StlTypes<Sc128BitData, Ta> {
    typedef Sonics::BigInt<128> Tdata;
    typedef Ta                  Taddr;
};

template<typename Tto, typename Tfrom>
struct Assignment {
    static const bool deepCopy = true;
    static void assign( Tto& to, const Tfrom& from );
    static void assign( Tto*& pTo, Tfrom* pFrom, int length );
};

// same types, trivial, do shallow copy
template<typename Tto>
struct Assignment<Tto, Tto> {
    static const bool deepCopy = false;
    static void assign( Tto& to, const Tto& from ) {
        to = from;
    }
    static void assign( Tto*& pTo, Tto* pFrom, int ) {
        pTo = pFrom;
    }
};

// for widths > integral types, data types start to differ
template<>
struct Assignment<Sc128BitData, Sonics::BigInt<128> > {
    static const bool deepCopy = true;
    static void assign( Sc128BitData& to, const Sonics::BigInt<128>& from ) {
        Sonics::BigInt<128> tmp = from;
        snx_uint64_t bothalf = tmp;
        tmp >>= 64;
        snx_uint64_t tophalf = tmp;
        to = tophalf;
        to = to << 64;
        to = to | bothalf;
    }
    static void assign( Sc128BitData*& pTo, Sonics::BigInt<128>* pFrom, int length ) {
        pTo = new Sc128BitData[length]();
        for (int i=0; i<length; ++i )
            assign( pTo[i] , pFrom[i] );
    }
};
}

namespace OcpIp {
//////////////////////////////////////////////////////////////////
// CLASS: MasterTL2DataFlow
// DESCRIPTION: utility class to manage/legalize request/response flow
// on a given master
// ARGUMENTS:
//////////////////////////////////////////////////////////////////
template <typename Td, typename Ta>
class MasterTL2DataFlow {
  public:
    friend class MasterTL2<Td, Ta>;
    typedef OCPTL2RequestGrp <Td,Ta>   RequestGrp;
    typedef OCPTL2ResponseGrp<Td>      ResponseGrp;
    typedef OCP_TL2_MasterIF <Td, Ta>  OcpInterface;

    MasterTL2DataFlow( MasterTL2<Td, Ta>& );
    void configure( const OCPParameters* );
    bool sendRequestBlocking( const RequestGrp& );
    template <typename TdMask, typename TdInfoMask>
    bool sendExpectRequestBlocking(
        const RequestGrp&, const deque<TdMask>&, const deque<TdInfoMask>& );
    snx_uint64_t  getPendingCount() const {
        return m_responsesExpected - m_responsesReceived;
    }
  protected:
    MasterTL2<Td, Ta>&    m_master;
    RequestGrp            m_currentRequest;
    bool prepareRequest();
    bool checkRequest( const RequestGrp& ) const;
    
    // response and expected data
    struct ExpectData {
        Td               data;
        Td               dataMask;
        snx_uint64_t     dataInfo;
        snx_uint64_t     dataInfoMask;
    };
    typedef std::pair<Td, snx_uint64_t> ResponseData;
    typedef Sonics::ExpectQueue<ExpectData, ResponseData> BurstExpect;
    typedef Sonics::ThreadedQueue<BurstExpect> ThreadedBurstQueue;

    ThreadedBurstQueue   m_burstQueue;
    // response counters (for wait response)
    snx_uint64_t m_responsesExpected;
    snx_uint64_t m_responsesReceived;

    template <typename TdMask, typename TdInfoMask>
    void prepareExpect( const RequestGrp&, BurstExpect&,
                        const deque<TdMask>&, const deque<TdInfoMask>& );
    bool checkExpect  ( ExpectData&, const ResponseData& );
    void processResponse();
};

template<typename Td, typename Ta>
MasterTL2DataFlow<Td, Ta>::MasterTL2DataFlow( MasterTL2<Td, Ta>& master ) :
    m_master( master )
{
}

//////////////////////////////////////////////////////////////////
// FUNCTION: MasterTL2DataFlow::configure
// DESCRIPTION: set ocp config, once it is known
// ARGUMENTS: OCPParameters pointer
//////////////////////////////////////////////////////////////////
template<typename Td, typename Ta>
void MasterTL2DataFlow<Td, Ta>::configure( const OCPParameters* pParams )
{
    assert( pParams != NULL );
    m_burstQueue.setNumThreads( pParams->threads );
}

//////////////////////////////////////////////////////////////////
// FUNCTION: MasterTL2DataFlow::prepareRequest
// DESCRIPTION: handle request prep. Type conversion, burst counting
// RETURNS: false if request should be skipped
// SIDE EFFECTS: request fields can be changed (byteen, addr)
//////////////////////////////////////////////////////////////////
template<typename Td, typename Ta>
bool MasterTL2DataFlow<Td, Ta>::prepareRequest()
{
    sc_port<OcpInterface>& ocp = m_master.ipP;
    const OCPParameters* pParams = ocp->GetParamCl();
    unsigned int  thread = m_currentRequest.MThreadID;
    int  byteenWidth     = OcpIp::getByteenWidth( pParams->data_wdth );
    bool isRead          = OcpIp::isRead( m_currentRequest.MCmd );

    // Byte Enable array if needed
    bool hasByteEn = pParams->byteen || ( !isRead && pParams->mdatabyteen );
    if ( !hasByteEn ) {
        if ( m_currentRequest.MByteEnPtr != NULL ) {
            for ( unsigned int i=0; i < m_currentRequest.DataLength; ++i ) {
                OcpIp::fillBitRange( m_currentRequest.MByteEnPtr[i],
                                     byteenWidth-1, 0 );
            }
        } else {
            OcpIp::fillBitRange( m_currentRequest.MByteEn, byteenWidth-1, 0 );
        }
    }

    // Burst tracking
    OcpIp::BurstCounter&      burstCounter  = m_master.m_burstCounter[thread];
    OcpIp::BurstSequence<Ta>& burstSequence = *(m_master.m_burstSequence[thread]);
    burstCounter.next( m_currentRequest );
    bool firstInBurst = ( burstCounter.count() == m_currentRequest.DataLength );
    m_currentRequest.LastOfBurst = burstCounter.last();
    if ( m_currentRequest.MBurstSingleReq ) {
        if ( isRead ) {
            if ( firstInBurst ) {
                if ( m_currentRequest.MBlockHeight < 1 )
                    m_currentRequest.MBlockHeight = 1;
                m_currentRequest.DataLength = m_currentRequest.MBurstLength
                    * m_currentRequest.MBlockHeight;
                m_currentRequest.LastOfBurst = true;
            } else {
                return false; // yep. Ignore subsequent SRMD read commands
            }
        } else if ( m_currentRequest.MBurstSeq != OCP_MBURSTSEQ_DFLT1 &&
                    m_currentRequest.MBurstSeq != OCP_MBURSTSEQ_DFLT2 &&
                    m_currentRequest.MBurstSeq != OCP_MBURSTSEQ_UNKN ) {                    
            // compute address of SRMD write burst
            if ( firstInBurst ) {
                burstSequence.init( m_currentRequest.MAddr,
                                    m_currentRequest.MBurstSeq,
                                    m_currentRequest.MBurstLength,
                                    m_currentRequest.MBurstPrecise,
                                    m_currentRequest.MBlockHeight,
                                    m_currentRequest.MBlockStride );
            }
            m_currentRequest.MAddr = burstSequence.next();
            for ( unsigned int i=1; i < m_currentRequest.DataLength; ++i )
                burstSequence.next(); // throw away
        }
    }
    return checkRequest( m_currentRequest );
}

//////////////////////////////////////////////////////////////////
// FUNCTION: MasterTL2DataFlow::checkRequest
// DESCRIPTION: Checks the request for validity with the config
// ARGUMENTS: RequestGrp
// RETURNS: true if valid, false if not
//////////////////////////////////////////////////////////////////
template<typename Td, typename Ta>
bool
MasterTL2DataFlow<Td, Ta>::checkRequest( const RequestGrp& request ) const
{
    const OCPParameters* pParams = m_master.ipP->GetParamCl();
    bool burst = request.MBurstLength > 1;
    bool hasErrors = false;
    // We have no list of checks that make sense here vs. the STL.
    // The entire issue of checking stimulus is still open. This is for PR 16336
    // until some better solution is in place
    if ( burst && request.MBurstSingleReq && !pParams->datahandshake ) {
//         report << QMASTER_INVALID_BURST << m_reader.getFileName()
//                << m_reader.getLineNum() << "datahandshake is disabled" << endm;
        hasErrors = true;
    }
    return !hasErrors;
}

//////////////////////////////////////////////////////////////////
// FUNCTION: MasterTL2DataFlow::sendRequestBlocking
// DESCRIPTION: arbitrates then sends request in blocking fashion
// ARGUMENTS: TL2 request (after preparation)
// RETURNS: Nothing
//////////////////////////////////////////////////////////////////
template<typename Td, typename Ta>
template<typename TdMask, typename TdInfoMask>
bool MasterTL2DataFlow<Td, Ta>::sendExpectRequestBlocking(
    const RequestGrp& request, const deque<TdMask>& dataMask,
    const deque<TdInfoMask>& dataInfoMask )
{
    m_currentRequest = request;
    if ( !prepareRequest() )
        return false;

    sc_port<OcpInterface>& ocp = m_master.ipP;
    const OCPParameters* pParams = ocp->GetParamCl();
    unsigned int thread = m_currentRequest.MThreadID;
    bool isRead   = OcpIp::isRead( m_currentRequest.MCmd );
    bool needsResponse = pParams->writeresp_enable || isRead;
    OcpIp::BurstCounter& burstCounter = m_master.m_burstCounter[thread];
    // <= because SRMD can have count=1 and a greater DataLength
    bool newBurst = burstCounter.count() <= m_currentRequest.DataLength;

    // get the thread semaphore
    // TODO: name the stream for DBG_MSG
    DBG_MSG( " stream waiting for semaphore on thread " << dec << thread );
    m_master.m_threadSemaphores[thread]->wait();

    // wait for arbitration (includes threadbusy assessment)
    DBG_MSG( " stream waiting for arbitration on thread " << dec << thread );
    m_master.m_threadArbiter.threadReady( thread );
    wait( m_master.m_threadArbiter.threadWinner( thread ) );

    // Fire request!
    DBG_MSG( " stream sending " << hex << m_currentRequest << dec );
    if ( !isRead )
        DBG_MSG( " data=" << hex << m_currentRequest.MDataPtr[0] << dec );

    if ( needsResponse ) {
        unsigned int responsesExpected = m_currentRequest.DataLength;
        if ( !isRead && m_currentRequest.MBurstSingleReq ) {
            responsesExpected = burstCounter.last() ? 1 : 0;
        }
        m_responsesExpected += responsesExpected;
        if ( newBurst ) {
            m_master.m_responseStream.enqueue( thread, this );
            m_burstQueue.enqueue( thread, BurstExpect() );
        }
        if (responsesExpected != 0) {
            prepareExpect( m_currentRequest, m_burstQueue.lastEnqueued( thread ),
                           dataMask, dataInfoMask );
        }
    }

    bool myretval = ocp->sendOCPRequestBlocking( m_currentRequest );
    assert(myretval == true );

    // release semaphore and wait to give other threads the chance
    m_master.m_threadSemaphores[thread]->post();
    return needsResponse;
}

template<typename Td, typename Ta>
bool MasterTL2DataFlow<Td, Ta>::sendRequestBlocking( const RequestGrp& request )
{
    return sendExpectRequestBlocking( request, deque<Td>(), deque<snx_uint64_t>() );
}

//////////////////////////////////////////////////////////////////
// FUNCTION: MasterTL2DataFlow::checkExpect
// DESCRIPTION: verifies that response matches expected data
// ARGUMENTS: ExpectData struct, Response
// RETURNS: yes or no
//////////////////////////////////////////////////////////////////
template<typename Td, typename Ta>
bool MasterTL2DataFlow<Td, Ta>::checkExpect(
    ExpectData& expect, const ResponseData& response )
{
    DBG_MSG( "Checking expected response" );

    if ( expect.dataMask != 0 ) {
        const Td& data = response.first;
        if ( ( data & expect.dataMask ) !=
             ( expect.data & expect.dataMask ) ) {
//             report << STL_UNEXPECTED_DATA << hex
//                    << data << expect.data << expect.dataMask << dec << endm;
        }
    }
    if ( expect.dataInfoMask != 0 ) {
        const snx_uint64_t& dataInfo  = response.second;
        if ( ( dataInfo & expect.dataInfoMask ) !=
             ( expect.dataInfo & expect.dataInfoMask ) ) {
//             report << STL_UNEXPECTED_DATAINFO << hex
//                    << dataInfo << expect.dataInfo
//                    << expect.dataInfoMask << dec << endm;
        }
    }
    return true;
}

//////////////////////////////////////////////////////////////////
// FUNCTION: MasterTL2DataFlow::prepareExpect
// DESCRIPTION: handle expect data extraction from transfer
// ARGUMENTS: STL transfer expect masks, Burst Expect Queue to append to
// RETURNS: Nothing
//////////////////////////////////////////////////////////////////
template<typename Td, typename Ta>
template<typename TdMask, typename TdInfoMask>
void MasterTL2DataFlow<Td, Ta>::prepareExpect(
    const RequestGrp& request, BurstExpect& burstExpect,
    const deque<TdMask>& expDataMask, const deque<TdInfoMask>& expDataInfoMask )
{
    typename deque<Td>::const_iterator  dataMaskIt     = expDataMask.begin();
    deque<snx_uint64_t>::const_iterator dataInfoMaskIt = expDataInfoMask.begin();
    bool isWrite = OcpIp::isWrite( request.MCmd );
    for ( unsigned int i = 0; i < request.DataLength; ++i ) {        
        ExpectData check;
        check.dataMask = 0;
        check.dataInfoMask = 0;
        if ( !isWrite ) {
            if ( dataMaskIt != expDataMask.end() )
                Assignment<Td, TdMask>::assign( check.dataMask, *(dataMaskIt++) );
            if ( dataInfoMaskIt != expDataInfoMask.end() )
                Assignment<snx_uint64_t, TdInfoMask>::assign( check.dataInfoMask,
                                                              *(dataInfoMaskIt++) );
            if ( i > 0 ) {
                // TL2 response grp has only one datainfo...
                check.dataInfoMask = 0;
            }
            if ( check.dataMask != 0 ) {
                assert( request.MDataPtr != NULL );
                check.data = request.MDataPtr[i];
            }

            if ( check.dataInfoMask != 0 ) {
                assert( request.MDataInfoPtr != NULL );
                check.dataInfo = request.MDataInfoPtr[i];
            }
        }
        DBG_MSG( "Expect Data MAddr=" << hex << request.MAddr << " i=" << i );
        burstExpect.expect( check, boost::bind( &MasterTL2DataFlow::checkExpect,
                                                this, _1, _2 ),
                            request.LastOfBurst );
        if ( isWrite && request.MBurstSingleReq )
            break; // only one response
    }
}

//////////////////////////////////////////////////////////////////
// FUNCTION: MasterTL2DataFlow::processResponse
// DESCRIPTION: Called back by master's response thread when response
// arrives for this stream. This method handles the required state
// for wait response
//////////////////////////////////////////////////////////////////
template<typename Td, typename Ta>
void
MasterTL2DataFlow<Td, Ta>::processResponse()
{
    ResponseGrp& response = m_master.m_currentResponse;
    DBG_MSG( "Processing " << hex << response << dec );
    unsigned int thread = response.SThreadID;

    for ( unsigned int i = 0; i < response.DataLength; ++i ) {
        ResponseData respData( ( response.SDataPtr != NULL ) ?
                               response.SDataPtr[i] : 0,
                               response.SDataInfo );
        m_burstQueue.peek( thread ).match( respData );
    }
    if ( m_burstQueue.peek( thread ).finished() ) {
        m_burstQueue.dequeue( thread );
    } else if ( response.LastOfBurst ) {
//         m_master.report << QMASTER_UNEXPECTED_RESPLAST_ON_THREAD << thread
//                         << m_master.endm;
    }

    m_responsesReceived += response.DataLength;
}   

//////////////////////////////////////////////////////////////////
// CLASS: MasterTL2InStream
// DESCRIPTION: utility class to play 1 stream on a given master
// ARGUMENTS:
//////////////////////////////////////////////////////////////////
template <typename Td, typename Ta>
class MasterTL2InStream : public sc_module {
  public:
    MasterTL2InStream( sc_module_name, MasterTL2<Td, Ta>& );
    virtual ~MasterTL2InStream() {}

  protected:
    MasterTL2<Td, Ta>&          m_master;
    sc_port<OCP_TL2_MasterIF<Td,Ta> >& ocp;
    MasterTL2DataFlow<Td, Ta>   m_dataFlow;
    SC_HAS_PROCESS( MasterTL2InStream );
    void mainThread();
    virtual void mainThreadGuts() {}
    virtual void end_of_elaboration();
};

template<typename Td, typename Ta>
MasterTL2InStream<Td, Ta>::MasterTL2InStream( sc_module_name name,
                                              MasterTL2<Td, Ta>& master ) :
    sc_module ( name ),
    m_master  ( master ),
    ocp       ( m_master.ipP ),
    m_dataFlow( m_master )
{
    SC_THREAD( mainThread );
}

template<typename Td, typename Ta>
void
MasterTL2InStream<Td, Ta>::mainThread()
{
    mainThreadGuts();
}

//////////////////////////////////////////////////////////////////
// FUNCTION: MasterTL2InStream::end_of_elaboration()
//////////////////////////////////////////////////////////////////
template<typename Td, typename Ta>
void
MasterTL2InStream<Td, Ta>::end_of_elaboration()
{
    m_dataFlow.configure( m_master.ipP->GetParamCl() );
}

//////////////////////////////////////////////////////////////////
// CLASS: MasterTL2TestStream
// DESCRIPTION: utility class to play 1 stream from a request fifo
// ARGUMENTS:
//////////////////////////////////////////////////////////////////
template <typename Td, typename Ta>
class MasterTL2TestStream : public MasterTL2InStream<Td, Ta> {
public:
    typedef MasterTL2DataFlow<Td, Ta> DataFlow;

    MasterTL2TestStream( sc_module_name name , MasterTL2<Td, Ta>& master ) :
        MasterTL2InStream<Td,Ta>( name, master ),
        m_testPort( "testPort" )
        {}
    virtual ~MasterTL2TestStream() {}
    sc_fifo_in<typename DataFlow::RequestGrp> m_testPort;

  protected:
    virtual void mainThreadGuts();
};

template<typename Td, typename Ta>
void
MasterTL2TestStream<Td, Ta>::mainThreadGuts()
{
    typename DataFlow::RequestGrp request;
    while ( true ) {
        request = m_testPort->read();
        this->m_dataFlow.sendRequestBlocking( request );
    }
}

//////////////////////////////////////////////////////////////////
// CLASS: MasterTL2StlStream
// DESCRIPTION: utility class to play 1 STL stream on given master
// ARGUMENTS:
//////////////////////////////////////////////////////////////////
template <typename Td, typename Ta>
class MasterTL2StlStream : public MasterTL2InStream<Td,Ta> {
  public:
    typedef OCPTL2RequestGrp <Td,Ta>   RequestGrp;
    typedef OCPTL2ResponseGrp<Td>      ResponseGrp;
    typedef std::pair<Td, snx_uint64_t> ResponseData;
    typedef MasterTL2InStream<Td,Ta>   Base;

    MasterTL2StlStream( sc_module_name, MasterTL2<Td, Ta>&, IStlReader* );
    
    virtual void mainThreadGuts();
    bool checkWaiters( const StlWaitCommand& );    
    template <typename StlTd, typename StlTa>
    void sendTransferBlocking( const StlTransferCommand<StlTd, StlTa>& );
  
  private:  
    IStlReader&  m_reader;
    RequestGrp   m_currentRequest;
};
}

template<typename Td, typename Ta>
MasterTL2StlStream<Td, Ta>::MasterTL2StlStream( sc_module_name name,
                                                MasterTL2<Td, Ta>& master,
                                                IStlReader* pReader ) :
    MasterTL2InStream<Td,Ta>( name, master ),
    m_reader( *pReader )
{
}

//////////////////////////////////////////////////////////////////
// FUNCTION: mainThreadGuts
// DESCRIPTION: the main process of an STL-player
// ARGUMENTS: None
//////////////////////////////////////////////////////////////////
template<typename Td, typename Ta>
void
MasterTL2StlStream<Td, Ta>::mainThreadGuts()
{
    //SimDriver _driver; // when all SimDrivers go out of scope simulation ends

    typedef typename StlTypes<Td, Ta>::Tdata StlTd;
    typedef typename StlTypes<Td, Ta>::Taddr StlTa;
    typedef StlTransferCommand<StlTd, StlTa> StlTransfer;
    MasterTL2<Td, Ta>&                 m_master   = this->m_master;
    sc_port<OCP_TL2_MasterIF<Td,Ta> >& ocp        = this->ocp;
    MasterTL2DataFlow<Td, Ta>          m_dataFlow = this->m_dataFlow;

    bool stleof;
    bool tbflags;
    bool timeStampEnable = m_master.m_config.timestamp_enable;
    SignalName signal;
    snx_uint64_t value, mask( 0 );
    const sc_time clkPeriod = ocp->getPeriod();
    assert( clkPeriod > SC_ZERO_TIME );

    m_reader.IStlReader::setConversionTypes<StlTd, StlTa>();
    StlCommandType cType; 

    while (true) {
        // get next stl line from file and check for end of file
        m_reader.next( stleof );
        if ( stleof )
            break;
        cType = m_reader.getCommandType();
        if ( cType == STL_INVALID ) continue;

        // wait for time stamp
        if ( timeStampEnable ) {
            double period = clkPeriod.value();
            double timenow = sc_time_stamp().value();
            double delay = m_reader.getCommandDelay() * period - timenow;
            delay /= period;
            if( delay > 0 )
                wait( delay * clkPeriod );
        }

        switch ( cType ) 
        {            
        case STL_WAIT:
            while( !checkWaiters( m_reader.getWaitCommand() ) )
                wait( clkPeriod );  // until fulfilled. This could be smarter...
            break;            
        case STL_SIGNAL:
            tbflags = false;
            signal = m_reader.getSignalCommand().getSignal( tbflags );

            if ( signal == S_MFlag || tbflags ) {
                value = m_reader.getSignalCommand().drivenValue( mask );
            } else {
                value = m_reader.getSignalCommand().drivenValue(); 
            }
            switch ( signal ) {
            case S_MFlag : m_master.setMFlag( value, mask ); break;
            case S_MError: ocp->MputMError( value ); break;
            default: ;
            }
            if ( tbflags )
                m_master.m_tbflags = ( m_master.m_tbflags & ~mask ) |
                    ( value & mask );

            break;
        case STL_TRANSFER:
        {
            StlTransfer transfer =
                m_reader.IStlReader::getTransferCommand<StlTd,StlTa>();
            if ( transfer.request.MCmd == OCP_MCMD_IDLE )
                wait(transfer.idleCycles * clkPeriod);
            else
                sendTransferBlocking( transfer );
        }
        break;
        case STL_RESET:
        {
            StlResetCommand resetCmd = m_reader.getResetCommand();
            ocp->MResetAssert();
            wait( clkPeriod * resetCmd.resetCycles );
            wait( SC_ZERO_TIME );
            ocp->MResetDeassert();
        }
        break;
        default:
            assert( false );
        }
    }    
    while ( m_dataFlow.getPendingCount() > 0 ) {
        wait( ocp->ResponseEndEvent() );
    }
    wait ( clkPeriod * 100 ); // extra slop. TODO, add user control to this
    DBG_MSG( m_reader.getFileName() << " stream terminated." );
}

//////////////////////////////////////////////////////////////////
// FUNCTION: MasterTL2StlStream::checkWaiters()
// DESCRIPTION: implement a STL wait command
// ARGUMENTS: STL wait command
// RETURNS: true or false
//////////////////////////////////////////////////////////////////
template<typename Td, typename Ta>
bool MasterTL2StlStream<Td, Ta>::checkWaiters( const StlWaitCommand& command )
{
    MasterTL2<Td, Ta>&                 m_master   = this->m_master;
    sc_port<OCP_TL2_MasterIF<Td,Ta> >& ocp        = this->ocp;
    MasterTL2DataFlow<Td, Ta>          m_dataFlow = this->m_dataFlow;

    deque<StlWaitCommand::Waiter> waiters ; 
    deque<StlWaitCommand::Waiter>::const_iterator it;
    StlWaitCommand::Waiter waiter ;
    waiters = command.waiters;
    assert( !waiters.empty() );
    for ( it = waiters.begin(); it != waiters.end(); ++it ) {
        bool fulfilled = false;
        waiter = *it;
        if ( waiter.m_response ) 
        {
            uint32_t pendingCount = m_dataFlow.getPendingCount();
            fulfilled = ( pendingCount < waiter.m_value );
        } else if ( waiter.m_tbflags ) {
            // temporary
            fulfilled = ( ( waiter.m_mask & m_master.m_tbflags ) ==
                          ( waiter.m_mask & waiter.m_value ) );    
        } else {
            switch ( waiter.m_signal ) {
            case S_SError:
                fulfilled = ( ocp->MgetSError() == waiter.m_value ); break;
            case S_SInterrupt:
                fulfilled = ( ocp->MgetSInterrupt() == waiter.m_value ); break;
            case S_SFlag:
                fulfilled = ( ( waiter.m_mask & ocp->MgetSFlag() ) == 
                              ( waiter.m_mask & waiter.m_value ) ); break;                              
            case S_SThreadBusy:
                fulfilled = ( ( waiter.m_mask & ocp->getSThreadBusy() ) == 
                              ( waiter.m_mask & waiter.m_value ) ); break;  
            default: ;
            }
        }
        if ( fulfilled && command.isOr() ){return true; }
        if ( !fulfilled && command.isAnd() ) { return false;} 
    }
    return command.isAnd(); // if it's or we should have exited when one met
} 

//////////////////////////////////////////////////////////////////
// FUNCTION: MasterTL2StlStream::sendTransferBlocking
// DESCRIPTION: sends transfer in blocking fashion
// ARGUMENTS: STL transfer
// RETURNS: Nothing
//////////////////////////////////////////////////////////////////
template<typename Td, typename Ta>
template<typename StlTd, typename StlTa>
void MasterTL2StlStream<Td, Ta>::sendTransferBlocking(
    const StlTransferCommand<StlTd, StlTa>&  transfer )
{    
    assert( sizeof( RequestGrp ) == sizeof( transfer.request ) );
    m_currentRequest = reinterpret_cast<const RequestGrp&>( transfer.request );
    Assignment<Td, StlTd>::assign( m_currentRequest.MDataPtr, transfer.request.MDataPtr,
                                   transfer.request.DataLength );

    bool needsResponse = this->m_dataFlow.sendExpectRequestBlocking(
        m_currentRequest, transfer.dataMask, transfer.dataInfoMask );
    wait( SC_ZERO_TIME );

    if ( !needsResponse && Assignment<Td, StlTd>::deepCopy ) {
        // if there was a deep copy, delete it now
        delete[] m_currentRequest.MDataPtr;
    }
}


#undef  DBG_STREAM
#define DBG_STREAM report
//////////////////////////////////////////////////////////////////
// FUNCTION: MasterTL2::constructor / destructor
// DESCRIPTION: Setup the data members, find STL streams to play
// ARGUMENTS: name and configuration core pointer
//////////////////////////////////////////////////////////////////
template<typename Td, typename Ta>
MasterTL2<Td, Ta>::MasterTL2( sc_module_name name, int testStreams ) :
    sc_module           ( name ),
    ipP                 ( "ip" ),
    m_curMFlag          ( 0 ),
    m_pThreadBusyAccess ( new SThreadBusyTL2<Td, Ta>( ipP ) ),
    m_threadArbiter     ( "threadbusy_arbiter", m_pThreadBusyAccess ),
    m_numTestStreams    ( testStreams ),
    m_lastTestStreamPort( 0 )
{
    SC_METHOD( responseAcceptMethod );
    sensitive << m_responseAcceptEvent;
    dont_initialize();
    
    SC_THREAD( responseThread );

    for ( int i=0; i < m_numTestStreams; ++i ) {
      char streamName[40];
      sprintf( streamName, "testStream_%d", i );
      MasterTL2InStream<Td, Ta>* pStream = new MasterTL2TestStream<Td, Ta>(
          streamName, *this );
      m_inStreams.push_back( pStream );
    }
}

template<typename Td, typename Ta>
MasterTL2<Td, Ta>::~MasterTL2()
{
    for ( int t=0; t<m_threads; ++t ) {
        delete m_threadSemaphores[t];
        delete m_burstSequence[t];
    }
    for ( typename vector<MasterTL2InStream<Td, Ta>*>::iterator it =
              m_inStreams.begin(); it != m_inStreams.end(); ++ it ) {
        delete *it;
    }
    delete m_pThreadBusyAccess;
}

//////////////////////////////////////////////////////////////////
// FUNCTION: MasterTL2::getTestStreamPort
// DESCRIPTION: when creating test streams in the constructor, these
// must be bound to a fifo channel
// ARGUMENTS: optional number of the stream port. If not given each
// invocation will return the next stream port
// RETURNS: a fifo input port to bind.
//////////////////////////////////////////////////////////////////
template<typename Td, typename Ta>
sc_fifo_in<typename MasterTL2<Td,Ta>::RequestGrp>*
MasterTL2<Td, Ta>::getTestStreamPort( int num )
{
    if ( num < 0 )
        num = m_lastTestStreamPort++;
    if ( num < m_numTestStreams ) {
        MasterTL2InStream<Td,Ta>* pStream = m_inStreams[num];
        assert( pStream != NULL );
        MasterTL2TestStream<Td,Ta>* pTStream =
            dynamic_cast<MasterTL2TestStream<Td,Ta>*>( pStream );
        if ( pStream != NULL )
            return &(pTStream->m_testPort);
    }
    return NULL;
}

//////////////////////////////////////////////////////////////////
// FUNCTION: MasterTL2::before_end_of_elaboration()
// DESCRIPTION: set data structures that need knowledge of the OCP
// configuration, now that the ipP port is bound
//////////////////////////////////////////////////////////////////
template<typename Td, typename Ta>
void MasterTL2<Td, Ta>::before_end_of_elaboration()
{
    const OCPParameters* pParams = ipP->GetParamCl();
    sc_module::end_of_elaboration();

    assert ( !pParams->burstseq || pParams->burstlength );
    m_threads = pParams->threads;    
    m_threadArbiter .setNumThreads( m_threads ); 
    m_responseStream.setNumThreads( m_threads );
    
    // Set up threaded data structures
    m_burstCounter.resize( m_threads );
    m_burstSequence.resize( m_threads, NULL );
    m_threadSemaphores.resize( m_threads, NULL );
    for (int t=0; t<m_threads; ++t) {
        m_burstSequence[t] = new OcpIp::BurstSequence<Ta>(
            pParams->data_wdth, pParams->addr_wdth, 0, OCP_MBURSTSEQ_INCR );
        m_threadSemaphores[t] = new sc_semaphore( 1 );
    }

    // Latencies
    MTimingGrp timing;
    timing.RqDL   = m_config.req2data_latency;
    timing.RqSndI = m_config.req_interval;
    timing.DSndI  = m_config.req_interval; // not explicit
    timing.RpAL   = m_config.respaccept_latency;
    ipP->putMasterTiming( timing );

    // Find STL programs
    deque<IStlReader*> stlReaders = IStlReader::findStlPrograms(
        basename(), false, *(ipP->GetParamCl()) );
    if ( stlReaders.empty() ) {
        //report << NO_STL_PROGRAM_FOUND << endm;
    } else {
        for ( deque<IStlReader*>::iterator stlIt = stlReaders.begin();
              stlIt != stlReaders.end(); ++stlIt ) {
            IStlReader* pReader = *stlIt;
            string legalName = pReader->getFileName();
            assert ( !legalName.empty() );
            for ( string::iterator it = legalName.begin(); it != legalName.end(); ++it ) {
                if ( *it == '.' ) *it = '_';
            }
            MasterTL2InStream<Td, Ta>* pStream = new MasterTL2StlStream<Td, Ta>(
                legalName.c_str(), *this, pReader );
            m_inStreams.push_back( pStream );
        }
    }
}

//////////////////////////////////////////////////////////////////
// FUNCTION: MasterTL2::responseThread
// DESCRIPTION: handle response accept and mthreadbusy
//////////////////////////////////////////////////////////////////
template<typename Td, typename Ta>
void MasterTL2<Td, Ta>::responseThread()
{
    const sc_time clkPeriod   = ipP->getPeriod();
    bool          respaccept  = ipP->GetParamCl()->respaccept;
    const int     latency     = respaccept ? m_config.respaccept_latency : 0;
    bool          mthreadbusy = ipP->GetParamCl()->mthreadbusy;
    const int     busyCycles  = m_config.mthreadbusy_cycles;

    while ( true ) {
        wait( ipP->ResponseStartEvent() );
        bool hasResponse = ipP->getOCPResponse( m_currentResponse );
        assert( hasResponse );
        unsigned int thread = m_currentResponse.SThreadID;
        if ( m_responseStream.size( thread ) == 0 ) {
            // report << QMASTER_UNEXPECTED_RESP_ON_THREAD << thread << endm;
        } else {
            MasterTL2DataFlow<Td, Ta>* pStream = m_responseStream.peek( thread );
            pStream->processResponse();
        }

        // accept response
        m_responseAcceptEvent.notify( ( latency < 1 ) ?
                                      SC_ZERO_TIME : latency * clkPeriod );

        if ( mthreadbusy && busyCycles > 0 ) {
            // MThreadBusy pulse
            ipP->putMThreadBusyBit( true, thread );
            wait ( busyCycles * clkPeriod );
            ipP->putMThreadBusyBit( false, thread );
        }
    }
}

template<typename Td, typename Ta>
void MasterTL2<Td, Ta>::responseAcceptMethod()
{
    DBG_MSG( "Accepted Response" );
    ipP->putMRespAccept();
}

template<typename Td, typename Ta>
void MasterTL2<Td, Ta>::setMFlag( snx_uint64_t value, snx_uint64_t mask )
{
    m_curMFlag = ( m_curMFlag & ~mask ) | ( value & mask );
    ipP->MputMFlag( m_curMFlag );
}
