///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// Copyright 2004 OCP-IP
// OCP-IP Confidential & Proprietary
//
//
//============================================================================
//      Project : OCP SLD WG
//       Author : Tim Kogel, CoWare, Inc.
//          Date: 11/26/2004
//
//  Description :  OCP Channel Transaction Recording Monitor
//	  This channel monitor is based on the transaction recording 
//	  API in the SystemC Verification (SCV) Library. It targets 
//	  performance analysis for architectural modeling.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef _OCP_TL2_PERF_MONITOR_H
#define _OCP_TL2_PERF_MONITOR_H

#include <systemc.h>

#include "ocp_perf_monitor_channel_scv.h"
#include "ocp_perf_monitor_system_registry.h"
#include "ocp_tl2_monitor_port.h"

/** This SystemC module represents the OCP channel monitor. For every
    OCP channel, which is shall be monitored, one
    OCP_Perf_Monitor_Channel has to be instantiated. The only port of
    this monitor has to be bound to the master interface of the OCP
    channel.
    
    The constructor arguments configure the transaction recording:
    - channel_recording : enables the recording of transaction on the
    connected OCP channel
    - system_recording: enables the system level recording of the
    transactions on the connected OCP channel
    - master_is_node: configures the system monitor, should be true if
    the module connected at the master side of the OCP channel is a
    communivation node
    - slave_is_node: configures the system monitor, should be true if
    the module connected at the slave side of the OCP channel is a
    communivation node
    - max_nbr_of_threads: the channel monitor separatly records the
    transactions from different threads. Make sure the MThreadID
    member is alaways smaller than max_nbr_of_threads!
    - burst_recording: enables separate recording of OCP busrts and
    TL2 chunks by evaluating the LastOfBurst member
    - attribute_recording: dumps the value of all fiels in the request
    and response data structures into the SCV database. Which values
    are dumped can be controlled by the partial template
    specialization of the scv_extension class in
    ocp_perf_monitor_extensions.h.

    During the end_of_elaboration() method the monitor is registerd
    to the connected OCP channel.

     Design Rules to obtain meangingful results:
    ===========================================
    - a transaction (both request and response) starts at the
    sendOCPRequest/sendOCPResponse and end with
    acceptRequest/acceptResponse. Hence immidiate accept results in
    zero-delay transactions and will confuse any post-processing tool
    - the system monitor relies on the TrHandle member to be
    maintained throughout the complete transaction, i.e.
      1. bus nodes must forwarding the TrHandle member
      2. slave modules must set the TrHandle member in the response
      3. slaves shall not send a response in case of OCP_MCMD_WR or
      OCP_MCMD_BCST
*/

template <class Tdata, class Taddr>
class OCP_TL2_Perf_Monitor :
  public sc_module
{
public:
  SC_HAS_PROCESS(OCP_TL2_Perf_Monitor);

  /** this port shall be bound to the monitor interface of the OCP channel */
  OCP_TL2_MonitorPort<Tdata,Taddr> p_mon;
  
  OCP_TL2_Perf_Monitor(sc_module_name name,
		       bool channel_recording  = true,
		       bool system_recording   = true,
		       bool master_is_node     = false,
		       bool slave_is_node      = false,
		       unsigned int  max_nbr_of_threads = 0,
		       bool burst_recording    = false,
		       bool attribute_recording= false) :
    sc_module(name),
    p_mon("p_mon"),
    m_channel_monitor(NULL),
    m_system_monitor(NULL),
    m_channel_recording(channel_recording),
    m_system_recording(system_recording),
    m_master_is_node(master_is_node),
    m_slave_is_node(slave_is_node)
  {
    m_channel_monitor = new OCP_Perf_Monitor_Channel_SCV<Tdata,Taddr>(name,max_nbr_of_threads,burst_recording,attribute_recording);

    if (m_system_recording) {
      m_system_monitor = OCP_Perf_Monitor_System_Registry<Tdata,Taddr>::instance();
      assert(m_system_monitor);
    }

  }
  
  virtual ~OCP_TL2_Perf_Monitor()
  {
    delete m_channel_monitor;
    OCP_Perf_Monitor_System_Registry<Tdata,Taddr>::delete_system_monitor();
  }

  void end_of_elaboration()
  {
    if (m_channel_recording) {
      p_mon->RegisterRequestStart(m_channel_monitor);
      p_mon->RegisterRequestEnd(m_channel_monitor);
      p_mon->RegisterResponseStart(m_channel_monitor);
      p_mon->RegisterResponseEnd(m_channel_monitor);
    }

    if (m_system_recording) {
      p_mon->RegisterRequestStart(m_system_monitor);
      p_mon->RegisterRequestEnd(m_system_monitor);
      p_mon->RegisterResponseStart(m_system_monitor);
      p_mon->RegisterResponseEnd(m_system_monitor);
    }

    m_channel_monitor->registerChannel(p_mon[0]);
    if (m_system_recording) {
      m_system_monitor->registerChannel(p_mon[0],
					m_master_is_node, 
					m_slave_is_node);
    }
  }

  void start_of_simulation()
  {
    if (m_system_recording) {
      m_system_monitor->start_of_simulation();
    }
  }

protected:
  OCP_TL2_Monitor_ObserverIF<Tdata,Taddr>* m_channel_monitor;
  OCP_TL2_Monitor_ObserverIF<Tdata,Taddr>* m_system_monitor;

  const bool m_channel_recording;
  const bool m_system_recording;
  const bool m_master_is_node;
  const bool m_slave_is_node;
};

#endif
