// 
//  Copyright 2003 OCP-IP
//
// ============================================================================
//      Project : OCP SLD WG, Layer adapter examples 
//      Authors : Yann Bajot, PROSILOG, bajot@prosilog.com 
//                Stphane Guntz, PROSILOG, guntz@prosilog.com
//         Date : 06/2003
//
//  Description : Transaction Level - Layer-1 to Layer-2 Slave Adapter

//    
//  Parameter    :
//    Template arguments.
//     - TdataCl_tl1: TL1 Data class containing data members and
//                access methods for the data exchanged between
//                Masters and Slaves
//	   - TdataCl_tl2: TL2 Data class containing data members and
//                access methods for the data exchanged between
//                Masters and Slaves
//     - Td: Data type 
//     - Ta: Address type
//                
//	Constructor arguments:
//      - sc_module_name: Name of the module instance
//	    - max_burst_length: Maximum size of burst length
//		- adapter_depth: Number of missed events and responses to be cached.
//						 The adapter stores new request/response infos which have arrived during the processing of the current request/response.
//						 It sends them when the current request/response is finished.
//		- writeResponse: boolean indicating if the slave connected to the adapter sends repsonses to write requests. In that case, the adapter has to send these responses to the master
//	 
//  The TL1 slave adapter has a TL2 slave interface on one side, and a TL1 master interface on the other side.
//  It must be connected to:
//      - a TL2 master through a TL2 channel
//      - a TL1 slave through a TL1 channel
//
//	The adapter retrieves TL2 requests and stores them in its internal buffer. It sends corresponding TL1 requests to the TL1 slave.
//	  
//  On the response side, the TL1 slave adapter retrieves the TL1 responses from the slave, packs them in burst and sends corresponding
//  TL2 responses to the master. 
//
// ============================================================================

#ifndef _OCP_TL1_TL2_SLAVE_ADAPTER_H
#define _OCP_TL1_TL2_SLAVE_ADAPTER_H

#include "ocp_tl1_globals.h"
#include "ocp_tl2_globals.h"
#include "tl_slave_if.h"
#include "tl_master_if.h"
#include "ocp_tl1_data_cl.h"
#include "ocp_tl2_data_cl.h"

//for debugging purposes, print messages on screen
//comment for no messages
//#define DEBUG_SLAVE_ADAPTER_TL1

#ifndef ADAPTER_TASK_ENUM
#define ADAPTER_TASK_ENUM

enum AdapterTask   {
	SEND_BURST_WITH_DATA = 0,
	SEND_BURST_WITHOUT_DATA,
	STORE_NEW_DATA
};

#endif

template <class TdataCl_tl1, class TdataCl_tl2> class OCP_TL1_TL2_Slave_Adapter
  : public sc_module
   , public MdirectIF<TdataCl_tl2>
   , public SdirectIF<TdataCl_tl1>
{
public:  
  //TL1 data type and address type
  typedef typename TdataCl_tl1::DataType Td_tl1;
  typedef typename TdataCl_tl1::AddrType Ta_tl1;

  //TL2 data type and address type
   typedef typename TdataCl_tl2::DataType Td_tl2;
  typedef typename TdataCl_tl2::AddrType Ta_tl2;
  

  //internal structure for data and request fields
  struct response_parameters  {
	OCPSRespType	m_SResp;
	int			    m_SThreadID;
  };

  struct request_parameters  {
	Ta_tl2			m_MAddr;
	int				m_MAddrSpace;
	OCPMBurstType	m_MBurst;
	int				m_MByteEn;
	OCPMCmdType		m_MCmd;
	int				m_MConnID;
	int			    m_MThreadID;
  };


  // TL2 Slave Port and TL1 Master port
  sc_port<TLslaveIF<TdataCl_tl2>, 1> SlaveP;
  sc_port<TLmasterIF<TdataCl_tl1>, 1> MasterP;

  //clock port
  sc_in_clk clk;

  SC_HAS_PROCESS(OCP_TL1_TL2_Slave_Adapter);

  // constructor
  OCP_TL1_TL2_Slave_Adapter(sc_module_name name, int max_burst_length, int adapter_depth, bool writeResponse);

  // destructor
  ~OCP_TL1_TL2_Slave_Adapter();

  // SystemC processes
  void MasterRequest();
  void SlaveRequest();
  void MasterResponse();
  void SlaveResponse();


  //direct interfaces methods
  virtual bool MputDirect(int, bool, Td_tl1*, Ta_tl1, int);
  virtual bool SputDirect(int, bool, Td_tl2*, Ta_tl2, int);

  //adapter methods
  void reset_response_parameters(int);
  bool same_response_fields(response_parameters&, response_parameters&);
  void put_TL2_response_fields(response_parameters&);
  AdapterTask define_adapter_task(void);

 private :
  void end_of_elaboration();

  // pointer to the TL1 and TL2 data structures of the channel
  TdataCl_tl1          *m_DataCl_tl1;
  TdataCl_tl2          *m_DataCl_tl2;
  
  //pointer to param class
  ParamCl<TdataCl_tl1> *m_ParamCl_tl1;
  ParamCl<TdataCl_tl2> *m_ParamCl_tl2;

  //pointer to communication class
  CommCl           *m_CommCl_tl1;
  CommCl           *m_CommCl_tl2;

  // parameters given as constructor parameters
  int max_burst_length;        //maximum length of a burst transaction
  int buffer_size;             //size of the internal buffer for data and request fields (MCmd, MAddr, MAddrSpace...)
  int adapter_depth;		  //depth of requests and response storage 
  bool writeResponse;		  //indicates if the slave sends responses to write requests 

  //internal parameters
  bool new_burst_transaction;  //indicates if the next transaction is the beginning of a new transaction
  bool last_datum_sent;		   // last datum of the burst is sent with the burst
  int m_NumBytesPerWord;	   // size of the data
  int nb_response;				// number of TL1 responses that the slave sends
  int count_response;

  //indicates a later release of the request or response thread, when the internal buffer for storing request/response is full
  bool release_request_thread; 
  bool release_response_thread; 
  
  //index of the currently processed request/response
  int current_processed_request;
  int current_processed_response;
  
  int beginning_burst_index, current_index;   //index for internal buffers
  
  //buffers for storing beginning index and length of request/response
  int* burst_response_index;		//beginning index of the TL2 response
  int* burst_response_length;		//length of the TL2 response
  int* request_index;	//beginning index of the group of TL1 requests
  int* request_length;	//number of TL1 requests to send

  int m_pending_request;   //indicates if there is a pending request which has arrived, during the processing of the current request
  int m_request_number;	   //ID of the request currently processed
  int m_pending_response;  //indicates if there is a pending response which has arrived, during the processing of the current request
  int m_response_number;   //ID of the response currently processed

  //internal buffers
  response_parameters* m_response_buffer;  //for storing TL1 response fields
  request_parameters* m_request_buffer;    //for storing TL2 request fields
  Td_tl2* m_data;							//for storing data
  Td_tl1* m_response_data;					//for storing response
  
  //events
  sc_event send_TL1_request_event;	//event for sending the group of TL1 requests, made from the TL2 request
  sc_event send_TL2_response_event; //event for sending TL2 response, made of TL1 atomic responses
  
};


#endif // _OCP_TL1_TL2_SLAVE_ADAPTER_H
