// 
//  Copyright 2003 OCP-IP
//
// ============================================================================
//      Project : OCP SLD WG, Layer adapter examples 
//      Authors : Yann Bajot, PROSILOG, bajot@prosilog.com 
//                Stphane Guntz, PROSILOG, guntz@prosilog.com
//         Date : 03/10/2003
//  Description : Transaction Level - Layer-0 to Layer-1 Slave Adapter 
//               (with Datahandshake support)
//
//     Features :
//    - Layer adapter used to connect a TL0 OCP Slave to a TL1 system
//
//    - Interface implements OCP Basic signals : Clk, MCmd, MAddr, MData,
//        SCmdAccept, SData, SResp & MRespAccept.
//        + Datahandshake signals: MDataValid, SDataAccept
//
//    - Glitches and propagation delays for input signals are removed using two
//      'Sample delay' parameters.  Signals coming from the TL0 slave are not
//      supposed to be clocked and therefore can change at any time during the
//      clock cycle. It is also possible that some glitches appear on them,
//      particulary in the case of co-simulation with a HDL-simulated RTL
//      Slave. 
//
//      Once a TL1 interface call has been done, it can not be cancelled. The
//        slave adapter issues three kinds of call to the TL1 channel:
//          * SputResponse()
//          * SreleaseData()
//          * Srelease() 
//
//      The 'SputResponse()' and associated data class calls must happen only
//        when TL0 associated signals 'SResp' and 'SData' are stable. Since the
//        propagation delays for these signals are slave dependant, a 'Response
//        Sample Delay' parameter is used to determine when these signals
//        should be sampled during the cycle.  Once sampled,'SResp' is tested;
//        if the response is valid (DVA value), dataclass response and
//        'SputResponse()' calls are done.
//
//      The 'Srelease()' call must happen only when 'SCmdAccept' is stable.
//        Similarly, 'SreleaseData()' call must wait 'SDataAccept' to be
//        stable. Since the propagation delay for these signals are slave
//        dependant, the same 'AcceptSampleDelay' parameter is used to
//        determine when 'SDataAccept' AND 'SCmdAccept' should be sampled
//        during the cycle.
//
//      The smallest value for the sample delays is sc_get_time_resolution() ;
//      it should be used in the case when TL0 master has pure 'zero-delay'
//      signals. Users should always use a timed sample delay (and not a
//      defined number of delta-cycles) since it is not possible to predict the
//      number of delta-cycles required for signal stability.
//
//      'Check_setup_time()' function is proposed for debugging purposes, to
//        ensure that TL0 slave output signals don't change after the sample
//        delays, which would cause the adapter to fail. A good way to proceed
//        is to perform the first simulation with checking function actived to
//        determine the appropriate sample delay value.  Once TL0 behaviour is
//        correct, checking function should be disabled to increase simulation
//        speed.
//                
// Parameters   : 
//   Template arguments.
//   - TdataCl: Data class containing data members and
//              access methods for the data exchanged between
//              Masters and Slaves
//   - Td: Data type
//   - Ta: Address type
//                
//   First Constructor arguments (with explicit sampling times):
//     - sc_module_name: Name of the module instance
//     - ID: Unique number identifying the Slave.
//           ID must be unique among all Slaves attached to the same Bus.
//     - AcceptSampleDelay: Time number specifying the delay between clock 
//           rising edge and the actual sampling time for 'scmdaccept' and
//           'sdataaccept' signals
//     - ResponseSampleDelay: Time number specifying the delay between clock 
//           rising edge and the actual sampling time for response signals
//     - Check_setup_time: boolean value specifying if 'check_setup_time()' 
//                 function is executed during simulation 
//
//   Second Constructor arguments (sampling times use default values):
//     - sc_module_name: Name of the module instance
//     - ID: Unique number identifying the Master.
//           ID must be unique among all Masters attached to the same Bus.
//     - Combinational: Boolean specifying if DataHandshake/Response occurs in the
//                 same clock cycle or sequentially. Sampling times will be
//                 set as follows:
//                    - Combinational:
//                          * RequestSampleDelay = ClockCycle*99/100
//                          * ResponseSampleDelay = ClockCycle*60/100
//                    - Sequential:
//                          * RequestSampleDelay = ClockCycle*99/100
//                          * ResponseSampleDelay = ClockCycle*60/100
//     - ClockCycle: sc_time number specifying the clock cycle
//     - Check_setup_time: boolean value specifying if 'check_setup_time()' 
//                 function is executed during simulation 
//
//
//     NOTE : The adapter uses events of the communication class to
//     synchronize.  TL1 channel connected to the slave port MUST have its
//     'syncevent' parameter set.
// ============================================================================

#include "ocp_tl0_tl1_slave_adapter_hs.h"
#include "ocp_tl1_data_cl.h"

//#define DEBUG_SLAVE_ADAPTER_HS

// ----------------------------------------------------------------------------
// Process : OCP_TL0_TL1_Slave_Adapter_Hs::OCP_TL0_TL1_Slave_Adapter_Hs
// 
// Constructor with explicit sampling timings
// ----------------------------------------------------------------------------
template<class TdataCl> OCP_TL0_TL1_Slave_Adapter_Hs<TdataCl>::OCP_TL0_TL1_Slave_Adapter_Hs
(
 sc_module_name name_
 , int ID
 , sc_time AcceptSampleDelay
 , sc_time ResponseSampleDelay
 , bool check_setup_time 
)
: sc_module         (name_)
  , SlaveP           ("SlavePort")
  , m_ID              (ID)
  , m_accept_sample_delay    (AcceptSampleDelay)
  , m_response_sample_delay    (ResponseSampleDelay)
  , m_check_active    (check_setup_time)
  , m_default_values  (false)
{

  if(m_response_sample_delay < sc_get_time_resolution() ) {
    cout << "Error: TL0 OCP Slave Adapter constructor (instance '" << name_ << "')"<< endl
      << "       'Response Sample Delay' parameter must be > 'sc_get_time_resolution()' " << endl 
      << "       'Response Sample Delay' =" << m_response_sample_delay
      << endl;
    assert(0);
  }

  if(m_accept_sample_delay < sc_get_time_resolution() ) {
    cout << "Error: TL0 OCP Slave Adapter constructor (instance '" << name_ << "')"<< endl
      << "       'CmdAccept Sample Delay' parameter must be > 'sc_get_time_resolution()' " << endl 
      << "       'CmdAccept Sample Delay' =" << m_accept_sample_delay
      << endl;
    assert(0);
  }


  // Init 'MCmd' as IDLE, 'MRespAccept' and 'MDataValid' as false 
  MCmd.initialize(OCP_MCMD_IDLE);
  MRespAccept.initialize(false);
  MDataValid.initialize(false);

  SC_METHOD(send_Response);
  sensitive << e_response_sample_event;

  SC_METHOD(set_MRespAccept);
  sensitive << SlaveP.ResponseEndEvent();
  dont_initialize();

  SC_METHOD(reset_TL0_signals);
  sensitive_pos << Clk;

  SC_METHOD(send_Request);
  sensitive << SlaveP.RequestStartEvent();
  dont_initialize();

  SC_METHOD(send_Data);
  sensitive << SlaveP.DataRequestStartEvent();
  dont_initialize();

  SC_METHOD(release_Request);
  sensitive << e_accept_sample_event;

  SC_METHOD(release_Data);
  sensitive << e_accept_sample_event;

  SC_METHOD(sample_events_trigger);
  dont_initialize();
  sensitive_pos << Clk;

  if(m_check_active) {
    SC_METHOD(check_accept_setup_time);
    sensitive << SCmdAccept << SDataAccept ;
    SC_METHOD(check_response_setup_time);
    sensitive << SData << SResp;
  }

}

// ----------------------------------------------------------------------------
// Process : OCP_TL0_TL1_Slave_Adapter_Hs::OCP_TL0_TL1_Slave_Adapter_Hs
// 
// Constructor with default sampling timings.  
// ----------------------------------------------------------------------------
template<class TdataCl> OCP_TL0_TL1_Slave_Adapter_Hs<TdataCl>::OCP_TL0_TL1_Slave_Adapter_Hs
(
 sc_module_name name_
 , int ID
 , bool Combinational
 , sc_time ClockCycle
 , bool check_setup_time 
)
: sc_module         (name_)
  , SlaveP           ("SlavePort")
  , m_ID              (ID)
  , m_check_active    (check_setup_time)
  , m_default_values  (true)
{

  // Setting default sampling values
  if(Combinational) {
    m_accept_sample_delay = sc_time(ClockCycle*double(0.99));
    m_response_sample_delay = sc_time(ClockCycle*double(0.60));
  } else {
    m_accept_sample_delay = sc_time(ClockCycle*double(0.99));
    m_response_sample_delay = sc_time(ClockCycle*double(0.60));
  }

  if(m_response_sample_delay < sc_get_time_resolution() ) {
    cout << "Error: TL0 OCP Slave Adapter constructor (instance '" << name_ << "')"<< endl
      << "       'Response Sample Delay' parameter must be > 'sc_get_time_resolution()' " << endl 
      << "       'Response Sample Delay' =" << m_response_sample_delay
      << endl;
    assert(0);
  }

  if(m_accept_sample_delay < sc_get_time_resolution() ) {
    cout << "Error: TL0 OCP Slave Adapter constructor (instance '" << name_ << "')"<< endl
      << "       'CmdAccept Sample Delay' parameter must be > 'sc_get_time_resolution()' " << endl 
      << "       'CmdAccept Sample Delay' =" << m_accept_sample_delay
      << endl;
    assert(0);
  }


  // Init 'MCmd' as IDLE, 'MRespAccept' and 'MDataValid' as false 
  MCmd.initialize(OCP_MCMD_IDLE);
  MRespAccept.initialize(false);
  MDataValid.initialize(false);

  SC_METHOD(send_Response);
  sensitive << e_response_sample_event;

  SC_METHOD(set_MRespAccept);
  sensitive << SlaveP.ResponseEndEvent();
  dont_initialize();

  SC_METHOD(reset_TL0_signals);
  sensitive_pos << Clk;

  SC_METHOD(send_Request);
  sensitive << SlaveP.RequestStartEvent();
  dont_initialize();

  SC_METHOD(send_Data);
  sensitive << SlaveP.DataRequestStartEvent();
  dont_initialize();

  SC_METHOD(release_Request);
  sensitive << e_accept_sample_event;

  SC_METHOD(release_Data);
  sensitive << e_accept_sample_event;

  SC_METHOD(sample_events_trigger);
  dont_initialize();
  sensitive_pos << Clk;

  if(m_check_active) {
    SC_METHOD(check_accept_setup_time);
    sensitive << SCmdAccept << SDataAccept ;
    SC_METHOD(check_response_setup_time);
    sensitive << SData << SResp;
  }

}


// ----------------------------------------------------------------------------
// Process : OCP_TL0_TL1_Slave_Adapter_Hs::~OCP_TL0_TL1_Slave_Adapter_Hs
// 
// Destructor
// ----------------------------------------------------------------------------
template<class TdataCl> OCP_TL0_TL1_Slave_Adapter_Hs<TdataCl>::~OCP_TL0_TL1_Slave_Adapter_Hs()
{
}

// ----------------------------------------------------------------------------
//  Method : OCP_TL0_TL1_Slave_Adapter_Hs::send_Response()
//
//  Send responses to TL1 channel 
//
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL0_TL1_Slave_Adapter_Hs<TdataCl>::send_Response()
{
  if(!SlaveP->SgetMbusy() && (SResp.read() != sc_bv<2>(OCP_SRESP_NULL) ) ) {
    // 'DVA' RESPONSE
    if(SResp.read() == sc_bv<2>(OCP_SRESP_DVA) ) {
      m_DataCl->SputSResp(OCP_SRESP_DVA);
      m_DataCl->SputSData((Td)SData.read());
      SlaveP->SputResponse();
    } else
      // 'ERR' RESPONSE
      if(SResp.read() == sc_bv<2>(OCP_SRESP_ERR) ) {
        m_DataCl->SputSResp(OCP_SRESP_ERR);
        m_DataCl->SputSData((Td)SData.read());
        SlaveP->SputResponse();
      } else {
        cout << "Error in 'OCP TL0 TL1 Slave Adapter' class 'send_Response' method " << endl
          << "       TL0 SResp signal value '" << SResp.read() 
          << "' is not allowed for OCP 1.0" << endl;
        assert(0);
      }
  }
} // end of SC_METHOD

// ----------------------------------------------------------------------------
//  Method : OCP_TL0_TL1_Slave_Adapter_Hs::set_MRespAccept()
//
//  Set 'MRespAccept' signal
//
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL0_TL1_Slave_Adapter_Hs<TdataCl>::set_MRespAccept()
{
    MRespAccept.write(true);
} // end of SC_METHOD

// ----------------------------------------------------------------------------
//  Method : OCP_TL0_TL1_Slave_Adapter_Hs::reset_TL0_signals()
//
//  Reset 'MRespAccept', 'MDataValid' and 'MCmd' signals 
//
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL0_TL1_Slave_Adapter_Hs<TdataCl>::reset_TL0_signals()
{
  MRespAccept.write(false);
  if(!m_CommCl->RequestPending) {
    MCmd.write(OCP_MCMD_IDLE);
  }

  if(!m_CommCl->DataRequestPending) {
    MDataValid.write(false);
  }
} // end of SC_METHOD

// ----------------------------------------------------------------------------
//  Method : OCP_TL0_TL1_Slave_Adapter_Hs::send_Request()
//
//  Send requests to TL0 slave 
//
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL0_TL1_Slave_Adapter_Hs<TdataCl>::send_Request()
{
    MCmd.write(m_DataCl->SgetMCmd());
    MAddr.write(m_DataCl->SgetMAddr());

} // end of SC_METHOD 

// ----------------------------------------------------------------------------
//  Method : OCP_TL0_TL1_Slave_Adapter_Hs::send_Data()
//
//  Send data request to TL0 slave 
//
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL0_TL1_Slave_Adapter_Hs<TdataCl>::send_Data()
{
    MDataValid.write(m_DataCl->SgetMDataValid());
    MData.write(m_DataCl->SgetMDataHS());

} // end of SC_METHOD 


// ----------------------------------------------------------------------------
//  Method : OCP_TL0_TL1_Slave_Adapter_Hs::release_Request()
//
//  Release request present on TL1 channel
//
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL0_TL1_Slave_Adapter_Hs<TdataCl>::release_Request()
{
  if(m_CommCl->RequestPending) {
    if(SCmdAccept.read()) {
      SlaveP->Srelease(); 
    }
  }
} // end of SC_METHOD 

// ----------------------------------------------------------------------------
//  Method : OCP_TL0_TL1_Slave_Adapter_Hs::release_Data()
//
//  Release data request present on TL1 channel
//
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL0_TL1_Slave_Adapter_Hs<TdataCl>::release_Data()
{
  if(m_CommCl->DataRequestPending) {
    if(SDataAccept.read()) {
      SlaveP->SreleaseData(); 
    }
  }
} // end of SC_METHOD 


// ----------------------------------------------------------------------------
//  Method : OCP_TL0_TL1_Slave_Adapter_Hs::sample_events_trigger()
//
//  Trigger 'e_cmdaccept_sample_event' and 'e_response_sample_event'
//  used to sample input signals 
//
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL0_TL1_Slave_Adapter_Hs<TdataCl>::sample_events_trigger()
{
#ifdef DEBUG_SLAVE_ADAPTER_HS
  cout << endl
    << "TL0 Slave Adapter: CLOCK cycle"
    << " delta " << simcontext()->delta_count()
    << " time " << sc_time_stamp().to_seconds()
    << endl;
#endif

  e_accept_sample_event.notify(m_accept_sample_delay);
  e_response_sample_event.notify(m_response_sample_delay);

  if(m_check_active)
    m_last_rising_edge = sc_time_stamp();

} // end of SC_METHOD 

// ----------------------------------------------------------------------------
//  Method : OCP_TL0_TL1_Slave_Adapter_Hs::check_response_setup_time()
//
//  Checks timing violations, i.e. if 'SResp' or 'SData' signals
//    change after the sample time
//    
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL0_TL1_Slave_Adapter_Hs<TdataCl>::check_response_setup_time()
{
  if( (sc_time_stamp()-m_last_rising_edge) >= m_response_sample_delay ) {
    cout << "TL0 Slave Adapter '" << name() << "': Setup Time Violation "
      << " delta " << simcontext()->delta_count()
      << " time " << sc_time_stamp().to_seconds() << endl
      << " last rising edge " << m_last_rising_edge.to_seconds() << endl
      << " one of the signal 'SResp' or 'SData' had a late change" << endl
      << " You have to adjust 'ResponseSampleDelay' parameter" 
      << endl;

   if(m_default_values) {
     cout << " Default values for sampling times do not work with that design." << endl
      << " You have to explicitly specify sampling times using the appropriate constructor" 
      << endl;
   }
    sc_stop();
  }

} // end of SC_METHOD 

// ----------------------------------------------------------------------------
//  Method : OCP_TL0_TL1_Slave_Adapter_Hs::check_accept_setup_time()
//
//  Checks timing violations, i.e. if 'SCmdAccept' or 'SDataAccept' signals
//    change after the sample time
//    
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL0_TL1_Slave_Adapter_Hs<TdataCl>::check_accept_setup_time()
{
  if( (sc_time_stamp()-m_last_rising_edge) >= m_accept_sample_delay ) {
    cout << "TL0 Slave Adapter '" << name() << "': Setup Time Violation "
      << " delta " << simcontext()->delta_count()
      << " time " << sc_time_stamp().to_seconds() << endl
      << " last rising edge " << m_last_rising_edge.to_seconds() << endl
      << " signal 'SCmdAccept' or 'SDataAccept' had a late change" << endl
      << " You have to adjust 'AcceptSampleDelay' parameter" 
      << endl;

   if(m_default_values) {
     cout << " Default values for sampling times do not work with that design." << endl
      << " You have to explicitly specify sampling times using the appropriate constructor" 
      << endl;
   }
    sc_stop();
  }

} // end of SC_METHOD 



// ----------------------------------------------------------------------------
//  Method : OCP_TL1_Slave_Async::MPutDirect()
//
//  Debug interface method.
//  Read/Write data, Master to Slave direction
//  Returns true for success and false for failure
//
// ----------------------------------------------------------------------------
template<class TdataCl> bool OCP_TL0_TL1_Slave_Adapter_Hs<TdataCl>::MputDirect(
    int MasterID, bool IsWrite, Td *Data, Ta Address, int NumWords)
{

  // not implemented
  return(false);
}

// ----------------------------------------------------------------------------
//  Method : OCP_TL0_TL1_Slave_Adapter_Hs::end_of_elaboration()
//
//  This method is activated at the end of the elaboration phase
//  (when all binding operations are finished).
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL0_TL1_Slave_Adapter_Hs<TdataCl>::end_of_elaboration()
{
  sc_module::end_of_elaboration();

  // Initialize debug interface
  SlaveP->SregisterDirectIF(this);

  // Get data structure
  m_DataCl = SlaveP->GetDataCl();

  // Get communication structure
  m_CommCl = SlaveP->GetCommCl();

  // Get system parameter structure
  m_ParamCl = SlaveP->GetParamCl();

  // Put parameter into Channel
  // No parameters needed
}


// ----------------------------------------------------------------------------
//
//  Instantiation of the Master
//
// ----------------------------------------------------------------------------
template class OCP_TL0_TL1_Slave_Adapter_Hs<TL1_TEMPL_DATA_CL >; // see ocp_tl1_globals.h
