// 
//  Copyright 2003 OCP-IP
//
// ============================================================================
//      Project : OCP SLD WG, Layer adapter examples 
//      Authors : Yann Bajot, PROSILOG, bajot@prosilog.com 
//                Stphane Guntz, PROSILOG, guntz@prosilog.com
//         Date : 03/10/2003
//  Description : Transaction Level - Layer-0 to Layer-1 Master Adapter
//               (with Datahandshake support)
//
//     Features :
//    - Layer adapter used to connect a TL0 OCP Master to a TL1 system
//
//    - Interface implements OCP Basic signals : Clk, MCmd, MAddr, MData,
//        SCmdAccept, SData, SResp & MRespAccept.
//        + Datahandshake signals: MDataValid, SDataAccept
//
//    - Glitches and propagation delays for input signals are removed using two
//        'Sample delay' parameters.  Signals coming from the TL0 master are
//        not supposed to be clocked and therefore can change at any time
//        during the clock cycle. It is also possible that some glitches appear
//        on them, particulary in the case of co-simulation with a
//        HDL-simulated RTL Master. 
//
//      Once a TL1 interface call has been done, it can not be cancelled. The
//      master adapter issues three kinds of call to the TL1 channel:
//          * Mput*Request()
//          * MputDataRequest()
//          * Mrelease() 
//
//      The 'Mput*Request()' and associated data class calls must happen only
//        when TL0 associated signals 'MCmd' and 'MAddr' are stable.  Similarly,
//        'MputDataRequest()' call must wait 'MDataValid' and 'MData' to be
//        stable.  Since the propagation delays for these signals are master
//        dependant, we use a 'Request Sample Delay' parameter to determine when
//        these signals should be sampled during the cycle.
//
//      The 'Mrelease()' call must happen only when 'MRespAccept' is stable.
//        Since the propagation delay for this signal is master dependant, a
//        'RespAcceptSampleDelay' parameter is used to determine when
//        'MRespAccept' should be sampled during the cycle.
//
//      The smallest value for the sample delays is sc_get_time_resolution() ;
//      it should be used in the case when TL0 master has pure 'zero-delay'
//      signals. Users should always use a timed sample delay (and not a
//      defined number of delta-cycles) since it is not possible to predict the
//      number of delta-cycles required for signal stability.
//
//      'Check_setup_time()' function is proposed for debugging purposes, to
//        ensure that TL0 master output signals don't change after the sample
//        delays, which would cause the adapter to fail. A good way to proceed
//        is to perform the first simulation with checking function actived to
//        determine the appropriate sample delay value.  Once TL0 behaviour is
//        correct, checking function should be disabled to increase simulation
//        speed.
//                
// Parameters   : 
//   Template arguments.
//   - TdataCl: Data class containing data members and
//              access methods for the data exchanged between
//              Masters and Slaves
//   - Td: Data type
//   - Ta: Address type
//                
//    First Constructor arguments (with explicit sampling times):
//     - sc_module_name: Name of the module instance
//     - ID: Unique number identifying the Master.
//           ID must be unique among all Masters attached to the same Bus.
//     - Priority: Positive number specifying the priority relative to the
//                 other Masters for Bus access. Higher numbers means higher
//                 priority. Masters can have the same priority.
//     - RequestSampleDelay: Time number specifying the delay between clock 
//                 rising edge and the actual sampling time for Request signals
//                 and Datahandshake signals
//     - RespAcceptSampleDelay: Time number specifying the delay between clock 
//                 rising edge and the actual sampling time for 'MRespAccept'
//                 signal
//     - Check_setup_time: boolean value specifying if 'check_setup_time()' 
//                 function is executed during simulation 
//
//   Second Constructor arguments (sampling times use default values):
//     - sc_module_name: Name of the module instance
//     - ID: Unique number identifying the Master.
//           ID must be unique among all Masters attached to the same Bus.
//     - Priority: Positive number specifying the priority relative to the
//                 other Masters for Bus access. Higher numbers means higher
//                 priority. Masters can have the same priority.
//     - Combinational: Boolean specifying if DataHandshake/Response occurs in the
//                 same clock cycle or sequentially. Sampling times will be
//                 set as follows:
//                    - Combinational:
//                          * RequestSampleDelay = ClockCycle*60/100
//                          * ResponseSampleDelay = ClockCycle*75/100
//                    - Sequential:
//                          * RequestSampleDelay = ClockCycle*60/100
//                          * ResponseSampleDelay = ClockCycle*75/100
//     - ClockCycle: sc_time number specifying the clock cycle
//     - Check_setup_time: boolean value specifying if 'check_setup_time()' 
//                 function is executed during simulation 
//
//
//     NOTE : The adapter uses events of the communication class to
//     synchronize.  TL1 channel connected to the master port MUST have its
//     'syncevent' parameter set.
// ============================================================================

#include "ocp_tl0_tl1_master_adapter_hs.h"
#include "ocp_tl1_data_cl.h"

//#define DEBUG_MASTER_ADAPTER_HS

// ----------------------------------------------------------------------------
// Process : OCP_TL0_TL1_Master_Adapter_Hs::OCP_TL0_TL1_Master_Adapter_Hs
// 
// Constructor with explicit sampling timings
// ----------------------------------------------------------------------------
template<class TdataCl> OCP_TL0_TL1_Master_Adapter_Hs<TdataCl>::OCP_TL0_TL1_Master_Adapter_Hs
      (
        sc_module_name name_
      , int ID
      , int Priority
      , sc_time RequestSampleDelay
      , sc_time RespAcceptSampleDelay
      , bool Check_setup_time
      )
        : sc_module         (name_)
        , MasterP           ("MasterPort")
        , m_ID              (ID)
        , m_Priority        (Priority)
        , m_request_sample_delay    (RequestSampleDelay)
        , m_respaccept_sample_delay (RespAcceptSampleDelay)
        , m_check_active    (Check_setup_time)
        , m_default_values  (false)
{

  if(m_request_sample_delay < sc_get_time_resolution() ) {
    cout << "Error: TL0 OCP Master Adapter constructor (instance '" << name_ << "')"<< endl
      << "       'Request Sample Delay' parameter must be > 'sc_get_time_resolution()' " << endl 
      << "       'Request Sample Delay' =" << m_request_sample_delay
      << endl;
    assert(0);
  }

  if(m_respaccept_sample_delay < sc_get_time_resolution() ) {
    cout << "Error: TL0 OCP Master Adapter constructor (instance '" << name_ << "')"<< endl
      << "       'RespAccept Sample Delay' parameter must be > 'sc_get_time_resolution()' " << endl 
      << "       'RespAccept Sample Delay' =" << m_respaccept_sample_delay
      << endl;
    assert(0);
  }

  // Init 'SCmdAccept', 'SDataAccept', 'SResp' and 'SData'
  SCmdAccept.initialize(false);
  SDataAccept.initialize(false);
  SResp.initialize(OCP_SRESP_NULL);
  SData.initialize(0);


  SC_METHOD(send_Request);
  sensitive << e_request_sample_event;

  SC_METHOD(send_Data);
  sensitive << e_request_sample_event;

  SC_METHOD(set_SCmdAccept);
  sensitive << MasterP.RequestEndEvent();
  dont_initialize();

  SC_METHOD(set_SDataAccept);
  sensitive << MasterP.DataRequestEndEvent();
  dont_initialize();

  SC_METHOD(reset_TL0_signals);
  sensitive_pos << Clk;

  SC_METHOD(send_Response);
  sensitive << MasterP.ResponseStartEvent();
  dont_initialize();

  SC_METHOD(release_Response);
  sensitive << e_respaccept_sample_event;

  SC_METHOD(sample_events_trigger);
  dont_initialize();
  sensitive_pos << Clk;

  if(m_check_active) {
    SC_METHOD(check_respaccept_setup_time);
    sensitive << MRespAccept ;
    SC_METHOD(check_request_setup_time);
    sensitive << MCmd << MAddr << MData << MDataValid;
  }

}

// ----------------------------------------------------------------------------
// Process : OCP_TL0_TL1_Master_Adapter_Hs::OCP_TL0_TL1_Master_Adapter_Hs
// 
// Constructor with default sampling timings.  
// ----------------------------------------------------------------------------
template<class TdataCl> OCP_TL0_TL1_Master_Adapter_Hs<TdataCl>::OCP_TL0_TL1_Master_Adapter_Hs
      (
        sc_module_name name_
      , int ID
      , int Priority
      , bool Combinational
      , sc_time ClockCycle
      , bool Check_setup_time
      )
        : sc_module         (name_)
        , MasterP           ("MasterPort")
        , m_ID              (ID)
        , m_Priority        (Priority)
        , m_check_active    (Check_setup_time)
        , m_default_values  (true)
{
  // Setting default sampling values
  if(Combinational) {
    m_request_sample_delay = sc_time(ClockCycle*double(0.60));
    m_respaccept_sample_delay = sc_time(ClockCycle*double(0.75));
  } else {
    m_request_sample_delay = sc_time(ClockCycle*double(0.60));
    m_respaccept_sample_delay = sc_time(ClockCycle*double(0.75));
  }


  if(m_request_sample_delay < sc_get_time_resolution() ) {
    cout << "Error: TL0 OCP Master Adapter constructor (instance '" << name_ << "')"<< endl
      << "       'Request Sample Delay' parameter must be > 'sc_get_time_resolution()' " << endl 
      << "       'Request Sample Delay' =" << m_request_sample_delay
      << endl;
    assert(0);
  }

  if(m_respaccept_sample_delay < sc_get_time_resolution() ) {
    cout << "Error: TL0 OCP Master Adapter constructor (instance '" << name_ << "')"<< endl
      << "       'RespAccept Sample Delay' parameter must be > 'sc_get_time_resolution()' " << endl 
      << "       'RespAccept Sample Delay' =" << m_respaccept_sample_delay
      << endl;
    assert(0);
  }

  // Init 'SCmdAccept', 'SDataAccept', 'SResp' and 'SData'
  SCmdAccept.initialize(false);
  SDataAccept.initialize(false);
  SResp.initialize(OCP_SRESP_NULL);
  SData.initialize(0);


  SC_METHOD(send_Request);
  sensitive << e_request_sample_event;

  SC_METHOD(send_Data);
  sensitive << e_request_sample_event;

  SC_METHOD(set_SCmdAccept);
  sensitive << MasterP.RequestEndEvent();
  dont_initialize();

  SC_METHOD(set_SDataAccept);
  sensitive << MasterP.DataRequestEndEvent();
  dont_initialize();

  SC_METHOD(reset_TL0_signals);
  sensitive_pos << Clk;

  SC_METHOD(send_Response);
  sensitive << MasterP.ResponseStartEvent();
  dont_initialize();

  SC_METHOD(release_Response);
  sensitive << e_respaccept_sample_event;

  SC_METHOD(sample_events_trigger);
  dont_initialize();
  sensitive_pos << Clk;

  if(m_check_active) {
    SC_METHOD(check_respaccept_setup_time);
    sensitive << MRespAccept ;
    SC_METHOD(check_request_setup_time);
    sensitive << MCmd << MAddr << MData << MDataValid;
  }

}


// ----------------------------------------------------------------------------
// Process : OCP_TL0_TL1_Master_Adapter_Hs::~OCP_TL0_TL1_Master_Adapter_Hs
// 
// Destructor
// ----------------------------------------------------------------------------
template<class TdataCl> OCP_TL0_TL1_Master_Adapter_Hs<TdataCl>::~OCP_TL0_TL1_Master_Adapter_Hs()
{
}

// ----------------------------------------------------------------------------
//  Method : OCP_TL0_TL1_Master_Adapter_Hs::send_Request()
//
//  Send requests to TL1 channel 
//
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL0_TL1_Master_Adapter_Hs<TdataCl>::send_Request()
{
  if(!MasterP->MgetSbusy() && (MCmd.read() != sc_bv<3>(OCP_MCMD_IDLE) ) ) {
    // WRITE OPERATION
    if(MCmd.read() == sc_bv<3>(OCP_MCMD_WR) ) {
      m_DataCl->MputMCmd(OCP_MCMD_WR);
      m_DataCl->MputMAddr((Ta)MAddr.read());
      MasterP->MputWriteRequest();
    } else
    // READ OPERATION
    if(MCmd.read() == sc_bv<3>(OCP_MCMD_RD) ) {
      m_DataCl->MputMCmd(OCP_MCMD_RD);
      m_DataCl->MputMAddr((Ta)MAddr.read());
      MasterP->MputReadRequest();
    } else
    // READ EX OPERATION
    if(MCmd.read() == sc_bv<3>(OCP_MCMD_RDEX) ) {
      m_DataCl->MputMCmd(OCP_MCMD_RDEX);
      m_DataCl->MputMAddr((Ta)MAddr.read());
      MasterP->MputReadRequest();
    } else
    // BROADCAST OPERATION
    if(MCmd.read() == sc_bv<3>(OCP_MCMD_BCST) ) {
      m_DataCl->MputMCmd(OCP_MCMD_BCST);
      m_DataCl->MputMAddr((Ta)MAddr.read());
      MasterP->MputWriteRequest();
    } else {
      cout << "Error in 'OCP TL0 TL1 Master Adapter' class 'send_Request' method " << endl
        << "       TL0 MCmd signal value '" << MCmd.read() 
        << "' is not allowed for OCP 1.0" << endl;
      assert(0);
    }
  }
} // end of SC_METHOD

// ----------------------------------------------------------------------------
//  Method : OCP_TL0_TL1_Master_Adapter_Hs::send_Data()
//
//  Send data requests to TL1 channel (handshake support)
//
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL0_TL1_Master_Adapter_Hs<TdataCl>::send_Data()
{
  if(!MasterP->MgetSbusy() && (MDataValid.read() == true) ) {
    m_DataCl->MputMDataHS((Td)MData.read());
    MasterP->MputDataRequest();
  }
} // end of SC_METHOD


// ----------------------------------------------------------------------------
//  Method : OCP_TL0_TL1_Master_Adapter_Hs::set_SCmdAccept()
//
//  Set 'SCmdAccept' signal
//
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL0_TL1_Master_Adapter_Hs<TdataCl>::set_SCmdAccept()
{
    SCmdAccept.write(true);
} // end of SC_METHOD

// ----------------------------------------------------------------------------
//  Method : OCP_TL0_TL1_Master_Adapter_Hs::set_SDataAccept()
//
//  Set 'SDataAccept' signal
//
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL0_TL1_Master_Adapter_Hs<TdataCl>::set_SDataAccept()
{
    SDataAccept.write(true);
} // end of SC_METHOD


// ----------------------------------------------------------------------------
//  Method : OCP_TL0_TL1_Master_Adapter_Hs::reset_TL0_signals()
//
//  Reset 'SCmdAccept', 'SData and 'SResp' signals 
//
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL0_TL1_Master_Adapter_Hs<TdataCl>::reset_TL0_signals()
{
  SCmdAccept.write(false);
  SDataAccept.write(false);
  if(!m_CommCl->ResponsePending) {
    SResp.write(sc_bv<3>(OCP_SRESP_NULL));
  }
} // end of SC_METHOD


// ----------------------------------------------------------------------------
//  Method : OCP_TL0_TL1_Master_Adapter_Hs::send_Response()
//
//  Send responses to TL0 master 
//
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL0_TL1_Master_Adapter_Hs<TdataCl>::send_Response()
{
    SData.write(m_DataCl->MgetSData());
    SResp.write(sc_bv<3>(OCP_SRESP_DVA));
} // end of SC_METHOD 

// ----------------------------------------------------------------------------
//  Method : OCP_TL0_TL1_Master_Adapter_Hs::release_Response()
//
//  Release response present on TL1 channel
//
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL0_TL1_Master_Adapter_Hs<TdataCl>::release_Response()
{
  if(m_CommCl->ResponsePending) {
    if(MRespAccept.read()) {
      MasterP->Mrelease(); 
    }
  }
} // end of SC_METHOD 


// ----------------------------------------------------------------------------
//  Method : OCP_TL0_TL1_Master_Adapter_Hs::sample_events_trigger()
//
//  Trigger 'e_respaccept_sample_event' and 'e_request_sample_event'
//  used to sample input signals 
//
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL0_TL1_Master_Adapter_Hs<TdataCl>::sample_events_trigger()
{
#ifdef DEBUG_MASTER_ADAPTER_HS
  cout << endl
    << "TL0 Master Adapter: CLOCK cycle"
    << " delta " << simcontext()->delta_count()
    << " time " << sc_time_stamp().to_seconds()
    << endl;
#endif

  e_respaccept_sample_event.notify(m_respaccept_sample_delay);
  e_request_sample_event.notify(m_request_sample_delay);

  if(m_check_active)
    m_last_rising_edge = sc_time_stamp();

} // end of SC_METHOD 

// ----------------------------------------------------------------------------
//  Method : OCP_TL0_TL1_Master_Adapter_Hs::check_request_setup_time()
//
//  Checks timing violations, i.e. if 'MCmd', 'MAddr', 'MData' or 'MDataValid'
//  signals change after the sample time
//    
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL0_TL1_Master_Adapter_Hs<TdataCl>::check_request_setup_time()
{
  if( (sc_time_stamp()-m_last_rising_edge) >= m_request_sample_delay ) {
    cout << "TL0 Master Adapter '" << name() << "': Setup Time Violation "
      << " delta " << simcontext()->delta_count()
      << " time " << sc_time_stamp().to_seconds() << endl
      << " last rising edge " << m_last_rising_edge.to_seconds() << endl
      << " one of the signal 'MCmd', 'MData' , 'MAddr' or 'MDataValid' had a late change" << endl
      << " You have to adjust 'RequestSampleDelay' parameter" 
      << endl;

   if(m_default_values) {
     cout << " Default values for sampling times do not work with that design." << endl
      << " You have to explicitly specify sampling times using the appropriate constructor" 
      << endl;
   }
 
    sc_stop();
  }

} // end of SC_METHOD 

// ----------------------------------------------------------------------------
//  Method : OCP_TL0_TL1_Master_Adapter_Hs::check_respaccept_setup_time()
//
//  Checks timing violations, i.e. if 'MRespAccept' signal
//    change after the sample time
//    
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL0_TL1_Master_Adapter_Hs<TdataCl>::check_respaccept_setup_time()
{
  if( (sc_time_stamp()-m_last_rising_edge) >= m_respaccept_sample_delay ) {
    cout << "TL0 Master Adapter: '" << name() << "' Setup Time Violation "
      << " delta " << simcontext()->delta_count()
      << " time " << sc_time_stamp().to_seconds() << endl
      << " last rising edge " << m_last_rising_edge.to_seconds() << endl
      << " signal 'MRespAccept' had a late change" << endl
      << " You have to adjust 'RespAcceptSampleDelay' parameter" 
      << endl;

   if(m_default_values) {
     cout << " Default values for sampling times do not work for that design." << endl
      << " You have to explicitly specify sampling times using the appropriate constructor" 
      << endl;
   }

    sc_stop();
  }

} // end of SC_METHOD 



// ----------------------------------------------------------------------------
//  Method : OCP_TL0_TL1_Master_Adapter_Hs::SputDirect()
//
//  Debug interface method.
//  Read/Write data, Slave to Master direction
//  Returns true for success and false for failure
//
// ----------------------------------------------------------------------------
template<class TdataCl> bool OCP_TL0_TL1_Master_Adapter_Hs<TdataCl>::SputDirect(
         int SlaveID, bool IsWrite, Td *Data, Ta Address, int NumWords)

{
  // not implemented
  return(false);
}


// ----------------------------------------------------------------------------
//  Method : OCP_TL0_TL1_Master_Adapter_Hs::end_of_elaboration()
//
//  This method is activated at the end of the elaboration phase
//  (when all binding operations are finished).
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL0_TL1_Master_Adapter_Hs<TdataCl>::end_of_elaboration()
{
  sc_module::end_of_elaboration();

  // Initialize debug interface
  MasterP->MregisterDirectIF(this);

  // Get data structure
  m_DataCl = MasterP->GetDataCl();

  // Get communication structure
  m_CommCl = MasterP->GetCommCl();

  // Get system parameter structure
  m_ParamCl = MasterP->GetParamCl();

  // Put parameter into Channel
  // No parameters needed
}


// ----------------------------------------------------------------------------
//
//  Instantiation of the Master
//
// ----------------------------------------------------------------------------
template class OCP_TL0_TL1_Master_Adapter_Hs<TL1_TEMPL_DATA_CL >; // see ocp_tl1_globals.h
