// 
//  Copyright 2003 OCP-IP
//
// ============================================================================
//      Project : OCP SLD WG, Layer adapter examples 
//      Authors : Yann Bajot, PROSILOG, bajot@prosilog.com 
//                Stephane Guntz, PROSILOG, guntz@prosilog.com
//         Date : 01/15/2004
//
//  Description : Layer Adapter, TL0-TL1 example
//              Top level, TL0 asynchronous Master <---> TL1 asynchronous Slave
//
//  This example presents a fully combinational OCP connexion between a
//    TL0 Master and a TL1 Slave, where requests and responses occur in
//    the same clock cycle.
//    
//  The master issues 5 writes followed by 5 reads, and checks received data.
//    
//  Operation timings are controlled by the following parameters: T1, T2, T3,
//  T4 ,T5 and T6
//    - T1, T4, T5 and T6 are expressed as 'sc_time' values
//    - T2 and T3 are expressed as a percentage (double) of the clock cycle 
//    - Clock cycle duration in this example is 10 ns
//  
//  OCP transfers occurs such as follows: 
//    * 'T1' ns after the clock rising edge, TL0 Master sends a request 
//    * 'T5' ns after the clock rising edge, TL0 Master Adapter samples
//        request signals group, and issues a request call to the TL1 channel
//        depending on the 'MCmd' value.
//    * After request reception, TL1 slave waits for:
//                  - 'T2*clock_cycle/100' ns and releases the request
//                  - 'T3*clock_cycle/100' ns and issues the response
//
//    * TL0 Master Adapter catches the TL1 response and drives corresponding
//      TL0 signals
//    * 'T4' ns after the response assertion, TL0 Master set the 'MRespAccept'
//      signal.
//    * 'T6' ns after the clock rising edge, TL0 Master Adapater samples
//      'MRespAccept' and conditionally releases the TL1 response.
//
//  T5 & T6 specifies the sample delays for the TL0 Master Adapter.
//  To get a correct behaviour, T5 must be larger than 'T1' (Request signals
//  group) and T6 must be larger than 'T5 + T3*clock_cycle/100 + T4'
//  ('MRespAccept' signal).
//
//  TL0 signals can be viewed using the VCD file 'top_master_async.vcd'
//    
// ============================================================================

#include <map>
#include <string>

#include "ocp_tl0_master_async.h"
#include "ocp_tl1_slave_async.h"
#include "ocp_tl0_tl1_master_adapter.h"
#include "ocp_tl1_data_cl.h"
#include "ocp_tl1_channel.h"  

// Timings 
const sc_time_unit TL_TIME_UNIT = SC_NS; 
const double TL_CLK_PERIOD = 10 ;     // clock period

void readMapFromFile(const string &myFileName, MapStringType &myParamMap) 
{
    // read pairs of data from the passed file
    string leftside;
    string rightside;
    
    // (1) open the file
    ifstream inputfile(myFileName.c_str());
    assert( inputfile );

    // set the formatting
    inputfile.setf(std::ios::skipws);

    // Now read through all the pairs of values and add them to the passed map
    while ( inputfile ) {
        inputfile >> leftside;
        inputfile >> rightside;
        myParamMap.insert(std::make_pair(leftside,rightside));
    }

    // All done, close up
    inputfile.close();
}


int sc_main(int, char*[])
{
   MapStringType   ocpParamMap;
   string  ocpParamFileName = "ocpParams_master_adapter_tl0_tl1";
   
   if ( ! ocpParamFileName.empty() ) {
        readMapFromFile(ocpParamFileName, ocpParamMap);
    }
 
    // TL1 channel
    //
    // 
    bool sync = true; // mandatory for OCP data classes
    bool defaultevent = true; // mandatory for the tl1 master used in this example 
    // 'SyncEvent' must be set for the adapter to work correctly
    bool syncevent = true;
    
    OCP_TL1_Channel<OCP_TL1_DataCl<int,int> > ch1("ch0", sync, syncevent, defaultevent, NULL, TL_CLK_PERIOD, TL_TIME_UNIT,"ch0.ocp",true);
    ch1.setConfiguration(ocpParamMap);
    
    // TL0 Asynchronous Master 
    //
    // 
    sc_time T1 = sc_time(1, SC_NS);   // Request delay
    sc_time T4 = sc_time(1, SC_NS);   // Response acknowledge delay
    OCP_TL0_Master_async<OCP_TL1_DataCl<int,int>  > ms0("ms0", T1, T4 );

    // TL1 Asynchronous Slave 
    //
    // 
    double T2 = 10;     // Request acknowledge is valid after 'T2'% of the clock
                        // cycle following the request detection
    double T3 = 20;     // Response is sent after 'T3'% of the clock cycle, following the request detection 
    OCP_TL1_Slave_Async<OCP_TL1_DataCl<int,int>  > sl1("sl1", 1, 0x0, 0x3ff, T2, T3);



    // TL0 Master Adapter
    //
    // Sample delay T5 is used to sample Request group signals: it must be > T1 in this case (fully asynchronous)
    // Sample delay T6 is used to sample MRespAccept signal: it must be > T5 + T3*clock_cycle/100 + T4 in this case (fully asynchronous)
    // check_setup_time parameter is actived to check TL0 setup time violations
    
    sc_time T5 = sc_time(2,SC_NS);
    sc_time T6 = sc_time(6,SC_NS);

    bool check_setup_time = true;
    // Here we use the explicit constructor to specify sampling times
    OCP_TL0_TL1_Master_Adapter<OCP_TL1_DataCl<int,int>  > ms_adapter("ms_adapter", 1, 1, T5, T6, check_setup_time );


    // Testbench building
    //
    //
    sc_clock clk("clock", TL_CLK_PERIOD, TL_TIME_UNIT);

    sc_signal<sc_bv<3> > 	s_MCmd;
    sc_signal<int>       	s_MAddr;
    sc_signal<int>       	s_MData;
    sc_signal<bool> 	    s_SCmdAccept;
    sc_signal<int>       	s_SData;
    sc_signal<sc_bv<2> >  s_SResp;
    sc_signal<bool>  	    s_MRespAccept;


    sl1.SlaveP(ch1);
    sl1.clk(clk);

    ms_adapter.MasterP(ch1);

    ms_adapter.Clk(clk);
    ms_adapter.MCmd(s_MCmd);
    ms_adapter.MAddr(s_MAddr);
    ms_adapter.MData(s_MData);
    ms_adapter.SCmdAccept(s_SCmdAccept);
    ms_adapter.SData(s_SData);
    ms_adapter.SResp(s_SResp);
    ms_adapter.MRespAccept(s_MRespAccept);

    ms0.Clk(clk);
    ms0.MCmd(s_MCmd);
    ms0.MAddr(s_MAddr);
    ms0.MData(s_MData);
    ms0.SCmdAccept(s_SCmdAccept);
    ms0.SData(s_SData);
    ms0.SResp(s_SResp);
    ms0.MRespAccept(s_MRespAccept);

    // VCD Trace file for TL0 signals
    //
    //
    sc_trace_file *vcd_file = sc_create_vcd_trace_file("top_master_async"); 

    sc_trace(vcd_file,clk,"Clk");
    sc_trace(vcd_file,s_MCmd,"MCmd");
    sc_trace(vcd_file,s_MAddr,"MAddr");
    sc_trace(vcd_file,s_MData,"MData");
    sc_trace(vcd_file,s_SCmdAccept,"SCmdAccept");
    sc_trace(vcd_file,s_SData,"SData");
    sc_trace(vcd_file,s_SResp,"SResp");
    sc_trace(vcd_file,s_MRespAccept,"MRespAccept");


    sc_start(500, SC_NS);

    sc_close_vcd_trace_file(vcd_file);

    return(0);
}
