// 
// Copyright 2003 OCP-IP
// OCP-IP Confidential & Proprietary
//
//
// ============================================================================
//      Project : OCP SLD WG, Layer adapter examples 
//      Authors : Yann Bajot, PROSILOG, bajot@prosilog.com 
//         Date : 10/16/2003
//
//  Description:  Regresion test for the TL2 data class
//                Tests that all members are correctly sent and read on each
//                side, for both request and response phases and sideband
//                signals
// ============================================================================


//#define DEBUG_C

#include <systemc.h>
#include "ocp_tl_param_cl.h"
#include "ocp_tl2_data_cl.h"
#include "tl_channel.h"  
#include "tl_master_port.h"
#include "tl_master_if.h"

// ----------------------------------------------------------------------------
// MASTER MODULE:
// ----------------------------------------------------------------------------


SC_MODULE(master) {

    // Port definitions
    sc_port<TLmasterIF<OCP_TL2_DataCl<unsigned int, unsigned int> > > ocp;


    // ----------------------------------------------------------------------------
    // Send all the request signals to the slave
    // ----------------------------------------------------------------------------
    void request_thread () {
        bool read_cycle = 0;
        int shift = 0;
        int trans_nb = 0;

        while (1) {

            ocp->GetDataCl()->MputMAddr           ( 0+trans_nb  ) ;
            ocp->GetDataCl()->MputMAddrSpace      ( 1+trans_nb  ) ;
            ocp->GetDataCl()->MputMAtomicLength   ( 2+trans_nb  ) ;
            ocp->GetDataCl()->MputMBurstSeq       ( (OCPMBurstSeqType) ((3+trans_nb)%8)  ) ;
            ocp->GetDataCl()->MputMByteEn         ( 4+trans_nb  ) ;
            ocp->GetDataCl()->MputMConnID         ( 6+trans_nb  ) ;
            ocp->GetDataCl()->MputMDataInfo       ( 7+trans_nb  ) ;
            ocp->GetDataCl()->MputMThreadID       ( 8+trans_nb  ) ;
            ocp->GetDataCl()->MputMBurstLength    ( 9+trans_nb  ) ;
            ocp->GetDataCl()->MputMBurstPrecise   ( (10+trans_nb)%2 ) ;
            ocp->GetDataCl()->MputMBurstSingleReq ( (11+trans_nb)%2 ) ;
            ocp->GetDataCl()->MputMReqInfo        ( 12+trans_nb ) ;
            ocp->GetDataCl()->MputMReqLast        ( (13+trans_nb)%2 ) ;

            // WRITE
            if (!read_cycle) {
                read_cycle = 1; // Toggle write and read cycles

                bool last_of_a_burst=false;

                if(shift==60)
                    last_of_a_burst=true;

                ocp->GetDataCl()->MputMData(&wdata[shift], 10, last_of_a_burst); 

                sc_assert(ocp->MputWriteRequestBlocking());

#ifdef DEBUG_C
                cout << "Master write request finished";
                cout << ", data = ";
                for (int i=shift;i<shift+10;i++)
                    cout << wdata[i] <<",";
                cout << "  at time  = " << sc_time_stamp().to_seconds() << endl;
#endif
                if (shift<90)
                    shift += 10;
                else
                    shift = 0;
            } // READ
            else {
                read_cycle = 0; // Toggle write and read cycles
                ocp->GetDataCl()->MputMReqChunkLen(10);
                ocp->GetDataCl()->MputMReqChunkLast( ((trans_nb%3)==0) );
                sc_assert(ocp->MputReadRequestBlocking());

#ifdef DEBUG_C
                cout << "Master read request finished";
                cout << " at time  = " << sc_time_stamp().to_seconds() << endl;
#endif
            }
            trans_nb++;
        }
    }

    // ----------------------------------------------------------------------------
    // Check all the received response signals from the slave
    // ----------------------------------------------------------------------------
    void response_thread () { 
        unsigned int len;
        unsigned int trans_nb=0;
        bool last_of_a_burst;

        while (1) {
            if (ocp->MgetResponseBlocking(0)) {
                ocp->Mrelease(sc_time(1000, SC_MS)); 

                sc_assert(ocp->GetDataCl()->MgetSDataInfo()==0+trans_nb);
                sc_assert(ocp->GetDataCl()->MgetSResp()==(OCPSRespType)((1+trans_nb)%4));
                sc_assert(ocp->GetDataCl()->MgetSRespInfo()==2+trans_nb);
                sc_assert(ocp->GetDataCl()->MgetSRespLast()==(3+trans_nb)%2);
                sc_assert(ocp->GetDataCl()->MgetSThreadID()==4+trans_nb);

                if(trans_nb%2 == 0) {
                    rdata = ocp->GetDataCl()->MgetSData(len,last_of_a_burst);
                    sc_assert(last_of_a_burst==true);
                } else {
                    rdata = ocp->GetDataCl()->MgetSData();
                    last_of_a_burst = ocp->GetDataCl()->MgetSRespChunkLast();
                    len = ocp->GetDataCl()->MgetSRespChunkLen();
                    sc_assert(last_of_a_burst==false);
                }
                sc_assert(len==10);

#ifdef DEBUG_C
                cout << "Master gets read response";
                cout << ", data = ";
                for (unsigned int i=0;i<len;i++)
                    cout << rdata[i] <<",";
                cout << " at time  = " << sc_time_stamp().to_seconds() << endl;
#endif
            }

            trans_nb++;
        }
    }

    // ----------------------------------------------------------------------------
    // Check all the received sideband signals from the slave
    // ----------------------------------------------------------------------------
    void get_slave_sb_thread() {  
        unsigned int trans_nb = 0;

        while(1) {

            wait(slave_sb_start_event);

            // 'SReset_n' test
            wait(ocp->GetDataCl()->SBResetEvent);
            sc_assert(ocp->GetDataCl()->MgetSReset_n() == trans_nb%2 == 0);

            // 'SFlag' test
            wait(ocp->GetDataCl()->SBSlaveEvent);
            sc_assert(ocp->GetDataCl()->MgetSFlag() == trans_nb+1);

            // 'SError' test
            wait(ocp->GetDataCl()->SBSlaveEvent);
            sc_assert(ocp->GetDataCl()->MgetSError() == (trans_nb+2)%2 );

            // 'SInterrupt' test
            wait(ocp->GetDataCl()->SBSlaveEvent);
            sc_assert(ocp->GetDataCl()->MgetSInterrupt() == (trans_nb+3)%2 );

            // 'SThreadBusy' test
            wait(ocp->GetDataCl()->SBSThreadBusyEvent);
            sc_assert(ocp->GetDataCl()->MgetSThreadBusy() == trans_nb%2 == 0);

            // 'CputStatus' test
            wait(ocp->GetDataCl()->SBStatusEvent);
            sc_assert(ocp->GetDataCl()->SysgetStatus() == trans_nb + 4);

            // 'CputStatusBusy' test
            wait(ocp->GetDataCl()->SBStatusEvent);
            sc_assert(ocp->GetDataCl()->SysgetStatusBusy() == trans_nb%2 == 0);

            // 'CputControlBusy' test
            wait(ocp->GetDataCl()->SBControlEvent);
            sc_assert(ocp->GetDataCl()->SysgetControlBusy() == trans_nb%2 == 0);

#ifdef DEBUG_C
            cout  << "get_slave_sb_thread test passed... " << trans_nb << "\n" ;
#endif

            trans_nb++;
        }
    }

    // ----------------------------------------------------------------------------
    // Check all the received sideband signals from the slave
    // ----------------------------------------------------------------------------
    void put_master_sb_thread() {  
        int trans_nb = 0;

        while(1) {

            wait(master_sb_start_event);

            // 'MReset_n' test
            wait(sc_time(10, SC_MS));
            ocp->GetDataCl()->MputMReset_n(trans_nb%2 == 0);

            // 'MFlag' test
            wait(sc_time(10, SC_MS));
            ocp->GetDataCl()->MputMFlag(trans_nb+1);

            // 'MError' test
            wait(sc_time(10, SC_MS));
            ocp->GetDataCl()->MputMError(trans_nb%2);

            // 'MThreadBusy' test
            wait(sc_time(10, SC_MS));
            ocp->GetDataCl()->MputMThreadBusy(trans_nb%2 == 0);

            // 'SysputControlWr' test
            wait(sc_time(10, SC_MS));
            ocp->GetDataCl()->SysputControlWr(trans_nb%2 == 0);

            // 'SysputControl' test
            wait(sc_time(10, SC_MS));
            ocp->GetDataCl()->SysputControl(trans_nb+3);

            // 'SysputStatusRd' test
            wait(sc_time(10, SC_MS));
            ocp->GetDataCl()->SysputStatusRd(trans_nb%2 == 0);

#ifdef DEBUG_C
            cout  << "put_master_sb_thread test passed... " << trans_nb << "\n" ;
#endif

            trans_nb++;
        }
    }

    // ----------------------------------------------------------------------------
    // Synchronizes 'master sideband' and 'slave sideband' communication phases
    // ----------------------------------------------------------------------------
    void sync_sb() {  
        while(true) {
            master_sb_start_event.notify();
            wait(sc_time(1000,SC_MS));
            slave_sb_start_event.notify();
            wait(sc_time(1000,SC_MS));
        }
    }


    // ----------------------------------------------------------------------------
    // Constructor 
    // ----------------------------------------------------------------------------
    SC_CTOR(master) {
        for (int i=0;i<100;i++)
            wdata[i] = i;
        SC_THREAD(get_slave_sb_thread);
        SC_THREAD(put_master_sb_thread);
        SC_THREAD(request_thread);
        SC_THREAD(response_thread);
        SC_THREAD(sync_sb);
    }

    unsigned int wdata[100], *rdata;
    sc_event slave_sb_start_event, master_sb_start_event;

};


// ----------------------------------------------------------------------------
// SLAVE MODULE:
// ----------------------------------------------------------------------------

#include "tl_slave_if.h"
#include "tl_slave_port.h"

SC_MODULE(slave) {

    // Port definitions
    sc_port<TLslaveIF<OCP_TL2_DataCl<unsigned int,unsigned int> > > ocp;

    // Globals
    unsigned int * data;
    int len;

    sc_event response_event;

    // ----------------------------------------------------------------------------
    // check all the received request signals from the master
    // ----------------------------------------------------------------------------
    void request_thread () {
        bool read_cycle = 0;
        int shift = 0;
        unsigned int trans_nb=0;

        while (true) {

            sc_assert(ocp->SgetRequestBlocking(0));
            ocp->Srelease(sc_time(1000, SC_MS)); 

            // Check request fields
            sc_assert( ocp->GetDataCl()->SgetMAddr           ()== 0+trans_nb  ) ;
            sc_assert( ocp->GetDataCl()->SgetMAddrSpace      ()== 1+trans_nb  ) ;
            sc_assert( ocp->GetDataCl()->SgetMAtomicLength   ()== 2+trans_nb  ) ;
            sc_assert( ocp->GetDataCl()->SgetMBurstSeq       ()== (OCPMBurstSeqType) ((3+trans_nb)%8) ) ;
            sc_assert( ocp->GetDataCl()->SgetMByteEn         ()== 4+trans_nb  ) ;

            if (!read_cycle) {
                sc_assert( ocp->GetDataCl()->SgetMCmd            ()== OCP_MCMD_WR ) ;
            } else {
                sc_assert( ocp->GetDataCl()->SgetMCmd            ()== OCP_MCMD_RD ) ;
            }
            sc_assert( ocp->GetDataCl()->SgetMConnID         ()== 6+trans_nb  ) ;
            sc_assert( ocp->GetDataCl()->SgetMDataInfo       ()== 7+trans_nb  ) ;
            sc_assert( ocp->GetDataCl()->SgetMThreadID       ()== 8+trans_nb  ) ;
            sc_assert( ocp->GetDataCl()->SgetMBurstLength    ()== 9+trans_nb  ) ;
            sc_assert( ocp->GetDataCl()->SgetMBurstPrecise   ()== (10+trans_nb)%2 ) ;
            sc_assert( ocp->GetDataCl()->SgetMBurstSingleReq ()== (11+trans_nb)%2 ) ;
            sc_assert( ocp->GetDataCl()->SgetMReqInfo        ()== 12+trans_nb ) ;
            sc_assert( ocp->GetDataCl()->SgetMReqLast        ()== (13+trans_nb)%2 ) ;

            // If Write request, check the received data
            if (!read_cycle) {
                read_cycle = 1; // Toggle write and read cycles

                // First Method
                unsigned int w;
                bool last_of_a_burst;
                data=ocp->GetDataCl()->SgetMData(w,last_of_a_burst); 
                sc_assert(w==10);

                if(shift==60) 
                    sc_assert(last_of_a_burst==true);
                else
                    sc_assert(last_of_a_burst==false);

                for ( unsigned int i=0; i<10 ;i++  )
                { sc_assert(data[i]==shift+i); }


                if (shift<90)
                    shift += 10;
                else
                    shift = 0;

#ifdef DEBUG_C
                cout << "Slave write request finished";
                cout << ", data = ";
                for (int i=0;i<10;i++)
                    cout << data[i] <<",";
                cout << "  at time  = " << sc_time_stamp().to_seconds() << endl;
#endif

            } else 
                // Read Request
            {
                read_cycle = 0; // Toggle write and read cycles

                sc_assert(ocp->GetDataCl()->SgetMReqChunkLen()==10); 
                sc_assert(ocp->GetDataCl()->SgetMReqChunkLast()==(trans_nb%3)==0); 

                response_event.notify(sc_time(100, SC_MS));

#ifdef DEBUG_C
                cout << "Slave read request finished";
                cout << "  at time  = " << sc_time_stamp().to_seconds() << endl;
#endif

            }


            trans_nb++;

        }
    }

    // ----------------------------------------------------------------------------
    // send all the response signals to the master
    // ----------------------------------------------------------------------------
    void response_thread () {
        int trans_nb=0;

        while (true) {

            wait (response_event);

            if ( trans_nb%2 == 0 )
            {
                ocp->GetDataCl()->SputSData(data, 10,true );
            }
            else
            {
                ocp->GetDataCl()->SputSData(data, 5, true );   // Bad values forced
                // Overwrites 'SDataChunkLen' and 'SDataChunkLast' values to test the
                // following functions :
                ocp->GetDataCl()->SputSRespChunkLen(10);
                ocp->GetDataCl()->SputSRespChunkLast(false);
            }
            ocp->GetDataCl()->SputSDataInfo(0+trans_nb);
            ocp->GetDataCl()->SputSResp((OCPSRespType)((1+trans_nb)%4));
            ocp->GetDataCl()->SputSRespInfo(2+trans_nb);
            ocp->GetDataCl()->SputSRespLast((3+trans_nb)%2);
            ocp->GetDataCl()->SputSThreadID(4+trans_nb);

            sc_assert(ocp->SputResponseBlocking());

#ifdef DEBUG_C
            cout << "Slave read response finished";
            cout << ", data = ";
            for (int i=0;i<10;i++)
                cout << data[i] <<",";
            cout << "  at time  = " << sc_time_stamp().to_seconds() << endl;
#endif

            trans_nb++;
        }
    }

    // ----------------------------------------------------------------------------
    // send all the slave sideband signals to the master
    // ----------------------------------------------------------------------------
    void put_slave_sb_thread() {
        int trans_nb = 0;

        while(1) {

            wait(slave_sb_start_event);

            // 'SReset_n' test
            wait(sc_time(5, SC_MS));
            ocp->GetDataCl()->SputSReset_n(trans_nb%2 == 0);

            // 'SFlag' test
            wait(sc_time(10, SC_MS));
            ocp->GetDataCl()->SputSFlag(trans_nb+1);

            // 'SError' test
            wait(sc_time(10, SC_MS));
            ocp->GetDataCl()->SputSError((trans_nb+2)%2);

            // 'SInterrupt' test
            wait(sc_time(10, SC_MS));
            ocp->GetDataCl()->SputSInterrupt((trans_nb+3)%2);

            // 'SThreadBusy' test
            wait(sc_time(10, SC_MS));
            ocp->GetDataCl()->SputSThreadBusy(trans_nb%2 == 0);

            // 'CputStatus' test
            wait(sc_time(10, SC_MS));
            ocp->GetDataCl()->CputStatus(trans_nb + 4);

            // 'CputStatusBusy' test
            wait(sc_time(10, SC_MS));
            ocp->GetDataCl()->CputStatusBusy(trans_nb%2 == 0);

            // 'CputControlBusy' test
            wait(sc_time(10, SC_MS));
            ocp->GetDataCl()->CputControlBusy(trans_nb%2 == 0);

#ifdef DEBUG_C
            cout  << "put_slave_sb_thread test passed... " << trans_nb << "\n" ;
#endif

            trans_nb ++ ;
        }
    }

    // ----------------------------------------------------------------------------
    // check all the received sideband signals from the master
    // ----------------------------------------------------------------------------
    void get_master_sb_thread() {
        unsigned int trans_nb = 0;

        while(1) {

            wait(master_sb_start_event);

            // 'MReset_n' test
            wait(ocp->GetDataCl()->SBResetEvent);
            sc_assert(ocp->GetDataCl()->SgetMReset_n() == (trans_nb%2 == 0));

            // 'MFlag' test
            wait(ocp->GetDataCl()->SBMasterEvent);
            sc_assert(ocp->GetDataCl()->SgetMFlag()==trans_nb+1);

            // 'MError' test
            wait(ocp->GetDataCl()->SBMasterEvent);
            sc_assert(ocp->GetDataCl()->SgetMError()==trans_nb%2);

            // 'MThreadBusy' test
            wait(ocp->GetDataCl()->SBMThreadBusyEvent);
            sc_assert(ocp->GetDataCl()->SgetMThreadBusy()== (trans_nb%2 == 0));

            // 'SysputControlWr' test
            wait(ocp->GetDataCl()->SBControlEvent);
            sc_assert(ocp->GetDataCl()->CgetControlWr()== (trans_nb%2 == 0));

            // 'SysputControl' test
            wait(ocp->GetDataCl()->SBControlEvent);
            sc_assert(ocp->GetDataCl()->CgetControl()== (trans_nb+3));

            // 'SysputStatusRd' test
            wait(ocp->GetDataCl()->SBStatusEvent);
            sc_assert(ocp->GetDataCl()->CgetStatusRd()== (trans_nb%2 == 0));

#ifdef DEBUG_C
            cout  << "get_master_sb_thread test passed... " << trans_nb << "\n" ;
#endif

            trans_nb++ ;
        }
    }

    // ----------------------------------------------------------------------------
    // Synchronizes 'master sideband' and 'slave sideband' communication phases
    // ----------------------------------------------------------------------------
    void sync_sb() {  
        while(true) {
            master_sb_start_event.notify();
            wait(sc_time(1000,SC_MS));
            slave_sb_start_event.notify();
            wait(sc_time(1000,SC_MS));
        }
    }


    // ----------------------------------------------------------------------------
    // Constructor 
    // ----------------------------------------------------------------------------
    SC_CTOR(slave) {
        SC_THREAD(put_slave_sb_thread);
        SC_THREAD(get_master_sb_thread);
        SC_THREAD(request_thread);
        SC_THREAD(response_thread);
        SC_THREAD(sync_sb);
    }
    sc_event slave_sb_start_event, master_sb_start_event;
};



// ----------------------------------------------------------------------------
// MAIN PROGRAM
// ----------------------------------------------------------------------------

int sc_main(int, char*[])
{

    bool sync = true;
    TL_Channel<OCP_TL2_DataCl<unsigned int, unsigned int> >  ch0("ch0", sync);

    slave sl1("sl1");
    master ms1("ms1");

    ms1.ocp(ch0);

    sl1.ocp(ch0);

    sc_start(20000, SC_MS);

    cout << endl << "DATA CLASS Regression test PASSED...." << endl;

    return(0);
}

