////////////////////////////////////////////////////////////////////////
//                                                                    //
//                                                                    //
// (c) Copyright OCP-IP 2003                                          //
// OCP-IP Confidential and Proprietary                                //
//                                                                    //
// Joe Chou, Sonics Inc., joechou@sonicsinc.com                       //
// Alan Kamas, for Sonics Inc., aok@sonicsinc.com, www.kamas.com      //
//                                                                    //
//                                                                    //
////////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include "SlaveSysC.h"
#include "ocp_globals.h"

// ----------------------------------------------------------------------------
// constructor
// ----------------------------------------------------------------------------
template<typename TdataCl>
SlaveSysC<TdataCl>::SlaveSysC(
    sc_module_name n,
    double         ocp_clock_period,
    sc_time_unit   ocp_clock_time_unit,
    int            id,
    Ta             memory_byte_size,
    ostream*       debug_os_ptr
) : sc_module(n),
    tpP("tpPort"),
    m_ID(id),
    m_ocpClkPeriod(ocp_clock_period),
    m_ocpClkTimeUnit(ocp_clock_time_unit),
    m_MemoryByteSize(memory_byte_size),
    m_requestQ(NULL),
    m_concurrentDataWaiting(false),
    m_writeReqAccepts(NULL),
    m_RespQueuePtr(NULL),
    m_Memory(NULL),
    m_debug_os_ptr(debug_os_ptr),
    m_curSThreadBusy(0),
    m_Reset(false),
    m_OCPParamP(NULL),
    m_threads(1),
    m_datahandshake(false),
    m_dataaccept(true),
    m_writeresp_enable(false),
    m_sthreadbusy(false),
    m_sthreadbusy_exact(false),
    m_mthreadbusy(false),
    m_cmdaccept(true),
    m_byteen(false),
    m_mdatabyteen(false),
    m_addrIncr(1),
    m_limitreq_enable(1),
    m_limitreq_max(4),
    m_Latency(NULL)
{
    // Note: member variables that depend on values of configuration parameters
    //       are constructed when those values are known - at the end of elaboration.

    // handles a reset from the channel
    SC_METHOD(slaveResetMethod);
    sensitive << tpP.ResetStartEvent();
    dont_initialize();

    // setup a SystemC thread process, which uses dynamic sensitive
    SC_THREAD(requestThreadProcess);

    // setup a SystemC thread process, which uses dynamic sensitive
    SC_THREAD(responseThreadProcess);

    // setup a SystemC thread process to handle incoming data handshakes (if any)
    SC_THREAD(dataHSThreadProcess);

    // setup a SystemC thread process to check and set sideband signals
    SC_THREAD(exerciseSidebandThreadProcess);

    // setup a SystemC thread to test sReset_n
    SC_THREAD(exerciseSResetProcess);
}

// --------------------------------------------------------------------------
// destructor
// --------------------------------------------------------------------------
template<typename TdataCl>
SlaveSysC<TdataCl>::~SlaveSysC()
{
    delete m_Memory;
    delete m_RespQueuePtr;
    delete[] m_Latency;
    delete[] m_writeReqAccepts;
    delete[] m_requestQ;
}

// --------------------------------------------------------------------------
//  SystemC Method SlaveSysC::end_of_elaboration()
// --------------------------------------------------------------------------
// 
//  At this point, everything has been built and connected.
//  We are now free to get our OCP parameters and to set up our
//  own variables that depend on them.
//
template<typename TdataCl>
void SlaveSysC<TdataCl>::end_of_elaboration()
{
    sc_module::end_of_elaboration();

    /////////////
    //
    // Process OCP Parameters from the port
    //
    /////////////

    m_OCPParamP = tpP->GetParamCl();

    // Set the number of threads
    m_threads = m_OCPParamP->threads;

    // Use the Threads value to set up our state that depends on it
    if (m_Latency) {
        delete[] m_Latency;
    }
    m_Latency = new int[m_threads];

    if (m_RespQueuePtr) {
        delete m_RespQueuePtr;
    }
    m_RespQueuePtr = new ThreadedRespQ<TdataCl>(m_threads);

    if (m_writeReqAccepts) {
        delete[] m_writeReqAccepts;
    }
    m_writeReqAccepts = new int[m_threads];
    for (int i=0; i<m_threads; i++) {
        m_writeReqAccepts[i]=0;
    }

    if (m_requestQ) {
        delete[] m_requestQ;
    }
    m_requestQ = new deque<ReqQElement>[m_threads];

    // Does the channel use data handshaking?
    m_datahandshake = m_OCPParamP->datahandshake;

    // Does the channel use data handshake accept?
    m_dataaccept = m_OCPParamP->dataaccept;

    // Do writes get reponses?
    m_writeresp_enable = m_OCPParamP->writeresp_enable;

    // is SThreadBusy part of the channel?
    m_sthreadbusy = m_OCPParamP->sthreadbusy;

    // is this slave expected to follow the threadbusy exact protocol?
    m_sthreadbusy_exact = m_OCPParamP->sthreadbusy_exact;

    // is MThreadBusy part of the channel?
    m_mthreadbusy = m_OCPParamP->mthreadbusy;

    // is SCmdAccept part of the channel?
    m_cmdaccept = m_OCPParamP->cmdaccept;

    // Is Byte Enable part of the channel?
    m_byteen = m_OCPParamP->byteen;

    // Is Data Byte Enable part of the channel?
    m_mdatabyteen = m_OCPParamP->mdatabyteen;

    // Data must be byte sized
    assert( (m_OCPParamP->data_wdth)%8 == 0 );
    // Amount to increment address for each incrementing burst
    m_addrIncr = (m_OCPParamP->data_wdth) / 8;

    // We only support precise bursts with specific lengths
    if (m_OCPParamP->burstseq) {
        assert(m_OCPParamP->burstprecise);
        assert(m_OCPParamP->burstlength);
    }

    /////////////
    //
    // Verify that this channel configuration is supported by this Slave
    //
    /////////////

    assert( !((m_datahandshake==true) && (m_dataaccept==false) && (m_cmdaccept==true)) );

    /////////////
    //
    // Process Slave Parameters
    //
    /////////////

    // For Debugging
    if (m_debug_os_ptr) {
        (*m_debug_os_ptr) << "DB (" << name() << "): "
            << "Configuring Slave." << endl;
        (*m_debug_os_ptr) << "DB (" << name() << "): was passed the following configuration map:" << endl;
        MapStringType::iterator map_it;
        for (map_it = m_ParamMap.begin(); map_it != m_ParamMap.end(); ++map_it) {
            (*m_debug_os_ptr) << "map[" << map_it->first << "] = " << map_it->second << endl;
        }
        cout << endl;
    }
    
    // Currently, myPrefix is not used. Retained for compatibility
    // with various parameter map formats.
    string myPrefix = "";
    string paramName = "undefined";
    char buffer[256];

    // latency(0), latency(1), ... , latency(N)
    // or latency0, latency1, latency2, latencyN
    for (int i=0; i<m_threads; i++) {
        // First try the old style:
        sprintf(buffer,"latency(%d)",i);
        paramName = buffer;
        if (!(ParamCl<TdataCl>::getIntOCPConfigValue(myPrefix, paramName, m_Latency[i], m_ParamMap,name())) ) {
            // Could not find it with parenthesis. Try without
            sprintf(buffer,"latency%d",i);
            paramName = buffer;
            if (!(ParamCl<TdataCl>::getIntOCPConfigValue(myPrefix, paramName, m_Latency[i], m_ParamMap, name())) ) {
                // Could not find the parameter so we must set it to a default
#ifndef NDEBUG 
                cout << "Warning: paramter \"" << paramName << "\" for Slave \"" << name() << "\" was not found in the parameter map." << endl;
                cout << "         setting missing parameter to 3." << endl;
#endif
                m_Latency[i] = 3;
            }
        }
    }

    // limitreq_enable
    paramName = "limitreq_enable";
    if (!(ParamCl<TdataCl>::getBoolOCPConfigValue(myPrefix, paramName, m_limitreq_enable, m_ParamMap, name())) ) {
        // Could not find the parameter so we must set it to a default
#ifndef NDEBUG 
        cout << "Warning: paramter \"" << paramName << "\" for Slave \"" << name() << "\" was not found in the parameter map." << endl;
        cout << "         setting missing parameter to false." << endl;
#endif
        m_limitreq_enable = false;
    }

    // limitreq_max
    paramName = "limitreq_max";
    if (!(ParamCl<TdataCl>::getIntOCPConfigValue(myPrefix, paramName, m_limitreq_max, m_ParamMap, name())) ) {
        // Could not find the parameter so we must set it to a default
#ifndef NDEBUG 
        cout << "Warning: paramter \"" << paramName << "\" for Slave \"" << name() << "\" was not found in the parameter map." << endl;
        cout << "         setting missing parameter to 4." << endl;
#endif
        m_limitreq_max = 4;
    }

    // Still have a limit on requests even if limitreq_enable is false to avoid
    // running out of memory.
    if (!m_limitreq_enable) {
        m_limitreq_max = 255;
    }

    /////////////
    //
    // Initialize the Slave with the New Parameters
    //
    /////////////

    // Create the memory:
    if (m_Memory) {
        // Just in case we are called multiple times.
        delete m_Memory;
    }
    char id_buff[10];
    sprintf(id_buff,"%d",m_ID);
    string my_id(id_buff);
    m_Memory = new MemoryCl<TdataCl>(my_id,m_OCPParamP->addr_wdth,sizeof(Td));
    
}

// --------------------------------------------------------------------------
// Reset State
// --------------------------------------------------------------------------
template<typename TdataCl>
void SlaveSysC<TdataCl>::resetState(void)
{
    // reset the request queue
    m_requestQ->clear();

    // No data waiting
    m_concurrentDataWaiting = false;

    // Tracking to make sure data is not accepted before the corresponding request
    for (int i=0; i<m_threads; i++) {
        m_writeReqAccepts[i]=0;
    }

    m_Memory->clear();

    m_curSThreadBusy = 0;
}

// --------------------------------------------------------------------------
// SetConfiguration
// --------------------------------------------------------------------------
template<typename TdataCl>
void SlaveSysC<TdataCl>::setConfiguration( MapStringType& passedMap )
{
    // would be nice to process the Map immediately to avoid copying it and
    // storing it, but some of the parameters, such as latency(x) are based
    // on the number of threads and that number is based on an OCP parameter.
    //
    // As long as Slave parameters are functions of OCP parameters, they will
    // have to be processed at the end of elaboration when the OCP parameters
    // are available.
    //
    // Save the Map until then
    m_ParamMap = passedMap;
}

// --------------------------------------------------------------------------
// SystemC Thread Process
// --------------------------------------------------------------------------
//

template<typename TdataCl>
void SlaveSysC<TdataCl>::slaveResetMethod()
{
    if (m_Reset) {

        if (m_debug_os_ptr) {
            (*m_debug_os_ptr) << "DB (" << name () << "): " << " exiting RESET at time " << sc_simulation_time() << endl;
        }

        // We were already in reset. Now we need to pull out of reset.
        assert(!(tpP->getReset()));  // Make sure channel really is out of reset
        // pull slave objects out of reset
        m_RespQueuePtr->endReset();
        // reset all state
        resetState();
        // unset the reset flag
        m_Reset = false;
    } else {

        if (m_debug_os_ptr) {
            (*m_debug_os_ptr) << "DB (" << name () << "): " << " entering RESET at time " << sc_simulation_time() << endl;
        }

        // We were not in reset - go into reset now.
        // set the reset flag
        m_Reset = true;
        // call reset on the slave objects
        m_RespQueuePtr->startReset();
        // reset all state
        resetState();
        // re-trigger for the end of reset event
        next_trigger(tpP->ResetEndEvent());
    }
}

template<typename TdataCl>
void SlaveSysC<TdataCl>::requestThreadProcess()
{
    // The new request we have just received
    OCPRequestGrp<Td,Ta> req;

    // Time after which the response can be sent or this request can be cleared from outgoing queue
    // so that the items behind it can be sent out.
    sc_time          send_time;

    // Set the initial value of theadbusy:
    m_curSThreadBusy = 0;
    if (m_sthreadbusy) {
        tpP->putNextSThreadBusy (m_curSThreadBusy);
    }

    // We are in the initialization call. Wait for the first simulation cycle.
    tpP->ocpWait();

    if (m_debug_os_ptr) {
        (*m_debug_os_ptr) << "DB (" << name () << "): " << " starting requestThreadProcess at time " << sc_simulation_time() << endl;
    }

    // New item to queue for request processing
    ReqQElement newRequest;

    // Number of Data/Responses associated with this request
    int dataCount =1;

    // main loop
    while (true) {

        if (m_debug_os_ptr) {
            (*m_debug_os_ptr) << "DB (" << name () << "): " << " starting requestThreadProcess loop. m_Reset = " << m_Reset << " at time " << sc_simulation_time() << endl;
        }

        if (m_Reset) {
            // We are in reset.
            // Reset my local state
            dataCount = 1;
            // wait to come out of reset
            // TODO: May want to add a slaveResetEndEvent 
            wait(tpP->ResetEndEvent());
            wait(SC_ZERO_TIME);
        }

        while (!m_Reset) {        

            // ----------------------------------------------------------
            // (1) Get the next request
            // ----------------------------------------------------------

            tpP->getOCPRequestBlocking(req,false);

            if (m_debug_os_ptr) {
                (*m_debug_os_ptr) << "DB (" << name() << "): " << " got new request." << endl;
                (*m_debug_os_ptr) << "DB (" << name() << "): " << "    t = " << sc_simulation_time() << " MCmd: " << req.MCmd << " MThreadID: " << req.MThreadID << endl;
                (*m_debug_os_ptr) << "DB (" << name() << "): " << "    MBurstSeq = " << req.MBurstSeq << " MBurstLength: " << req.MBurstLength << " MBurstSingleReq: " << req.MBurstSingleReq << endl;
            }

            // ----------------------------------------------------------
            // (2) if our queue is full, generate back pressure halt
            //     the flow of requests. Otherwise, accept the request
            //     and move on.
            // ----------------------------------------------------------

            int myThread = req.MThreadID;
            unsigned int threadBit = 1 << myThread;

            // Do we need to set SThreadBusy??
            if (m_sthreadbusy && (int (m_RespQueuePtr->length(myThread) + m_requestQ[myThread].size()) >= m_limitreq_max)) {
                m_curSThreadBusy = m_curSThreadBusy | threadBit;

                if (m_debug_os_ptr) {
                    (*m_debug_os_ptr) << "DB (" << name () << "): " << " for Thread#" << myThread << " setting sthreadbusy = " << m_curSThreadBusy << " (queue size is " << int(m_RespQueuePtr->length(myThread) + m_requestQ[myThread].size()) << ")" << endl;
                }
                tpP->putNextSThreadBusy (m_curSThreadBusy);
            }

            // Should we accept this command?
            if ( m_cmdaccept ) {
                // if queue is full, delay accepting request
                while ( int(m_RespQueuePtr->length(myThread) + m_requestQ[myThread].size() ) >= m_limitreq_max) {
                    // Our queue is full. Wait for this to change.

                    if (m_debug_os_ptr) {
                        (*m_debug_os_ptr) << "DB (" << name () << "): "
                            << " request queue " << myThread << " full at time " << sc_simulation_time() << ": waiting for it to lower." << endl;
                        (*m_debug_os_ptr) << "DB (" << name () << "): " << " m_RespQueuePtr->length("<<myThread<<") = "<< m_RespQueuePtr->length(myThread) << " m_requestQ[" << myThread << "].size() = " << m_requestQ[myThread].size() << endl;
                    }

                    tpP->ocpWait();
                    // The slave may have reset while waiting
                    if (m_Reset) { 
                        // we have reset - get out of the loop
                        break; 
                    }
                }
                // now it is okay to accept the request

                if (m_debug_os_ptr) {
                    (*m_debug_os_ptr) << "DB (" << name () << "): accepts request at time " << sc_simulation_time() << endl;
                }
                tpP->putSCmdAccept();
            }

            if (m_dataaccept) {
                // request was just accepted
                // The data can now be accepted if this was a write request
                if ((req.MCmd == OCP_MCMD_WR) || (req.MCmd == OCP_MCMD_WRNP) || (req.MCmd == OCP_MCMD_WRC) || (req.MCmd == OCP_MCMD_BCST)) {
                    // This was a write request. It must have had data with it.
                    if (m_writeReqAccepts[myThread] == -1) {
                        // A data handshake is already waiting to be accepted. Accept it now.
                        if (m_debug_os_ptr) {
                            (*m_debug_os_ptr) << "DB (" << name () << "): waited to accept data at time " << sc_simulation_time() << endl;
                        }
                        tpP->putSDataAccept();
                    }
                    // Increase our count of how far we are ahead.
                    if (req.MBurstSingleReq) {
                        m_writeReqAccepts[myThread] += req.MBurstLength;
                    } else {
                        m_writeReqAccepts[myThread]++;
                    }
                    if (m_debug_os_ptr) {
                        (*m_debug_os_ptr) << "DB (" << name () << "): Incrementing writeReqAccepts for thread " << myThread << " to: " << m_writeReqAccepts[myThread] << " at time " << sc_simulation_time() << endl;
                    }
                } 
            } 

            // ----------------------------------------------------------
            // (3) send the new request over to the processing queue to
            //     wait for data handshake (if any)
            // ----------------------------------------------------------

            // Expand the request into multiple requests if 
            // Single request / Multiple data is used

            if (req.MBurstSingleReq) {
                dataCount=req.MBurstLength;
                assert(dataCount>0);
                assert(req.MBurstSeq == OCP_MBURSTSEQ_INCR);
            } else {
                dataCount = 1;
            }

            // A single request could expand into a set of requests
            // if the channel has single request/multiple data on.
            for (int count=1; count <= dataCount; count++) {

                // compute the address of the request
                req.MAddr += (count-1) * m_addrIncr;

                // compute pipelined response delay
                send_time = sc_time_stamp() + sc_time(count * m_Latency[myThread], m_ocpClkTimeUnit);

                // If write with response is on, only send a response for the last request of the set.
                if (m_writeresp_enable) {
                    if (count != dataCount) {
                        // Not the last one - do not send response
                        if (req.MCmd == OCP_MCMD_WRNP) {
                            req.MCmd = OCP_MCMD_WR;
                        }
                    } else {
                        // Is the last one - do send response
                        if ((req.MCmd == OCP_MCMD_WR) || (req.MCmd == OCP_MCMD_WRC) || (req.MCmd == OCP_MCMD_BCST) ) {
                            req.MCmd = OCP_MCMD_WRNP;
                        }
                    }
                }

                // Queue the request to wait for data handshake.
                // If no data handshake - process immediately.
                if (m_datahandshake) {
                    // Our channel gets data through data handshake - requests will not have it.
                    if ( (req.MCmd == OCP_MCMD_WR) || (req.MCmd == OCP_MCMD_WRNP) || (req.MCmd == OCP_MCMD_WRC) || (req.MCmd == OCP_MCMD_BCST) ) {
                        // This is a "write" type command and data will be coming.
                        if (m_concurrentDataWaiting) {
                            // Data Handshake came at same time as request - use it now
                            assert(req.MThreadID == m_dataHeld.MDataThreadID);
                            req.HasMData = true;
                            req.MData = m_dataHeld.MData;
                            req.MDataInfo = m_dataHeld.MDataInfo;
                            // use MDataByteEn for writes
                            if (m_mdatabyteen) {
                                req.MByteEn = m_dataHeld.MDataByteEn;
                            } else {
                                req.MByteEn = 0xFFFFFFFF;
                            }
                            m_concurrentDataWaiting = false;
                        } else {
                            // Data Handshake will come later
                            req.HasMData = false;
                        }
                    } else {
                        // This is a "read" type command. No data will be coming
                        // and it's an error if data came with this request.
                        assert(m_concurrentDataWaiting == false);
                        req.HasMData = true;
                        // use MByteEn for reads
                        if (!m_byteen) {
                            req.MByteEn = 0xFFFFFFFF;
                        }
                    }
                    // Add element to the back of the queue of requests waiting to be processed.
                    newRequest.request = req;
                    newRequest.respSendTime = send_time;

                    // DEBUG logging
                    if (m_debug_os_ptr) {
                        (*m_debug_os_ptr) << "DB (" << name () << "): putting request on wait for data queue. timeNow = " << sc_simulation_time() << " send_time= " << send_time << endl;
                    }

                    m_requestQ[myThread].push_back(newRequest);
                    // Process the request Q in case the request can be used immediately.
                    processReqQ();
                } else {
                    // No data handshake. All requests should already have data and can be processed immediately.
                    // No data handshake: use MByteEn for reads & writes
                    if (!m_byteen) {
                        req.MByteEn = 0xFFFFFFFF;
                    }
                    processRequest(req, send_time);
                }
            }
        }
    }
}

template<typename TdataCl>
void SlaveSysC<TdataCl>::processRequest( OCPRequestGrp<Td,Ta>& req, sc_time& send_time)
{
    // The response to the new request
    OCPResponseGrp<Td>   resp;

    // ----------------------------------------------------------
    // (1) process the Request
    // ----------------------------------------------------------

    // compute the word address

    // Simple memory wrapping
    if (req.MAddr >= m_MemoryByteSize) {
        req.MAddr = req.MAddr - m_MemoryByteSize;
    }

    // ----------------------------------------------------------
    // (2) Write data to memory or read data from memory 
    // ----------------------------------------------------------

    // write to or read from the memory
    switch (req.MCmd) {
        case OCP_MCMD_WR:
        case OCP_MCMD_BCST:
            // posted write to memory
            m_Memory->write(req.MAddr,req.MData,req.MByteEn);

            // note that posted writes do not have responses. However,
            // they do have a processing delay that can contribute to
            // a max request limit back up.
            // To solve this problem, requests that have no response
            // generate a dummy respose with SRESP=NULL which is defined
            // as "No response".
            // Dummy responses are never sent out on the channel.
            resp.SResp = OCP_SRESP_NULL;
            resp.SThreadID = req.MThreadID;
            break;

        case OCP_MCMD_RD:
        case OCP_MCMD_RDEX:
            // NOTE that for a single threaded slave, Read-EX works just like Read 
            // read from memory
            m_Memory->read(req.MAddr,resp.SData,req.MByteEn);
            // setup a read response
            resp.SResp = OCP_SRESP_DVA;
            resp.SThreadID = req.MThreadID;
            break;

        case OCP_MCMD_WRNP:
            // posted write to memory
            m_Memory->write(req.MAddr,req.MData,req.MByteEn);

            // Generate an acknowledgement response
            resp.SResp = OCP_SRESP_DVA;
            resp.SThreadID = req.MThreadID;
            resp.SData = 0;
            break;

        default:
            cout << "ERROR: SlaveSysC - MCmd #" << req.MCmd << " not supported yet." << endl;
            sc_stop();
            break;
    }

    // ----------------------------------------------------------
    // (3) Add response to the response queue
    // ----------------------------------------------------------

    // purge the queue of any posted write place holder responses
    // that have reached their send times
    m_RespQueuePtr->purgePlaceholders();
    freeSThreadBusy();

    m_RespQueuePtr->enqueueBlocking(resp, send_time);
}

// remove and process any requests that now have their data (or that were
// waiting for the requests in front to get their data).
template<typename TdataCl>
void SlaveSysC<TdataCl>::processReqQ(void)
{
    for (int i=0; i<m_threads; i++) {
        while ( (!m_requestQ[i].empty()) && (m_requestQ[i].front().request.HasMData) ) {
            // This request has data. Process it.
            processRequest( m_requestQ[i].front().request, m_requestQ[i].front().respSendTime);
            // Request has been processed. Remove from queue.
            m_requestQ[i].pop_front();
        }
    }
}

template<typename TdataCl>
void SlaveSysC<TdataCl>::dataHSThreadProcess(void)
{
    // The new data we have just received
    OCPDataHSGrp<Td> dataHs;

    // Was the new data able to be placed into a request?
    bool dataUsed = false;

    // Iterator for moving accoss the requestQ
    typename std::deque<ReqQElement>::iterator pos; 

    // We are in the initialization call. Wait for the first simulation cycle.
    tpP->ocpWait();

    // main loop
    while (true) {

        if (m_Reset) {
            // We are in reset.
            // Reset my local state
            dataUsed = false;
            // wait to come out of reset
            // TODO: May want to add a slaveResetEndEvent 
            wait(tpP->ResetEndEvent());
            wait(SC_ZERO_TIME);
            wait(SC_ZERO_TIME);
        }

        while (!m_Reset) {

            // ----------------------------------------------------------
            // (1) Get the new data
            //     Blocks forever if no data handshake
            // ----------------------------------------------------------

            // Always accept new data - use cmdaccept and mthreadbusy for backflow
            // This slave does not use SDataThreadBusy

            if (m_debug_os_ptr) {
                (*m_debug_os_ptr) << "DB (" << name() << "): "
                    << " waiting for data handshake." << endl;
            }

            // Cannot accept data until after request accepted
            tpP->getOCPDataHSBlocking(dataHs, false);
            // TODO: may be able to remove this catch if there is no harm in letting it fall through
            if (m_Reset) {
                // Slave went into reset during the blocking call
                break;
            }

            if (m_debug_os_ptr) {
                (*m_debug_os_ptr) << "DB (" << name() << "): "
                    << " got data handshake." << endl;
                (*m_debug_os_ptr) << "DB (" << name() << "): "
                    << "    t = " << sc_simulation_time() << endl;
                (*m_debug_os_ptr) << "DB (" << name() << "): "
                    << "    MData: " << dataHs.MData << endl;
            }

            // ----------------------------------------------------------
            // (2) use the new data to complete a request
            // ----------------------------------------------------------
            dataUsed = false; 
            int myThread = dataHs.MDataThreadID;
            for ( pos = m_requestQ[myThread].begin(); pos != m_requestQ[myThread].end(); ++pos ) {
                // Look for a request on the same thread that needs data
                if ( (!pos->request.HasMData) && (pos->request.MThreadID == dataHs.MDataThreadID) ) {
                    // A spot has been found. Fill in the data.
                    dataUsed = true;
                    pos->request.MData = dataHs.MData;
                    pos->request.MDataInfo = dataHs.MDataInfo;
                    if (m_mdatabyteen) {
                        pos->request.MByteEn = dataHs.MDataByteEn;
                    } else {
                        pos->request.MByteEn = 0xFFFFFFFF;
                    }
                    pos->request.HasMData = true;
                    break;
                }
            }

            // ------------------------------------------------------------
            // (3) If the data has been added to the requestQ - process it.
            // ------------------------------------------------------------

            // If the data was not used, then perhaps this data was sent at the same time
            // as the request and we just happened to get the data first.
            if (!dataUsed) {
                // hold the data for a request this cycle
                // (any concurrent data held for a previous cycle should have already been used.
                assert(m_concurrentDataWaiting == false); 
                m_concurrentDataWaiting = true;
                m_dataHeld = dataHs;
            } else {
                processReqQ();
            }

            //-------------------------------------------------------------
            // (4) Accept Data (if part of channel and timing is right) 
            //-------------------------------------------------------------
            // According to the OCP spec, a datahandshake cannot be accepted until after its
            // corresponing write request has been accepted. (Okay to accept both in same cycle).
            // As a result, we can only accept the data if the write request accepts are ahead
            // of the data handshake requests.
            if (m_dataaccept) {
                // Data Accept is part of the channel
                if (m_writeReqAccepts[myThread] > 0) {
                    // The write request associated with this data has already been accepted.
                    // Okay to accept this data.
                    if (m_debug_os_ptr) {
                        (*m_debug_os_ptr) << "DB (" << name () << "):  accepts data at time " << sc_simulation_time() << endl;
                        (*m_debug_os_ptr) << "DB (" << name () << "): writeReqAccepts is " << m_writeReqAccepts[myThread] << endl;
                    }
                    tpP->putSDataAccept();
                    m_writeReqAccepts[myThread]--;
                } else {
                    // go negative. This means that the data will be accepted when the request is.
                    m_writeReqAccepts[myThread]--;
                }
                // Can only be one acceptance in the hole. Any more is an error.
                assert(m_writeReqAccepts[myThread] > -2);
            }
        }
    }
}


template<typename TdataCl>
void SlaveSysC<TdataCl>::freeSThreadBusy(void)
{
    // Checks to see if SThreadBusy can be freed, and then
    // releases it if it can

    // No point to this rountine if SThreadBusy no part of the channel.
    if (!m_sthreadbusy) {
        return;
    }

    for (int i=0; i<m_threads; i++) {
        unsigned int threadBit = 1 << i;
        if ( (m_curSThreadBusy & threadBit) && ( int(m_RespQueuePtr->length(i) + m_requestQ[i].size() ) < m_limitreq_max) ) {
            // This thread's queue has been shortened. Clear threadBusy.
            m_curSThreadBusy &= (~threadBit);
            tpP->putNextSThreadBusy(m_curSThreadBusy);

            if (m_debug_os_ptr) {
	        (*m_debug_os_ptr) << "DB (" << name () << "): " << " freeing Thread#" << i << " by setting sthreadbusy = " << m_curSThreadBusy << endl;

            }
        }
    }
}

template<typename TdataCl>
void SlaveSysC<TdataCl>::responseThreadProcess()
{
    OCPResponseGrp<Td>   resp;
    sc_time          send_time;
    sc_time          CurTime;
    unsigned int     mthreadbusy;

    tpP->ocpWait();

    // main loop
    while (true) {
        if (m_Reset) {
            // We are in reset.
            // Reset my local state
            // wait to come out of reset
            // TODO: May want to add a slaveResetEndEvent 
            wait(tpP->ResetEndEvent());
            // now wait until the slaveReset, requestProcessing and dataProcessing have started
            wait(SC_ZERO_TIME);
            wait(SC_ZERO_TIME);
            wait(SC_ZERO_TIME);
        }
        while (!m_Reset) {   

            // -------------------------------------------------
            // (1) Find a response to place on the channel
            // -------------------------------------------------

            // Get to next response (wait for one, if necessary).

            // First, clear any stale write latency waits
            m_RespQueuePtr->purgePlaceholders();

            // If our queues have dropped, clear threadbusy bits
            freeSThreadBusy();

            // TODO: clear this debug
            if (m_debug_os_ptr) {
                (*m_debug_os_ptr) << "DB (" << name() << "): "
                    << "Waiting for response to send." << endl;
            }

            // Get the next response to send. Aribtration is handled by
            // the ResponseQueue

            m_RespQueuePtr->dequeueBlocking(resp,send_time);

            // TODO: clear this debug
            if (m_debug_os_ptr) {
                (*m_debug_os_ptr) << "DB (" << name() << "): "
                    << "Can send response. SResp = " << resp.SResp << " Data = " << resp.SData << " Send time = " << send_time << endl;
            }

            // check if we still need to wait
            CurTime = sc_time_stamp();
            if (send_time > CurTime) {
                if ( !(tpP->ocpSafeWait((send_time.value() - CurTime.value())/1000))) {
                    // Slave was reset during the wait. Give up.
                    break;
                }
            }

            if (m_debug_os_ptr) {
                (*m_debug_os_ptr) << "DB (" << name() << "): "
                    << "slave wait time = "
                    << send_time.value() << endl;
            }

            // The response could be a place holder response 
            // used to implement write latency. If this is the case,
            // skip the rest of the steps.

            if (resp.SResp == OCP_SRESP_NULL) {
                if (m_debug_os_ptr) {
                    (*m_debug_os_ptr) << "DB (" << name() << "): "
                        << "finished Write Latency waiting." << endl;
                }
            } else {

                // ----------------------------------
                // (2) is MThreadBusy?
                // ----------------------------------

                if (m_mthreadbusy) {
                    mthreadbusy = tpP->getMThreadBusy();
                    int mThreadBit = 1 << resp.SThreadID;
                    while (mthreadbusy & mThreadBit) {
                        if (!(tpP->ocpSafeWait())) {
                            // reset occurred while we were waiting. Bail out.
                            break;
                        }
                        mthreadbusy = tpP->getMThreadBusy();
                    }
                }

                // ----------------------------------
                // (3) return a response
                // ----------------------------------

                if (m_debug_os_ptr) {
                    (*m_debug_os_ptr) << "DB (" << name() << "): "
                        << "send response." << endl;
                    (*m_debug_os_ptr) << "DB (" << name() << "): "
                        << "    t = " << sc_simulation_time() << endl;
                    (*m_debug_os_ptr) << "DB (" << name() << "): "
                        << "    SResp: " << resp.SResp << endl;
                    (*m_debug_os_ptr) << "DB (" << name() << "): "
                        << "    SData: " << resp.SData << endl;
                }

                // Send out the response
                tpP->startOCPResponseBlocking(resp);
            }

            // We must be able to clear ThreadBusy now as we just sent a request (or cleared a write latency)
            freeSThreadBusy();

            // wait until next cycle to send out the next response (if any)
            tpP->ocpWait();
        }
    }
}

// Exercises the sideband signals by setting them with a recurring pattern
// Also loops back error signal from the master if both Master and Slave 
// versions (MError and SError) are configured into the channel 
template<typename TdataCl>
void SlaveSysC<TdataCl>::exerciseSidebandThreadProcess()
{
    // Systematically send out sideband signals on any signals that are attached to us.
    tpP->ocpWait(10);
    int tweakCounter =0;
    bool hasMError = m_OCPParamP->merror;
    bool hasSError = m_OCPParamP->serror;
    bool nextSError = false;
    bool hasSInterrupt = m_OCPParamP->interrupt;
    bool nextSInterrupt = false;
    bool hasSFlag = m_OCPParamP->sflag;
    int numSFlag = m_OCPParamP->sflag_wdth;
    unsigned int nextSFlag = 0;
    unsigned int maxSFlag = (1 << numSFlag) -1; 


    // main loop
    while (true) {
        if (m_Reset) {
            // Channel is in reset.
            // wait until reset is over
            wait(tpP->ResetEndEvent());
            wait(SC_ZERO_TIME);
            // reset my local state
            tweakCounter =0;
            nextSError = false;
            nextSInterrupt = false;
            nextSFlag = 0;
        }

        while (!m_Reset) {

            // wait 10 cycles
            if( !(tpP->ocpSafeWait(10)) ) {
                // The channel went into reset while we waited.
                break;
            }

            // Now count through my sideband changes
            tweakCounter++;

            // Drive SError every time we are called
            if (hasSError) {
                if (hasMError) {
                    // loop MError back through SError
                    nextSError=tpP->SgetMError();
                    tpP->SputSError(nextSError);
                } else {
                    // Toggle SError
                    nextSError = !nextSError;
                    tpP->SputSError(nextSError);
                }
            }

            // Drive SInterrupt
            if (hasSInterrupt) {
                // Drive every other time we are called
                if (tweakCounter%2 == 0) {
                    // Toggle SInterrupt
                    nextSInterrupt = !nextSInterrupt;
                    tpP->SputSInterrupt(nextSInterrupt);
                }
            }

            // Drive SFlag
            if (hasSFlag) {
                // Drive every fourth time we are called
                if (tweakCounter%4 == 0) {
                    nextSFlag += 1;
                    if (nextSFlag > maxSFlag) {
                        nextSFlag = 0;
                    }
                    tpP->SputSFlag(nextSFlag);
                }
            }
        } // end while(!Reset)
    } // end while(true)
}

// Exercises the SReset_n signal.
template<typename TdataCl>
void SlaveSysC<TdataCl>::exerciseSResetProcess()
{
    // Systematically send out sideband signals on any signals that are attached to us.
    tpP->ocpWait(1);
    bool hasSReset = m_OCPParamP->sreset;
        // TODO: remove this debug
        if (m_debug_os_ptr) {
            (*m_debug_os_ptr) << "DB (" << name () << "): " << " Slave Exercise Reset Process has started. SReset = " << hasSReset << endl;
        }
    tpP->ocpWait(500);
    if (hasSReset) {
        // TODO: remove this debug
        if (m_debug_os_ptr) {
            (*m_debug_os_ptr) << "DB (" << name () << "): " << " Slave Exercise Reset Asserting RESET at time = " << sc_simulation_time() << endl;
        }
        tpP->SResetAssert();
    }
    tpP->ocpWait(50);
    if (hasSReset) {
        // TODO: remove this debug
        if (m_debug_os_ptr) {
            (*m_debug_os_ptr) << "DB (" << name () << "): " << " Slave Exercise Reset De-asserting RESET at time = " << sc_simulation_time() << endl;
        }
        tpP->SResetDeassert();
    }
    // That's it. This thread is done.
}
        
// --------------------------------------------------------------------------
//  Method
// --------------------------------------------------------------------------

template<typename TdataCl>
bool SlaveSysC<TdataCl>::MputDirect(
    int MasterID, bool IsWrite, Td *Data, Ta Address, int NumWords)
{
    return (true);
}

#include "ocp_tl1_data_cl.h"

// --------------------------------------------------------------------------
// explicit instantiation of the SlaveSysC template class
// -------------------------------------------------------------------------
template class SlaveSysC< OCP_TL1_SIGNAL_CL >;
