// 
//  Copyright 2003 OCP-IP
//
// ============================================================================
//      Project : OCP SLD WG, Layer adapter examples 
//      Authors : Yann Bajot, PROSILOG, bajot@prosilog.com 
//                Stephane Guntz, PROSILOG, guntz@prosilog.com
//         Date : 01/13/2004
//  Description : Transaction Level - Layer-0 to Layer-1 Master Adapter (OCP 2.0)
//
//     Features :
//
//    - Uses the OCP-specific TL1 Channel and associated API
//
//    - Layer adapter used to connect a TL0 OCP Master to a TL1 system
//
//    - Interface implements OCP Basic signals : Clk, MCmd, MAddr, MData,
//        SCmdAccept, SData, SResp & MRespAccept.
//
//    - Glitches and propagation delays for input signals are removed using two
//        'Sample delay' parameters.  Signals coming from the TL0 master are
//        not supposed to be clocked and therefore can change at any time
//        during the clock cycle. It is also possible that some glitches appear
//        on them, particulary in the case of co-simulation with a
//        HDL-simulated RTL Master. 
//
//      Once a TL1 interface call has been done, it can not be cancelled. The
//      master adapter issues two kinds of call to the TL1 channel:
//          *startOCPRequest()
//          *putMRespAccept() 
//
//      The 'startOCPRequest()' and associated API calls must happen only
//        when TL0 associated signals 'MCmd', 'MData' and 'MAddr' are stable.
//        Since the propagation delays for these signals are master dependant,
//        a 'Request Sample Delay' parameter is used to determine when these
//        signals should be sampled during the cycle. Once sampled, 'MCmd' is
//        tested and corresponding 'startOCPRequest()' call is issued to the TL1
//        channel.
//
//      The 'putMRespAccept()' call must happen only when 'MRespAccept' is stable.
//        Since the propagation delay for this signal is master dependant, a
//        'RespAcceptSampleDelay' parameter is used to determine when
//        'MRespAccept' should be sampled during the cycle.
//
//      The smallest value for the sample delays is sc_get_time_resolution() ;
//      it should be used in the case when TL0 master has pure 'zero-delay'
//      signals. Users should always use a timed sample delay (and not a
//      defined number of delta-cycles) since it is not possible to predict the
//      number of delta-cycles required for signal stability.
//        
//      'Check_setup_time()' function is proposed for debugging purposes, to
//        ensure that TL0 master output signals don't change after the sample
//        delays, which would cause the adapter to fail. A good way to proceed
//        is to perform the first simulation with checking function actived to
//        determine the appropriate sample delay value.  Once TL0 behaviour is
//        correct, checking function should be disabled to increase simulation
//        speed.
//                
// Parameters   : 
//   Template arguments.
//   - TdataCl: Data class containing data members and
//              access methods for the data exchanged between
//              Masters and Slaves
//   - Td: Data type
//   - Ta: Address type
//                
//    First Constructor arguments (with explicit sampling times):
//     - sc_module_name: Name of the module instance
//     - ID: Unique number identifying the Master.
//           ID must be unique among all Masters attached to the same Bus.
//     - Priority: Positive number specifying the priority relative to the
//                 other Masters for Bus access. Higher numbers means higher
//                 priority. Masters can have the same priority.
//     - RespAcceptSampleDelay: Time number specifying the delay between clock 
//                 rising edge and the actual sampling time for 'MRespAccept'
//                 signal
//     - RequestSampleDelay: Time number specifying the delay between clock 
//                 rising edge and the actual sampling time for Request signals
//     - Check_setup_time: boolean value specifying if 'check_setup_time()' 
//                 function is executed during simulation 
//
//   Second Constructor arguments (sampling times use default values):
//     - sc_module_name: Name of the module instance
//     - ID: Unique number identifying the Master.
//           ID must be unique among all Masters attached to the same Bus.
//     - Priority: Positive number specifying the priority relative to the
//                 other Masters for Bus access. Higher numbers means higher
//                 priority. Masters can have the same priority.
//     - Combinational: Boolean specifying if Request/Response occurs in the
//                 same clock cycle or sequentially. Sampling times will be
//                 set as follows:
//                    - Combinational:
//                          * RequestSampleDelay = ClockCycle*60/100
//                          * ResponseSampleDelay = ClockCycle*75/100
//                    - Sequential:
//                          * RequestSampleDelay = ClockCycle*60/100
//                          * ResponseSampleDelay = ClockCycle*75/100
//     - ClockCycle: sc_time number specifying the clock cycle
//     - Check_setup_time: boolean value specifying if 'check_setup_time()' 
//                 function is executed during simulation 
//
//     NOTE : The adapter uses events of the communication class to
//     synchronize.  TL1 channel connected to the master port MUST have its
//     'syncevent' parameter set.
// ============================================================================

#include "ocp_tl0_tl1_master_adapter.h"
#include "ocp_tl1_data_cl.h"

// Used for debug purposes
//#define DEBUG_MASTER_ADAPTER

// ----------------------------------------------------------------------------
// Process : OCP_TL0_TL1_Master_Adapter::OCP_TL0_TL1_Master_Adapter
// 
// Constructor with explicit sampling timings
// ----------------------------------------------------------------------------
template<class TdataCl> OCP_TL0_TL1_Master_Adapter<TdataCl>::OCP_TL0_TL1_Master_Adapter
(
 sc_module_name name_
 , int ID
 , int Priority
 , sc_time RequestSampleDelay
 , sc_time RespAcceptSampleDelay
 , bool Check_setup_time
)
: sc_module         (name_)
    , MasterP           ("MasterPort")
    , m_ID              (ID)
    , m_Priority        (Priority)
    , m_request_sample_delay    (RequestSampleDelay)
    , m_respaccept_sample_delay (RespAcceptSampleDelay)
    , m_check_active    (Check_setup_time)
, m_default_values  (false)
{

    if(m_request_sample_delay < sc_get_time_resolution() ) {
        cout << "Error: TL0 OCP Master Adapter constructor (instance '" << name_ << "')"<< endl
            << "       'Request Sample Delay' parameter must be > 'sc_get_time_resolution()' " << endl 
            << "       'Request Sample Delay' =" << m_request_sample_delay
            << endl;
        exit(-1);
    }

    if(m_respaccept_sample_delay < sc_get_time_resolution() ) {
        cout << "Error: TL0 OCP Master Adapter constructor (instance '" << name_ << "')"<< endl
            << "       'RespAccept Sample Delay' parameter must be > 'sc_get_time_resolution()' " << endl 
            << "       'RespAccept Sample Delay' =" << m_respaccept_sample_delay
            << endl;
        exit(-1);
    }

    // Init TL0 output signals:
    //    * 'SCmdAccept' as false
    //    * 'SResp' as OCP_SRESP_NULL
    //    * 'SData' as '0'

    SCmdAccept.initialize(false);
    SResp.initialize(OCP_SRESP_NULL);
    SData.initialize(0);

    // Reset the request structure
    TL1_request.reset();

    // SC_METHODs

    SC_METHOD(send_Request);
    sensitive << e_request_sample_event;

    SC_METHOD(set_SCmdAccept);
    sensitive << MasterP.RequestEndEvent();
    dont_initialize();

    SC_METHOD(reset_TL0_signals);
    sensitive_pos << Clk;

    SC_METHOD(send_Response);
    sensitive << MasterP.ResponseStartEvent();
    dont_initialize();

    SC_METHOD(release_Response);
    sensitive << e_respaccept_sample_event;

    SC_METHOD(sample_events_trigger);
    dont_initialize();
    sensitive_pos << Clk;

    if(m_check_active) {
        SC_METHOD(check_respaccept_setup_time);
        sensitive << MRespAccept ;
        SC_METHOD(check_request_setup_time);
        sensitive << MCmd << MAddr << MData;
    }

}

// ----------------------------------------------------------------------------
// Process : OCP_TL0_TL1_Master_Adapter::OCP_TL0_TL1_Master_Adapter
// 
// Constructor with default sampling timings.  
// ----------------------------------------------------------------------------
template<class TdataCl> OCP_TL0_TL1_Master_Adapter<TdataCl>::OCP_TL0_TL1_Master_Adapter
(
 sc_module_name name_
 , int ID
 , int Priority
 , bool Combinational
 , sc_time ClockCycle
 , bool Check_setup_time
)
: sc_module         (name_)
    , MasterP           ("MasterPort")
    , m_ID              (ID)
    , m_Priority        (Priority)
    , m_check_active    (Check_setup_time)
, m_default_values  (true)
{
    // Setting default sampling values
    if(Combinational) {
        m_request_sample_delay = sc_time(ClockCycle*double(0.60));
        m_respaccept_sample_delay = sc_time(ClockCycle*double(0.75));
    } else {
        m_request_sample_delay = sc_time(ClockCycle*double(0.60));
        m_respaccept_sample_delay = sc_time(ClockCycle*double(0.75));
    }

    if(m_request_sample_delay < sc_get_time_resolution() ) {  
        cout << "Error: TL0 OCP Master Adapter constructor (instance '" << name_ << "')"<< endl
            << "       'Request Sample Delay' parameter must be > 'sc_get_time_resolution()' " << endl 
            << "       'Request Sample Delay' =" << m_request_sample_delay
            << endl;
        exit(-1);
    }

    if(m_respaccept_sample_delay < sc_get_time_resolution() ) {
        cout << "Error: TL0 OCP Master Adapter constructor (instance '" << name_ << "')"<< endl
            << "       'RespAccept Sample Delay' parameter must be > 'sc_get_time_resolution()' " << endl 
            << "       'RespAccept Sample Delay' =" << m_respaccept_sample_delay
            << endl;
        exit(-1);
    }

    // Init TL0 output signals:
    //    * 'SCmdAccept' as false
    //    * 'SResp' as OCP_SRESP_NULL
    //    * 'SData' as '0'

    SCmdAccept.initialize(false);
    SResp.initialize(OCP_SRESP_NULL);
    SData.initialize(0);

    // Reset the request structure
    TL1_request.reset();

    // SC_METHODs

    SC_METHOD(send_Request);
    sensitive << e_request_sample_event;

    SC_METHOD(set_SCmdAccept);
    sensitive << MasterP.RequestEndEvent();
    dont_initialize();

    SC_METHOD(reset_TL0_signals);
    sensitive_pos << Clk;

    SC_METHOD(send_Response);
    sensitive << MasterP.ResponseStartEvent();
    dont_initialize();

    SC_METHOD(release_Response);
    sensitive << e_respaccept_sample_event;

    SC_METHOD(sample_events_trigger);
    dont_initialize();
    sensitive_pos << Clk;

    if(m_check_active) {
        SC_METHOD(check_respaccept_setup_time);
        sensitive << MRespAccept ;
        SC_METHOD(check_request_setup_time);
        sensitive << MCmd << MAddr << MData;
    }

}


// ----------------------------------------------------------------------------
// Process : OCP_TL0_TL1_Master_Adapter::~OCP_TL0_TL1_Master_Adapter
// 
// Destructor
// ----------------------------------------------------------------------------
template<class TdataCl> OCP_TL0_TL1_Master_Adapter<TdataCl>::~OCP_TL0_TL1_Master_Adapter()
{
}

// ----------------------------------------------------------------------------
//  Method : OCP_TL0_TL1_Master_Adapter::send_Request()
//
//  Catch the TL0 requests and send them to the TL1 channel 
//
//  Sensitivity: 
//      sensitive << e_request_sample_event;
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL0_TL1_Master_Adapter<TdataCl>::send_Request()
{
    if(!MasterP->getSBusy() && (MCmd.read() != sc_bv<3>(OCP_MCMD_IDLE) ) ) {
        // WRITE OPERATION
        if(MCmd.read() == sc_bv<3>(OCP_MCMD_WR) ) {
            TL1_request.MCmd=OCP_MCMD_WR;
            TL1_request.MAddr=(Ta)MAddr.read();
            TL1_request.MData=(Td)MData.read();
        } else
            // READ OPERATION
            if(MCmd.read() == sc_bv<3>(OCP_MCMD_RD) ) {
                TL1_request.MCmd=OCP_MCMD_RD;
                TL1_request.MAddr=(Ta)MAddr.read();
            } else
                // READ EX OPERATION
                if(MCmd.read() == sc_bv<3>(OCP_MCMD_RDEX) ) {
                    TL1_request.MCmd=OCP_MCMD_RDEX;
                    TL1_request.MAddr=(Ta)MAddr.read();
                } else
                    // BROADCAST OPERATION
                    if(MCmd.read() == sc_bv<3>(OCP_MCMD_BCST) ) {
                        TL1_request.MCmd=OCP_MCMD_BCST;
                        TL1_request.MAddr=(Ta)MAddr.read();
                        TL1_request.MData=(Td)MData.read();
                    } else {
                        cout << "Error in 'OCP TL0 TL1 Master Adapter' class 'send_Request' method " << endl
                            << "       TL0 MCmd signal value '" << MCmd.read() 
                            << "' is not allowed" << endl;
                        exit(-1);
                    }
                bool ret = MasterP->startOCPRequest(TL1_request);

                // For Debug purposes only
                assert(ret);

    }
} // end of SC_METHOD

// ----------------------------------------------------------------------------
//  Method : OCP_TL0_TL1_Master_Adapter::set_SCmdAccept()
//
//  Set 'SCmdAccept' TL0 signal
//
//  Sensitivity: 
//      sensitive << MasterP.RequestEndEvent()
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL0_TL1_Master_Adapter<TdataCl>::set_SCmdAccept()
{
    SCmdAccept.write(true);
} // end of SC_METHOD

// ----------------------------------------------------------------------------
//  Method : OCP_TL0_TL1_Master_Adapter::reset_TL0_signals()
//
//  Reset 'SCmdAccept' and 'SResp' TL0 signals 
//
//  Sensitivity: 
//      sensitive_pos << Clk;
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL0_TL1_Master_Adapter<TdataCl>::reset_TL0_signals()
{
    SCmdAccept.write(false);
    if(!m_CommCl->ResponseStart) {
        SResp.write(sc_bv<3>(OCP_SRESP_NULL));
    }
} // end of SC_METHOD


// ----------------------------------------------------------------------------
//  Method : OCP_TL0_TL1_Master_Adapter::send_Response()
//
//  Catch responses from the TL1 Slave and send them to the TL0 master 
//
//  Sensitivity: 
//      sensitive << MasterP.ResponseStartEvent();
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL0_TL1_Master_Adapter<TdataCl>::send_Response()
{
    if(MasterP->getOCPResponse(TL1_response)) {
        SData.write(TL1_response.SData);
        SResp.write(sc_bv<3>(TL1_response.SResp)); 
    }
} // end of SC_METHOD 

// ----------------------------------------------------------------------------
//  Method : OCP_TL0_TL1_Master_Adapter::release_Response()
//
//  Release response present on TL1 channel
//
//  Sensitivity: 
//      sensitive << e_respaccept_sample_event;
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL0_TL1_Master_Adapter<TdataCl>::release_Response()
{
    if(m_CommCl->ResponseStart) {
        if(MRespAccept.read()) {
            MasterP->putMRespAccept(); 
        }
    }
} // end of SC_METHOD 


// ----------------------------------------------------------------------------
//  Method : OCP_TL0_TL1_Master_Adapter::sample_events_trigger()
//
//  Trigger 'e_respaccept_sample_event' and 'e_request_sample_event'
//  used to sample input signals 
//
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL0_TL1_Master_Adapter<TdataCl>::sample_events_trigger()
{
#ifdef DEBUG_MASTER_ADAPTER
    cout << endl
        << "TL0 Master Adapter: CLOCK cycle"
        << " delta " << simcontext()->delta_count()
        << " time " << sc_time_stamp().to_seconds()
        << endl;
#endif

    e_respaccept_sample_event.notify(m_respaccept_sample_delay);
    e_request_sample_event.notify(m_request_sample_delay);

    if(m_check_active)
        m_last_rising_edge = sc_time_stamp();

} // end of SC_METHOD 

// ----------------------------------------------------------------------------
//  Method : OCP_TL0_TL1_Master_Adapter::check_request_setup_time()
//
//  Checks timing violations, i.e. if 'MCmd', 'MAddr' or 'MData' signals
//    change after the sample time
//    
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL0_TL1_Master_Adapter<TdataCl>::check_request_setup_time()
{
    if( (sc_time_stamp()-m_last_rising_edge) >= m_request_sample_delay ) {
        cout << "TL0 Master Adapter '" << name() << "': Setup Time Violation "
            << " delta " << simcontext()->delta_count()
            << " time " << sc_time_stamp().to_seconds() << endl
            << " last rising edge " << m_last_rising_edge.to_seconds() << endl
            << " one of the signal 'MCmd', 'MData' or 'MAddr' had a late change" << endl
            << " You have to adjust 'RequestSampleDelay' parameter" 
            << endl;

        if(m_default_values) {
            cout << " Default values for sampling times do not work with that design." << endl
                << " You have to explicitly specify sampling times using the appropriate constructor" 
                << endl;
        }
        sc_stop();
    }

} // end of SC_METHOD 

// ----------------------------------------------------------------------------
//  Method : OCP_TL0_TL1_Master_Adapter::check_respaccept_setup_time()
//
//  Checks timing violations, i.e. if 'MRespAccept' signal
//    change after the sample time
//    
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL0_TL1_Master_Adapter<TdataCl>::check_respaccept_setup_time()
{
    if( (sc_time_stamp()-m_last_rising_edge) >= m_respaccept_sample_delay ) {
        cout << "TL0 Master Adapter: '" << name() << "' Setup Time Violation "
            << " delta " << simcontext()->delta_count()
            << " time " << sc_time_stamp().to_seconds() << endl
            << " last rising edge " << m_last_rising_edge.to_seconds() << endl
            << " signal 'MRespAccept' had a late change" << endl
            << " You have to adjust 'RespAcceptSampleDelay' parameter" 
            << endl;

        if(m_default_values) {
            cout << " Default values for sampling times do not work for that design." << endl
                << " You have to explicitly specify sampling times using the appropriate constructor" 
                << endl;
        }

        sc_stop();
    }

} // end of SC_METHOD 



// ----------------------------------------------------------------------------
//  Method : OCP_TL0_TL1_Master_Adapter::SputDirect()
//
//  Debug interface method.
//  Read/Write data, Slave to Master direction
//  Returns true for success and false for failure
//
// ----------------------------------------------------------------------------
template<class TdataCl> bool OCP_TL0_TL1_Master_Adapter<TdataCl>::SputDirect(
        int SlaveID, bool IsWrite, Td *Data, Ta Address, int NumWords)

{
    // not implemented
    return(false);
}


// ----------------------------------------------------------------------------
//  Method : OCP_TL0_TL1_Master_Adapter::end_of_elaboration()
//
//  This method is activated at the end of the elaboration phase
//  (when all binding operations are finished).
// ----------------------------------------------------------------------------
template<class TdataCl> void OCP_TL0_TL1_Master_Adapter<TdataCl>::end_of_elaboration()
{
    sc_module::end_of_elaboration();

    // Get communication structure
    m_CommCl = MasterP->GetCommCl();

    // Put parameter into Channel
    // No parameters needed
}


// ----------------------------------------------------------------------------
//
//  Instantiation of the Master
//
// ----------------------------------------------------------------------------
template class OCP_TL0_TL1_Master_Adapter<OCP_TL1_DataCl<int,int> >; 
