///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// Copyright 2004 OCP-IP
// OCP-IP Confidential & Proprietary
//
//
//============================================================================
//      Project : OCP SLD WG
//       Author : Alan Kamas for Sonics, Inc.
//          $Id :
//
//  Description :  OCP TL2 SystemC Channel Transaction Monitor
//        For use with the performance OCP TL2 channel.
//        Outputs a view of each transaction moving through the OCP channel.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef _OCP_TL2_TMON_H
#define _OCP_TL2_TMON_H

#include <iomanip>
#include "ocp_param.h"
#include "ocp_globals.h"
#include <string>
#include <deque>

// -----------------------------------------------------------------

// forward declaration needed for the monitor class below:
template <class Tdata, class Taddr> class OCP_TL2_Channel;

// Used for transaction logging
template <class Td, class Ta>
struct WaitReq
{
    double startCycle;
    unsigned int MConnID;
    OCPMCmdType MCmd;
    OCPMBurstSeqType MBurstSeq;
    unsigned int tCount;
    unsigned int DataLength;
    bool lastReqOfBurst;
    unsigned int MAddrSpace;
    Ta MAddr;
    unsigned int MByteEn;
    unsigned long long MReqInfo;
    unsigned int RqDL;
    unsigned int RqAL;
    unsigned int DAL;
    Td data;
    bool fillerReq;
    bool fillerResp;
};

template <class Tdata, class Taddr>
class OCP_TL2_TMon
{

  public:
    // Constructor
    OCP_TL2_TMon( ostream *traceStreamPtr, OCP_TL2_Channel<Tdata,Taddr> *m_channel ) :
        m_tLogPtr(traceStreamPtr),
        m_chPtr(m_channel),
        m_tLogWaitQ(NULL)
    { 
        // nothing to do 
    }

    //---------------------------------------------------------------
    // destructor
    //---------------------------------------------------------------
    ~OCP_TL2_TMon()
    { 
        if (m_tLogWaitQ) {
            delete [] m_tLogWaitQ;
        }
    }

    //---------------------------------------------------------------
    // Transaction Logging Functions
    //---------------------------------------------------------------
    //
    // Uses the new "ocpdis2 -w transfer" transfer style format to
    // display the transaction log.
    // 
    // The output has the following format:
    // Field(full name) / display width / base / [condition (if any)]
    // SimTime / 9 / dec
    // Cycle / 7 / dec
    // ( (left parenthesis) / 1 / char / burstsinglereq == true
    // T(Thread) / (ocpParams.threads-1)/10 +1 / dec / ocpParams.threads > 1
    // C(ConnID) / (connid_wdth-1)/4 +1 / hex / connid == true
    // Cmd (MCmd) / 4 / char
    // BuS (MBurstSeq) / 4 / char 
    // Burst (transfer count/DataLength) / 
    //          5 (tcount=2 right, "/"=1, dLen=2 left) / dec
    // l (last request of LastOfBurst=true request) / 1 / dec
    // AS (MAddressSpace) / (addrspace_wdth-1)/4 +1 / hex / addrspace
    // Addr (MAddr) / (addr_wdth-1)/4 +1 / hex
    // ) (right parenthesis) / 1 / char / burstsinglereq == true
    // BE (byteen) / (data_wdth-1)/32 +1 / hex / byteen
    // Ri(MReqInfo) / (reqinfo_wdth-1)/4 +1 / hex / reqinfo
    // RqDL (Request Data Latency) / 4 / dec
    // RqAL (Request Accept Latency) / 4 / dec
    // DAL (Data Accept Latency) / 3 / dec
    // Data (MData or SData) / (data_wdth-1)/4 +1 
    // ( (left parenthesis) / 1 / char / burstsinglereq == true
    // Resp(SResp) / 4 / char
    // RI (SRespInfo) / (respinfo_wdth-1)/4 +1 / hex / respinfo
    // L (last response of LastOfBurst=true response) / 1 / dec
    // ) (right parenthesis) / 1 / char / burstsinglereq == true
    // RqRpL (Request Response Latency / 5 / dec
    // RpAL (Response Accept Latency) / 4 / dec
    // MF (MFlag) / (mflag_wdth-1)/4 +1 / hex / mflag
    // SF (SFlag) / (sflag_wdth-1)/4 +1 / hex / sflag
    // I (SInterrupt) / 1 / dec / interrupt

    // Prints the Monitor header lines and computes each of the field
    // widths.
    // NOTE that a field width of zero means that the field is not being used.
    void tLogHeader( void )
    {
        (*m_tLogPtr).setf(std::ios::right);
        (*m_tLogPtr) << "##" << endl;
        (*m_tLogPtr) << 
            "# Legend for \"transfer\" write style (sorted by request cycle)" 
            << endl;
        (*m_tLogPtr) << "##" << endl;
        (*m_tLogPtr) << 
            "# SimTime  :                        Simulation Time : Time"
            << endl;
        (*m_tLogPtr) << 
            "# Cycle    :                       Cycle Valid Time : Cycle"
            << endl;
        if ((m_chPtr->ocpParams).threads > 1) {
            (*m_tLogPtr) << 
                "# T        :             Master and Slave Thread ID : ThreadID"
                << endl;
        }
        if ((m_chPtr->ocpParams).connid) {
            (*m_tLogPtr) << 
                "# C        :                 Connection indentifier : MConnID"
                << endl;
        }
        (*m_tLogPtr) << 
            "# Cmd      :                         Master Command : MCmd"
            << endl;
        (*m_tLogPtr) << 
            "# BuS      :                    Burst Sequence Type : MBurstSeq"
            << endl;
        (*m_tLogPtr) << 
            "# Burst    :  req burst transfer count/burst length : transfer count/DataLength"
            << endl;
        (*m_tLogPtr) << 
            "# l        :last transfer of req w/LastOfBurst true : request LastOfBurst"
            << endl;
        if ((m_chPtr->ocpParams).addrspace) {
            (*m_tLogPtr) << 
                "# AS       :                   Master Address Space : MAddrSpace"
                << endl;
        }
        (*m_tLogPtr) << 
            "# Addr     :                         Master Address : MAddr"
            << endl;
        if ((m_chPtr->ocpParams).byteen) {
            (*m_tLogPtr) << 
                "# BE       :                           Byte Enables : MByteEn"
                << endl;
        }
        if ((m_chPtr->ocpParams).reqinfo) {
            (*m_tLogPtr) << 
                "# Ri       :                           Request Info : MReqInfo"
                << endl;
        }
        (*m_tLogPtr) << 
            "# RqDL     :                   Request Data Latency : RqDL" 
            << endl;
        (*m_tLogPtr) << 
            "# RqAL     :                 Request Accept Latency : RqAL" 
            << endl;
        (*m_tLogPtr) << 
            "# DAL      :                    Data Accpet Latency : DAL" 
            << endl;
        (*m_tLogPtr) << 
            "# Data     :    MData (if write) or SData (if read) : Data "
            << endl;
        (*m_tLogPtr) << 
            "# Resp     :                         Slave Response : SResp "
            << endl;
        if ((m_chPtr->ocpParams).respinfo) {
            (*m_tLogPtr) << 
                "# RI       :                          Response Info : MRespInfo"
                << endl;
        }
        (*m_tLogPtr) << 
            "# L        :last transfer of resp w/LastOfBurst true: response LastOfBurst"
            << endl;
        (*m_tLogPtr) << 
            "# RqRpL    :               Request Response Latency : RqRpL" 
            << endl;
        (*m_tLogPtr) << 
            "# RpAL     :                Resposne Accept Latency : RpAL" 
            << endl;
        if ((m_chPtr->ocpParams).mflag) {
            (*m_tLogPtr) << 
                "# MF       :                           Master Flags : MFlag" 
                << endl;
        }
        if ((m_chPtr->ocpParams).sflag) {
            (*m_tLogPtr) << 
                "# SF       :                            Slave Flags : SFlag" 
                << endl;
        }
        if ((m_chPtr->ocpParams).interrupt) {
            (*m_tLogPtr) << 
                "# I        :                        Slave Interrupt : SInterrupt" 
                << endl;
        }
        (*m_tLogPtr) << endl;

        // Print Header Row
        (*m_tLogPtr) << std::setfill(' ');
        // SimTime / 9 / dec
        (*m_tLogPtr) << "  SimTime" << " ";
        // Cycle / 7 / dec
        (*m_tLogPtr) << "  Cycle" << " ";
        // ( (left parenthesis) / 1 / char / burstsinglereq == true
        if ((m_chPtr->ocpParams).burstsinglereq) {
            (*m_tLogPtr) << " " << " ";
        }
        // T(Thread) / ((m_chPtr->ocpParams).threads-1)/10 +1 / dec / (m_chPtr->ocpParams).threads > 1
        m_tLogThreadW = 0;
        if ((m_chPtr->ocpParams).threads>1) {
            m_tLogThreadW = int(((m_chPtr->ocpParams).threads-1)/10) +1;
            (*m_tLogPtr) << std::setw(m_tLogThreadW);
            (*m_tLogPtr) << "T" << " ";
        }
        // C(ConnID) / (connid_wdth-1)/4 +1 / hex / connid == true
        m_tLogConnW = 0;
        if ((m_chPtr->ocpParams).connid) {
            m_tLogConnW = int(((m_chPtr->ocpParams).connid_wdth-1)/4) +1;
            (*m_tLogPtr) << std::setw(m_tLogConnW);
            (*m_tLogPtr) << "C" << " ";
        }
        // Cmd (MCmd) / 4 / char
        (*m_tLogPtr) << " CMD" << " ";
        // BuS (MBurstSeq) / 4 / char 
        (*m_tLogPtr) << " BuS" << " ";
        // Burst (transfer count/DataLength) / 
        //          5 (tcount=2 right, "/"=1, dLen=2 left) / dec
        (*m_tLogPtr) << "Burst" << " ";
        // l (last transfer of LastOfBurst=true request) / 1 / dec
        (*m_tLogPtr) << "l" << " ";
        // AS (MAddressSpace) / (addrspace_wdth-1)/4 +1 / hex / addrspace
        m_tLogAddrSpaceW = 0;
        if ((m_chPtr->ocpParams).addrspace) {
            m_tLogAddrSpaceW = int(((m_chPtr->ocpParams).addrspace_wdth-1)/4) +1;
            if (m_tLogAddrSpaceW < 2 ) {
                m_tLogAddrSpaceW = 2;
            }
            (*m_tLogPtr) << std::setw(m_tLogAddrSpaceW);
            (*m_tLogPtr) << "AS" << " ";
        }
        // Addr (MAddr) / (addr_wdth-1)/4 +1 / hex
        m_tLogAddrW = int(((m_chPtr->ocpParams).addr_wdth-1)/4) +1;
        if (m_tLogAddrW < 4 ) {
            m_tLogAddrW = 4;
        }
        (*m_tLogPtr) << std::setw(m_tLogAddrW);
        (*m_tLogPtr) << "Addr" << " ";
        // ) (right parenthesis) / 1 / char / burstsinglereq == true
        if ((m_chPtr->ocpParams).burstsinglereq) {
            (*m_tLogPtr) << " " << " ";
        }
        // BE (byteen) / (data_wdth-1)/32 +1 / hex / byteen
        m_tLogByteEnW = 0;
        if ((m_chPtr->ocpParams).byteen) {
            m_tLogByteEnW = int(((m_chPtr->ocpParams).data_wdth-1)/32) +1;
            if (m_tLogByteEnW < 2 ) {
                m_tLogByteEnW = 2;
            }
            (*m_tLogPtr) << std::setw(m_tLogByteEnW);
            (*m_tLogPtr) << "BE" << " ";
        }
        // Ri(MReqInfo) / (reqinfo_wdth-1)/4 +1 / hex / reqinfo
        m_tLogReqInfoW = 0;
        if ((m_chPtr->ocpParams).reqinfo) {
            m_tLogReqInfoW = int(((m_chPtr->ocpParams).reqinfo_wdth-1)/4) +1;
            if (m_tLogReqInfoW < 2 ) {
                m_tLogReqInfoW = 2;
            }
            (*m_tLogPtr) << std::setw(m_tLogReqInfoW);
            (*m_tLogPtr) << "Ri" << " ";
        }
        // RqDL (Request Data Latency) / 4 / dec
        (*m_tLogPtr) << "RqDL" << " ";
        // RqAL (Request Accept Latency) / 4 / dec
        (*m_tLogPtr) << "RqAL" << " ";
        // DAL (Data Accept Latency) / 3 / dec
        (*m_tLogPtr) << "DAL" << " ";
        // Data (MData or SData) / (data_wdth-1)/4 +1 
        m_tLogDataW = int(((m_chPtr->ocpParams).data_wdth-1)/4) +1;
        if (m_tLogDataW < 4 ) {
            m_tLogDataW = 4;
        }
        (*m_tLogPtr) << std::setw(m_tLogDataW);
        (*m_tLogPtr) << "Data" << " ";
        // ( (left parenthesis) / 1 / char / burstsinglereq == true
        if ((m_chPtr->ocpParams).burstsinglereq) {
            (*m_tLogPtr) << " " << " ";
        }
        // Resp(SResp) / 4 / char
        (*m_tLogPtr) << "Resp" << " ";
        // RI (SRespInfo) / (respinfo_wdth-1)/4 +1 / hex / respinfo
        m_tLogRespInfoW = 0;
        if ((m_chPtr->ocpParams).respinfo) {
            m_tLogRespInfoW = int(((m_chPtr->ocpParams).respinfo_wdth-1)/4) +1;
            if (m_tLogRespInfoW < 2 ) {
                m_tLogRespInfoW = 2;
            }
            (*m_tLogPtr) << std::setw(m_tLogRespInfoW);
            (*m_tLogPtr) << "RI" << " ";
        }
        // L (last transfer of LastOfBurst=true resposne) / 1 / dec
        (*m_tLogPtr) << "L" << " ";
        // ) (right parenthesis) / 1 / char / burstsinglereq == true
        if ((m_chPtr->ocpParams).burstsinglereq) {
            (*m_tLogPtr) << " " << " ";
        }
        // RqRpL (Request Response Latency / 5 / dec
        (*m_tLogPtr) << "RqRpL" << " ";
        // RpAL (Response Accept Latency) / 4 / dec
        (*m_tLogPtr) << "RpAL" << " ";
        // MF (MFlag) / (mflag_wdth-1)/4 +1 / hex / mflag
        m_tLogMFW = 0;
        if ((m_chPtr->ocpParams).mflag) {
            m_tLogMFW = int(((m_chPtr->ocpParams).mflag_wdth-1)/4) +1;
            if (m_tLogMFW < 2 ) {
                m_tLogMFW = 2;
            }
            (*m_tLogPtr) << std::setw(m_tLogMFW);
            (*m_tLogPtr) << "MF" << " ";
        }
        // SF (SFlag) / (sflag_wdth-1)/4 +1 / hex / sflag
        m_tLogSFW = 0;
        if ((m_chPtr->ocpParams).sflag) {
            m_tLogSFW = int(((m_chPtr->ocpParams).sflag_wdth-1)/4) +1;
            if (m_tLogSFW < 2 ) {
                m_tLogSFW = 2;
            }
            (*m_tLogPtr) << std::setw(m_tLogSFW);
            (*m_tLogPtr) << "SF" << " ";
        }
        // I (SInterrupt) / 1 / dec / interrupt
        if ((m_chPtr->ocpParams).interrupt) {
            (*m_tLogPtr) << "I";
        }
        // Done with this line
        (*m_tLogPtr) << endl;

        // Now print out the dashed underlines
        (*m_tLogPtr) << std::setfill('-');
        // SimTime / 9 / dec
        (*m_tLogPtr) << "---------";
        // Cycle / 7 / dec
        (*m_tLogPtr) << " -------";
        // ( (left parenthesis) / 1 / char / burstsinglereq == true
        if ((m_chPtr->ocpParams).burstsinglereq) {
            (*m_tLogPtr) << " -";
        }
        // T(Thread) / ((m_chPtr->ocpParams).threads-1)/10 +1 / dec / (m_chPtr->ocpParams).threads > 1
        if (m_tLogThreadW) {
            (*m_tLogPtr) << " ";
            (*m_tLogPtr) << std::setw(m_tLogThreadW);
            (*m_tLogPtr) << "-";
        }
        // C(ConnID) / (connid_wdth-1)/4 +1 / hex / connid == true
        if (m_tLogConnW) {
            (*m_tLogPtr) << " ";
            (*m_tLogPtr) << std::setw(m_tLogConnW);
            (*m_tLogPtr) << "-";
        }
        // Cmd (MCmd) / 4 / char
        (*m_tLogPtr) << " ----";
        // BuS (MBurstSeq) / 4 / char 
        (*m_tLogPtr) << " ----";
        // Burst (transfer count/DataLength) / 
        //          5 (tcount=2 right, "/"=1, dLen=2 left) / dec
        (*m_tLogPtr) << " -----";
        // l (last transfer of LastOfBurst=true request) / 1 / dec
        (*m_tLogPtr) << " -";
        // AS (MAddressSpace) / (addrspace_wdth-1)/4 +1 / hex / addrspace
        if (m_tLogAddrSpaceW) {
            (*m_tLogPtr) << " ";
            (*m_tLogPtr) << std::setw(m_tLogAddrSpaceW);
            (*m_tLogPtr) << "-";
        }
        // Addr (MAddr) / (addr_wdth-1)/4 +1 / hex
        (*m_tLogPtr) << " ";
        (*m_tLogPtr) << std::setw(m_tLogAddrW);
        (*m_tLogPtr) << "-";
        // ) (right parenthesis) / 1 / char / burstsinglereq == true
        if ((m_chPtr->ocpParams).burstsinglereq) {
            (*m_tLogPtr) << " -";
        }
        // BE (byteen) / (data_wdth-1)/32 +1 / hex / byteen
        if (m_tLogByteEnW) {
            (*m_tLogPtr) << " ";
            (*m_tLogPtr) << std::setw(m_tLogByteEnW);
            (*m_tLogPtr) << "-";
        }
        // Ri(MReqInfo) / (reqinfo_wdth-1)/4 +1 / hex / reqinfo
        if (m_tLogReqInfoW) {
            (*m_tLogPtr) << " ";
            (*m_tLogPtr) << std::setw(m_tLogReqInfoW);
            (*m_tLogPtr) << "-";
        }
        // RqDL (Request Data Latency) / 4 / dec
        (*m_tLogPtr) << " ----";
        // RqAL (Request Accept Latency) / 4 / dec
        (*m_tLogPtr) << " ----";
        // DAL (Data Accept Latency) / 3 / dec
        (*m_tLogPtr) << " ---";
        // Data (MData or SData) / (data_wdth-1)/4 +1 
        (*m_tLogPtr) << " ";
        (*m_tLogPtr) << std::setw(m_tLogDataW);
        (*m_tLogPtr) << "-";
        // ( (left parenthesis) / 1 / char / burstsinglereq == true
        if ((m_chPtr->ocpParams).burstsinglereq) {
            (*m_tLogPtr) << " -";
        }
        // Resp(SResp) / 4 / char
        (*m_tLogPtr) << " ----";
        // RI (SRespInfo) / (respinfo_wdth-1)/4 +1 / hex / respinfo
        if (m_tLogRespInfoW) {
            (*m_tLogPtr) << " ";
            (*m_tLogPtr) << std::setw(m_tLogRespInfoW);
            (*m_tLogPtr) << "-";
        }
        // L (last transfer of LastOfBurst=true resposne) / 1 / dec
        (*m_tLogPtr) << " -";
        // ) (right parenthesis) / 1 / char / burstsinglereq == true
        if ((m_chPtr->ocpParams).burstsinglereq) {
            (*m_tLogPtr) << " -";
        }
        // RqRpL (Request Response Latency / 5 / dec
        (*m_tLogPtr) << " -----";
        // RpAL (Response Accept Latency) / 4 / dec
        (*m_tLogPtr) << " ----";
        // MF (MFlag) / (mflag_wdth-1)/4 +1 / hex / mflag
        if (m_tLogMFW) {
            (*m_tLogPtr) << " ";
            (*m_tLogPtr) << std::setw(m_tLogMFW);
            (*m_tLogPtr) << "-";
        }
        // SF (SFlag) / (sflag_wdth-1)/4 +1 / hex / sflag
        if (m_tLogSFW) {
            (*m_tLogPtr) << " ";
            (*m_tLogPtr) << std::setw(m_tLogSFW);
            (*m_tLogPtr) << "-";
        }
        // I (SInterrupt) / 1 / dec / interrupt
        if ((m_chPtr->ocpParams).interrupt) {
            (*m_tLogPtr) << " -";
        }
        // Done with this line
        (*m_tLogPtr) << endl;
        // Return to normal fill style
        (*m_tLogPtr) << std::setfill(' ');
    }

    void tLogSetUp( void )
    {
        // Set up request-waiting-for-resp-queues. One for each thread.
        if (m_tLogWaitQ) {
            // possible that we could be configured twice.
            delete [] m_tLogWaitQ;
        }
        m_tLogWaitQ = new deque< WaitReq<Tdata,Taddr> >[(m_chPtr->ocpParams).threads];
    }

    const string tLogCmdName(OCPMCmdType myType) const
    {
        switch (myType) {
            case OCP_MCMD_IDLE:
                return "IDLE";
            case OCP_MCMD_WR:
                return "  WR";
            case OCP_MCMD_RD:
                return "  RD";
            case OCP_MCMD_RDEX:
                return "RDEX";
            case OCP_MCMD_RDL:
                return " RDL";
            case OCP_MCMD_WRNP:
                return "WRNP";
            case OCP_MCMD_WRC:
                return " WRC";
            case OCP_MCMD_BCST:
                return "BCST";
        }
        return "UNKNOWN";
    }

    const string tLogRespName(OCPSRespType myType) const
    {
        switch (myType) {
            case OCP_SRESP_NULL:
                return "NULL";
            case OCP_SRESP_DVA:
                return " DVA";
            case OCP_SRESP_FAIL:
                return "FAIL";
            case OCP_SRESP_ERR:
                return " ERR";
        }
        return "UNKNOWN";
    }

    const string tLogBSeqName(OCPMBurstSeqType myType) const
    {
        switch (myType) {
            case OCP_MBURSTSEQ_INCR:
                return "INCR";
            case OCP_MBURSTSEQ_DFLT1:
                return "DFT1";
            case OCP_MBURSTSEQ_WRAP:
                return "WRAP";
            case OCP_MBURSTSEQ_DFLT2:
                return "DFT2";
            case OCP_MBURSTSEQ_XOR:
                return " XOR";
            case OCP_MBURSTSEQ_STRM:
                return "STRM";
            case OCP_MBURSTSEQ_UNKN:
                return "UNKN";
            case OCP_MBURSTSEQ_RESERVED:
                return "RSVD";
        }
        return "UNKNOWN";
    }

    bool tLogIsWrite(OCPMCmdType myType) const
    {
        if ((myType == OCP_MCMD_WR) ||
            (myType == OCP_MCMD_WRNP) ||
            (myType == OCP_MCMD_WRC) ||
            (myType == OCP_MCMD_BCST) ) {
            return true;
        }
        return false;
    }

    void tLogNewReq(void)
    {
        // We have a new request. Either it needs a response and 
        // printing must be delayed until the response arrives or
        // it does not need a response in which case it may be printed
        // immediately.

        double myStart = sc_simulation_time();
        // NOTE: start of the "request" could be the time when the 
        // request if first presented on the channel. It could also be
        // the time that the associated data is first put on the channel.
        // here, we'll use the start of the request.

        // Set up the address incrementing
        // NOTE: currently no support for addressing types other than
        // INCR, STRM, and WRAP.
        int myAddrInc = 0;
        if ((m_chPtr->m_currentRequest).MBurstSeq != OCP_MBURSTSEQ_STRM) {
            myAddrInc = ((m_chPtr->ocpParams).data_wdth)/8;
        }

        // Does this request require a response?
        
        if ( (m_chPtr->m_currentRequest).MCmd == OCP_MCMD_IDLE ) {
            // An IDLE command. Do nothing.
        } else if ( (!(m_chPtr->ocpParams).writeresp_enable) && 
                    (((m_chPtr->m_currentRequest).MCmd == OCP_MCMD_WR)
                      || ((m_chPtr->m_currentRequest).MCmd == OCP_MCMD_BCST)) ) {
            // This is a basic posted write command.
            // There will be no response so this transfer is now complete.

            // Print it out now.
            for (unsigned int i=0;i<(m_chPtr->m_currentRequest).DataLength;i++) {
                // NOTE: for now, cycle time and start time are the same
                // SimTime / 9 / dec
                (*m_tLogPtr).setf(std::ios::dec, std::ios::basefield);
                (*m_tLogPtr) << std::setw(9) << myStart + i*(m_chPtr->getWDI());
                // Cycle / 7 / dec
                (*m_tLogPtr) << " " << std::setw(7) << myStart + i*(m_chPtr->getWDI());

                // ( (left parenthesis) / 1 / char 
                if ((m_chPtr->ocpParams).burstsinglereq) {
                    if ( (i!=0) ) {
                        // This is a "filler" request.
                        // Since the channel is SRMD, only the first request
                        // was actually sent.
                        (*m_tLogPtr) << " (";
                    } else {
                        // This is the first request of the burst.
                        (*m_tLogPtr) << "  ";
                    }
                }
                // T(Thread) / ((m_chPtr->ocpParams).threads-1)/10 +1 / dec 
                if (m_tLogThreadW) {
                    (*m_tLogPtr) << " " << std::setw(m_tLogThreadW);
                    (*m_tLogPtr) << (m_chPtr->m_currentRequest).MThreadID;
                }
                // C(ConnID) / (connid_wdth-1)/4 +1 / hex
                if (m_tLogConnW) {
                    (*m_tLogPtr).setf(std::ios::hex, std::ios::basefield);
                    (*m_tLogPtr) << " " << std::setw(m_tLogConnW);
                    (*m_tLogPtr) << (m_chPtr->m_currentRequest).MConnID;
                }
                // Cmd (MCmd) / 4 / char
                (*m_tLogPtr) << " " << tLogCmdName((m_chPtr->m_currentRequest).MCmd);
                // BuS (MBurstSeq) / 4 / char 
                (*m_tLogPtr) << " " << tLogBSeqName((m_chPtr->m_currentRequest).MBurstSeq);
                // Burst (transfer count/DataLength) 
                (*m_tLogPtr).setf(std::ios::dec, std::ios::basefield);
                (*m_tLogPtr) << " " << std::setw(2) << i+1 
                    << "/";
                (*m_tLogPtr) << std::setw(2) 
                    << resetiosflags(std::ios::adjustfield)
                    << setiosflags(std::ios::left) 
                    << (m_chPtr->m_currentRequest).DataLength;
                (*m_tLogPtr) << resetiosflags(std::ios::adjustfield)
                    << setiosflags(std::ios::right);
                // l (last transfer of LastOfBurst=true request) / 1 / dec
                if ((m_chPtr->m_currentRequest).LastOfBurst 
                        && (i==((m_chPtr->m_currentRequest).DataLength-1))) {
                    (*m_tLogPtr) << " 1";
                } else {
                    (*m_tLogPtr) << " 0";
                }
                (*m_tLogPtr).setf(std::ios::hex, std::ios::basefield);
                // AS (MAddressSpace)/(addrspace_wdth-1)/4 +1/hex/addrspace
                if (m_tLogAddrSpaceW) {
                    (*m_tLogPtr) << " " << std::setw(m_tLogAddrSpaceW);
                    (*m_tLogPtr) << (m_chPtr->m_currentRequest).MAddrSpace;
                }
                // Addr (MAddr) / (addr_wdth-1)/4 +1 / hex
                (*m_tLogPtr) << std::setfill('0');
                (*m_tLogPtr) << " " << std::setw(m_tLogAddrW);
                (*m_tLogPtr) << (m_chPtr->m_currentRequest).MAddr + i*myAddrInc;
                // ) (right parenthesis) / 1 / char / burstsinglereq == true
                (*m_tLogPtr) << std::setfill(' ');
                if ((m_chPtr->ocpParams).burstsinglereq) {
                    if (i!=0) {
                        // This is a "filler" request.
                        // Since the channel is SRMD, only the first request
                        // was actually sent.
                        (*m_tLogPtr) << " )";
                    } else {
                        // This is the first request of the burst.
                        (*m_tLogPtr) << "  ";
                    }
                }
                (*m_tLogPtr) << std::setfill('0');
                // BE (byteen) / (data_wdth-1)/32 +1 / hex / byteen
                if (m_tLogByteEnW) {
                    (*m_tLogPtr) << " " << std::setw(m_tLogByteEnW);
                    if ( (m_chPtr->m_currentRequest).MByteEnPtr == NULL ) {
                        // no pointer used - same for whole burst
                        (*m_tLogPtr) << (m_chPtr->m_currentRequest).MByteEn;
                    } else {
                        (*m_tLogPtr) << (m_chPtr->m_currentRequest).MByteEnPtr[i];
                    }
                }
                // Ri(MReqInfo) / (reqinfo_wdth-1)/4 +1 / hex / reqinfo
                if (m_tLogReqInfoW) {
                    (*m_tLogPtr) << " " << std::setw(m_tLogReqInfoW);
                    (*m_tLogPtr) << (m_chPtr->m_currentRequest).MReqInfo;
                }
                (*m_tLogPtr).setf(std::ios::dec, std::ios::basefield);
                (*m_tLogPtr) << std::setfill(' ');
                // RqAL (Request Accept Latency) / 4 / dec
                (*m_tLogPtr) << " " << std::setw(4);
                (*m_tLogPtr) << (m_chPtr->m_sTiming).RqAL;
                // RqDL (Request Data Latency) / 4 / dec
                (*m_tLogPtr) << " " << std::setw(4);
                (*m_tLogPtr) << (m_chPtr->m_mTiming).RqDL;
                // DAL (Data Accept Latency) / 3 / dec
                (*m_tLogPtr) << " " << std::setw(3);
                (*m_tLogPtr) << (m_chPtr->m_sTiming).DAL;
                // Data (MData or SData) / (data_wdth-1)/4 +1 
                (*m_tLogPtr).setf(std::ios::hex, std::ios::basefield);
                (*m_tLogPtr) << std::setfill('0');
                (*m_tLogPtr) << " " << std::setw(m_tLogDataW);
                (*m_tLogPtr) << (m_chPtr->m_currentRequest).MDataPtr[i];
                // Request is over. Fill rest with blanks
                // ( (left parenthesis) / 1 / char / burstsinglereq == true
                (*m_tLogPtr) << std::setfill(' ');
                if ((m_chPtr->ocpParams).burstsinglereq) {
                    (*m_tLogPtr) << "  ";
                }
                // Resp(SResp) / 4 / char
                (*m_tLogPtr) << "    .";
                // RI (SRespInfo) / (respinfo_wdth-1)/4 +1 / hex / respinfo
                if (m_tLogRespInfoW) {
                    (*m_tLogPtr) << " ";
                    (*m_tLogPtr) << std::setw(m_tLogRespInfoW);
                    (*m_tLogPtr) << ".";
                }
                // L (last transfer of LastOfBurst=true resposne) / 1 / dec
                (*m_tLogPtr) << " .";
                // ) (right parenthesis) / 1 / char / burstsinglereq == true
                if ((m_chPtr->ocpParams).burstsinglereq) {
                    (*m_tLogPtr) << "  ";
                }
                // RqRpL (Request Response Latency / 5 / dec
                (*m_tLogPtr) << std::setw(5) << "     .";
                // RpAL (Response Accept Latency) / 4 / dec
                (*m_tLogPtr) << std::setw(4) << "    .";
                // MF (MFlag) / (mflag_wdth-1)/4 +1 / hex / mflag
                if (m_tLogMFW) {
                    (*m_tLogPtr) << " ";
                    (*m_tLogPtr) << std::setw(m_tLogMFW);
                    (*m_tLogPtr) << (m_chPtr->m_sideband).MFlag ;
                }
                // SF (SFlag) / (sflag_wdth-1)/4 +1 / hex / sflag
                if (m_tLogSFW) {
                    (*m_tLogPtr) << " ";
                    (*m_tLogPtr) << std::setw(m_tLogSFW);
                    (*m_tLogPtr) << (m_chPtr->m_sideband).SFlag ;
                }
                // I (SInterrupt) / 1 / dec / interrupt
                if ((m_chPtr->ocpParams).interrupt) {
                    (*m_tLogPtr) << " " << (m_chPtr->m_sideband).SInterrupt;
                }
                // Done with this line
                (*m_tLogPtr) << endl;

                // Go on to the next request in the burst
            }

        } else {

            // This request will require a response.
            // Store it for now and then print it once the response
            // arrives.
            for (unsigned int i=0;i<(m_chPtr->m_currentRequest).DataLength;i++) {
                // Reuse the temp_waitReq structure for efficiency
                if (tLogIsWrite((m_chPtr->m_currentRequest).MCmd)) {
                    // Write commands
                    temp_waitReq.startCycle = myStart + i*(m_chPtr->getWDI());
                } else if ((m_chPtr->ocpParams).burstsinglereq) {
                    // SRMD Read Command
                    temp_waitReq.startCycle = myStart;
                } else {
                    // MRMD Read Command
                    temp_waitReq.startCycle = myStart
                        + (i * max((m_chPtr->m_mTiming).RqSndI,(m_chPtr->m_sTiming).RqAL));
                }
                temp_waitReq.MConnID = (m_chPtr->m_currentRequest).MConnID;
                temp_waitReq.MCmd = (m_chPtr->m_currentRequest).MCmd;
                temp_waitReq.MBurstSeq =  (m_chPtr->m_currentRequest).MBurstSeq;
                temp_waitReq.tCount = i+1;
                temp_waitReq.DataLength = (m_chPtr->m_currentRequest).DataLength;
                temp_waitReq.lastReqOfBurst = (m_chPtr->m_currentRequest).LastOfBurst
                        && (i == ((m_chPtr->m_currentRequest).DataLength-1));
                temp_waitReq.MAddrSpace = (m_chPtr->m_currentRequest).MAddrSpace;
                temp_waitReq.MAddr = (m_chPtr->m_currentRequest).MAddr + i*myAddrInc;
                if ( ((m_chPtr->m_currentRequest).MByteEnPtr != NULL )
                        && (tLogIsWrite((m_chPtr->m_currentRequest).MCmd)) ) {
                    // use the pointer for ByteEn as we have an array of them for
                    // a write command
                    temp_waitReq.MByteEn = (m_chPtr->m_currentRequest).MByteEnPtr[i];
                } else {
                    temp_waitReq.MByteEn = (m_chPtr->m_currentRequest).MByteEn;
                }
                temp_waitReq.MReqInfo = (m_chPtr->m_currentRequest).MReqInfo;
                temp_waitReq.RqDL = (m_chPtr->m_mTiming).RqDL;
                temp_waitReq.RqAL = (m_chPtr->m_sTiming).RqAL;
                temp_waitReq.DAL = (m_chPtr->m_sTiming).DAL;
                if (tLogIsWrite((m_chPtr->m_currentRequest).MCmd)) {
                    temp_waitReq.data = (m_chPtr->m_currentRequest).MDataPtr[i];
                } else {
                    temp_waitReq.data = 0;
                }
                temp_waitReq.fillerReq = (m_chPtr->ocpParams).burstsinglereq && (i!=0);
                if (tLogIsWrite((m_chPtr->m_currentRequest).MCmd)) {
                   if ((m_chPtr->ocpParams).burstsinglereq) {
                        temp_waitReq.fillerResp = 
                            (i != ((m_chPtr->m_currentRequest).DataLength -1));
                   }
                } else {
                    temp_waitReq.fillerResp = false;
                }

                // Add the response to the queue
                m_tLogWaitQ[(m_chPtr->m_currentRequest).MThreadID].push_back(temp_waitReq);

                // Now on to the next request in the burst
            }
        }
    }

    void tLogNewResp( void )
    {
        // A response has come in. 

        // How long is the burst response?
        unsigned int respCount = (m_chPtr->m_currentResponse).DataLength;

        // Each individual response in the response burst should have a
        // individual request stored up and waiting for it. 
        // Test: are there enough requests waiting for this reponse?
        assert(m_tLogWaitQ[(m_chPtr->m_currentResponse).SThreadID].size() >= respCount);

        // When is this response burst start?
        double timeNow = sc_simulation_time();

        // print out each individual transaction of the burst
        for (unsigned int i=0;i<respCount;i++) {
            // Get the next request that's waiting.
            temp_waitReq = m_tLogWaitQ[(m_chPtr->m_currentResponse).SThreadID].front();

            // NOTE: for now, cycle time and start time are the same
            // SimTime / 9 / dec
            (*m_tLogPtr).setf(std::ios::dec, std::ios::basefield);
            (*m_tLogPtr) << std::setw(9) << temp_waitReq.startCycle;

            // Cycle / 7 / dec
            (*m_tLogPtr) << " " << std::setw(7) << temp_waitReq.startCycle;

            // ( (left parenthesis) / 1 / char 
            if ((m_chPtr->ocpParams).burstsinglereq) {
                if (temp_waitReq.fillerReq) {
                    (*m_tLogPtr) << " (";
                } else {
                    (*m_tLogPtr) << "  ";
                }
            }
            // T(Thread) / ((m_chPtr->ocpParams).threads-1)/10 +1 / dec 
            if (m_tLogThreadW) {
                (*m_tLogPtr) << " " << std::setw(m_tLogThreadW);
                (*m_tLogPtr) << (m_chPtr->m_currentResponse).SThreadID;
            }
            // C(ConnID) / (connid_wdth-1)/4 +1 / hex
            if (m_tLogConnW) {
                (*m_tLogPtr).setf(std::ios::hex, std::ios::basefield);
                (*m_tLogPtr) << " " << std::setw(m_tLogConnW);
                (*m_tLogPtr) << temp_waitReq.MConnID;
            }
            // Cmd (MCmd) / 4 / char
            (*m_tLogPtr) << " " << tLogCmdName(temp_waitReq.MCmd);
            // BuS (MBurstSeq) / 4 / char 
            (*m_tLogPtr) << " " << tLogBSeqName(temp_waitReq.MBurstSeq);
            // Burst (transfer count/DataLength) 
            (*m_tLogPtr).setf(std::ios::dec, std::ios::basefield);
            (*m_tLogPtr) << " " << std::setw(2) << temp_waitReq.tCount
                << "/";
            (*m_tLogPtr) << resetiosflags(std::ios::adjustfield)
                << setiosflags(std::ios::left) << std::setw(2)
                << temp_waitReq.DataLength;
            (*m_tLogPtr) << resetiosflags(std::ios::adjustfield)
                << setiosflags(std::ios::right);
            // l (last transfer of LastOfBurst=true request) / 1 / dec
            if (temp_waitReq.lastReqOfBurst) {
                (*m_tLogPtr) << " 1";
            } else {
                (*m_tLogPtr) << " 0";
            }
            // AS (MAddressSpace)/(addrspace_wdth-1)/4 +1/hex/addrspace
            (*m_tLogPtr).setf(std::ios::hex, std::ios::basefield);
            if (m_tLogAddrSpaceW) {
                (*m_tLogPtr) << " " << std::setw(m_tLogAddrSpaceW);
                (*m_tLogPtr) << temp_waitReq.MAddrSpace;
            }
            // Addr (MAddr) / (addr_wdth-1)/4 +1 / hex
            (*m_tLogPtr) << std::setfill('0');
            (*m_tLogPtr) << " " << std::setw(m_tLogAddrW);
            (*m_tLogPtr) << temp_waitReq.MAddr;
            (*m_tLogPtr) << std::setfill(' ');
            // ) (right parenthesis) / 1 / char / burstsinglereq == true
            if ((m_chPtr->ocpParams).burstsinglereq) {
                if (temp_waitReq.fillerReq) {
                    (*m_tLogPtr) << " )";
                } else {
                    (*m_tLogPtr) << "  ";
                }
            }
            (*m_tLogPtr) << std::setfill('0');
            // BE (byteen) / (data_wdth-1)/32 +1 / hex / byteen
            if (m_tLogByteEnW) {
                (*m_tLogPtr) << " " << std::setw(m_tLogByteEnW);
                (*m_tLogPtr) << temp_waitReq.MByteEn;
            }
            // Ri(MReqInfo) / (reqinfo_wdth-1)/4 +1 / hex / reqinfo
            if (m_tLogReqInfoW) {
                (*m_tLogPtr) << " " << std::setw(m_tLogReqInfoW);
                (*m_tLogPtr) << temp_waitReq.MReqInfo;
            }
            // RqAL (Request Accept Latency) / 4 / dec
            (*m_tLogPtr).setf(std::ios::dec, std::ios::basefield);
            (*m_tLogPtr) << std::setfill(' ');
            (*m_tLogPtr) << " " << std::setw(4);
            (*m_tLogPtr) << (m_chPtr->m_sTiming).RqAL;
            // RqDL (Request Data Latency) / 4 / dec
            (*m_tLogPtr) << " " << std::setw(4);
            (*m_tLogPtr) << temp_waitReq.RqDL;
            // DAL (Data Accept Latency) / 3 / dec
            (*m_tLogPtr) << " " << std::setw(3);
            (*m_tLogPtr) << temp_waitReq.DAL;
            // Data (MData or SData) / (data_wdth-1)/4 +1 
            (*m_tLogPtr).setf(std::ios::hex, std::ios::basefield);
            (*m_tLogPtr) << std::setfill('0');
            (*m_tLogPtr) << " " << std::setw(m_tLogDataW);
            if (tLogIsWrite(temp_waitReq.MCmd)) {
                // This was a write command, there should be data
                // in the request
                (*m_tLogPtr) << temp_waitReq.data;
            } else {
                // This was a read command, there should be data
                // in the response
                assert((m_chPtr->m_currentResponse).SDataPtr != NULL );
                (*m_tLogPtr) << (m_chPtr->m_currentResponse).SDataPtr[i];
            }
            (*m_tLogPtr) << std::setfill(' ');

            // Print the response portion of the transaction
            
            // ( (left parenthesis) / 1 / char / burstsinglereq == true
            if ((m_chPtr->ocpParams).burstsinglereq) {
                if (temp_waitReq.fillerResp) {
                    (*m_tLogPtr) << " (";
                } else {
                    (*m_tLogPtr) << "  ";
                }
            }
            // Resp(SResp) / 4 / char
            (*m_tLogPtr) << " " << tLogRespName((m_chPtr->m_currentResponse).SResp);
            // RI (SRespInfo) / (respinfo_wdth-1)/4 +1 / hex / respinfo
            if (m_tLogRespInfoW) {
                (*m_tLogPtr) << " ";
                (*m_tLogPtr) << std::setw(m_tLogRespInfoW);
                (*m_tLogPtr) << (m_chPtr->m_currentResponse).SRespInfo;
            }
            // L (last transfer of LastOfBurst=true resposne) / 1 / dec
            if ((m_chPtr->m_currentResponse).LastOfBurst && (i==(respCount-1))) {
                (*m_tLogPtr) << " 1";
            } else {
                (*m_tLogPtr) << " 0";
            }
            // ) (right parenthesis) / 1 / char / burstsinglereq == true
            if ((m_chPtr->ocpParams).burstsinglereq) {
                if (temp_waitReq.fillerResp) {
                    (*m_tLogPtr) << " )";
                } else {
                    (*m_tLogPtr) << "  ";
                }
            }
            // RqRpL (Request Response Latency / 5 / dec
            if ((m_chPtr->ocpParams).burstsinglereq && tLogIsWrite(temp_waitReq.MCmd)) {
                // There is only one response and thus only one response time
                (*m_tLogPtr) << " " << std::setw(5)
                    << (timeNow + (respCount-1)*(m_chPtr->getRDI())) 
                           - temp_waitReq.startCycle; 
            } else {
                // There are multiple responses:
                (*m_tLogPtr) << " " << std::setw(5) 
                    << (timeNow + i*(m_chPtr->getRDI())) - temp_waitReq.startCycle; 
            }
            // RpAL (Response Accept Latency) / 4 / dec
            (*m_tLogPtr) << " " << std::setw(4) << (m_chPtr->m_mTiming).RpAL;
            // MF (MFlag) / (mflag_wdth-1)/4 +1 / hex / mflag
            if (m_tLogMFW) {
                (*m_tLogPtr) << " ";
                (*m_tLogPtr) << std::setw(m_tLogMFW);
                (*m_tLogPtr) << (m_chPtr->m_sideband).MFlag ;
            }
            // SF (SFlag) / (sflag_wdth-1)/4 +1 / hex / sflag
            if (m_tLogSFW) {
                (*m_tLogPtr) << " ";
                (*m_tLogPtr) << std::setw(m_tLogSFW);
                (*m_tLogPtr) << (m_chPtr->m_sideband).SFlag ;
            }
            // I (SInterrupt) / 1 / dec / interrupt
            if ((m_chPtr->ocpParams).interrupt) {
                (*m_tLogPtr) << " " << (m_chPtr->m_sideband).SInterrupt;
            }
            // Done with this line
            (*m_tLogPtr) << endl;

            // All done. Remove this is request from the queue.
            m_tLogWaitQ[(m_chPtr->m_currentResponse).SThreadID].pop_front();

            // Go on to the next request in the burst
        }
    }


  protected:
    ostream *m_tLogPtr;
    OCP_TL2_Channel<Tdata, Taddr> *m_chPtr;
    deque< WaitReq<Tdata,Taddr> > *m_tLogWaitQ;
    unsigned int m_tLogThreadW;
    unsigned int m_tLogConnW;
    unsigned int m_tLogAddrSpaceW;
    unsigned int m_tLogAddrW;
    unsigned int m_tLogByteEnW;
    unsigned int m_tLogReqInfoW;
    unsigned int m_tLogDataW;
    unsigned int m_tLogRespInfoW;
    unsigned int m_tLogMFW;
    unsigned int m_tLogSFW;
    WaitReq<Tdata, Taddr> temp_waitReq;
};

#endif  // _OCP_TL2_TMON_H
