// 
//  (c) Copyright OCP-IP 2003
//  OCP-IP Confidential and Proprietary
//
// ============================================================================
//      Project : OCP SLD WG
//       Author : Alan Kamas for Sonics, Inc.
//                based on previous work by Yann Bajot, Norman Weyrich, 
//                Anssi Haverinen, and Joe Chou.
//          $Id :
//
//  Description : OCP TL2 Slave Interface
//
// ============================================================================

#ifndef _OCP_TL2_SLAVE_IF
#define _OCP_TL2_SLAVE_IF

#include "systemc.h"
#include "ocp_globals.h"
#include "ocp_param.h"

template <class Tdata, class Taddr>
class OCP_TL2_SlaveIF :  virtual public sc_interface
{
  public:

    //////////////////////////////////////////////////////////////
    // OCP TL2 Specific Slave Methods
    //////////////////////////////////////////////////////////////

    virtual bool getOCPRequest(OCPTL2RequestGrp<Tdata,Taddr>& req) =0;
    virtual bool getOCPRequestBlocking(OCPTL2RequestGrp<Tdata,Taddr>& req) =0;

    virtual bool sendOCPResponse(const OCPTL2ResponseGrp<Tdata>& resp) =0;
    virtual bool sendOCPResponseBlocking(const OCPTL2ResponseGrp<Tdata>& resp) =0;

    virtual bool acceptRequest(void) = 0;
    virtual bool acceptRequest(const sc_time& time) = 0;
    virtual bool acceptRequest(int cycles) = 0;

    virtual bool responseInProgress(void) = 0;
    virtual bool requestInProgress(void) = 0;

    // Get OCP Channel Parameter Pointer 
    virtual OCPParameters* GetParamCl(void) =0;

    // TL2 Timing Methods.
    virtual void getMasterTiming(MTimingGrp& mTimes) const = 0;
    virtual void getSlaveTiming(STimingGrp& sTimes) const = 0;
    virtual void putSlaveTiming(const STimingGrp& sTimes) = 0;
    virtual const sc_time& getPeriod(void) const = 0;

    // TL2 Tming Helper Functions
    virtual int getWDI(void) const = 0;
    virtual int getRqI(void) const = 0;
    virtual int getTL2ReqDuration(void) const = 0;
    virtual int getRDI(void) const = 0;
    virtual int getTL2RespDuration(void) const = 0;

    // ThreadBusy Commands
    virtual bool getMThreadBusyBit(unsigned int ThreadID = 0) const = 0;
    virtual bool getSThreadBusyBit(unsigned int ThreadID = 0) const = 0;
    virtual void putSThreadBusyBit(bool nextBitValue, unsigned int ThreadID = 0) = 0;
    
    // Reset Commands
    virtual void SResetAssert(void) =0;
    virtual void SResetDeassert(void) =0;
    virtual bool getReset(void) = 0;

    // SideBand Group Access methods
    virtual bool SgetMError(void) const = 0;
    virtual unsigned long long int SgetMFlag(void) const = 0;
    virtual void SputSError(bool nextValue) = 0;
    virtual void SputSFlag(unsigned long long int nextValue) = 0;
    virtual void SputSInterrupt(bool nextValue) = 0;
    virtual void SysputControl(unsigned int nextValue) = 0;
    virtual bool SysgetControlBusy(void) const = 0;
    virtual void SysputControlWr(bool nextValue) = 0;
    virtual unsigned int SysgetStatus(void) const = 0;
    virtual bool SysgetStatusBusy(void) const = 0;
    virtual void SysputStatusRd(bool nextValue) = 0;
    virtual unsigned int CgetControl(void) const = 0;
    virtual void CputControlBusy(bool nextValue) = 0;
    virtual bool CgetControlWr(void) const = 0;
    virtual void CputStatus(unsigned int nextValue) = 0;
    virtual void CputStatusBusy(bool nextValue) = 0;
    virtual bool CgetStatusRd(void) const = 0;

    // Event access
    virtual const sc_event& RequestStartEvent(void) const = 0; 
    virtual const sc_event& RequestEndEvent(void) const = 0; 
    virtual const sc_event& ResponseStartEvent(void) const = 0; 
    virtual const sc_event& ResponseEndEvent(void) const = 0; 
    virtual const sc_event& MThreadBusyEvent(void) const = 0; 
    virtual const sc_event& MasterTimingEvent(void) const = 0; 
    virtual const sc_event& ResetStartEvent(void) const = 0; 
    virtual const sc_event& ResetEndEvent(void) const = 0; 
    virtual const sc_event& SidebandMasterEvent(void) const = 0; 
    virtual const sc_event& SidebandSystemEvent(void) const = 0;
    virtual const sc_event& SidebandCoreEvent(void) const = 0;
    
    //////////////////////////////////////////////////////////////
    // OCP Original TL2 Slave Methods for backward compatibility
    //////////////////////////////////////////////////////////////

    virtual bool getOCPRequest(OCPRequestGrp<Tdata,Taddr>& req, 
            bool accept, 
            unsigned int& ReqChunkLen, 
            bool& last_chunk_of_a_burst) = 0;

    virtual bool getOCPRequestBlocking(
            OCPRequestGrp<Tdata,Taddr>& req, 
            bool accept, 
            unsigned int& ReqChunkLen, 
            bool& last_chunk_of_a_burst) = 0;

    virtual bool sendOCPResponse(const OCPResponseGrp<Tdata>& resp, 
            unsigned int RespChunkLen = 1, 
            bool last_chunk_of_a_burst = true) = 0;

    virtual bool startOCPResponse(const OCPResponseGrp<Tdata>& resp, 
            unsigned int RespChunkLen = 1, 
            bool last_chunk_of_a_burst = true) = 0;

    virtual bool sendOCPResponseBlocking(
            const OCPResponseGrp<Tdata>& resp, 
            unsigned int RespChunkLen = 1, 
            bool last_chunk_of_a_burst = true) = 0;

    virtual bool startOCPResponseBlocking(
            const OCPResponseGrp<Tdata>& resp, 
            unsigned int RespChunkLen = 1, 
            bool last_chunk_of_a_burst = true) = 0;

    virtual bool getMBusy(void) const = 0;

    virtual bool putSCmdAccept() = 0;

    virtual bool putSCmdAccept(const sc_time&) = 0;

    virtual bool getMRespAccept(void) const = 0;

    virtual void waitMRespAccept(void) = 0;

    // Old Style ThreadBusy Commands 
    virtual void putSThreadBusy(unsigned int nextMThreadBusy) = 0;
    virtual unsigned int getMThreadBusy(void) const = 0;
    virtual unsigned int getSThreadBusy(void) const = 0;

    // Original TL2 Reset commands for backward compatibility
    virtual void SputSReset_n(bool nextValue) = 0;
    virtual bool SgetMReset_n(void) const = 0;
    // Generic Reset commands
    // Generic reset, cycles through within one delta cycle
    virtual void Reset(void) = 0;
    virtual void reset(void) = 0;
    virtual void remove_reset(void) =0;
    virtual bool get_reset(void) = 0;

    // Original TL2 Sideband Events - no longer supported
    virtual const sc_event& SidebandControlEvent(void) const = 0;
    virtual const sc_event& SidebandStatusEvent(void) const = 0;

    //////////////////////////////////////////////////////////////
    // Generic Style Data Class Methods
    //////////////////////////////////////////////////////////////
    // These methods are not compatible with the performance TL2 channel
    // as the channel no longer uses the request/update scheme required
    // for these methods.
    // They produce an error message when run.
    // Request Group Access methods
    virtual  void GetRequestFields(OCPRequestGrp<Tdata,Taddr>& req, bool mdata_copy, unsigned int& ReqChunkLen, bool& last_chunk_of_a_burst) = 0;
    virtual void SetResponseFields(OCPResponseGrp<Tdata>& resp, unsigned int RespChunkLen = 1, bool last_chunk_of_a_burst = true) = 0;
    virtual  Taddr SgetMAddr() = 0;
    virtual  unsigned int SgetMAddrSpace() = 0;
    virtual  unsigned int SgetMAtomicLength() = 0;
    virtual  OCPMBurstSeqType SgetMBurstSeq() = 0;
    virtual  unsigned int SgetMByteEn() = 0;
    virtual  OCPMCmdType SgetMCmd() = 0;
    virtual  unsigned int SgetMConnID() = 0;
    virtual  Tdata* SgetMData(unsigned int& w, bool& last_of_a_burst) = 0;
    virtual  Tdata* SgetMData(unsigned int& w) = 0;
    virtual  Tdata* SgetMData() = 0;
    virtual  unsigned int SgetMThreadID() = 0;
    virtual  unsigned int SgetMBurstLength() = 0;
    virtual  unsigned int SgetMBurstPrecise() = 0;
    virtual  unsigned int SgetMBurstSingleReq() = 0;
    virtual  unsigned int SgetMReqInfo() = 0;
    virtual  unsigned int SgetMReqLast() = 0;
    // Response Group Access methods
    virtual void SputSData(Tdata* d, unsigned int w = 1, bool last_of_a_burst = true) = 0;
    virtual void SputSDataInfo(unsigned int a) = 0;
    virtual void SputSResp(OCPSRespType a) = 0;
    virtual void SputSRespInfo(unsigned int a) = 0;
    virtual void SputSRespLast(bool a) = 0;
    virtual void SputSThreadID(unsigned int a) = 0;
    // TL2-specific burst chunks methods
    virtual void SputSRespChunkLen(unsigned int w) = 0;
    virtual void SputSRespChunkLast(bool w) = 0;
    virtual unsigned int SgetMReqChunkLen() const = 0;
    virtual bool SgetMReqChunkLast() const = 0;
    // DataHS Group Access methods
    virtual  unsigned int SgetMDataInfo() = 0;
    // ThreadBusy Group Access methods
    virtual unsigned int SgetMThreadBusy(void) const = 0;
    virtual void SputSThreadBusy(unsigned int mthreadbusy) = 0;


    // Methods for getting requests and sending responses.
    virtual void Srelease() = 0;
    virtual void SreleasePE() = 0;
    virtual void SunreleasePE() = 0; // AHa Feb 04 Added
    virtual void Srelease(sc_time) = 0;
    virtual bool SgetRequest(bool) = 0;
    virtual bool SgetRequestBlocking(bool) = 0;
    virtual bool SgetRequestPE() = 0;
    virtual bool SputResponse() = 0;
    virtual bool SputResponseBlocking() = 0;
    virtual bool IsWrite() = 0;
    virtual bool SgetMbusy() const = 0;
    
    // second request channel (data handshake)
    virtual void SreleaseData() = 0;
    virtual void SreleaseDataPE() = 0;
    virtual void SunreleaseDataPE() = 0; 
    virtual void SreleaseData(sc_time) = 0;
    virtual bool SgetDataRequest(bool) = 0;
    virtual bool SgetDataRequestBlocking(bool) = 0;
    virtual bool SgetDataRequestPE() = 0;
    
    // No default event for this channel
    // virtual const sc_event& default_event() const = 0;
    
    // Access methods to the user defined data structures of the Channel.
    // Error message functions not defined due to template issues
    // virtual TdataCl          *GetDataCl()  = 0;
    // virtual ParamCl<TdataCl> *GetParamCl() = 0;
    // virtual CommCl           *GetCommCl()  = 0;
    // Direct access method Slave to Master direction.
    // virtual void SregisterDirectIF(SdirectIF<TdataCl>*) = 0;
    virtual bool SputDirect(int, bool, Tdata*, Taddr, int) = 0;
};

#endif // _OCP_TL2_SLAVE_IF

