// 
//  Copyright 2003 OCP-IP
//  OCP-IP Confidential & Proprietary
//
// ============================================================================
//      Project : OCP SLD WG
//      Authors : Yann Bajot, PROSILOG
//                Alan Kamas, Sonics Inc.
//         $Id: MasterSysC.cpp,v 1.2 2004/09/05 17:57:56 halexan Exp $
//
//  Description:  TL2 Master
//
//                This TL2 master emulates a '3 threads' master. It sends
//                requests labelled with a MThread ID varying from 0 to 2.
//                Depending on the current thread, each request targets a
//                different location in the target memory space (no overlap
//                between threads).
//
//                The master uses two SystemC threads, one for the requests and
//                one for the responses. The response thread checks that read
//                responses are correct. 
//                
//
//  Parameters:
//                The master accepts the following parameters:
//                  * mrespaccept_delay
//                  * mrespaccept_fixeddelay
//                  * command_cycles
//
//                The first two parameters are described in section 6.1.3 of
//                the OCP API documentation. Note that for TL2, delays are not
//                expressed in terms of clock cycles but as absolute timings
//                (unit is SC_NS in the master). 'Command_cycles' specifies the
//                number of times the predefined TL2 requests sequence is sent.
//
// ============================================================================



#include "MasterSysC.h"
#include "ocp_globals.h"
#include "ocp_tl2_data_cl.h"

#define NUM_REQUESTS        12 
#define MAX_THREADS         10

// ----------------------------------------------------------------------------
// constructor
// ----------------------------------------------------------------------------
MasterSysC::MasterSysC(
        sc_module_name name,
        ostream*       debug_os_ptr
        ) : sc_module(name),
        ipP("ipPort"),
        m_data_offset_cycle(0),
        m_debug_os_ptr(debug_os_ptr),
        m_OCPParamP(NULL),
        m_threads(1),
        m_addrspace(false),
        m_sthreadbusy(false),
        m_must_use_sthreadbusy(false),
        m_sdatathreadbusy(false),
        m_sdatathreadbusy_exact(false),
        m_mthreadbusy(false),
        m_must_use_mthreadbusy(false),
        m_respaccept(true),
        m_datahandshake(false),
        m_writeresp_enable(false),
        m_readex_enable(false),
        m_writenonpost_enable(false),
        m_max_burst_len(1),
        m_burstseq(false),
        m_burstsinglereq(false),
        m_address_incr(1),
        m_respaccept_fixeddelay(true),
        m_respaccept_delay(0),
        m_command_cycles(0),
m_command_cycles_persistent(0)
{
    // setup a SystemC thread process, which uses dynamic sensitive
    SC_THREAD(requestThreadProcess);

    // setup a SystemC thread process, which uses dynamic sensitive
    SC_THREAD(responseThreadProcess);
}

// ----------------------------------------------------------------------------
// destructor
// ----------------------------------------------------------------------------
MasterSysC::~MasterSysC()
{
    // do nothing
}

// ----------------------------------------------------------------------------
// SystemC Method MasterSysC::end_of_elaboration()
// ----------------------------------------------------------------------------
// 
//  At this point, everything has been built and connected.
//  We are now free to get our OCP parameters and to set up our
//  own variables that depend on them.
//
void MasterSysC::end_of_elaboration()
{
    // Call the System C version of this function first
    sc_module::end_of_elaboration();

    //-----------------------------------------
    //  OCP Parameters
    //-----------------------------------------

    // This Master adjusts to the OCP it is connected to.

    // Now get my OCP parameters from the port.
    m_OCPParamP = ipP->GetParamCl();

    // Get the number of threads
    m_threads = m_OCPParamP->threads;

    // is the MAddrSpace field part of the OCP channel?
    m_addrspace = m_OCPParamP->addrspace;

    // is SThreadBusy part of the channel?
    m_sthreadbusy = m_OCPParamP->sthreadbusy;

    // Is SThreadBusy compliance required?
    if (m_OCPParamP->sthreadbusy_exact) {
        m_must_use_sthreadbusy = true;
    } else if ( (m_OCPParamP->sthreadbusy) && (! m_OCPParamP->cmdaccept) ) {
        m_must_use_sthreadbusy = true;
    } else {
        m_must_use_sthreadbusy = false;
    }

    // is SDataThreadBusy part of the channel?
    m_sdatathreadbusy = m_OCPParamP->sdatathreadbusy;

    // Is SDataThreadBusy compliance required?
    m_sdatathreadbusy_exact = m_OCPParamP->sdatathreadbusy_exact;

    // is MThreadBusy part of the channel?
    m_mthreadbusy = m_OCPParamP->mthreadbusy;

    // is MRespAccept part of the channel?
    m_respaccept = m_OCPParamP->respaccept;

    // Is MThreadBusy compliance required?
    if (m_OCPParamP->mthreadbusy_exact) {
        m_must_use_mthreadbusy = true;
    } else if ( (m_OCPParamP->mthreadbusy) && (! m_OCPParamP->respaccept) ) {
        m_must_use_mthreadbusy = true;
    } else {
        m_must_use_mthreadbusy = false;
    }

    // is Data Handshake part of the channel?
    m_datahandshake = m_OCPParamP->datahandshake;

    // is write response part of the channel?
    m_writeresp_enable = m_OCPParamP->writeresp_enable;

    // is READ-EX part of the channel?
    m_readex_enable = m_OCPParamP->readex_enable;

    // Are non-posted writes (write commands that receive responses) part of the channel?
    m_writenonpost_enable = m_OCPParamP->writenonpost_enable;

    // Burst Parameters
    // The Slave only supports precise bursts with a maximum length of at least 4.
    m_max_burst_len = (1 << (m_OCPParamP->burstlength_wdth));
    m_burstseq = m_OCPParamP->burstseq;
    if (m_burstseq) {
        assert(m_OCPParamP->burstprecise);
        assert(m_OCPParamP->burstlength);
        assert(m_max_burst_len >= 4);
        assert(m_OCPParamP->burstseq_incr_enable);
        // data width must be a byte length
        assert( (m_OCPParamP->data_wdth)%8 == 0 );
    }
    // amount to increment the address by each burst transaction
    m_address_incr = (m_OCPParamP->data_wdth) / 8;
    // Support for single request / multiple data
    m_burstsinglereq = m_OCPParamP->burstsinglereq;

    // Single Request / Multiple Data not supported in TL2
    sc_assert(!m_burstsinglereq);

    // burstdatalast, burstreqlast, and burstresplast are not supported
    assert(!m_OCPParamP->datalast);
    assert(!m_OCPParamP->reqlast);
    assert(!m_OCPParamP->resplast);

    // writeresp_enable not supported
    assert(!m_writeresp_enable);

    // Thread Busy signals not supported 
    assert(!m_mthreadbusy);
    assert(!m_sthreadbusy);

    //-----------------------------------------
    //  Master Specific Parameters
    //-----------------------------------------

    // Retrieve any configuration parameters that were passed to this block
    // in the setConfiguration command.

#ifdef DEBUG
    cout << "I am configuring a Master!" << endl;
    cout << "Here is my configuration map for Master >" << name() << "< that was passed to me." << endl;
    MapStringType::iterator map_it;
    for (map_it = m_ParamMap.begin(); map_it != m_ParamMap.end(); ++map_it) {
        cout << "map[" << map_it->first << "] = " << map_it->second << endl;
    }
    cout << endl;
#endif

    string myPrefix = "";
    string paramName = "undefined";

    // MRespAccept delay in OCP cycles 
    paramName = "mrespaccept_delay";
    if (!(ParamCl<OCP_TL2_DataCl<Td,Ta> >::getIntOCPConfigValue(myPrefix, paramName, m_respaccept_delay, m_ParamMap)) ) {
        // Could not find the parameter so we must set it to a default
#ifdef DEBUG 
        cout << "Warning: master paramter \"" << paramName << "\" for Master \"" << name() << "\" was not found in the parameter map." << endl;
        cout << "         setting missing parameter to 1." << endl;
#endif
        m_respaccept_delay = 1;
    }

    // MRespAccept Delay Style. 1=fixed delay : 0=random delay
    paramName = "mrespaccept_fixeddelay";
    if (!(ParamCl<OCP_TL2_DataCl<Td,Ta> >::getBoolOCPConfigValue(myPrefix, paramName, m_respaccept_fixeddelay, m_ParamMap)) ) {
        // Could not find the parameter so we must set it to a default
#ifdef DEBUG 
        cout << "Warning: master paramter \"" << paramName << "\" for Master \"" << name() << "\" was not found in the parameter map." << endl;
        cout << "         setting missing parameter to 1 (fixed delay)." << endl;
#endif
        m_respaccept_fixeddelay = true;
    }

    // Command Set Repeat Count: Number of times to repeat the set of Master Commands. >0 = fixed number of cycles. zero = infinite repeats
    paramName = "command_cycles";
    if (!(ParamCl<OCP_TL2_DataCl<Td,Ta> >::getIntOCPConfigValue(myPrefix, paramName, m_command_cycles, m_ParamMap)) ) {
        // Could not find the parameter so we must set it to a default
#ifdef DEBUG 
        cout << "Warning: master paramter \"" << paramName << "\" for Master \"" << name() << "\" was not found in the parameter map." << endl;
        cout << "         setting missing parameter to 0 (continuously repeating)." << endl;
#endif
        m_command_cycles = 0;
    }

    m_command_cycles_persistent = m_command_cycles;
}


// --------------------------------------------------------------------------
// SetConfiguration
// --------------------------------------------------------------------------
void MasterSysC::setConfiguration( MapStringType& passedMap )
{
    // would be nice to process the Map immediately to avoid copying it and
    // storing it, but some of the parameters may be based on an OCP parameter.
    //
    // As long as Master parameters could be functions of OCP parameters, they will
    // have to be processed at the end of elaboration when the OCP parameters
    // are available.
    //
    // Save the Map until then
    m_ParamMap = passedMap;
}

// ----------------------------------------------------------------------------
// SystemC Thread Processes
// ----------------------------------------------------------------------------

void MasterSysC::requestThreadProcess()
{
    bool isWriteCmd;

    // Wait time before next request
    sc_time NumWait[NUM_REQUESTS];
    NumWait[0]=sc_time((double) 10 , SC_NS);
    NumWait[1]=sc_time((double) 10 , SC_NS);
    NumWait[2]=sc_time((double) 10 , SC_NS);
    NumWait[3]=sc_time((double) 10 , SC_NS);
    NumWait[4]=sc_time((double) 10 , SC_NS);
    NumWait[5]=sc_time((double) 10 , SC_NS);
    NumWait[6]=sc_time((double) 10 , SC_NS);
    NumWait[7]=sc_time((double) 10 , SC_NS);
    NumWait[8]=sc_time((double) 10 , SC_NS);
    NumWait[9]=sc_time((double) 10 , SC_NS);
    NumWait[10]=sc_time((double) 10, SC_NS);

    // Request Commands 
    OCPMCmdType Commands[NUM_REQUESTS] = { OCP_MCMD_WR, OCP_MCMD_WR, OCP_MCMD_WR, OCP_MCMD_WR, OCP_MCMD_RD, OCP_MCMD_RD, OCP_MCMD_RD, OCP_MCMD_WR, OCP_MCMD_WR, OCP_MCMD_RD, OCP_MCMD_RD, OCP_MCMD_RD };

    // The thread to use (if available) for each request
    unsigned int TestThread[NUM_REQUESTS] = {0, 1, 2, 0, 1, 2, 0, 1, 2, 0, 1, 2};

    // Request chunk length for each request
    unsigned int ReqChunkLen[NUM_REQUESTS] = {4, 2, 3, 3, 1, 2, 2, 4, 4, 1, 3, 1};

    // Start address for each request
    unsigned int StartAdress[NUM_REQUESTS] = {0, 100, 200, 16, 104, 200, 8, 116, 216, 24, 120, 224};


    // -----------------------------------
    // (1) processing and preparation step
    // -----------------------------------

    // initialize data
    OCPRequestGrp<Td,Ta> req;
    int              Nr = 0;
    sc_time          old_time;
    sc_time          current_time;

    // my_data[] contains the next data to be sent for each thread 
    unsigned int     my_data[MAX_THREADS] = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0};

    unsigned int     myThread = 0;

    // calculate the new waiting time
    sc_time  wait_for = NumWait[Nr];

    // Initial wait before the request thread starts
    wait(10,SC_NS);

    // main loop
    while (true) {
        // wait for the time to send the current request

        if (m_debug_os_ptr) {
            (*m_debug_os_ptr) << "DB (" << name() << "): "
                << "master wait_for = " << wait_for << endl;
        }

        wait(wait_for);

        // remember the time
        old_time = sc_time_stamp();

        // Set the thread for this test
        myThread = TestThread[Nr];
        if (int(myThread) >= m_threads) {
            // The thread for this test is out of the range of available threads.
            // Use "old reliable" thread 0 instead.
            myThread = 0;
            cout << "ThreadID value exceed ocp 'threads' parameter" << endl;
            sc_stop();
        }

        // ------------------------------------------------
        // (2) send a request chunk
        // ------------------------------------------------

        // Compute the next request
        req.MCmd = Commands[Nr];
        req.MThreadID = myThread;
        req.MAddr = StartAdress[Nr];

        // is this an extended command to be sent over a basic channel?
        if ( (!m_readex_enable) && (req.MCmd == OCP_MCMD_RDEX) ) {
            // channel cannot handle READ-EX. Send simple READ.
            req.MCmd = OCP_MCMD_RD;
        } else if ( (!m_writenonpost_enable) && (req.MCmd == OCP_MCMD_WRNP) ) {
            // channel cannout handle WRITE-NP. Send simple WRITE.
            req.MCmd = OCP_MCMD_WR;
        }

        req.MByteEn = 0xf;

        // for burst
        if (m_burstseq) {
            // we use an incremental address sequence
            req.MBurstSeq = OCP_MBURSTSEQ_INCR;
        }   // anyway, TL2 slave will consider the address sequence is 'INCR'

        // compute the data
        Td data_sent[4];

        switch (req.MCmd) {
            case OCP_MCMD_WR:
            case OCP_MCMD_WRNP:
            case OCP_MCMD_WRC:
            case OCP_MCMD_BCST:
                // This is a write command - it has data
                isWriteCmd = true;

                // put the data into the request
                // NOTE for TL2:
                // since we send a request chunk, we use the MDataPtr field of the request instead of the MData (TL1).
                for(unsigned int i = 0; i < ReqChunkLen[Nr]; i++) {
                    data_sent[i] = my_data[myThread] + myThread*100 + m_data_offset_cycle;
                    my_data[myThread]++;
                }
                req.MDataPtr = data_sent;
                break;
            case OCP_MCMD_RD:
            case OCP_MCMD_RDEX:
            case OCP_MCMD_RDL:
                // this is a read command - no data.
                isWriteCmd = false;
                req.MDataPtr = NULL;
                break;
            default:
                cout << "ERROR: Master \"" << name() << "\" generates unknown command #"<< req.MCmd << endl;
                sc_stop();
        }

        // send the request 
        // NOTE for TL2: we send a complete request chunk, in place of several TL1 requests.

        ipP->sendOCPRequestBlocking(req,ReqChunkLen[Nr],true);

        if (m_debug_os_ptr) {
            (*m_debug_os_ptr) << "DB (" << name() << "): " << "sent request with data." << endl;
            (*m_debug_os_ptr) << "DB (" << name() << "): " << "    t = " << sc_simulation_time() << " MCmd: " << req.MCmd << endl;

            if(isWriteCmd) {
                (*m_debug_os_ptr) << "DB (" << name() << "): " << "     MData:";
                for(unsigned int i=0;i<ReqChunkLen[Nr];i++) {
                    (*m_debug_os_ptr) << req.MDataPtr[i] << " ";
                }
            }

            (*m_debug_os_ptr) << " MByteEn: " << req.MByteEn << endl;
            (*m_debug_os_ptr) << "DB (" << name() << "): " << "    MAddr: " << req.MAddr << " MThreadID: " << req.MThreadID << endl;
        }


        // -------------------------------
        // (1) processing and preparation step
        // -------------------------------

        // Go on to the next request
        if (++Nr >= NUM_REQUESTS) {
            // we have finished the cycle.  Should we start another?
            m_command_cycles--;

            m_data_offset_cycle += 1000;
            for(unsigned int i = 0; i < MAX_THREADS; i++) {
                my_data[i] = 0;
            }


            if (m_command_cycles == 0) {
                // no, that was our last cycle of commands
                // Work for this thread is finished. Now just sit here and rest.
                if (m_debug_os_ptr) {
                    (*m_debug_os_ptr) << "DB (" << name() << "): "
                        << "ALL REQUESTS HAVE BEEN SENT ....." << endl;
                }

                while (true) {
                    // Done sending commands. Sleep.
                    wait(10000,SC_NS);
                }
            } else {
                // Start another cycle
                Nr = 0;
            }
        }

        // calculate the new waiting time
        wait_for = NumWait[Nr];
        current_time = sc_time_stamp();
        sc_time delta_time = (current_time - old_time) ;
        if (delta_time >= wait_for) {
            wait_for = sc_time(0,SC_NS);
        } else {
            wait_for = wait_for - delta_time;
        }
    }
}

void MasterSysC::responseThreadProcess()
{

    // initialization
    OCPResponseGrp<Td> resp;
    sc_time         wait_for;

    // Initial wait before the response thread starts
    wait(10,SC_NS);

    // main loop
    while (true) {
        // ------------------------------------------------
        // (1) wait for a response (blocking wait)
        // ------------------------------------------------

        if (m_debug_os_ptr) {
            (*m_debug_os_ptr) << "DB (" << name() << "): "
                << "waiting for response." << endl;
        }

        // get the next response
        unsigned int chunk_length;
        bool chunk_last;

        ipP->getOCPResponseBlocking(resp,false,chunk_length,chunk_last);

        if (m_debug_os_ptr) {
            (*m_debug_os_ptr) << "DB (" << name() << "): "
                << "got response." << endl;
            (*m_debug_os_ptr) << "DB (" << name() << "): "
                << "    t = " << sc_simulation_time() << endl;

            (*m_debug_os_ptr) << "DB (" << name() << "): " << "    SData: ";

            for(unsigned int i = 0; i < chunk_length; i++) {
                (*m_debug_os_ptr) << resp.SDataPtr[i] << " ";
            }

            (*m_debug_os_ptr) << endl;
        }

        // ------------------------
        // (2) process the response
        // ------------------------

        // compute the response acceptance time

        // NOTE for TL2: to mimic the TL1 temporal behaviour, response
        // acceptance time for a response chunk equals the sum of the TL1
        // time for each response of this chunk

        if (m_respaccept_fixeddelay) {
            wait_for = sc_time(m_respaccept_delay * chunk_length,SC_NS);
        } else {
            // Go random up to max delay
            wait_for = sc_time( (int)((m_respaccept_delay+1) * rand() / (RAND_MAX + 1.0)) * chunk_length, SC_NS);
        }

        // --------------------------------------------------
        // (3) accept the response
        // --------------------------------------------------

        if (m_respaccept) {
            if (wait_for.value() == 0) {
                // Accept immediately
                ipP->putMRespAccept();
            } else {
                // Accept later
                wait(wait_for);
                ipP->putMRespAccept();
            }
        }

    }
}

