//
//  (c) Copyright OCP-IP 2003
//  OCP-IP Confidential and Proprietary
//  $Id :

#ifndef _SONICS_MEMORYCL_H
#define _SONICS_MEMORYCL_H

#include <ostream.h>
#include <map>
#include <string>
#include <bitset>
#include <math.h>

// define the MemoryCl class
template <typename TdataCl>
class MemoryCl {
  public:
    // --------------------------
    // public members and methods
    // --------------------------

    // type definitions
    typedef typename TdataCl::DataType Td;
    typedef typename TdataCl::AddrType Ta;
    typedef map< Ta, Td > MemMapType;

    // constructor
    // - address width in number of bits
    // - word size in number of bytes
    MemoryCl(string name, unsigned int addr_width, unsigned int word_size)
      : m_Name(name),
        m_WordSize(word_size),
        m_LastByteEn(0),
        m_LastOldDataMask(0),
        m_LastNewDataMask(0)
    {
        checkNumericTypes();

        // setup content of the number of shift bits
        m_AddrShiftRight = shiftBits(word_size);

        // setup the byte enable mask
        setupByteEnMasks(word_size);

        // ---------------------------------
        // setup content of the address mask
        // ---------------------------------

        // adjust the address width, if needed
        if (addr_width > (sizeof(Ta)*8)) {
            cout << "Address Width (" << addr_width
                 << ") cannot be greater than "
                 << (sizeof(Ta)*8) << endl;
            addr_width = sizeof(Ta)*8;
        }

        // calculate the address mask
        m_AddrMask = fillOnes<Ta>( addr_width );
    }

    // destructor
    virtual ~MemoryCl(void)
    { }

    void
    clear(void)
    {
        // erases the memory
        m_Memory.clear();
        m_LastByteEn = 0;
        m_LastOldDataMask = 0;
        m_LastNewDataMask = 0;
    }
        
    void
    dumpState(ostream& os = std::cout)
    {
        typename MemMapType::iterator it;
        unsigned int         i = 0;

        os << hex << "Memory " << m_Name << endl;
        for (it = m_Memory.begin(); it != m_Memory.end(); ++it) {
            os << "Memory[" << it->first << "] = 0x" << hex
               << it->second << endl;
            i++;
        }
        os << "Memory size = " << i << endl;
    }

    // methods
    void
    read(Ta addr, Td& data, unsigned int byteen)
    {
        Td data_value;

        //
        byteen &= m_ByteEnMask;
        if (byteen == 0) {
            // return all-zeros data
            data = 0;
            return;
        }

        if (byteen != m_LastByteEn) {
            // handle the special (which is very common)
            if (byteen == m_ByteEnMask) {
                m_LastNewDataMask = m_OldDataMask;
                m_LastOldDataMask = 0;
            } else {
                recalculateByteMasks(byteen);
            }

            // save the byteen value
            m_LastByteEn = byteen;
        }

        // check whether the memory location has be initialized
        Ta my_addr = (addr >> m_AddrShiftRight);
        if (m_Memory.count(my_addr) == 0) {
            // read an uninitialized memory location

            // return all-zeros data
            data = 0;
            return;
        }

        // get the data value first
        data_value = m_Memory[my_addr];

        // only return the byte enabled part
        data = (m_LastNewDataMask & data_value) |
               (m_LastOldDataMask & data);
    }

    void
    write(Ta addr, Td data, unsigned int byteen)
    {
        Td data_value;

        //
        byteen &= m_ByteEnMask;
        if (byteen == 0) {
            // return all-zeros data
            data = 0;
            return;
        }

        //cout << << "m_LastNewDataMask " << m_LastNewDataMask << endl;
        //cout << "m_LastOldDataMask " << m_LastOldDataMask << endl;

        if (byteen != m_LastByteEn) {
            // handle the special (which is very common)
            if (byteen == m_ByteEnMask) {
                m_LastNewDataMask = m_OldDataMask;
                m_LastOldDataMask = 0;
            } else {
                recalculateByteMasks(byteen);
            }

            // save the byteen value
            m_LastByteEn = byteen;
        }

        // check whether the memory location has be initialized
        Ta my_addr = (addr >> m_AddrShiftRight);
        if (m_Memory.count(my_addr) == 0) {
            // write to an uninitialized memory location
            m_Memory[my_addr] = data & m_LastNewDataMask;
        } else {
            // write to an used memory location

            // get the data value first
            data_value = m_Memory[my_addr];

            // only update the byte enabled part
            m_Memory[my_addr] = (m_LastNewDataMask & data) |
                                (m_LastOldDataMask & data_value);
        }

        //
        //cout << "Memory[" << my_addr <<"] " << m_Memory[my_addr] << endl;
        //cout << "m_LastNewDataMask " << m_LastNewDataMask << endl;
        //cout << "m_LastOldDataMask " << m_LastOldDataMask << endl;
        //cout << "addr " << addr << ", data " << data << ", byteen "
        //     << byteen << endl;

        //dumpState();
    }

  private:

    // Utility to fill a number with ones
    template<typename numT>
    static numT fillOnes( unsigned int numbits = 8*sizeof( numT ) )
    {
        // Only works up to 64 bits
        unsigned long long tmp = 0ULL;
        if ( numbits < 8*sizeof( numT ) ) {
            tmp = ( 0x1ULL << numbits ) - 1;
        } else {
            memset( &tmp, 0xff, sizeof( unsigned long long ) );
        }
        return static_cast<numT>( tmp );
    }

    static bool checkNumericTypes()
    {
        return sizeof( Ta ) <= 8;
    }

    //
    void
    recalculateByteMasks(unsigned int byteen)
    {
        // re-calculate the masks
        bitset<16>    my_byte_en = byteen;
        unsigned int  i;
        Td            curByteMask;
        Td            curByteNegMask;

        // reset the masks
        m_LastNewDataMask = 0;
        m_LastOldDataMask = m_OldDataMask;
        curByteMask = 0xff;
        curByteNegMask = ~m_OldDataMask;

        // rebuild the masks
        for (i = 0; i < m_NumByteEnBits; i++) {
            // update the masks, if the current byte is enabled
            if (my_byte_en[i]) {
                m_LastNewDataMask |= curByteMask;
                m_LastOldDataMask &= curByteNegMask;
            }

            // reset the current byteen
            my_byte_en[i] = 0;
            if (my_byte_en == 0) {
                // done
                return;
            }

            // update the current byte mask and negative mask
            curByteNegMask <<= 8;
            curByteNegMask |= 0xff;
            curByteMask <<= 8;
        }
    }

    //
    unsigned int
    shiftBits(unsigned int word_size)
    {
        switch (word_size) {
        case 1: return (0); break;
        case 2: return (1); break;
        case 4: return (2); break;
        case 8: return (3); break;
        case 16: return (4); break;
        case 32: return (5); break;
        default:
            // not supported
            cout << "Error: word size (bytes): " << word_size
                 << " is not supported." << endl;
            return (0);
            break;
        }
    }

    void
    setupByteEnMasks(unsigned int word_size)
    {
        switch (word_size) {
        case 1:
            m_ByteEnMask = 0x1;
            m_NumByteEnBits = 1;
            break;
        case 2:
            m_ByteEnMask = 0x3;
            m_NumByteEnBits = 2;
            break;
        case 4:
            m_ByteEnMask = 0xf;
            m_NumByteEnBits = 4;
            break;
        case 8:
            m_ByteEnMask = 0xff;
            m_NumByteEnBits = 8;
            break;
        //case 16:
        //    m_ByteEnMask = 0xffff;
        //    m_NumByteEnBits = 16;
        //    m_OldDataMask = ??;
        //    break;
        default:
            cout << "Non-supported Word Size ("
                 << word_size << ") found." << endl;
            exit(-1);
            break;
        }
        m_OldDataMask = fillOnes<Td>();
    }

  private:
    //
    string       m_Name;
    unsigned int m_WordSize;  // in bytes
    unsigned int m_AddrShiftRight;
    unsigned int m_ByteEnMask;
    unsigned int m_NumByteEnBits;
    Ta           m_AddrMask;
    Td           m_OldDataMask;

    //
    unsigned int m_LastByteEn;
    Td           m_LastOldDataMask;
    Td           m_LastNewDataMask;
    MemMapType   m_Memory;
};

#endif // _SONICS_MEMORYCL_H
