// ============================================================================
//      Project :  OCP SLD WG, OCP Transaction Level
//       Author : Norman Weyrich, Synopsys Inc., weyrich@synopsys.com
//                Anssi Haverinen, Nokia Inc., anssi.haverinen@nokia.com
//                Joe Chou, Sonics Inc., joechou@sonicsinc.com
//         Date : 03/31/2003
//
//  Description : Transaction Level - Layer-2 data class to model
//                the Open Core Protocol (OCP) 1.0
//                 The data class can be used in conjunction with
//                OCP transaction channel (tl_channel.h) to transport
//                data between Initiator and Target modules.
//                Contains member variables and access methods
//                to set or querry member variables.
//                The member variables are split in to two groups:
//                - Request-Channel members (set by Master read by Slave)
//                - Response-Channel members (set by Slave, read by Master)
//
//                See definition of these groups in OCP 1.0 specification
// ============================================================================
//
//  Copyright @ 2003 OCP-IP
//
// ============================================================================

// This file contains the definition of the classes:
// template <class Td, class Ta> class OCP_TL2_DataCl

#ifndef _OCP_TL2_DATA_CL
#define _OCP_TL2_DATA_CL

#include "systemc.h"

// -------------------------------------------------
// MCmd encoding (OCP 1.0)
// -------------------------------------------------
enum OCPMCmdType {
  OCP_MCMD_IDLE = 0,
  OCP_MCMD_WR,
  OCP_MCMD_RD,
  OCP_MCMD_RDEX,
  OCP_MCMD_RESERVED4,
  OCP_MCMD_WRNP,
  OCP_MCMD_RESERVED6,
  OCP_MCMD_BCST
};

// -------------------------------------------------
// SResp encoding (OCP 1.0)
// -------------------------------------------------
enum OCPSRespType {
  OCP_SRESP_NULL = 0,
  OCP_SRESP_DVA,
  OCP_SRESP_RESERVED2,
  OCP_SRESP_ERR
};

// -------------------------------------------------
// MBurst encoding (OCP 1.0)
// -------------------------------------------------
enum OCPMBurstType {
  OCP_MBURST_LAST = 0,
  OCP_MBURST_DFLT1,
  OCP_MBURST_TWO,
  OCP_MBURST_DFLT2,
  OCP_MBURST_FOUR,
  OCP_MBURST_STRM,
  OCP_MBURST_EIGHT,
  OCP_MBURST_CONT
};

// -------------------------------------------------
//
// -------------------------------------------------
template <class Td, class Ta> class OCP_TL2_DataCl : public sc_prim_channel
{
public:

  typedef Td DataType;
  typedef Ta AddrType;

  // -------------------------------------------------
  // sideband signals
  // -------------------------------------------------
  struct OCPSidebandSignals {
    // reset signal
    bool Reset_n;

    // system/core signals
    int  Control;
    bool ControlWr;
    bool ControlBusy;
    int  Status;
    bool StatusRd;
    bool StatusBusy;

    // sideband signals
    int  MFlag;
    bool MError;
    int  SFlag;
    bool SError;
    bool SInterrupt;

    void reset()
    {
      SInterrupt = false;
      SError = false;
      MError = false;
      MFlag = 0;
      SFlag = 0;
      Control = 0;;
      ControlWr = false;
      ControlBusy = false;
      Status = 0;
      StatusRd = false;
      StatusBusy = false;
    }

    void copy(const OCPSidebandSignals& src)
    {
      Reset_n = src.Reset_n;
      SInterrupt= src.SInterrupt;
      SError= src.SError;
      MError= src.MError;
      SFlag= src.SFlag;
      MFlag= src.MFlag;
      Control= src.Control;
      ControlWr= src.ControlWr;
      ControlBusy= src.ControlBusy;
      Status= src.Status;
      StatusRd= src.StatusRd;
      StatusBusy= src.StatusBusy;
    }

    OCPSidebandSignals()
    {
      Reset_n = true;
      reset();
    }
  };

  // Constructor
  OCP_TL2_DataCl()
  {
    Reset();
  }

  // Destructor
  ~OCP_TL2_DataCl()
  {
  }

  void Reset()
  {
    // Request Signal Group
    m_MAddr[0] = m_MAddr[1]           = 0;
    m_MAddrSpace[0] = m_MAddrSpace[1] = 0;
    m_MBurst[0] = m_MBurst[1]         = OCP_MBURST_LAST;
    m_MByteEn[0] = m_MByteEn[1]       = 0;
    m_MCmd[0] = m_MCmd[1]             = OCP_MCMD_IDLE;
    m_MConnID[0] = m_MConnID[1]       = 0;
    m_MData[0] = m_MData[1]           = NULL;
    m_MThreadID[0] = m_MThreadID[1]   = 0;
    
    //
    m_MDataLast[0] = m_MDataLast[1] = true;
    m_MDataWordLength[0] = m_MDataWordLength[1] = 0;

    //
    m_MThreadBusy[0] = m_MThreadBusy[1] = false;

    // Response Signal Group
    m_SData[0] = m_SData[1] = NULL;
    m_SResp[0] = m_SResp[1] = OCP_SRESP_NULL;
    m_SThreadID[0] = m_SThreadID[1] = 0;

    //
    m_SAddr[0] = m_SAddr[1]           = 0;
    m_SCmd[0] = m_SCmd[1]             = OCP_MCMD_IDLE;
    m_SConnID[0] = m_SConnID[1]       = 0;

    //
    m_SThreadBusy[0] = m_SThreadBusy[1] = false;

    //
    m_SDataLast[0] = m_SDataLast[1] = true;
    m_SDataWordLength[0] = m_SDataWordLength[1] = 0;

    // initialize private members
    m_Synchron = false; // hard coded

    m_Request  = false;
    m_ReqToggle = 0;

    m_Response = false;
    m_ResToggle = 0;

    m_UgRequest = false;
    m_UgReqToggle = 0;

    m_UgResponse = false;
    m_UgResToggle = 0;

    //
    m_CurSidebands.reset();
    m_NextSidebands.reset();
  }

//---------------------------------------------------------------------
// -------------- Start of access methods -------------------------------

// These three methods are used in the Channel,
// and hence they must always be present!

  bool IsWriteRequest()
  {
    return((m_MCmd[1 - m_ReqToggle] == OCP_MCMD_WR) ? true : false);
  }
  void SetWriteRequest()
  {
    m_MCmd[m_ReqToggle] = OCP_MCMD_WR;
  }
  void SetReadRequest() 
  {
    m_MCmd[m_ReqToggle] = OCP_MCMD_RD;
  }

  // Makes new initiator request data visible to the target.
  // ToggleRequest() must be called for: 
  // SetWriteRequest(), SetReadRequest(), MputMCmd()
  void ToggleRequest()
  {
    m_Request = true;
    if (m_Synchron) request_update();
    else update();
  }

  void ToggleDataRequest()
  {
    cout << "Error: OCP_TL2_DataCL::ToggleDataRequest() is not supported"
         << " <" << name() << ">" << endl;
    assert(0);
  }

  // Makes new target response data visible to the initiator.
  // ToggleRequest() must be called for: SputSResp()
  void ToggleResponse()
  {
    m_Response = true;
    if (m_Synchron) request_update();
    else update();
  }

  // Time stamp methods: These methods could e.g. be used by
  // Masters and Slaves to indicate a duration of a packet.
  void MputEndTime(sc_time tt)
  {
    ReqEndTime = tt;
  }
  sc_time SgetEndTime()
  {
    return ReqEndTime;
  }
  void SputEndTime(sc_time tt)
  {
    ResEndTime = tt;
  }
  sc_time MgetEndTime()
  {
    return ResEndTime;
  }

//-----------------------------------------------------------
// The following methods are needed to run the examples
// We recommend to use them in user applications as well
//
// These are access methods for data fields to
//   - make code more readable and
//   - make Master and Slave code independent of the
//     data structure. If a different data structure is used
//     only the data class must be changed but not Master and Slave code.


  // ---------- Request Signal Group ----------
  // Request toggling is done whenever channel put request calls are used


  // Access methods for MAddr
  void MputMAddr(Ta a)
  {
    m_MAddr[m_ReqToggle] = a;
  }
  Ta SgetMAddr()
  {
    return m_MAddr[1 - m_ReqToggle];
  }

  // Access methods for MAddrSpace
  void MputMAddrSpace(int a)
  {
    m_MAddrSpace[m_ReqToggle] = a;
  }
  int SgetMAddrSpace()
  {
    return m_MAddrSpace[1 - m_ReqToggle];
  }

  // Access methods for MBurst
  void MputMBurst(OCPMBurstType a)
  {
    m_MBurst[m_ReqToggle] = a;
  }
  OCPMBurstType SgetMBurst()
  {
    return m_MBurst[1 - m_ReqToggle];
  }

  // Access methods for MByteEn
  void m_MputMByteEn(int a)
  {
    m_MByteEn[m_ReqToggle] = a;
  }
  int SgetMByteEn()
  {
    return m_MByteEn[1 - m_ReqToggle];
  }

  // Access methods for MCmd
  void MputMCmd(OCPMCmdType a)
  {
    m_MCmd[m_ReqToggle] = a;
  }
  OCPMCmdType SgetMCmd()
  {
    OCPMCmdType a = MCmd[1 - m_ReqToggle];
    MCmd[1 - m_ReqToggle] = OCP_MCMD_IDLE; // reset
    return a;
  }

  // Access methods for MConnID
  void MputMConnID(int a)
  {
    m_MConnID[m_ReqToggle] = a;
  }
  int SgetMConnID()
  {
    return m_MConnID[1 - m_ReqToggle];
  }

  // Automic length. Not part of OCP 1.0, but necessary for TL2 to work
  void MputAtomicLen(int w) {
    m_MDataWordLength[m_ReqToggle] = w;
  }
  int SgetAtomicLen() {
    return m_MDataWordLength[1 - m_ReqToggle];
  }
  void SputAtomicLen(int w) {
    m_SDataWordLength[m_ResToggle] = w;
  }
  int MgetAtomicLen() {
    return m_SDataWordLength[1 - m_ResToggle];
  }


  // Access methods for MData
  void MputMData(Td* d, int w = 1, bool last_of_a_burst = true)
  {
    m_MData[m_ReqToggle] = d;
    m_MDataWordLength[m_ReqToggle] = w;
    m_MDataLast[m_ReqToggle] = last_of_a_burst;
  }
  Td* SgetMData(int& w, bool& last_of_a_burst)
  {
    Td* data;
    int i = 1 - m_ReqToggle;

    data = m_MData[i];
    w = m_MDataWordLength[i];
    last_of_a_burst = m_MDataLast[i];

    return (data);
  }
  Td* SgetMData(int& w)
  {
    Td* data;
    int i = 1 - m_ReqToggle;

    data = m_MData[i];
    w = m_MDataWordLength[i];

    return (data);
  }
  Td* SgetMData()
  {
    Td* data;
    int i = 1 - m_ReqToggle;

    data = m_MData[i];
    return (data);
  }

  // Access methods for MThreadID
  void MputMThreadID(int a)
  {
    m_MThreadID[m_ReqToggle] = a;
  }
  int SgetMThreadID()
  {
    return m_MThreadID[1 - m_ReqToggle];
  }


  // ---------- Response Signal Group ----------
  // Response toggling is done whenever channel response methods is called

  // Access methods for SData
  void SputSData(Td* d, int w = 1, bool last_of_a_burst = true)
  {
    m_SData[m_ResToggle] = d;
    m_SDataWordLength[m_ResToggle] = w;
    m_SDataLast[m_ResToggle] = last_of_a_burst;
  }
  Td* MgetSData(int& w, bool& last_of_a_burst)
  {
    Td* data;
    int i = 1 - m_ResToggle;

    data = m_SData[i];
    w = m_SDataWordLength[i];
    last_of_a_burst = m_SDataLast[i];

    return (data);
  }
  Td* MgetSData(int& w)
  {
    Td* data;
    int i = 1 - m_ResToggle;

    data = m_SData[i];
    w = m_SDataWordLength[i];

    return (data);
  }
  Td* MgetSData()
  {
    Td* data;
    int i = 1 - m_ResToggle;

    data = m_SData[i];

    return (data);
  }


  // Access methods for SResp
  void SputSResp(OCPSRespType a)
  {
    m_SResp[m_ResToggle] = a;
  }
  OCPSRespType MgetSResp()
  {
    OCPSRespType a = m_SResp[1 - m_ResToggle];
    m_SResp[1 - m_ResToggle] = OCP_SRESP_NULL; // reset
    return a;
  }

  // Access methods for SThreadID
  void SputSThreadID(int a)
  {
    m_SThreadID[m_ResToggle] = a;
  }
  int MgetSThreadID()
  {
    return m_SThreadID[1 - m_ResToggle];
  }

  // Auxiliary methods for SAddr, not part of OCP 1.0
  void SputSAddr(Ta a)
  {
    m_SAddr[m_ResToggle] = a;
  }
  Ta MgetSAddr()
  {
    return m_SAddr[1 - m_ResToggle];
  }

  // Auxiliary methods to SCmd, not part of OCP 1.0
  bool IsWriteResponse()
  {
    return((m_SCmd[1 - m_ResToggle] == OCP_MCMD_WR) ? true : false);
  }
  void SputWriteResponse()
  {
    m_SCmd[m_ResToggle] = OCP_MCMD_WR;
  }
  void SputReadResponse()
  {
    m_SCmd[m_ResToggle] = OCP_MCMD_RD;
  }
  void SputSCmd(OCPMCmdType a)
  {
    m_SCmd[m_ResToggle] = a;
  }
  OCPMCmdType MgetSCmd()
  {
    return m_SCmd[1 - m_ResToggle];
  }

  // Auxiliary methods to SConnID, not part of OCP 1.0
  void SputSConnID(int a)
  {
    m_SConnID[m_ResToggle] = a;
  }
  int MgetSConnID()
  {
    return m_SConnID[1 - m_ResToggle];
  }

  
//---------------------------------------------------------------------
// The following members have not been throughly tested
// for this release, and may change in later releases.
// They are included for completeness.
//---------------------------------------------------------------------

  //---------------------------------------------------------------------
  // reset (Alpha version)
  //---------------------------------------------------------------------

  bool getReset_n()
  { return (m_CurSidebands.Reset_n); }

  void putReset_n(bool reset_n)
  { m_NextSidebands.Reset_n = reset_n; }

  //---------------------------------------------------------------------
  // system/core signaling (Alpha version)
  //---------------------------------------------------------------------

  int  CgetControl()
  { return (m_CurSidebands.Control); }

  void SysputControl(int control)
  { m_NextSidebands.Control = control; }

  int  CgetControlWr()
  { return (m_CurSidebands.ControlWr); }

  void SysputControlWr(bool controlwr)
  { m_NextSidebands.ControlWr = controlwr; }

  int  SysgetControlBusy()
  { return (m_CurSidebands.ControlBusy); }

  void CputControlBusy(bool controlbusy)
  { m_NextSidebands.ControlBusy = controlbusy; }

  int  SysgetStatus()
  { return (m_CurSidebands.Status); }

  void CputStatus(int status)
  { m_NextSidebands.Status = status; }

  int  SysgetStatusBusy()
  { return (m_CurSidebands.StatusBusy); }

  void CputStatusBusy(int statusbusy)
  { m_NextSidebands.StatusBusy = statusbusy; }

  int  CgetStatusRd()
  { return (m_CurSidebands.StatusRd); }

  void SysputStatusRd(int statusrd)
  { m_NextSidebands.StatusRd = statusrd; }

 
  //---------------------------------------------------------------------
  // threadbusy signaling
  //---------------------------------------------------------------------

  bool MgetSThreadBusy()
  {
    return m_SThreadBusy[1 - m_UgResToggle];
  }

  void SputSThreadBusy(int sthreadbusy)
  {
    m_SThreadBusy[m_UgResToggle] = sthreadbusy;
    m_UgResponse = true;
    if (m_Synchron) request_update();
    else update();
  }

  bool SgetMThreadBusy()
  {
    return m_MThreadBusy[1 - m_UgReqToggle];
  }

  void MputMThreadBusy(int mthreadbusy)
  {
    m_MThreadBusy[m_UgReqToggle] = mthreadbusy;
    m_UgRequest = true;
    if (m_Synchron) request_update();
    else update();
  }

  //---------------------------------------------------------------------
  // sideband signaling
  //---------------------------------------------------------------------

  bool MgetSError()
  { return (m_CurSidebands.SError); }

  void SputSError(bool serror)
  { m_NextSidebands.SError = serror; }

  bool MgetSInterrupt()
  { return (m_CurSidebands.SInterrupt); }

  void SputSInterrupt(bool sinterrupt)
  { m_NextSidebands.SInterrupt = sinterrupt; }

  int  MgetSFlag()
  { return (m_CurSidebands.SFlag); }

  void SputSFlag(int sflag)
  { m_NextSidebands.SFlag = sflag; }

  bool SgetMError()
  { return (m_CurSidebands.MError); }

  void MputMError(bool merror)
  { m_NextSidebands.MError = merror; }

  int  SgetMFlag()
  { return (m_CurSidebands.MFlag); }

  void MputMFlag(int mflag)
  { m_NextSidebands.MFlag = mflag; }


  //---------------------------------------------------------------------
  //
  //---------------------------------------------------------------------
  void update()
  {
    if (m_Request)  m_ReqToggle = 1 - m_ReqToggle;
    if (m_Response) m_ResToggle = 1 - m_ResToggle;
    if (m_UgRequest)  m_UgReqToggle = 1 - m_UgReqToggle;
    if (m_UgResponse) m_UgResToggle = 1 - m_UgResToggle;
    m_Request  = false;
    m_Response = false;
    m_UgRequest  = false;
    m_UgResponse = false;

    // ---------------
    // sideband update
    // ---------------
    m_CurSidebands.copy(m_NextSidebands);
    m_NextSidebands.ControlWr = false;
    m_NextSidebands.StatusRd = false;
  }

// -------------- End of access methods -------------------------------

private:

// --------------- Data members --------------- 

  // Request Signal Group
  Ta            m_MAddr[2];
  int           m_MAddrSpace[2];
  OCPMBurstType m_MBurst[2];
  int           m_MByteEn[2];
  OCPMCmdType   m_MCmd[2];
  int           m_MConnID[2];
  Td*           m_MData[2];
  int           m_MThreadID[2];

  // Response Signal Group
  Td*           m_SData[2];
  OCPSRespType  m_SResp[2];
  int           m_SThreadID[2];

  // helping signal value
  Ta            m_SAddr[2];
  OCPMCmdType   m_SCmd[2];
  int           m_SConnID[2];

  // ungrouped threadbusy signals
  bool          m_MThreadBusy[2];
  bool          m_SThreadBusy[2];

  //
  bool m_Synchron;

  // request channel update
  bool m_Request;
  int  m_ReqToggle;

  // response channel update
  bool m_Response;
  int  m_ResToggle;

  // ungrouped threadbusy signals update
  bool m_UgRequest;
  int  m_UgReqToggle;
  bool m_UgResponse;
  int  m_UgResToggle;

  // for sideband and threadbusy signals
  OCPSidebandSignals m_CurSidebands;
  OCPSidebandSignals m_NextSidebands;

  // -------------------------------------------------------------
  // for burst transactions (atomic and/or non-atomic)
  // -------------------------------------------------------------

  // for non-atomic burst transactions
  bool  m_MDataLast[2];
  bool  m_SDataLast[2];

  //
  int   m_MDataWordLength[2];
  int   m_SDataWordLength[2];

  // -------------------------------------------------------------
  // -------------------------------------------------------------

  // The ReqEndTime time stamps can be used to pass on information
  // about the durations of request packets to other modules.
  sc_time ReqEndTime;

  // The ResEndTime time stamps can be used to pass on information
  // about the durations of response packets to other modules.
  sc_time ResEndTime;
};

#endif // _OCP_TL2_DATA_CL
