///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// Copyright 2004 OCP-IP
// OCP-IP Confidential & Proprietary
//
//
//============================================================================
//      Project : OCP SLD WG
//       Author : Alan Kamas for Sonics, Inc.
//          $Id :
//
//  Description :  OCP TL2 SystemC Channel Transaction Monitor
//        For use with the performance OCP TL2 channel.
//        Outputs a view of each transaction moving through the OCP channel.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef _OCP_TL2_TRACE_MONITOR_H
#define _OCP_TL2_TRACE_MONITOR_H

#include <iomanip>
#include "ocp_param.h"
#include "ocp_globals.h"
#include <string>
#include <deque>
#include <vector>

#include "ocp_tl2_monitor_observer_if.h"
#include "ocp_tl2_monitor_port.h"

// -----------------------------------------------------------------

// forward declaration needed for the monitor class below:
template <class Tdata, class Taddr> class OCP_TL2_Channel;

// Used for transaction logging
template <class Td, class Ta>
struct WaitReq
{
    double startCycle;
    unsigned int MConnID;
    OCPMCmdType MCmd;
    OCPMBurstSeqType MBurstSeq;
    unsigned int tCount;
    unsigned int DataLength;
    unsigned int MBurstLength;
    bool lastReqOfBurst;
    unsigned int MAddrSpace;
    Ta MAddr;
    unsigned int MByteEn;
    unsigned long long MReqInfo;
    unsigned int RqDL;
    unsigned int RqAL;
    unsigned int DAL;
    Td data;
    bool fillerReq;
    bool fillerResp;
};

template <class Tdata, class Taddr>
class OCP_TL2_Trace_Monitor :
  public OCP_TL2_Monitor_ObserverIF<Tdata,Taddr>,
  public sc_module
{

  public:
  typedef typename OCP_TL2_Monitor_ObserverIF<Tdata,Taddr>::tl2_peek_type tl2_peek_type;

  SC_HAS_PROCESS(OCP_TL2_Trace_Monitor);
    // Constructor
    OCP_TL2_MonitorPort<Tdata,Taddr> p_mon;

    OCP_TL2_Trace_Monitor( sc_module_name name, ostream *traceStreamPtr ) :
        sc_module(name),
	p_mon("p_mon"),
        m_tLogPtr(traceStreamPtr),
        m_tLogWaitQ(NULL)
    { 
    }

    //---------------------------------------------------------------
    // destructor
    //---------------------------------------------------------------
    ~OCP_TL2_Trace_Monitor()
    { 
        if (m_tLogWaitQ) {
            delete [] m_tLogWaitQ;
        }
    }

  void end_of_elaboration() 
  {
    p_mon->RegisterRequestStart(this);
    p_mon->RegisterResponseStart(this);

    tLogSetUp();
    tLogHeader();
  }

    //---------------------------------------------------------------
    // Transaction Logging Functions
    //---------------------------------------------------------------
    //
    // Uses the new "ocpdis2 -w transfer" transfer style format to
    // display the transaction log.
    // 
    // The output has the following format:
    // Field(full name) / display width / base / [condition (if any)]
    // SimTime / 9 / dec
    // Cycle / 7 / dec
    // ( (left parenthesis) / 1 / char / burstsinglereq == true
    // T(Thread) / (ocpParams.threads-1)/10 +1 / dec / ocpParams.threads > 1
    // C(ConnID) / (connid_wdth-1)/4 +1 / hex / connid == true
    // Cmd (MCmd) / 4 / char
    // BuS (MBurstSeq) / 4 / char 
    // Burst (transfer count/MBurstLength) / 
    //          5 (tcount=2 right, "/"=1, dLen=2 left) / dec
    // l (last request of LastOfBurst=true request) / 1 / dec
    // AS (MAddressSpace) / (addrspace_wdth-1)/4 +1 / hex / addrspace
    // Addr (MAddr) / (addr_wdth-1)/4 +1 / hex
    // ) (right parenthesis) / 1 / char / burstsinglereq == true
    // BE (byteen) / (data_wdth-1)/32 +1 / hex / byteen
    // Ri(MReqInfo) / (reqinfo_wdth-1)/4 +1 / hex / reqinfo
    // RqDL (Request Data Latency) / 4 / dec
    // RqAL (Request Accept Latency) / 4 / dec
    // DAL (Data Accept Latency) / 3 / dec
    // Data (MData or SData) / (data_wdth-1)/4 +1 
    // ( (left parenthesis) / 1 / char / burstsinglereq == true
    // Resp(SResp) / 4 / char
    // RI (SRespInfo) / (respinfo_wdth-1)/4 +1 / hex / respinfo
    // L (last response of LastOfBurst=true response) / 1 / dec
    // ) (right parenthesis) / 1 / char / burstsinglereq == true
    // RqRpL (Request Response Latency / 5 / dec
    // RpAL (Response Accept Latency) / 4 / dec
    // MF (MFlag) / (mflag_wdth-1)/4 +1 / hex / mflag
    // SF (SFlag) / (sflag_wdth-1)/4 +1 / hex / sflag
    // I (SInterrupt) / 1 / dec / interrupt

    // Prints the Monitor header lines and computes each of the field
    // widths.
    // NOTE that a field width of zero means that the field is not being used.
    void tLogHeader( void )
    {
        (*m_tLogPtr).setf(std::ios::right);
        (*m_tLogPtr) << "##" << endl;
        (*m_tLogPtr) << 
            "# Legend for \"transfer\" write style (sorted by request cycle)" 
            << endl;
        (*m_tLogPtr) << "##" << endl;
        (*m_tLogPtr) << 
            "# SimTime  :                        Simulation Time : Time"
            << endl;
        (*m_tLogPtr) << 
            "# Cycle    :                       Cycle Valid Time : Cycle"
            << endl;
        if (p_mon->GetParamCl()->threads > 1) {
            (*m_tLogPtr) << 
                "# T        :             Master and Slave Thread ID : ThreadID"
                << endl;
        }
        if (p_mon->GetParamCl()->connid) {
            (*m_tLogPtr) << 
                "# C        :                 Connection indentifier : MConnID"
                << endl;
        }
        (*m_tLogPtr) << 
            "# Cmd      :                         Master Command : MCmd"
            << endl;
        (*m_tLogPtr) << 
            "# BuS      :                    Burst Sequence Type : MBurstSeq"
            << endl;
        (*m_tLogPtr) << 
            "# Burst    :  req burst transfer count/burst length : transfer count/MBurstLength"
            << endl;
        (*m_tLogPtr) << 
            "# l        :last transfer of req w/LastOfBurst true : request LastOfBurst"
            << endl;
        if (p_mon->GetParamCl()->addrspace) {
            (*m_tLogPtr) << 
                "# AS       :                   Master Address Space : MAddrSpace"
                << endl;
        }
        (*m_tLogPtr) << 
            "# Addr     :                         Master Address : MAddr"
            << endl;
        if (p_mon->GetParamCl()->byteen) {
            (*m_tLogPtr) << 
                "# BE       :                           Byte Enables : MByteEn"
                << endl;
        }
        if (p_mon->GetParamCl()->reqinfo) {
            (*m_tLogPtr) << 
                "# Ri       :                           Request Info : MReqInfo"
                << endl;
        }
        (*m_tLogPtr) << 
            "# RqDL     :                   Request Data Latency : RqDL" 
            << endl;
        (*m_tLogPtr) << 
            "# RqAL     :                 Request Accept Latency : RqAL" 
            << endl;
        (*m_tLogPtr) << 
            "# DAL      :                    Data Accpet Latency : DAL" 
            << endl;
        (*m_tLogPtr) << 
            "# Data     :    MData (if write) or SData (if read) : Data "
            << endl;
        (*m_tLogPtr) << 
            "# Resp     :                         Slave Response : SResp "
            << endl;
        if (p_mon->GetParamCl()->respinfo) {
            (*m_tLogPtr) << 
                "# RI       :                          Response Info : MRespInfo"
                << endl;
        }
        (*m_tLogPtr) << 
            "# L        :last transfer of resp w/LastOfBurst true: response LastOfBurst"
            << endl;
        (*m_tLogPtr) << 
            "# RqRpL    :               Request Response Latency : RqRpL" 
            << endl;
        (*m_tLogPtr) << 
            "# RpAL     :                Resposne Accept Latency : RpAL" 
            << endl;
        if (p_mon->GetParamCl()->mflag) {
            (*m_tLogPtr) << 
                "# MF       :                           Master Flags : MFlag" 
                << endl;
        }
        if (p_mon->GetParamCl()->sflag) {
            (*m_tLogPtr) << 
                "# SF       :                            Slave Flags : SFlag" 
                << endl;
        }
        if (p_mon->GetParamCl()->interrupt) {
            (*m_tLogPtr) << 
                "# I        :                        Slave Interrupt : SInterrupt" 
                << endl;
        }
        (*m_tLogPtr) << endl;

        // Print Header Row
        (*m_tLogPtr) << std::setfill(' ');
        // SimTime / 9 / dec
        (*m_tLogPtr) << "  SimTime" << " ";
        // Cycle / 7 / dec
        (*m_tLogPtr) << "  Cycle" << " ";
        // ( (left parenthesis) / 1 / char / burstsinglereq == true
        if (p_mon->GetParamCl()->burstsinglereq) {
            (*m_tLogPtr) << " " << " ";
        }
        // T(Thread) / (p_mon->GetParamCl()->threads-1)/10 +1 / dec / p_mon->GetParamCl()->threads > 1
        m_tLogThreadW = 0;
        if (p_mon->GetParamCl()->threads>1) {
            m_tLogThreadW = int((p_mon->GetParamCl()->threads-1)/10) +1;
            (*m_tLogPtr) << std::setw(m_tLogThreadW);
            (*m_tLogPtr) << "T" << " ";
        }
        // C(ConnID) / (connid_wdth-1)/4 +1 / hex / connid == true
        m_tLogConnW = 0;
        if (p_mon->GetParamCl()->connid) {
            m_tLogConnW = int((p_mon->GetParamCl()->connid_wdth-1)/4) +1;
            (*m_tLogPtr) << std::setw(m_tLogConnW);
            (*m_tLogPtr) << "C" << " ";
        }
        // Cmd (MCmd) / 4 / char
        (*m_tLogPtr) << " CMD" << " ";
        // BuS (MBurstSeq) / 4 / char 
        (*m_tLogPtr) << " BuS" << " ";
        // Burst (transfer count/MBurstLength) / 
        //          5 (tcount=2 right, "/"=1, dLen=2 left) / dec
        (*m_tLogPtr) << "Burst" << " ";
        // l (last transfer of LastOfBurst=true request) / 1 / dec
        (*m_tLogPtr) << "l" << " ";
        // AS (MAddressSpace) / (addrspace_wdth-1)/4 +1 / hex / addrspace
        m_tLogAddrSpaceW = 0;
        if (p_mon->GetParamCl()->addrspace) {
            m_tLogAddrSpaceW = int((p_mon->GetParamCl()->addrspace_wdth-1)/4) +1;
            if (m_tLogAddrSpaceW < 2 ) {
                m_tLogAddrSpaceW = 2;
            }
            (*m_tLogPtr) << std::setw(m_tLogAddrSpaceW);
            (*m_tLogPtr) << "AS" << " ";
        }
        // Addr (MAddr) / (addr_wdth-1)/4 +1 / hex
        m_tLogAddrW = int((p_mon->GetParamCl()->addr_wdth-1)/4) +1;
        if (m_tLogAddrW < 4 ) {
            m_tLogAddrW = 4;
        }
        (*m_tLogPtr) << std::setw(m_tLogAddrW);
        (*m_tLogPtr) << "Addr" << " ";
        // ) (right parenthesis) / 1 / char / burstsinglereq == true
        if (p_mon->GetParamCl()->burstsinglereq) {
            (*m_tLogPtr) << " " << " ";
        }
        // BE (byteen) / (data_wdth-1)/32 +1 / hex / byteen
        m_tLogByteEnW = 0;
        if (p_mon->GetParamCl()->byteen) {
            m_tLogByteEnW = int((p_mon->GetParamCl()->data_wdth-1)/32) +1;
            if (m_tLogByteEnW < 2 ) {
                m_tLogByteEnW = 2;
            }
            (*m_tLogPtr) << std::setw(m_tLogByteEnW);
            (*m_tLogPtr) << "BE" << " ";
        }
        // Ri(MReqInfo) / (reqinfo_wdth-1)/4 +1 / hex / reqinfo
        m_tLogReqInfoW = 0;
        if (p_mon->GetParamCl()->reqinfo) {
            m_tLogReqInfoW = int((p_mon->GetParamCl()->reqinfo_wdth-1)/4) +1;
            if (m_tLogReqInfoW < 2 ) {
                m_tLogReqInfoW = 2;
            }
            (*m_tLogPtr) << std::setw(m_tLogReqInfoW);
            (*m_tLogPtr) << "Ri" << " ";
        }
        // RqDL (Request Data Latency) / 4 / dec
        (*m_tLogPtr) << "RqDL" << " ";
        // RqAL (Request Accept Latency) / 4 / dec
        (*m_tLogPtr) << "RqAL" << " ";
        // DAL (Data Accept Latency) / 3 / dec
        (*m_tLogPtr) << "DAL" << " ";
        // Data (MData or SData) / (data_wdth-1)/4 +1 
        m_tLogDataW = int((p_mon->GetParamCl()->data_wdth-1)/4) +1;
        if (m_tLogDataW < 4 ) {
            m_tLogDataW = 4;
        }
        (*m_tLogPtr) << std::setw(m_tLogDataW);
        (*m_tLogPtr) << "Data" << " ";
        // ( (left parenthesis) / 1 / char / burstsinglereq == true
        if (p_mon->GetParamCl()->burstsinglereq) {
            (*m_tLogPtr) << " " << " ";
        }
        // Resp(SResp) / 4 / char
        (*m_tLogPtr) << "Resp" << " ";
        // RI (SRespInfo) / (respinfo_wdth-1)/4 +1 / hex / respinfo
        m_tLogRespInfoW = 0;
        if (p_mon->GetParamCl()->respinfo) {
            m_tLogRespInfoW = int((p_mon->GetParamCl()->respinfo_wdth-1)/4) +1;
            if (m_tLogRespInfoW < 2 ) {
                m_tLogRespInfoW = 2;
            }
            (*m_tLogPtr) << std::setw(m_tLogRespInfoW);
            (*m_tLogPtr) << "RI" << " ";
        }
        // L (last transfer of LastOfBurst=true resposne) / 1 / dec
        (*m_tLogPtr) << "L" << " ";
        // ) (right parenthesis) / 1 / char / burstsinglereq == true
        if (p_mon->GetParamCl()->burstsinglereq) {
            (*m_tLogPtr) << " " << " ";
        }
        // RqRpL (Request Response Latency / 5 / dec
        (*m_tLogPtr) << "RqRpL" << " ";
        // RpAL (Response Accept Latency) / 4 / dec
        (*m_tLogPtr) << "RpAL" << " ";
        // MF (MFlag) / (mflag_wdth-1)/4 +1 / hex / mflag
        m_tLogMFW = 0;
        if (p_mon->GetParamCl()->mflag) {
            m_tLogMFW = int((p_mon->GetParamCl()->mflag_wdth-1)/4) +1;
            if (m_tLogMFW < 2 ) {
                m_tLogMFW = 2;
            }
            (*m_tLogPtr) << std::setw(m_tLogMFW);
            (*m_tLogPtr) << "MF" << " ";
        }
        // SF (SFlag) / (sflag_wdth-1)/4 +1 / hex / sflag
        m_tLogSFW = 0;
        if (p_mon->GetParamCl()->sflag) {
            m_tLogSFW = int((p_mon->GetParamCl()->sflag_wdth-1)/4) +1;
            if (m_tLogSFW < 2 ) {
                m_tLogSFW = 2;
            }
            (*m_tLogPtr) << std::setw(m_tLogSFW);
            (*m_tLogPtr) << "SF" << " ";
        }
        // I (SInterrupt) / 1 / dec / interrupt
        if (p_mon->GetParamCl()->interrupt) {
            (*m_tLogPtr) << "I";
        }
        // Done with this line
        (*m_tLogPtr) << endl;

        // Now print out the dashed underlines
        (*m_tLogPtr) << std::setfill('-');
        // SimTime / 9 / dec
        (*m_tLogPtr) << "---------";
        // Cycle / 7 / dec
        (*m_tLogPtr) << " -------";
        // ( (left parenthesis) / 1 / char / burstsinglereq == true
        if (p_mon->GetParamCl()->burstsinglereq) {
            (*m_tLogPtr) << " -";
        }
        // T(Thread) / (p_mon->GetParamCl()->threads-1)/10 +1 / dec / p_mon->GetParamCl()->threads > 1
        if (m_tLogThreadW) {
            (*m_tLogPtr) << " ";
            (*m_tLogPtr) << std::setw(m_tLogThreadW);
            (*m_tLogPtr) << "-";
        }
        // C(ConnID) / (connid_wdth-1)/4 +1 / hex / connid == true
        if (m_tLogConnW) {
            (*m_tLogPtr) << " ";
            (*m_tLogPtr) << std::setw(m_tLogConnW);
            (*m_tLogPtr) << "-";
        }
        // Cmd (MCmd) / 4 / char
        (*m_tLogPtr) << " ----";
        // BuS (MBurstSeq) / 4 / char 
        (*m_tLogPtr) << " ----";
        // Burst (transfer count/DataLength) / 
        //          5 (tcount=2 right, "/"=1, dLen=2 left) / dec
        (*m_tLogPtr) << " -----";
        // l (last transfer of LastOfBurst=true request) / 1 / dec
        (*m_tLogPtr) << " -";
        // AS (MAddressSpace) / (addrspace_wdth-1)/4 +1 / hex / addrspace
        if (m_tLogAddrSpaceW) {
            (*m_tLogPtr) << " ";
            (*m_tLogPtr) << std::setw(m_tLogAddrSpaceW);
            (*m_tLogPtr) << "-";
        }
        // Addr (MAddr) / (addr_wdth-1)/4 +1 / hex
        (*m_tLogPtr) << " ";
        (*m_tLogPtr) << std::setw(m_tLogAddrW);
        (*m_tLogPtr) << "-";
        // ) (right parenthesis) / 1 / char / burstsinglereq == true
        if (p_mon->GetParamCl()->burstsinglereq) {
            (*m_tLogPtr) << " -";
        }
        // BE (byteen) / (data_wdth-1)/32 +1 / hex / byteen
        if (m_tLogByteEnW) {
            (*m_tLogPtr) << " ";
            (*m_tLogPtr) << std::setw(m_tLogByteEnW);
            (*m_tLogPtr) << "-";
        }
        // Ri(MReqInfo) / (reqinfo_wdth-1)/4 +1 / hex / reqinfo
        if (m_tLogReqInfoW) {
            (*m_tLogPtr) << " ";
            (*m_tLogPtr) << std::setw(m_tLogReqInfoW);
            (*m_tLogPtr) << "-";
        }
        // RqDL (Request Data Latency) / 4 / dec
        (*m_tLogPtr) << " ----";
        // RqAL (Request Accept Latency) / 4 / dec
        (*m_tLogPtr) << " ----";
        // DAL (Data Accept Latency) / 3 / dec
        (*m_tLogPtr) << " ---";
        // Data (MData or SData) / (data_wdth-1)/4 +1 
        (*m_tLogPtr) << " ";
        (*m_tLogPtr) << std::setw(m_tLogDataW);
        (*m_tLogPtr) << "-";
        // ( (left parenthesis) / 1 / char / burstsinglereq == true
        if (p_mon->GetParamCl()->burstsinglereq) {
            (*m_tLogPtr) << " -";
        }
        // Resp(SResp) / 4 / char
        (*m_tLogPtr) << " ----";
        // RI (SRespInfo) / (respinfo_wdth-1)/4 +1 / hex / respinfo
        if (m_tLogRespInfoW) {
            (*m_tLogPtr) << " ";
            (*m_tLogPtr) << std::setw(m_tLogRespInfoW);
            (*m_tLogPtr) << "-";
        }
        // L (last transfer of LastOfBurst=true resposne) / 1 / dec
        (*m_tLogPtr) << " -";
        // ) (right parenthesis) / 1 / char / burstsinglereq == true
        if (p_mon->GetParamCl()->burstsinglereq) {
            (*m_tLogPtr) << " -";
        }
        // RqRpL (Request Response Latency / 5 / dec
        (*m_tLogPtr) << " -----";
        // RpAL (Response Accept Latency) / 4 / dec
        (*m_tLogPtr) << " ----";
        // MF (MFlag) / (mflag_wdth-1)/4 +1 / hex / mflag
        if (m_tLogMFW) {
            (*m_tLogPtr) << " ";
            (*m_tLogPtr) << std::setw(m_tLogMFW);
            (*m_tLogPtr) << "-";
        }
        // SF (SFlag) / (sflag_wdth-1)/4 +1 / hex / sflag
        if (m_tLogSFW) {
            (*m_tLogPtr) << " ";
            (*m_tLogPtr) << std::setw(m_tLogSFW);
            (*m_tLogPtr) << "-";
        }
        // I (SInterrupt) / 1 / dec / interrupt
        if (p_mon->GetParamCl()->interrupt) {
            (*m_tLogPtr) << " -";
        }
        // Done with this line
        (*m_tLogPtr) << endl;
        // Return to normal fill style
        (*m_tLogPtr) << std::setfill(' ');
    }

    void tLogSetUp( void )
    {
        // Set up request-waiting-for-resp-queues. One for each thread.
        if (m_tLogWaitQ) {
            // possible that we could be configured twice.
            delete [] m_tLogWaitQ;
        }
        m_tLogWaitQ = new deque< WaitReq<Tdata,Taddr> >[p_mon->GetParamCl()->threads];
       
        // Set up burst transfer counters
        xferCount.clear();
        for (int i=0; i< p_mon->GetParamCl()->threads; i++) {
            xferCount.push_back(0);
        }
    }

    const string tLogCmdName(OCPMCmdType myType) const
    {
        switch (myType) {
            case OCP_MCMD_IDLE:
                return "IDLE";
            case OCP_MCMD_WR:
                return "  WR";
            case OCP_MCMD_RD:
                return "  RD";
            case OCP_MCMD_RDEX:
                return "RDEX";
            case OCP_MCMD_RDL:
                return " RDL";
            case OCP_MCMD_WRNP:
                return "WRNP";
            case OCP_MCMD_WRC:
                return " WRC";
            case OCP_MCMD_BCST:
                return "BCST";
        }
        return "UNKNOWN";
    }

    const string tLogRespName(OCPSRespType myType) const
    {
        switch (myType) {
            case OCP_SRESP_NULL:
                return "NULL";
            case OCP_SRESP_DVA:
                return " DVA";
            case OCP_SRESP_FAIL:
                return "FAIL";
            case OCP_SRESP_ERR:
                return " ERR";
        }
        return "UNKNOWN";
    }

    const string tLogBSeqName(OCPMBurstSeqType myType) const
    {
        switch (myType) {
            case OCP_MBURSTSEQ_INCR:
                return "INCR";
            case OCP_MBURSTSEQ_DFLT1:
                return "DFT1";
            case OCP_MBURSTSEQ_WRAP:
                return "WRAP";
            case OCP_MBURSTSEQ_DFLT2:
                return "DFT2";
            case OCP_MBURSTSEQ_XOR:
                return " XOR";
            case OCP_MBURSTSEQ_STRM:
                return "STRM";
            case OCP_MBURSTSEQ_UNKN:
                return "UNKN";
            case OCP_MBURSTSEQ_RESERVED:
                return "RSVD";
        }
        return "UNKNOWN";
    }

    bool tLogIsWrite(OCPMCmdType myType) const
    {
        if ((myType == OCP_MCMD_WR) ||
            (myType == OCP_MCMD_WRNP) ||
            (myType == OCP_MCMD_WRC) ||
            (myType == OCP_MCMD_BCST) ) {
            return true;
        }
        return false;
    }


    virtual void NotifyRequestStart(tl2_peek_type *tl2_channel)
    {
        // We have a new request. Either it needs a response and 
        // printing must be delayed until the response arrives or
        // it does not need a response in which case it may be printed
        // immediately.

      cout << __PRETTY_FUNCTION__ << ", req : " << p_mon->peekOCPResponse() << endl;
        double myStart = sc_simulation_time();
        int myThread = (p_mon->peekOCPRequest()).MThreadID;
        // NOTE: start of the "request" could be the time when the 
        // request if first presented on the channel. It could also be
        // the time that the associated data is first put on the channel.
        // here, we'll use the start of the request.

        // Set up the address incrementing
        // NOTE: currently no support for addressing types other than
        // INCR, STRM, and WRAP.
        int myAddrInc = 0;
        if ((p_mon->peekOCPRequest()).MBurstSeq != OCP_MBURSTSEQ_STRM) {
            myAddrInc = (p_mon->GetParamCl()->data_wdth)/8;
        }

        // Does this request require a response?
        
        if ( (p_mon->peekOCPRequest()).MCmd == OCP_MCMD_IDLE ) {
            // An IDLE command. Do nothing.
        } else if ( (!p_mon->GetParamCl()->writeresp_enable) && 
                    (((p_mon->peekOCPRequest()).MCmd == OCP_MCMD_WR)
                      || ((p_mon->peekOCPRequest()).MCmd == OCP_MCMD_BCST)) ) {
            // This is a basic posted write command.
            // There will be no response so this transfer is now complete.

            // Print it out now.
            for (unsigned int i=0;i<(p_mon->peekOCPRequest()).DataLength;i++) {
                // NOTE: for now, cycle time and start time are the same
                // SimTime / 9 / dec
                (*m_tLogPtr).setf(std::ios::dec, std::ios::basefield);
                (*m_tLogPtr) << std::setw(9) << myStart + i*(p_mon->getWDI());
                // Cycle / 7 / dec
                (*m_tLogPtr) << " " << std::setw(7) << myStart + i*(p_mon->getWDI());

                // ( (left parenthesis) / 1 / char 
                if (p_mon->GetParamCl()->burstsinglereq ) {
                    if ( ((p_mon->peekOCPRequest()).MBurstSingleReq) && (i!=0) ) {
                        // This is a "filler" request.
                        // Since the request is SRMD, only the first request
                        // was actually sent.
                        (*m_tLogPtr) << " (";
                    } else {
                        // This is the first request of the burst.
                        // or the request is not SRMD
                        (*m_tLogPtr) << "  ";
                    }
                }
                // T(Thread) / (p_mon->GetParamCl()->threads-1)/10 +1 / dec 
                if (m_tLogThreadW) {
                    (*m_tLogPtr) << " " << std::setw(m_tLogThreadW);
                    (*m_tLogPtr) << myThread;
                }
                // C(ConnID) / (connid_wdth-1)/4 +1 / hex
                if (m_tLogConnW) {
                    (*m_tLogPtr).setf(std::ios::hex, std::ios::basefield);
                    (*m_tLogPtr) << " " << std::setw(m_tLogConnW);
                    (*m_tLogPtr) << (p_mon->peekOCPRequest()).MConnID;
                }
                // Cmd (MCmd) / 4 / char
                (*m_tLogPtr) << " " << tLogCmdName((p_mon->peekOCPRequest()).MCmd);
                // BuS (MBurstSeq) / 4 / char 
                (*m_tLogPtr) << " " << tLogBSeqName((p_mon->peekOCPRequest()).MBurstSeq);

                // Burst (transfer count/MBurstLength) 
                (*m_tLogPtr).setf(std::ios::hex, std::ios::basefield);
                (*m_tLogPtr) << " " << std::setw(2) 
                    << (i+1) + xferCount[myThread] 
                    << "/";
                if (!(p_mon->peekOCPRequest().MBurstPrecise)
                        && (i != ((p_mon->peekOCPRequest()).DataLength - 1) ) ) {
                    // This is a chunk of imprecise burst transfers and this is not 
                    // the last transfer of the chunk.
                    // Print out a ". " as we only have a valid MBurstLength value 
                    // for the last transfer of the chunk
                    (*m_tLogPtr) << ". ";
                } else {
                    // MBurstLength is valid. Print it out
                    (*m_tLogPtr) << std::setw(2) 
                        << resetiosflags(std::ios::adjustfield)
                        << setiosflags(std::ios::left) 
                        << (p_mon->peekOCPRequest()).MBurstLength; 
                    (*m_tLogPtr) << resetiosflags(std::ios::adjustfield)
                        << setiosflags(std::ios::right);
                }

                // l (last transfer of LastOfBurst=true request) / 1 / dec
                if ((p_mon->peekOCPRequest()).LastOfBurst 
                        && (i==((p_mon->peekOCPRequest()).DataLength-1))) {
                    (*m_tLogPtr) << " 1";
                } else {
                    (*m_tLogPtr) << " 0";
                }
                (*m_tLogPtr).setf(std::ios::hex, std::ios::basefield);
                // AS (MAddressSpace)/(addrspace_wdth-1)/4 +1/hex/addrspace
                if (m_tLogAddrSpaceW) {
                    (*m_tLogPtr) << " " << std::setw(m_tLogAddrSpaceW);
                    (*m_tLogPtr) << (p_mon->peekOCPRequest()).MAddrSpace;
                }
                // Addr (MAddr) / (addr_wdth-1)/4 +1 / hex
                (*m_tLogPtr) << std::setfill('0');
                (*m_tLogPtr) << " " << std::setw(m_tLogAddrW);
                (*m_tLogPtr) << (p_mon->peekOCPRequest()).MAddr + i*myAddrInc;
                // ) (right parenthesis) / 1 / char / burstsinglereq == true
                (*m_tLogPtr) << std::setfill(' ');
                if (p_mon->GetParamCl()->burstsinglereq) { 
                    if ( ((p_mon->peekOCPRequest()).MBurstSingleReq) && (i!=0)) {
                        // This is a "filler" request.
                        // Since the channel is SRMD, only the first request
                        // was actually sent.
                        (*m_tLogPtr) << " )";
                    } else {
                        // This is the first request of the burst.
                        // or this is a not an SRMD request
                        (*m_tLogPtr) << "  ";
                    }
                }
                (*m_tLogPtr) << std::setfill('0');
                // BE (byteen) / (data_wdth-1)/32 +1 / hex / byteen
                if (m_tLogByteEnW) {
                    (*m_tLogPtr) << " " << std::setw(m_tLogByteEnW);
                    if ( (p_mon->peekOCPRequest()).MByteEnPtr == NULL ) {
                        // no pointer used - same for whole burst
                        (*m_tLogPtr) << (p_mon->peekOCPRequest()).MByteEn;
                    } else {
                        (*m_tLogPtr) << (p_mon->peekOCPRequest()).MByteEnPtr[i];
                    }
                }
                // Ri(MReqInfo) / (reqinfo_wdth-1)/4 +1 / hex / reqinfo
                if (m_tLogReqInfoW) {
                    (*m_tLogPtr) << " " << std::setw(m_tLogReqInfoW);
                    (*m_tLogPtr) << (p_mon->peekOCPRequest()).MReqInfo;
                }
                (*m_tLogPtr).setf(std::ios::dec, std::ios::basefield);
                (*m_tLogPtr) << std::setfill(' ');
                // RqDL (Request Data Latency) / 4 / dec
                (*m_tLogPtr) << " " << std::setw(4);
                (*m_tLogPtr) << (p_mon->peekMasterTiming()).RqDL;
                // RqAL (Request Accept Latency) / 4 / dec
                (*m_tLogPtr) << " " << std::setw(4);
                (*m_tLogPtr) << (p_mon->peekSlaveTiming()).RqAL;
                // DAL (Data Accept Latency) / 3 / dec
                (*m_tLogPtr) << " " << std::setw(3);
                (*m_tLogPtr) << (p_mon->peekSlaveTiming()).DAL;
                // Data (MData or SData) / (data_wdth-1)/4 +1 
                (*m_tLogPtr).setf(std::ios::hex, std::ios::basefield);
                (*m_tLogPtr) << std::setfill('0');
                (*m_tLogPtr) << " " << std::setw(m_tLogDataW);
		if ((p_mon->peekOCPRequest()).MDataPtr) {
		  (*m_tLogPtr) << (p_mon->peekOCPRequest()).MDataPtr[i];
		} else {
		  (*m_tLogPtr) << 0;
		}
                // Request is over. Fill rest with blanks
                // ( (left parenthesis) / 1 / char / burstsinglereq == true
                (*m_tLogPtr) << std::setfill(' ');
                if (p_mon->GetParamCl()->burstsinglereq ) {
                    (*m_tLogPtr) << "  ";
                }
                // Resp(SResp) / 4 / char
                (*m_tLogPtr) << "    .";
                // RI (SRespInfo) / (respinfo_wdth-1)/4 +1 / hex / respinfo
                if (m_tLogRespInfoW) {
                    (*m_tLogPtr) << " ";
                    (*m_tLogPtr) << std::setw(m_tLogRespInfoW);
                    (*m_tLogPtr) << ".";
                }
                // L (last transfer of LastOfBurst=true resposne) / 1 / dec
                (*m_tLogPtr) << " .";
                // ) (right parenthesis) / 1 / char / burstsinglereq == true
                if (p_mon->GetParamCl()->burstsinglereq ) {
                    (*m_tLogPtr) << "  ";
                }
                // RqRpL (Request Response Latency / 5 / dec
                (*m_tLogPtr) << std::setw(5) << "     .";
                // RpAL (Response Accept Latency) / 4 / dec
                (*m_tLogPtr) << std::setw(4) << "    .";
                // MF (MFlag) / (mflag_wdth-1)/4 +1 / hex / mflag
                if (m_tLogMFW) {
                    (*m_tLogPtr) << " ";
                    (*m_tLogPtr) << std::setw(m_tLogMFW);
                    (*m_tLogPtr) << (p_mon->peekSideband()).MFlag ;
                }
                // SF (SFlag) / (sflag_wdth-1)/4 +1 / hex / sflag
                if (m_tLogSFW) {
                    (*m_tLogPtr) << " ";
                    (*m_tLogPtr) << std::setw(m_tLogSFW);
                    (*m_tLogPtr) << (p_mon->peekSideband()).SFlag ;
                }
                // I (SInterrupt) / 1 / dec / interrupt
                if (p_mon->GetParamCl()->interrupt) {
                    (*m_tLogPtr) << " " << (p_mon->peekSideband()).SInterrupt;
                }
                // Done with this line
                (*m_tLogPtr) << endl;

                // Go on to the next request in the burst
            }

        } else {

            // This request will require a response.
            // Store it for now and then print it once the response
            // arrives.
            for (unsigned int i=0;i<(p_mon->peekOCPRequest()).DataLength;i++) {
                // Reuse the temp_waitReq structure for efficiency
                if (tLogIsWrite((p_mon->peekOCPRequest()).MCmd)) {
                    // Write commands
                    temp_waitReq.startCycle = myStart + i*(p_mon->getWDI());
                } else if (p_mon->GetParamCl()->burstsinglereq
                        && (p_mon->peekOCPRequest()).MBurstSingleReq) {
                    // SRMD Read Command
                    temp_waitReq.startCycle = myStart;
                } else {
                    // MRMD Read Command
                    temp_waitReq.startCycle = myStart
                        + (i * max((p_mon->peekMasterTiming()).RqSndI,(p_mon->peekSlaveTiming()).RqAL));
                }
                temp_waitReq.MConnID = (p_mon->peekOCPRequest()).MConnID;
                temp_waitReq.MCmd = (p_mon->peekOCPRequest()).MCmd;
                temp_waitReq.MBurstSeq =  (p_mon->peekOCPRequest()).MBurstSeq;
                temp_waitReq.tCount = (i+1) + xferCount[myThread];
                temp_waitReq.DataLength = (p_mon->peekOCPRequest()).DataLength;

                if (!(p_mon->peekOCPRequest().MBurstPrecise)
                        && (i != ((p_mon->peekOCPRequest()).DataLength - 1) ) ) {
                    // This is a chunk of imprecise burst transfers and this is not 
                    // the last transfer of the chunk.
                    // Print out a ". " as we only have a valid MBurstLength value 
                    // for the last transfer of the chunk
                    temp_waitReq.MBurstLength = 0;
                } else {
                    // MBurstLength is valid - save it
                    temp_waitReq.MBurstLength = (p_mon->peekOCPRequest()).MBurstLength; 
                }

                temp_waitReq.lastReqOfBurst = (p_mon->peekOCPRequest()).LastOfBurst
                        && (i == ((p_mon->peekOCPRequest()).DataLength-1));
                temp_waitReq.MAddrSpace = (p_mon->peekOCPRequest()).MAddrSpace;
                temp_waitReq.MAddr = (p_mon->peekOCPRequest()).MAddr + i*myAddrInc;
                if ( ((p_mon->peekOCPRequest()).MByteEnPtr != NULL ) ) {
                    // use the pointer for ByteEn as we have an array of them
                    temp_waitReq.MByteEn = (p_mon->peekOCPRequest()).MByteEnPtr[i];
                } else {
                    temp_waitReq.MByteEn = (p_mon->peekOCPRequest()).MByteEn;
                }
                temp_waitReq.MReqInfo = (p_mon->peekOCPRequest()).MReqInfo;
                temp_waitReq.RqDL = (p_mon->peekMasterTiming()).RqDL;
                temp_waitReq.RqAL = (p_mon->peekSlaveTiming()).RqAL;
                temp_waitReq.DAL = (p_mon->peekSlaveTiming()).DAL;
                if (tLogIsWrite((p_mon->peekOCPRequest()).MCmd)) {
                    temp_waitReq.data = (p_mon->peekOCPRequest()).MDataPtr[i];
                } else {
                    temp_waitReq.data = 0;
                }
                if (tLogIsWrite((p_mon->peekOCPRequest()).MCmd)) {
                   temp_waitReq.fillerResp = 
                        p_mon->GetParamCl()->burstsinglereq
                        && (p_mon->peekOCPRequest()).MBurstSingleReq 
                        && (i != ((p_mon->peekOCPRequest()).DataLength -1));
                } else {
                    temp_waitReq.fillerReq = p_mon->GetParamCl()->burstsinglereq
                        && (p_mon->peekOCPRequest()).MBurstSingleReq && (i!=0);
                }

                // Add the response to the queue
                m_tLogWaitQ[(p_mon->peekOCPRequest()).MThreadID].push_back(temp_waitReq);

                // Now on to the next request in the burst
            }
        }
        // Update transfer counter going forward
        if ( (p_mon->peekOCPRequest()).LastOfBurst ) {
            xferCount[myThread] = 0;
        } else {
            xferCount[myThread] += (p_mon->peekOCPRequest()).DataLength;
        }
    }

    //    void tLogNewResp( void )
    virtual void NotifyResponseStart(tl2_peek_type *tl2_channel)
    {
        // A response has come in. 
      cout << __PRETTY_FUNCTION__ << " resp " << p_mon->peekOCPResponse() << endl;
        // How long is the burst response?
        unsigned int respCount = (p_mon->peekOCPResponse()).DataLength;

        // Each individual response in the response burst should have a
        // individual request stored up and waiting for it. 
        // Test: are there enough requests waiting for this reponse?
        assert(m_tLogWaitQ[(p_mon->peekOCPResponse()).SThreadID].size() >= respCount);

        // When is this response burst start?
        double timeNow = sc_simulation_time();

        // print out each individual transaction of the burst
        for (unsigned int i=0;i<respCount;i++) {
            // Get the next request that's waiting.
            temp_waitReq = m_tLogWaitQ[(p_mon->peekOCPResponse()).SThreadID].front();

            // NOTE: for now, cycle time and start time are the same
            // SimTime / 9 / dec
            (*m_tLogPtr).setf(std::ios::dec, std::ios::basefield);
            (*m_tLogPtr) << std::setw(9) << temp_waitReq.startCycle;

            // Cycle / 7 / dec
            (*m_tLogPtr) << " " << std::setw(7) << temp_waitReq.startCycle;

            // ( (left parenthesis) / 1 / char 
            if (p_mon->GetParamCl()->burstsinglereq) {
                if (temp_waitReq.fillerReq) {
                    (*m_tLogPtr) << " (";
                } else {
                    (*m_tLogPtr) << "  ";
                }
            }
            // T(Thread) / (p_mon->GetParamCl()->threads-1)/10 +1 / dec 
            if (m_tLogThreadW) {
                (*m_tLogPtr) << " " << std::setw(m_tLogThreadW);
                (*m_tLogPtr) << (p_mon->peekOCPResponse()).SThreadID;
            }
            // C(ConnID) / (connid_wdth-1)/4 +1 / hex
            if (m_tLogConnW) {
                (*m_tLogPtr).setf(std::ios::hex, std::ios::basefield);
                (*m_tLogPtr) << " " << std::setw(m_tLogConnW);
                (*m_tLogPtr) << temp_waitReq.MConnID;
            }
            // Cmd (MCmd) / 4 / char
            (*m_tLogPtr) << " " << tLogCmdName(temp_waitReq.MCmd);
            // BuS (MBurstSeq) / 4 / char 
            (*m_tLogPtr) << " " << tLogBSeqName(temp_waitReq.MBurstSeq);
            // Burst (transfer count/MBurstLength) 
            (*m_tLogPtr).setf(std::ios::hex, std::ios::basefield);
            (*m_tLogPtr) << " " << std::setw(2) << temp_waitReq.tCount
                << "/";
            // Only print out MBurstLength if we had a value for this transfer to store
            if (temp_waitReq.MBurstLength > 0) {
                (*m_tLogPtr) << resetiosflags(std::ios::adjustfield)
                    << setiosflags(std::ios::left) << std::setw(2)
                    << temp_waitReq.MBurstLength;
                (*m_tLogPtr) << resetiosflags(std::ios::adjustfield)
                    << setiosflags(std::ios::right);
            } else {
                (*m_tLogPtr) << ". ";
            }

            // l (last transfer of LastOfBurst=true request) / 1 / dec
            if (temp_waitReq.lastReqOfBurst) {
                (*m_tLogPtr) << " 1";
            } else {
                (*m_tLogPtr) << " 0";
            }
            // AS (MAddressSpace)/(addrspace_wdth-1)/4 +1/hex/addrspace
            (*m_tLogPtr).setf(std::ios::hex, std::ios::basefield);
            if (m_tLogAddrSpaceW) {
                (*m_tLogPtr) << " " << std::setw(m_tLogAddrSpaceW);
                (*m_tLogPtr) << temp_waitReq.MAddrSpace;
            }
            // Addr (MAddr) / (addr_wdth-1)/4 +1 / hex
            (*m_tLogPtr) << std::setfill('0');
            (*m_tLogPtr) << " " << std::setw(m_tLogAddrW);
            (*m_tLogPtr) << temp_waitReq.MAddr;
            (*m_tLogPtr) << std::setfill(' ');
            // ) (right parenthesis) / 1 / char / burstsinglereq == true
            if (p_mon->GetParamCl()->burstsinglereq) {
                if (temp_waitReq.fillerReq) {
                    (*m_tLogPtr) << " )";
                } else {
                    (*m_tLogPtr) << "  ";
                }
            }
            (*m_tLogPtr) << std::setfill('0');
            // BE (byteen) / (data_wdth-1)/32 +1 / hex / byteen
            if (m_tLogByteEnW) {
                (*m_tLogPtr) << " " << std::setw(m_tLogByteEnW);
                (*m_tLogPtr) << temp_waitReq.MByteEn;
            }
            // Ri(MReqInfo) / (reqinfo_wdth-1)/4 +1 / hex / reqinfo
            if (m_tLogReqInfoW) {
                (*m_tLogPtr) << " " << std::setw(m_tLogReqInfoW);
                (*m_tLogPtr) << temp_waitReq.MReqInfo;
            }
            // RqDL (Request Data Latency) / 4 / dec
            (*m_tLogPtr) << " " << std::setw(4);
            (*m_tLogPtr) << temp_waitReq.RqDL;
            // RqAL (Request Accept Latency) / 4 / dec
            (*m_tLogPtr).setf(std::ios::dec, std::ios::basefield);
            (*m_tLogPtr) << std::setfill(' ');
            (*m_tLogPtr) << " " << std::setw(4);
            (*m_tLogPtr) << (p_mon->peekSlaveTiming()).RqAL;
            // DAL (Data Accept Latency) / 3 / dec
            (*m_tLogPtr) << " " << std::setw(3);
            (*m_tLogPtr) << temp_waitReq.DAL;
            // Data (MData or SData) / (data_wdth-1)/4 +1 
            (*m_tLogPtr).setf(std::ios::hex, std::ios::basefield);
            (*m_tLogPtr) << std::setfill('0');
            (*m_tLogPtr) << " " << std::setw(m_tLogDataW);
            if (tLogIsWrite(temp_waitReq.MCmd)) {
                // This was a write command, there should be data
                // in the request
                (*m_tLogPtr) << temp_waitReq.data;
            } else {
                // This was a read command, there should be data
                // in the response
		if ((p_mon->peekOCPResponse()).SDataPtr) {
		  (*m_tLogPtr) << (p_mon->peekOCPResponse()).SDataPtr[i];
		} else {
		  (*m_tLogPtr) << 0;
		}
            }
            (*m_tLogPtr) << std::setfill(' ');

            // Print the response portion of the transaction
            
            // ( (left parenthesis) / 1 / char / burstsinglereq == true
            if (p_mon->GetParamCl()->burstsinglereq) {
                if (temp_waitReq.fillerResp) {
                    (*m_tLogPtr) << " (";
                } else {
                    (*m_tLogPtr) << "  ";
                }
            }
            // Resp(SResp) / 4 / char
            (*m_tLogPtr) << " " << tLogRespName((p_mon->peekOCPResponse()).SResp);
            // RI (SRespInfo) / (respinfo_wdth-1)/4 +1 / hex / respinfo
            if (m_tLogRespInfoW) {
                (*m_tLogPtr) << " ";
                (*m_tLogPtr) << std::setw(m_tLogRespInfoW);
                (*m_tLogPtr) << (p_mon->peekOCPResponse()).SRespInfo;
            }
            // L (last transfer of LastOfBurst=true resposne) / 1 / dec
            if ((p_mon->peekOCPResponse()).LastOfBurst && (i==(respCount-1))) {
                (*m_tLogPtr) << " 1";
            } else {
                (*m_tLogPtr) << " 0";
            }
            // ) (right parenthesis) / 1 / char / burstsinglereq == true
            if (p_mon->GetParamCl()->burstsinglereq) {
                if (temp_waitReq.fillerResp) {
                    (*m_tLogPtr) << " )";
                } else {
                    (*m_tLogPtr) << "  ";
                }
            }
            // RqRpL (Request Response Latency / 5 / dec
            if (p_mon->GetParamCl()->burstsinglereq && tLogIsWrite(temp_waitReq.MCmd)) {
                // There is only one response and thus only one response time
                (*m_tLogPtr) << " " << std::setw(5)
                    << (timeNow + (respCount-1)*(p_mon->getRDI())) 
                           - temp_waitReq.startCycle; 
            } else {
                // There are multiple responses:
                (*m_tLogPtr) << " " << std::setw(5) 
                    << (timeNow + i*(p_mon->getRDI())) - temp_waitReq.startCycle; 
            }
            // RpAL (Response Accept Latency) / 4 / dec
            (*m_tLogPtr) << " " << std::setw(4) << (p_mon->peekMasterTiming()).RpAL;
            // MF (MFlag) / (mflag_wdth-1)/4 +1 / hex / mflag
            if (m_tLogMFW) {
                (*m_tLogPtr) << " ";
                (*m_tLogPtr) << std::setw(m_tLogMFW);
                (*m_tLogPtr) << (p_mon->peekSideband()).MFlag ;
            }
            // SF (SFlag) / (sflag_wdth-1)/4 +1 / hex / sflag
            if (m_tLogSFW) {
                (*m_tLogPtr) << " ";
                (*m_tLogPtr) << std::setw(m_tLogSFW);
                (*m_tLogPtr) << (p_mon->peekSideband()).SFlag ;
            }
            // I (SInterrupt) / 1 / dec / interrupt
            if (p_mon->GetParamCl()->interrupt) {
                (*m_tLogPtr) << " " << (p_mon->peekSideband()).SInterrupt;
            }
            // Done with this line
            (*m_tLogPtr) << endl;

            // All done. Remove this is request from the queue.
            m_tLogWaitQ[(p_mon->peekOCPResponse()).SThreadID].pop_front();

            // Go on to the next request in the burst
        }
    }


  protected:
    ostream *m_tLogPtr;
    deque< WaitReq<Tdata,Taddr> > *m_tLogWaitQ;
    unsigned int m_tLogThreadW;
    unsigned int m_tLogConnW;
    unsigned int m_tLogAddrSpaceW;
    unsigned int m_tLogAddrW;
    unsigned int m_tLogByteEnW;
    unsigned int m_tLogReqInfoW;
    unsigned int m_tLogDataW;
    unsigned int m_tLogRespInfoW;
    unsigned int m_tLogMFW;
    unsigned int m_tLogSFW;
    WaitReq<Tdata, Taddr> temp_waitReq;
    vector<int> xferCount;
};

#endif  // _OCP_TL2_TMON_H
