///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// Copyright 2004 OCP-IP
// OCP-IP Confidential & Proprietary
//
//
//============================================================================
//      Project : OCP SLD WG
//       Author : Tim Kogel, CoWare, Inc.
//          Date: 11/26/2004
//
//  Description :  OCP Channel Transaction Recording Monitor
//	  This channel monitor is based on the transaction recording 
//	  API in the SystemC Verification (SCV) Library. It targets 
//	  performance analysis for architectural modeling.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef _OCP_PERF_MONITOR_CHANNEL_SCV_H
#define _OCP_PERF_MONITOR_CHANNEL_SCV_H


#include "ocp_tl2_monitor_observer_if.h"
#include "ocp_perf_monitor_extensions.h"
#include "scv.h"

template <class, class> class OCP_TL2_MonitorIF;

template <class Tdata, class Taddr>
class OCP_Perf_Monitor_Channel_SCV: 
  public OCP_TL2_Monitor_ObserverIF<Tdata,Taddr>
{
public:
  typedef typename OCP_TL2_Monitor_ObserverIF<Tdata,Taddr>::tl2_peek_type tl2_peek_type;

  OCP_Perf_Monitor_Channel_SCV(const char* name,
			       int  max_nbr_of_threads,
			       bool burst_recording,
			       bool attribute_recording); 
  virtual ~OCP_Perf_Monitor_Channel_SCV(); 

  virtual void NotifyRequestStart (tl2_peek_type *);
  virtual void NotifyRequestEnd   (tl2_peek_type *);
  virtual void NotifyResponseStart(tl2_peek_type *);
  virtual void NotifyResponseEnd  (tl2_peek_type *);

  virtual void registerChannel(tl2_peek_type *,
			       bool master_is_node, 
			       bool slave_is_node);

protected:
  void registerMaster(const char* name);
  void registerSlave (const char* name);

  const char*				m_name;
  scv_tr_db*				m_db; 
  scv_tr_stream**			m_req_burst_stream;
  scv_tr_stream**			m_req_chunk_stream;
  scv_tr_stream**			m_resp_burst_stream;
  scv_tr_stream**			m_resp_chunk_stream;
  scv_tr_generator<unsigned int>**	m_req_burst_read_gen;
  scv_tr_generator<unsigned int>**	m_req_burst_write_gen;
  scv_tr_generator<unsigned int>**	m_req_chunk_read_gen;
  scv_tr_generator<unsigned int>**	m_req_chunk_write_gen;
  scv_tr_generator<unsigned int>**	m_resp_burst_gen;
  scv_tr_generator<unsigned int>**	m_resp_chunk_gen;

  scv_tr_handle*			m_req_burst_handle;
  scv_tr_handle*			m_req_chunk_handle;
  scv_tr_handle*			m_resp_burst_handle;
  scv_tr_handle*			m_resp_chunk_handle;

  const char*				m_master_name;
  const char*				m_slave_name;
  unsigned int				m_thread_id;

  bool					m_master_registered;
  bool					m_slave_registered;

  bool*					m_ongoing_req_burst;
  bool*					m_ongoing_resp_burst;

  const unsigned int			m_max_nbr_of_threads;
  const bool				m_burst_recording;
  const bool				m_attribute_recording;
};

template <class Tdata, class Taddr>
OCP_Perf_Monitor_Channel_SCV<Tdata, Taddr>::OCP_Perf_Monitor_Channel_SCV(const char* name,int max_nbr_of_threads, bool burst_recording, bool attribute_recording):
  m_name(name),
  m_req_burst_stream(NULL),
  m_req_chunk_stream(NULL),
  m_resp_burst_stream(NULL),
  m_resp_chunk_stream(NULL),
  m_req_burst_read_gen(NULL),
  m_req_burst_write_gen(NULL),
  m_req_chunk_read_gen(NULL),
  m_req_chunk_write_gen(NULL),
  m_resp_burst_gen(NULL),
  m_resp_chunk_gen(NULL),
  m_thread_id(0),
  m_master_registered(false),
  m_slave_registered(false),
  m_max_nbr_of_threads(max_nbr_of_threads),
  m_burst_recording(burst_recording),
  m_attribute_recording(attribute_recording)
{
  /*   // multiple db's not supported -> use default db */
  /*   string db_name(m_name); */
  /*   db_name += ".db"; */
  /*   m_db = new scv_tr_db(db_name.c_str()); */
  cout << " m_max_nbr_of_threads : " << m_max_nbr_of_threads << endl;
  unsigned int nbr_of_threads = 1;
  if (m_max_nbr_of_threads) {
    // per-thread recording enabled, 
    // allocate separete data-structures per thread
    nbr_of_threads = m_max_nbr_of_threads;
  }

  m_req_chunk_handle = new scv_tr_handle[nbr_of_threads];
  m_resp_chunk_handle = new scv_tr_handle[nbr_of_threads];

  if (m_burst_recording) {
    m_resp_burst_handle = new scv_tr_handle[nbr_of_threads];
    m_req_burst_handle = new scv_tr_handle[nbr_of_threads];
    m_ongoing_req_burst = new bool[nbr_of_threads];
    m_ongoing_resp_burst = new bool[nbr_of_threads];
    for (unsigned int i = 0; i < nbr_of_threads; i++) {
      m_ongoing_req_burst[i]=false;
      m_ongoing_resp_burst[i]=false;
    }
  }
}

template <class Tdata, class Taddr>
OCP_Perf_Monitor_Channel_SCV<Tdata, Taddr>::~OCP_Perf_Monitor_Channel_SCV()
{
  // delte stuff
  if (m_burst_recording) {
    delete [] m_resp_burst_handle;
    delete [] m_req_burst_handle;
    delete [] m_ongoing_req_burst;
    delete [] m_ongoing_resp_burst;
  }
  delete [] m_req_chunk_handle;
  delete [] m_resp_chunk_handle;
 
  unsigned int nbr_of_threads = 1;
  if (m_max_nbr_of_threads) {
    nbr_of_threads = m_max_nbr_of_threads;
  }
  if (m_master_registered) {
    for (unsigned int i = 0; i < nbr_of_threads; i++) {
      delete m_req_chunk_stream[i];
      delete m_req_chunk_read_gen[i];
      delete m_req_chunk_write_gen[i];
      if (m_burst_recording) {
	delete m_req_burst_stream[i];
	delete m_req_burst_read_gen[i];
	delete m_req_burst_write_gen[i];
      } 
    }

    delete [] m_req_chunk_stream;
    delete [] m_req_chunk_read_gen;
    delete [] m_req_chunk_write_gen;

    if (m_burst_recording) {
      delete [] m_req_burst_stream;
      delete [] m_req_burst_read_gen;
      delete [] m_req_burst_write_gen;
    }
  }
  if (m_slave_registered) {
    for (unsigned int i = 0; i < nbr_of_threads; i++) {
      delete m_resp_chunk_stream[i];
      delete m_resp_chunk_gen[i];
      if (m_burst_recording) {
	delete 	m_resp_burst_stream[i];
	delete 	m_resp_burst_gen[i];
      }
    }
    delete [] m_resp_chunk_stream;
    delete [] m_resp_chunk_gen;
    if (m_burst_recording) {
      delete [] m_resp_burst_stream;
      delete [] m_resp_burst_gen;
    }
  }
}

template <class Tdata, class Taddr>
void
OCP_Perf_Monitor_Channel_SCV<Tdata, Taddr>::registerMaster(const char* name)
{
  cout << __PRETTY_FUNCTION__ << " master " << name << endl;
  if (m_master_registered) {
    cout << "master " << name << " is already registered with name " 
	 << m_master_name << endl;
    return;
  }
  m_master_registered = true;
  m_master_name = name;
  std::string stream_name(name);
  std::string thread_str;
  std::string chunk_str;
  if (m_burst_recording) 
    chunk_str = "(chunk)";

  unsigned int nbr_of_threads = 1;
  if (m_max_nbr_of_threads) {
    nbr_of_threads = m_max_nbr_of_threads;
  }

  m_req_chunk_stream = new scv_tr_stream*[nbr_of_threads];
  m_req_chunk_read_gen = new scv_tr_generator<unsigned int>*[nbr_of_threads];
  m_req_chunk_write_gen = new scv_tr_generator<unsigned int>*[nbr_of_threads];
  
  if (m_burst_recording) {
    m_req_burst_stream = new scv_tr_stream*[nbr_of_threads];
    m_req_burst_read_gen = new scv_tr_generator<unsigned int>*[nbr_of_threads];
    m_req_burst_write_gen = new scv_tr_generator<unsigned int>*[nbr_of_threads];
  } // end if m_burst_recording

  for (unsigned int i = 0; i < nbr_of_threads; i++) {
    if (nbr_of_threads > 1) {
      char thread_char[20];
      sprintf(thread_char,",thread_%d",i);
      thread_str = thread_char;
    }
    m_req_chunk_stream[i]    = new scv_tr_stream((stream_name+thread_str+chunk_str).c_str(),"ocp_chunk");
    m_req_chunk_read_gen[i]  = new scv_tr_generator<unsigned int>("read", *m_req_chunk_stream[i],"addr");
    m_req_chunk_write_gen[i] = new scv_tr_generator<unsigned int>("write",*m_req_chunk_stream[i],"addr");
    
    if (m_burst_recording) {
      m_req_burst_stream[i]    = new scv_tr_stream((stream_name+thread_str+"(burst)").c_str(),"ocp_burst");
      m_req_burst_read_gen[i]  = new scv_tr_generator<unsigned int>("read", *m_req_burst_stream[i],"addr");
      m_req_burst_write_gen[i] = new scv_tr_generator<unsigned int>("write",*m_req_burst_stream[i],"addr");
    } // end if m_burst_recording
  }
}

template <class Tdata, class Taddr>
void
OCP_Perf_Monitor_Channel_SCV<Tdata, Taddr>::registerSlave(const char* name)
{
  cout << __PRETTY_FUNCTION__ << " slave " << name << endl;
  if (m_slave_registered) {
    cout << "slave " << name << " is already registered with name " 
	 << m_slave_name << endl;
    return;
  }
  m_slave_registered = true;
  m_slave_name = name;

  std::string stream_name(name);
  std::string thread_str;
  std::string chunk_str;
  if (m_burst_recording) 
    chunk_str = "(chunk)";

  unsigned int nbr_of_threads = 1;
  if (m_max_nbr_of_threads) {
    nbr_of_threads = m_max_nbr_of_threads;
  }

  m_resp_chunk_stream = new scv_tr_stream*[nbr_of_threads];
  m_resp_chunk_gen = new scv_tr_generator<unsigned int>*[nbr_of_threads];
  if (m_burst_recording) {
    m_resp_burst_stream = new scv_tr_stream*[nbr_of_threads];
    m_resp_burst_gen = new scv_tr_generator<unsigned int>*[nbr_of_threads];
  }
  for (unsigned int i = 0; i < nbr_of_threads; i++) {
    if (nbr_of_threads > 1) {
      char thread_char[20];
      sprintf(thread_char,",thread_%d",i);
      thread_str = thread_char;
    }
    m_resp_chunk_stream[i] = new scv_tr_stream((stream_name+thread_str+chunk_str).c_str(),"ocp_chunk");
    m_resp_chunk_gen[i] = new scv_tr_generator<unsigned int>("read",*m_resp_chunk_stream[i],"addr");
    if (m_burst_recording) {
      m_resp_burst_stream[i] = new scv_tr_stream((stream_name+thread_str+"(burst)").c_str(),"ocp_burst");
      m_resp_burst_gen[i] = new scv_tr_generator<unsigned int>("read",*m_resp_burst_stream[i],"addr");
    }
  }
}

template <class Tdata, class Taddr>
void
OCP_Perf_Monitor_Channel_SCV<Tdata, Taddr>::registerChannel(tl2_peek_type *tl2_channel,
							    bool master_is_node, 
							    bool slave_is_node)
{
  registerMaster(tl2_channel->peekMasterPortName().c_str());
  registerSlave(tl2_channel->peekSlavePortName().c_str());
}

template <class Tdata, class Taddr>
void
OCP_Perf_Monitor_Channel_SCV<Tdata, Taddr>::NotifyRequestStart(tl2_peek_type *tl2_channel)
{
  const OCPTL2RequestGrp<Tdata,Taddr>& req = tl2_channel->peekOCPRequest();
  m_thread_id = 0;
  if (m_max_nbr_of_threads) {
    assert ( req.MThreadID < m_max_nbr_of_threads );
    m_thread_id = req.MThreadID;
  }
  bool is_read= (req.MCmd==OCP_MCMD_RD)||(req.MCmd==OCP_MCMD_RDEX)||(req.MCmd==OCP_MCMD_RDL);
  if (is_read) {
    m_req_chunk_handle[m_thread_id] = m_req_chunk_read_gen[m_thread_id]->begin_transaction(req.MAddr);
  } else {
    m_req_chunk_handle[m_thread_id] = m_req_chunk_write_gen[m_thread_id]->begin_transaction(req.MAddr);
  }
  if (m_attribute_recording) {
    m_req_chunk_handle[m_thread_id].record_attribute("Req",req);
  }
  if (m_burst_recording && !m_ongoing_req_burst[m_thread_id]) {
    m_ongoing_req_burst[m_thread_id] = true;
    // this is the 1st packet of a new OCP burst
    if (is_read) {
      m_req_burst_handle[m_thread_id] = m_req_burst_read_gen[m_thread_id]->begin_transaction(req.MAddr);
    } else {
      m_req_burst_handle[m_thread_id] = m_req_burst_write_gen[m_thread_id]->begin_transaction(req.MAddr);
    }
  }
}

template <class Tdata, class Taddr>
void 
OCP_Perf_Monitor_Channel_SCV<Tdata, Taddr>::NotifyRequestEnd(tl2_peek_type *tl2_channel)
{
  const OCPTL2RequestGrp<Tdata,Taddr>& req = tl2_channel->peekOCPRequest();
  m_thread_id = 0;
  if (m_max_nbr_of_threads) {
    assert ( req.MThreadID < m_max_nbr_of_threads );
    m_thread_id = req.MThreadID;
  }
  if (!m_req_chunk_handle[m_thread_id].is_active()) {
#ifdef MONITOR_DEBUG
    cout << "ERROR! endRequest, handle " <<  m_req_chunk_handle[m_thread_id].get_id() << " is not active!" << endl;
#endif
    return;
  }
  m_req_chunk_handle[m_thread_id].end_transaction();
  if (m_burst_recording && req.LastOfBurst) {
    // this is the last packet of an OCP burst
    m_ongoing_req_burst[m_thread_id] = false;
    m_req_burst_handle[m_thread_id].end_transaction();
  }
}

template <class Tdata, class Taddr>
void 
OCP_Perf_Monitor_Channel_SCV<Tdata, Taddr>::NotifyResponseStart(tl2_peek_type *tl2_channel)
{
  const OCPTL2ResponseGrp<Tdata>& resp = tl2_channel->peekOCPResponse();
  m_thread_id = 0;
  if (m_max_nbr_of_threads) {
    assert (resp.SThreadID < m_max_nbr_of_threads);
    m_thread_id = resp.SThreadID;
  }
  m_resp_chunk_handle[m_thread_id] = m_resp_chunk_gen[m_thread_id]->begin_transaction(resp.DataLength);
  if (m_attribute_recording) {
    m_resp_chunk_handle[m_thread_id].record_attribute("Resp",resp);
  }
  if (m_burst_recording && !m_ongoing_resp_burst[m_thread_id]) {
    m_ongoing_resp_burst[m_thread_id] = true;
    m_resp_burst_handle[m_thread_id] = m_resp_burst_gen[m_thread_id]->begin_transaction(resp.DataLength);
  }
}

template <class Tdata, class Taddr>
void 
OCP_Perf_Monitor_Channel_SCV<Tdata, Taddr>::NotifyResponseEnd(tl2_peek_type *tl2_channel)
{
  const OCPTL2ResponseGrp<Tdata>& resp = tl2_channel->peekOCPResponse();
  m_thread_id = 0;
  if (m_max_nbr_of_threads) {
    assert (resp.SThreadID < m_max_nbr_of_threads);
    m_thread_id = resp.SThreadID;
  }
  if (!m_resp_chunk_handle[m_thread_id].is_active()) {
#ifdef MONITOR_DEBUG
    cout << "ERROR! endResponse, handle " <<  m_resp_chunk_handle[m_thread_id].get_id() << " is not active!" << endl;
#endif
    return;
  }
  m_resp_chunk_handle[m_thread_id].end_transaction();
  if (m_burst_recording && resp.LastOfBurst) {
    m_ongoing_resp_burst[m_thread_id] = false;
    m_resp_burst_handle[m_thread_id].end_transaction();
  }
}

#endif
