// 
//  (c) Copyright OCP-IP 2003, 2004, 2005
//  OCP-IP Confidential and Proprietary
//
// ============================================================================
//
//      Project : OCP SLD WG, Layer adapters
//      Authors : Herve Alexanian, SONICS, herve@sonicsinc.com
//                Stephane Guntz, PROSILOG, guntz@prosilog.com
//		  Yann Bajot, PROSILOG, bajot@prosilog.com
//
//  Description : Transaction Level - Layer-1 to Layer-2 OCP Slave Adapter
//  $Id:
//
// ============================================================================

#include "ocp_tl1_tl2_slave_adapter.h"

#define MAX_THREADS          16
#define POOL_SIZE            1024

//---------------------------------------------------------------------------
// Tl2RequestExpander constructor
// Note: The TL2 request is copied. This may cost some performance
// but would require some infrastructure to address with shared pointers
// and a memory pool for example. Could use boost I suppose, but would need
// to clear that with the committee
//---------------------------------------------------------------------------
template<typename Td, typename Ta>
Tl2RequestExpander<Td, Ta>::Tl2RequestExpander(
    const OCPTL2RequestGrp<Td, Ta>& req ) :
    m_numExpanded( 0 ),
    m_isRead( OcpIp::isRead( req.MCmd ) ),
    m_tl2Req( req ),
    m_tl1Req( req ), // to initialize the TL1 with TL2 fields by default
    m_pBurstSequence( NULL ),
    m_pParams( NULL )
{
    if (!m_tl2Req.MBurstPrecise) {
        // We will count down the burst length so need to start
        // at one before
        m_tl1Req.MBurstLength++;
    }
}

template<typename Td, typename Ta>
OCPRequestGrp<Td, Ta>
Tl2RequestExpander<Td, Ta>::next()
{
    assert( !finished() );
    m_tl1Req.HasMData = false;
    if ( !m_pParams->datahandshake ) {
        m_tl1Req.HasMData = isWrite();
        // assign the length sensitive fields
        if ( m_tl2Req.MByteEnPtr   != NULL )
            m_tl1Req.MByteEn        = *( m_tl2Req.MByteEnPtr++ );
        if ( m_tl2Req.MDataPtr     != NULL )
            m_tl1Req.MData          = *( m_tl2Req.MDataPtr++ );
        if ( m_tl2Req.MDataInfoPtr != NULL )
            m_tl1Req.MDataInfo      = *( m_tl2Req.MDataInfoPtr++ );        
    }

    // Calculate next address.
    assert( m_pBurstSequence != NULL );
    if ( m_tl2Req.MBurstSeq != OCP_MBURSTSEQ_UNKN ) {
        m_tl1Req.MAddr = m_pBurstSequence->next();
    }

    if ( m_tl2Req.MBurstSingleReq )
        m_numExpanded = m_tl2Req.DataLength;
    else
        ++m_numExpanded;

    // Calculate last of burst/row
    m_tl1Req.MReqLast = m_tl2Req.MBurstSingleReq ||
        ( m_tl2Req.LastOfBurst && m_numExpanded == m_tl2Req.DataLength );
    m_tl1Req.MReqRowLast = m_tl1Req.MReqLast;
    if ( m_tl1Req.MBlockHeight > 1 )
        m_tl1Req.MReqRowLast |= // m_tl2Req.LastOfRow &&
            ( m_numExpanded % m_tl2Req.MBurstLength ) == 0;

    if ( !m_tl2Req.MBurstPrecise ) {
        if ( m_tl1Req.MReqLast )
            m_tl1Req.MBurstLength = 1;
        else {
            m_tl1Req.MBurstLength--;
            // Cannot allow burstlength to fall to one 
            // as this is not the last of burst
            if ( m_tl1Req.MBurstLength < 2 ) {
                m_tl1Req.MBurstLength = 2;
            }
        }
    } else {
        // for precise, MBurstLength remains the same and was taken care of
        // at construction time
    }

    return m_tl1Req;
}

//---------------------------------------------------------------------------
// Tl2RequestExpanderDh constructor
// Note: The TL2 request is copied. See comment above in Tl2RequestExpander
// class
//---------------------------------------------------------------------------
template<typename Td, typename Ta>
Tl2RequestExpanderDh<Td, Ta>::Tl2RequestExpanderDh(
    const OCPTL2RequestGrp<Td, Ta>& req ) :
    m_numExpanded( 0 ),
    m_requestsSent( 0 ),
    m_tl2Req( req )
{
    m_tl1Dh.MDataThreadID = req.MThreadID;
    m_tl1Dh.MDataTagID    = req.MTagID;
    m_tl1Dh.MDataByteEn   = req.MByteEn;
    m_tl1Dh.MDataValid    = 1;  // is this even necessary ?
}

template<typename Td, typename Ta>
OCPDataHSGrp<Td>
Tl2RequestExpanderDh<Td, Ta>::next()
{
    // assign the length sensitive fields
    if ( m_tl2Req.MByteEnPtr   != NULL )
        m_tl1Dh.MDataByteEn     = *( m_tl2Req.MByteEnPtr++ );
    if ( m_tl2Req.MDataPtr     != NULL )
        m_tl1Dh.MData           = *( m_tl2Req.MDataPtr++ );
    if ( m_tl2Req.MDataInfoPtr != NULL )
        m_tl1Dh.MDataInfo       = *( m_tl2Req.MDataInfoPtr++ );
    ++m_numExpanded;

    // calculate last data of burst/row
    m_tl1Dh.MDataLast = m_tl2Req.LastOfBurst && m_numExpanded == m_tl2Req.DataLength;
    m_tl1Dh.MDataRowLast = m_tl1Dh.MDataLast;
    if ( m_tl2Req.MBlockHeight > 1 )
        m_tl1Dh.MDataRowLast = // m_tl2Req.LastOfRow &&
            ( m_numExpanded % m_tl2Req.MBurstLength ) == 0;

    return m_tl1Dh;
}

//---------------------------------------------------------------------------
// Tl1ResponseAggregator constructor
// Note: The TL1 requests are aggregated into one expected TL2 response
//---------------------------------------------------------------------------
template<typename Td, typename Ta>
Tl1ResponseAggregator<Td,Ta>::Tl1ResponseAggregator(
    const OCPTL2RequestGrp<Td, Ta>& req, Td* pDataPtr ) :
    m_numAggregated(0)
{
    m_tl2Resp.DataLength   = req.DataLength;
    if ( OcpIp::isWrite( req.MCmd ) && req.MBurstSingleReq )
        m_tl2Resp.DataLength = 1;
    m_tl2Resp.LastOfBurst  = req.LastOfBurst;
    m_tl2Resp.LastOfRow    = req.LastOfRow;
    m_tl2Resp.SThreadID    = req.MThreadID;
    m_tl2Resp.STagID       = req.MTagID;
    if ( OcpIp::isRead( req.MCmd ) ) {
        m_tl2Resp.SDataPtr = pDataPtr;
    }
}

template<typename Td, typename Ta>
void
Tl1ResponseAggregator<Td,Ta>::next( const OCPResponseGrp<Td>& resp )
{
    if ( m_tl2Resp.SDataPtr != NULL ) 
      m_tl2Resp.SDataPtr[m_numAggregated] = resp.SData;

    // only sample these once
    if ( m_numAggregated == 0 ) {
        m_tl2Resp.SResp         = resp.SResp;
        m_tl2Resp.SDataInfo     = resp.SDataInfo;
        m_tl2Resp.SRespInfo     = resp.SRespInfo;
    }

    // failure/error responses become sticky
    if ( m_tl2Resp.SResp == OCP_SRESP_FAIL )
        m_tl2Resp.SResp = resp.SResp;
    if ( m_tl2Resp.SResp == OCP_SRESP_ERR )
        m_tl2Resp.SResp = resp.SResp;

    ++m_numAggregated;
}

//---------------------------------------------------------------------------
// OCP_TL1_TL2_Slave_Adapter constructor
//---------------------------------------------------------------------------
template<class TdataCl_tl1 >
OCP_TL1_TL2_Slave_Adapter<TdataCl_tl1>::OCP_TL1_TL2_Slave_Adapter(
    sc_module_name name, int max_chunk_length, int, int): 
  sc_module(name),
  MasterP("MasterP"),
  SlaveP("SlaveP"),
  m_maxChunkLength(max_chunk_length),
  m_tl1RequestThreadId( 0 ),
  m_tl1DataHsThreadId( 0 ),
  m_tl2ResponseThreadId( 0 ),
  m_bindGetSThreadBusy( MasterP ),
  m_bindPutSThreadBusy( SlaveP ),
  m_sThreadBusyRelay( "SThreadBusy", MasterP.SThreadBusyEvent(),
                      m_bindGetSThreadBusy, m_bindPutSThreadBusy ),
  m_bindGetMThreadBusy( SlaveP ),
  m_bindPutMThreadBusy( MasterP ),
  m_mThreadBusyRelay( "MThreadBusy", SlaveP.MThreadBusyEvent(),
                      m_bindGetMThreadBusy, m_bindPutMThreadBusy )
{
    SC_METHOD(SlaveRequest);
    sensitive<<SlaveP.RequestStartEvent();
    dont_initialize();

    SC_METHOD(MasterResponse);
    sensitive << MasterP.ResponseStartEvent();
    dont_initialize();

#define SC_THREADN(func, name)                      \
    declare_thread_process( func ## _handle,        \
                            name,                   \
                            SC_CURRENT_USER_MODULE, \
                            func )

    for ( int i=0; i < MAX_THREADS; ++i ) {
        char mrname[40], mdname[40], srname[40];
        sprintf( mrname, "MasterRequest_%d", i );
        SC_THREADN(MasterRequest, mrname);
        sprintf( mdname, "MasterDataHs_%d", i );               
        SC_THREADN(MasterDataHs, mdname);
        sprintf( srname, "SlaveResponse_%d", i );
        SC_THREADN(SlaveResponse, srname);
    }

    SC_METHOD(SCmdAccept);
    sensitive << m_tl2CmdAcceptEvent;
    dont_initialize();

    SC_METHOD(MRespAccept);
    sensitive << m_tl1RespAcceptEvent;
    dont_initialize();

    SC_METHOD(MputReset);
    sensitive << SlaveP.ResetStartEvent() << SlaveP.ResetEndEvent();
    dont_initialize();

    SC_METHOD(SputReset);
    sensitive << MasterP.ResetStartEvent() << MasterP.ResetEndEvent();
    dont_initialize();

    // sideband processes
    SC_METHOD(MputFlags);
    sensitive << SlaveP.SidebandMasterEvent() ;

    SC_METHOD(SputSFlag);
    sensitive << MasterP.SidebandSFlagEvent() ;

    SC_METHOD(SputSError);
    sensitive << MasterP.SidebandSErrorEvent() ;

    SC_METHOD(SputSInterrupt);
    sensitive << MasterP.SidebandSInterruptEvent() ;
}

//---------------------------------------------------------------------------
// OCP_TL1_TL2_Slave_Adapter destructor
//---------------------------------------------------------------------------
template<class TdataCl_tl1 >
OCP_TL1_TL2_Slave_Adapter<TdataCl_tl1>::~OCP_TL1_TL2_Slave_Adapter()    
{
    vector<sc_event*>::iterator it;
    for ( it = m_newTl2ReqEvent.begin(); it != m_newTl2ReqEvent.end(); ++it )
        delete *it;
    for ( it = m_newTl2RespEvent.begin(); it != m_newTl2RespEvent.end(); ++it )
        delete *it;
    for ( it = m_requestSentEvent.begin(); it != m_requestSentEvent.end(); ++it )
        delete *it;
}

//---------------------------------------------------------------------------
// Method: OCP_TL1_TL2_Slave_Adapter::SlaveRequest
// get the TL2 request from the master
//---------------------------------------------------------------------------
template<class TdataCl_tl1 >
void OCP_TL1_TL2_Slave_Adapter<TdataCl_tl1>::SlaveRequest()
{
    if ( !SlaveP->getOCPRequest( m_tl2Request ) )
        return;

    unsigned int threadId = m_tl2Request.MThreadID;
    assert( threadId < m_tl2RequestExpanderQueues.size() );
    OcpIp::BurstCounter&      burstCounter  = m_tl2BurstCounter[threadId];
    OcpIp::BurstSequence<Ta>& burstSequence = m_burstSequence  [threadId];
    bool continuingSRMD = m_tl2Request.MBurstSingleReq &&
        ( burstCounter.count() > 0 ) && !burstCounter.last() ;
    if ( burstCounter.count() == 0 || burstCounter.last() ) {
        burstSequence.init( m_tl2Request.MAddr, m_tl2Request.MBurstSeq,
                            m_tl2Request.MBurstLength,
                            m_tl2Request.MBurstPrecise,
                            m_tl2Request.MBlockHeight,
                            m_tl2Request.MBlockStride );
    }
    // Queue up the TL2 request (unless it's a SRMD continuation)
    if ( !continuingSRMD ) {
        Tl2RequestExpander<Td, Ta> expander( m_tl2Request );
        expander.setParamCl( m_params );
        expander.setSequenceCalc( burstSequence );
        m_tl2RequestExpanderQueues[threadId].push_back( expander );
    }

    // Prepare Data handshake
    if ( OcpIp::isWrite( m_tl2Request.MCmd ) && m_params->datahandshake ) {
        Tl2RequestExpanderDh<Td, Ta> expanderDh( m_tl2Request );
        if ( continuingSRMD ) {
            expanderDh.sentRequest( m_tl2Request.MBurstLength );
        }
        m_tl2RequestExpanderDhQueues[threadId].push_back( expanderDh );
    }

    // Prepare Response
    if ( OcpIp::isRead( m_tl2Request.MCmd ) || m_params->writeresp_enable ) {
        // for the moment, only handle chopping/aggregating of SRMD RD
        //   responses based on m_maxChunkLength
        assert( m_maxChunkLength > 0 );
        if ( m_tl2Request.MBurstSingleReq &&
             OcpIp::isRead( m_tl2Request.MCmd ) &&
             ( ( (unsigned int) m_maxChunkLength )
               < m_tl2Request.DataLength ) ) {
            // need chopping on the response side
            int chunks;
            int savedDataLength = m_tl2Request.DataLength;

            // set the new DataLength and LastOfBurst values
            m_tl2Request.DataLength = m_maxChunkLength;
            assert( m_tl2Request.LastOfBurst );
            m_tl2Request.LastOfBurst = false;

            // allocate enough entries
            for ( chunks = 0; chunks < savedDataLength; ) {
                chunks += m_maxChunkLength;
                if ( chunks >= savedDataLength ) {
                    // this is the final chunk
                    m_tl2Request.DataLength = m_maxChunkLength -
                        ( chunks - savedDataLength );
                    m_tl2Request.LastOfBurst = true;
                }

                Tl1ResponseAggregator<Td, Ta> aggregator( m_tl2Request,
                                                          m_tl2SDataPool[threadId] );
                m_tl1ResponseAggregatorQueues[threadId].push_back( aggregator );
            }

            // restore the original SRMD RD request
            m_tl2Request.DataLength = savedDataLength;
            m_tl2Request.LastOfBurst = true;           
        } else if ( m_tl2Request.MBurstSingleReq &&
                    OcpIp::isWrite( m_tl2Request.MCmd ) ) {
            // single aggregator for SRMD write
            if ( m_tl2Request.LastOfBurst ) {
                Tl1ResponseAggregator<Td, Ta> aggregator( m_tl2Request,
                                                          m_tl2SDataPool[threadId] );
                m_tl1ResponseAggregatorQueues[threadId].push_back( aggregator );
            }
        } else {
            Tl1ResponseAggregator<Td, Ta> aggregator( m_tl2Request,
                                                      m_tl2SDataPool[threadId] );
            m_tl1ResponseAggregatorQueues[threadId].push_back( aggregator );
        }
    }
    burstCounter.next( m_tl2Request );
    m_newTl2ReqEvent[m_tl2Request.MThreadID]->notify( SC_ZERO_TIME );
} //end of SlaveRequest method

//---------------------------------------------------------------------------
// Method: OCP_TL1_TL2_Slave_Adapter::MasterResponse
// send TL2 response made of atomic TL1 response
//---------------------------------------------------------------------------
template< class TdataCl_tl1 >
void OCP_TL1_TL2_Slave_Adapter<TdataCl_tl1>::MasterResponse()
{
    if ( !MasterP->getOCPResponse( m_tl1Response ) )
        return;

    m_tl1RespAcceptEvent.notify( m_times.RpAL * SlaveP->getPeriod() );

    // Queue up the TL2 request
    unsigned int threadId = m_tl1Response.SThreadID;
    assert( threadId < m_tl1ResponseAggregatorQueues.size() );

    Tl1ResponseAggregatorQueue& queue = m_tl1ResponseAggregatorQueues[threadId];
    if ( queue.empty() ) {
        cerr << sc_time_stamp() << ": "
             << name() << ": WARNING: Ignoring unexpected TL1 response received\n"
             << hex << m_tl1Response << endl;
        return;
    }

    // Look for the first incomplete aggregator and stuff the response in there
    typename Tl1ResponseAggregatorQueue::iterator qit;
    for ( qit = queue.begin(); qit < queue.end(); ++qit ) {        
        Tl1ResponseAggregator<Td, Ta>& aggregator = *qit;
        if ( !aggregator.finished() ) {
            aggregator.next( m_tl1Response );
            if ( aggregator.finished() )
                m_newTl2RespEvent[threadId]->notify( SC_ZERO_TIME );
            break;
        }
    }
    if ( qit == queue.end() ) {
        cerr << sc_time_stamp() << ": "
             << name() << ": WARNING: Ignoring unexpected TL1 response received\n"
             << hex << m_tl1Response << endl;
        return;
    }
} //end of MasterResponse method

//---------------------------------------------------------------------------
// Thread: OCP_TL1_TL2_Slave_Adapter::MasterRequest
// send the group of TL1 requests to the slave, made from the TL2 request from
// the master 
//---------------------------------------------------------------------------
template<class TdataCl_tl1 >
void OCP_TL1_TL2_Slave_Adapter<TdataCl_tl1>::MasterRequest()
{
    int threadId = m_tl1RequestThreadId++;
    if ( threadId >= m_params->threads ) {
        // See comment at beginning of SlaveResponse
        return;
    }

    assert( threadId < static_cast<int>( m_tl2RequestExpanderQueues.size() ) );
    Tl2RequestExpanderQueue&     queue = m_tl2RequestExpanderQueues[threadId];
    Tl2RequestExpanderDhQueue& dhqueue = m_tl2RequestExpanderDhQueues[threadId];
    sc_event& newRequestEvent          = *(m_newTl2ReqEvent[threadId]);
    TdataCl_tl1* pDataCl               = MasterP->GetDataCl();
    
    OCPRequestGrp<Td,Ta> tl1req;

    while ( true ) {
        while ( queue.empty() )
            wait( newRequestEvent );
        Tl2RequestExpander<Td, Ta>& expander = queue.front();
        tl1req = expander.next();

        if ( m_params->sthreadbusy ) {
            while ( pDataCl->isThisThreadBusy( threadId,
                                               MasterP->getSThreadBusy() ) )
	      wait( MasterP->SThreadBusyEvent() );
        }
        // TODO: warning: the code below will override any thread busy signal
        // that gets set while it waits
        while ( !MasterP->startOCPRequest( tl1req ) )
            wait( MasterP->RequestEndEvent() );
        // if data handshake, notify that request sent, for phase ordering
        bool dataHandshake = ( m_params->datahandshake && expander.isWrite() );
        if ( dataHandshake ) {
            assert( !tl1req.HasMData );
            assert( !dhqueue.empty() );
            dhqueue.front().sentRequest();
            m_requestSentEvent[threadId]->notify( SC_ZERO_TIME );
        }
        MasterP->waitSCmdAccept();
        wait( clk->posedge_event() ); // re-sync to clock edge
        if ( expander.finished() ) {
            if ( !dataHandshake )
                m_tl2CmdAcceptEvent.notify( SC_ZERO_TIME );
            queue.pop_front();
        }
    }
} //end of MasterRequest method

//---------------------------------------------------------------------------
// Thread: OCP_TL1_TL2_Slave_Adapter::MasterDataHs
// send the group of TL1 datahs to the slave, made from the TL2 request from
// the master 
//---------------------------------------------------------------------------
template<class TdataCl_tl1 >
void OCP_TL1_TL2_Slave_Adapter<TdataCl_tl1>::MasterDataHs()
{
    int threadId = m_tl1DataHsThreadId++;
    if ( threadId >= m_params->threads ) {
        return;
    }

    assert( threadId < static_cast<int>( m_tl2RequestExpanderDhQueues.size() ) );
    Tl2RequestExpanderDhQueue& queue = m_tl2RequestExpanderDhQueues[threadId];
    sc_event& newRequestEvent        = *(m_newTl2ReqEvent[threadId]);
    sc_event& requestSentEvent       = *(m_requestSentEvent[threadId]);
    TdataCl_tl1* pDataCl             = MasterP->GetDataCl();
    MTimingGrp mTiming;
    SlaveP->getMasterTiming( mTiming );

    OCPDataHSGrp<Td> tl1dh;
    while ( true ) {
        while ( queue.empty() )
            wait( newRequestEvent );
        Tl2RequestExpanderDh<Td, Ta>& expander = queue.front();
        // wait for phase ordering
        while ( !expander.ready() )
            wait( requestSentEvent );
        tl1dh = expander.next();

        // Request-Data latency
        for ( int i=0; i < mTiming.RqDL; ++i ) {
            wait( clk->posedge_event() );
        }
        if ( m_params->sdatathreadbusy ) {
            while ( pDataCl->isThisThreadBusy( threadId,
                                               MasterP->getSDataThreadBusy() ) )
                wait( MasterP->SDataThreadBusyEvent() );
        }
        while ( !MasterP->startOCPDataHS( tl1dh ) )
            wait( MasterP->DataHSEndEvent() );
        MasterP->waitSDataAccept();
        wait( clk->posedge_event() );

        if ( expander.finished() ) {
            m_tl2CmdAcceptEvent.notify( SC_ZERO_TIME );
            queue.pop_front();
        }
    }
}

//---------------------------------------------------------------------------
// Thread: OCP_TL1_TL2_Slave_Adapter::SlaveResponse 
// get the TL1 responses, store them in the internal buffer, prepare TL2
// response to be sent to the TL2 master
//---------------------------------------------------------------------------
template<class TdataCl_tl1 >
void OCP_TL1_TL2_Slave_Adapter<TdataCl_tl1>::SlaveResponse()
{
    int threadId = m_tl2ResponseThreadId++;
    if ( threadId >= m_params->threads ) {
        // there is one SlaveResponse SC_THREAD per ocp thread
        // but technically, we start them at construction time when we don't
        // know the number of threads yet. So we start a fixed number and let
        // those who exceed the thread count die now
        return;
    }

    assert( threadId < static_cast<int>( m_tl2RequestExpanderQueues.size() ) );
    Tl1ResponseAggregatorQueue& queue = m_tl1ResponseAggregatorQueues[threadId];
    sc_event& newRespEvent            = *(m_newTl2RespEvent[threadId]);
    TdataCl_tl1* pDataCl              = MasterP->GetDataCl();
    OCPTL2ResponseGrp<Td> tl2resp;

    while ( true ) {
        if ( queue.empty() || !queue.front().finished())
            wait( newRespEvent );
        Tl1ResponseAggregator<Td, Ta>& aggregator = queue.front();
        assert( aggregator.finished() );
        tl2resp = aggregator.get();

        if ( m_params->mthreadbusy ) {
            while ( pDataCl->isThisThreadBusy( threadId, SlaveP->getMThreadBusy() ) )
	      wait( SlaveP->MThreadBusyEvent() );
        }
        while ( !SlaveP->sendOCPResponse( tl2resp ) )
            wait( SlaveP->ResponseEndEvent() );
        SlaveP->waitMRespAccept();
        // re-sync to clock edge
        wait( clk->posedge_event() );
        queue.pop_front();
    }

}  //end of SlaveResponse

template<class TdataCl_tl1 >
void OCP_TL1_TL2_Slave_Adapter<TdataCl_tl1>::SCmdAccept()
{
    SlaveP->putSCmdAccept();
}

template<class TdataCl_tl1 >
void OCP_TL1_TL2_Slave_Adapter<TdataCl_tl1>::MRespAccept()   
{
    if ( m_params->respaccept )
        MasterP->putMRespAccept();
}

// ----------------------------------------------------------------------------
// sideband processes
// ----------------------------------------------------------------------------
template<class TdataCl_tl1 >
void OCP_TL1_TL2_Slave_Adapter<TdataCl_tl1>::MputReset()
{
    if ( !m_params->mreset ) return;
    // be cautious to only move the reset signal if it affects something
    // or we risk an infinite loop if both mreset and sreset are enabled
    if ( SlaveP->getReset() ) {
        if ( !MasterP->getReset() )
            MasterP->MResetAssert();
    } else {
        if ( MasterP->getReset() )
            MasterP->MResetDeassert();
    }
}

template<class TdataCl_tl1 >
void OCP_TL1_TL2_Slave_Adapter<TdataCl_tl1>::SputReset()
{
    if ( !m_params->sreset ) return;
    if ( MasterP->getReset() ) {
        if ( !SlaveP->getReset() )        
            SlaveP->SResetAssert();
    } else {
        if ( SlaveP->getReset() )
            SlaveP->SResetDeassert();
    }
}

template<class TdataCl_tl1 >
void OCP_TL1_TL2_Slave_Adapter<TdataCl_tl1>::MputFlags()
{
    MasterP->MputMFlag(SlaveP->SgetMFlag());
    MasterP->MputMError(SlaveP->SgetMError());
}

template<class TdataCl_tl1 >
void OCP_TL1_TL2_Slave_Adapter<TdataCl_tl1>::SputSFlag()
{
    SlaveP->SputSFlag(MasterP->MgetSFlag());
}
template<class TdataCl_tl1 >
void OCP_TL1_TL2_Slave_Adapter<TdataCl_tl1>::SputSError()
{
    SlaveP->SputSError(MasterP->MgetSError());
}
template<class TdataCl_tl1 >
void OCP_TL1_TL2_Slave_Adapter<TdataCl_tl1>::SputSInterrupt()
{
    SlaveP->SputSInterrupt(MasterP->MgetSInterrupt());
}

// ----------------------------------------------------------------------------
//  OCP_TL1_TL2_Slave_Adapter::end_of_elaboration()
//
//  This method is activated at the end of the elaboration phase
//  (when all binding operations are finished).
// ----------------------------------------------------------------------------
template<class TdataCl_tl1 >
void OCP_TL1_TL2_Slave_Adapter<TdataCl_tl1>::end_of_elaboration()
{
    sc_module::end_of_elaboration();

    // Get channel params and timing
    m_params = SlaveP->GetParamCl();
    SlaveP->getMasterTiming( m_times );
 
    int threads = m_params->threads;
    m_tl2RequestExpanderQueues.resize     ( threads );
    m_tl2RequestExpanderDhQueues.resize   ( threads );
    m_tl1ResponseAggregatorQueues.resize  ( threads );
    m_tl2BurstCounter.resize              ( threads );
    m_burstSequence.resize                ( threads,
                                            OcpIp::BurstSequence<Ta> (
                                                m_params->data_wdth,
                                                m_params->addr_wdth, 0, 0 ) );
    m_tl2SDataPool.resize                 ( threads );
    m_newTl2ReqEvent.resize               ( threads );
    m_newTl2RespEvent.resize              ( threads );
    m_requestSentEvent.resize             ( threads );

    for ( int t=0; t<threads; ++t) {
        m_newTl2ReqEvent[t]   = new sc_event();
        m_newTl2RespEvent[t]  = new sc_event();
        m_requestSentEvent[t] = new sc_event();
        m_tl2SDataPool[t]     = new Td[POOL_SIZE];        
    }
}
