// 
//  (c) Copyright OCP-IP 2003, 2004, 2005
//  OCP-IP Confidential and Proprietary
//
// ============================================================================
//      Project : OCP SLD WG, Layer adapters
//      Authors : Herve Alexanian, SONICS, herve@sonicsinc.com
//                Stephane Guntz, PROSILOG, guntz@prosilog.com
//		  Yann Bajot, PROSILOG, bajot@prosilog.com
//
//  Description : Transaction Level - Layer-1 to Layer-2 OCP Master Adapter
//  $Id:
//	 
//  The TL1/TL2 master adapter is a systemc module with a OCP TL2 master
//  port and a OCP TL1 slave port.
//  It must be connected to:
//      - a TL1 master through a TL1 channel
//      - a TL2 slave  through a TL2 channel
//
//  The adapter is defined in a templated class OCP_TL1_TL2_Master_Adapter,
//  and the template argument is the TL1 channel data class.
//
//  Constructor arguments
//   - module name
//   - max_chunk_length: the maximum number of TL1 requests pertaining to the
//     same burst that the adapter aggregates before passing to TL2
//   - adapter_depth and time_out are still present for code compatibility but
//     are deprecated and have no effect
//
// ============================================================================

#ifndef _OCP_TL1_TL2_MASTER_ADAPTER_H
#define _OCP_TL1_TL2_MASTER_ADAPTER_H

#include "systemc.h"
#include "ocp_globals.h"
#include "ocp_tl1_data_cl.h"
#include "ocp_tl1_slave_port.h"
#include "ocp_tl2_master_port.h"
#include "ocp_tl_param_cl.h"
#include "async_signal_relay.h"
#include "ocp_utils.h"
#include <deque>
#include <vector>

template <typename Td, typename Ta>
class Tl1RequestAggregator
{
  public:
    Tl1RequestAggregator( int maxChunkLength, bool lastOfBurst = false );
    void setDataPtr    ( Td*           p ) { m_tl2Req.MDataPtr     = p; }
    void setByteEnPtr  ( unsigned int* p ) { m_tl2Req.MByteEnPtr   = p; }
    void setDataInfoPtr( uint64*       p ) { m_tl2Req.MDataInfoPtr = p; }
    bool finishedRequest() const;
    bool finished() const;
    void setParamCl( const OCPParameters* pParams ) { m_pParams = pParams; }
    void setBurstPosition( unsigned int pos ) { m_burstPosition = pos; }

    // the workhorses
    void next    ( const OCPRequestGrp<Td, Ta>& tl1Req    );
    void nextData( const OCPDataHSGrp <Td    >& tl1DataHs );
    OCPTL2RequestGrp<Td, Ta>  get() {
        return m_tl2Req;
    }

  private:
    unsigned int             m_numAggregated;
    unsigned int             m_numDataAggregated;
    unsigned int             m_burstPosition;
    int                      m_maxChunkLength;
    OCPTL2RequestGrp<Td, Ta> m_tl2Req;
    const OCPParameters     *m_pParams;
};

template <typename Td>
class Tl2ResponseExpander
{
  public:
    Tl2ResponseExpander( const OCPTL2ResponseGrp<Td>& resp, Td* pDataPtr );
    bool finished() const { return m_numExpanded == m_tl2Resp.DataLength; }
    bool isRead()  const { return m_isRead; }
    bool isWrite() const { return !m_isRead; }
    OCPResponseGrp<Td> next(); // the workhorse
    const OCPTL2ResponseGrp<Td>& get() const { return m_tl2Resp; }
    template <typename Ta>
    void setTurnAroundReq( const OCPTL2RequestGrp<Td, Ta>& req ) {
        m_turnAroundBurstLength = req.MBurstLength;
    }

  private:
    unsigned int  m_numExpanded;
    unsigned int  m_turnAroundBurstLength;
    bool m_isRead;
    OCPTL2ResponseGrp<Td> m_tl2Resp;
    OCPResponseGrp<Td>    m_tl1Resp;
};

template <typename Td> class ArrayStack;
template <class TdataCl_tl1> class OCP_TL1_TL2_Master_Adapter
: public sc_module  
{
 public:  
    //TL1 data type and address type
    typedef typename TdataCl_tl1::DataType Td;
    typedef typename TdataCl_tl1::AddrType Ta;
    typedef OCP_TL1_SlaveIF<TdataCl_tl1>   SlaveIF;
    typedef OCP_TL2_MasterIF<Td, Ta>       MasterIF;

    // TL1 Slave Port and TL2 Master port
    OCP_TL1_SlavePort<TdataCl_tl1 > SlaveP;
    OCP_TL2_MasterPort<Td, Ta > MasterP;

    //clock port
    sc_in_clk clk;

    SC_HAS_PROCESS(OCP_TL1_TL2_Master_Adapter);

    // constructor. Keep compatible for now. Last 2 arguments are deprecated
    OCP_TL1_TL2_Master_Adapter(sc_module_name name, int max_chunk_length,
                               int adapter_depth=-1, sc_time timeout = sc_time( 1, SC_NS ) );

    // destructor
    ~OCP_TL1_TL2_Master_Adapter();  
  
  private:
    void end_of_elaboration();

    // Settings
    const OCPParameters* m_params;   
    STimingGrp m_times;
 
    // Storage pools
    ArrayStack<unsigned int>* m_ptl2MByteEnPool;
    ArrayStack<Td>*           m_ptl2MDataPool;
    ArrayStack<Td>*           m_ptl2SDataPool;
    ArrayStack<uint64>*       m_ptl2MDataInfoPool;

    // TL1-TL2 request path
    OCPRequestGrp<Td,Ta> m_tl1Request;
    typedef deque<Tl1RequestAggregator<Td, Ta> > Tl1RequestAggregatorQueue;
    vector<Tl1RequestAggregatorQueue>            m_tl1RequestAggregatorQueues;
    vector<sc_event*>                            m_newTl2ReqEvent;
    vector<OcpIp::BurstCounter>                  m_burstCount;
    OcpIp::BurstSequence<Ta>*                    m_pAddrSequence;
    sc_event                                     m_tl1CmdAcceptEvent;
    void SlaveRequest();
    void SCmdAccept();
    void MasterRequest();

    // TL1-TL2 data hs path
    OCPDataHSGrp<Td>          m_tl1DataHs;
    sc_event                  m_tl1DataAcceptEvent;
    sc_event                  m_slaveDataHsStartEventPlusZero;
    void SlaveDataHs();
    void SlaveResyncDataHs();
    void SDataAccept();

    // TL2-TL1 response path
    OCPTL2ResponseGrp<Td> m_tl2Response;
    typedef deque<Tl2ResponseExpander<Td> >   Tl2ResponseExpanderQueue;
    vector<Tl2ResponseExpanderQueue>          m_tl2ResponseExpanderQueues;
    vector<sc_event*>    m_newTl2RespEvent;
    sc_event             m_tl2RespAcceptEvent;
    void MasterResponse();
    void SlaveResponse();
    void MRespAccept();

    // TL2 response turnaround queues
    vector<deque<OCPTL2RequestGrp<Td, Ta> > > m_tl2ReqPendingResp;
    vector<bool>                              m_tl2BurstOpen;

    // Sideband
    void SputFlags();
    void SputMFlag();
    void SputMError();
    void MputReset();
    void SputReset();

  private:
    // only to assign the right ocp thread to each master/datahs SC_THREADN
    // do not use after the threads have started
    int m_tl2RequestThreadId;
    int m_tl1ResponseThreadId;

    int m_maxChunkLength;   // maximum length of a TL2 chunk

    // pass threadbusy signals through
    OcpIp::PortVoidFunctor< MasterIF, unsigned int,
        &MasterIF::getSThreadBusy> m_bindGetSThreadBusy;
    OcpIp::PortFunctor< SlaveIF, void, unsigned int,
        &SlaveIF::putSThreadBusy> m_bindPutSThreadBusy;
    OcpIp::AsyncSignalRelay<unsigned int>  m_sThreadBusyRelay;

    OcpIp::PortVoidFunctor< SlaveIF, unsigned int,
        &SlaveIF::getMThreadBusy> m_bindGetMThreadBusy;
    OcpIp::PortFunctor< MasterIF, void, unsigned int,
        &MasterIF::putMThreadBusy> m_bindPutMThreadBusy;
    OcpIp::AsyncSignalRelay<unsigned int>  m_mThreadBusyRelay;
};

#endif // _OCP_TL1_TL2_MASTER_ADAPTER_H
