#ifndef _params_h_
#define _params_h_

#include <vector>
#include <utility>
#include "NamedIndex.h"

//! \brief Store parameters indexed by values of a template enumerated type
//! \param typeT the type to store parameters. This class is limited to
//! homogeneous parameters (all of the same type)
//! \param nameT enumerated value listing the names of the parameters
template<typename typeT, typename nameT>
class ParamSet : protected NamedIndex<nameT> {
  protected:
    typedef nameT                                ParamIndex;
    typedef typename NamedIndex<nameT>::Iterator ParamIterator;
  private:
    // Params are represented as a pair for the default value and actual value
    typedef pair<typeT, typeT>    ParamPair ;
    vector<ParamPair>  m_params;

  protected:
    void          setDefault( int index, typeT def );
    //! \brief set all defaults. Does nothing unless overridden by derived class
    virtual void  setDefaults() {}

  public:
    typeT         getDefault( ParamIndex index ) const;
    typeT         getParam  ( ParamIndex index ) const;
    void          setParam  ( ParamIndex index, typeT value );

    ParamSet() :
        m_params( NamedIndex<nameT>::end() ) {}
    virtual ~ParamSet() {}
};

//! \brief Set a default value for a parameter
//! \param index parameter index (of enum type paramNameT)
//! \param def default value
template<typename typeT, typename nameT>
void
ParamSet<typeT, nameT>::setDefault( int index, typeT def )
{
    int nindex = static_cast<int>( index );
    m_params[nindex].first  = def;
    m_params[nindex].second = def;
}

//! \brief Set the value for a parameter
//! \param index parameter index (of enum type paramNameT)
//! \param value parameter value
template<typename typeT, typename nameT>
void
ParamSet<typeT, nameT>::setParam( ParamIndex index, typeT value )
{
    int nindex = static_cast<int>( index );
    m_params[nindex].second = value;
}

//! \brief Get the default value for a parameter
//! \param index parameter index (of enum type paramNameT)
template<typename typeT, typename nameT>
typeT
ParamSet<typeT, nameT>::getDefault( ParamIndex index ) const {
    int nindex = static_cast<int>( index );
    return m_params[nindex].first;
}

//! \brief Get the value for a parameter
//! \param index parameter index (of enum type paramNameT)
template<typename typeT, typename nameT>
typeT
ParamSet<typeT, nameT>::getParam( ParamIndex index ) const {
    int nindex = static_cast<int>( index );
    return m_params[nindex].second;
}

#endif /* _params_h_ */
