// 
//  Copyright 2008 OCP-IP
//
// ============================================================================
//      Project : OCP SLD WG
//       Author : Anssi Haverinen, Nokia Inc.
//                Robert Guenzel (from TU of Braunschweig) for Greensocs Ltd.
//           $Id: slave.cc,v 1.1 2005/01/07 03:42:33 Anssi Exp $
//
//  Description : OCP API - TL1 profile example
// ============================================================================

#include "slave.h"

// ----------------------------------------------------------------------------
// Constructor
// ----------------------------------------------------------------------------
Slave::Slave (sc_core::sc_module_name name_)
  : sc_core::sc_module (name_), tpP("tpPort"), req(NULL), ongoing_rsp(false) {

  // initialize common members
  lock=false;
  reset=false;
  address = 0;
  burststart = true;
  last_request = tlm::TLM_IGNORE_COMMAND;
  unlock_object=lock_object=NULL;
  bytemask = 0;
  b_length=1;
  for (int i=0;i<1024;i++)
    memory[i] = 0;
  reqcnt=0;
  SC_METHOD(proc);
  sensitive<<clk.pos();
  dont_initialize(); // make sure we execute only at clock edge
  
  tpP.register_nb_transport_fw(this, &Slave::nb_transport);
  tpP.activate_synchronization_protection();
}


// ----------------------------------------------------------------------------
// Destructor
// ----------------------------------------------------------------------------
Slave::~Slave(){}


// ----------------------------------------------------------------------------
//  Method : Slave::proc()
//
//  Synchronous slave process
//
// ----------------------------------------------------------------------------

void Slave::proc(){
  if (reset){
    return;
  }


  // If request was write
  if (last_request == tlm::TLM_WRITE_COMMAND) {
    assert(req_data);
    assert(tpP.get_extension<ocpip::posted>(*req_data)); //we do not support nonposted writes
    if (lock && address==locked_address){
      assert(lock_object==unlock_object); //make sure the unlocking write was really locked to the read
      lock = false; // ReadEx handling is not quite complete here
      lock_object=unlock_object=NULL;
    }
    unsigned int effective_cnt=reqcnt-1;
    if (current_sequence==ocpip::WRAP){
      effective_cnt=address-(~(b_length-1) & address);
    }
    
    memory[address] = (*(((Td*)(req_data->get_data_ptr()))+effective_cnt)) & bytemask;
    
    last_request = tlm::TLM_IGNORE_COMMAND;
    req_data->release(); //we are done so we can release now
    req_data=NULL;
  }
  else 
  if (last_request == tlm::TLM_READ_COMMAND) {
    assert(rsp);
    if (!ongoing_rsp) {
      // Set OCP response
      if (lock)
        rsp->set_response_status(tlm::TLM_GENERIC_ERROR_RESPONSE); // readex must be followed by write
      else
        rsp->set_response_status(tlm::TLM_OK_RESPONSE);
      unsigned int effective_cnt=reqcnt-1;
      if (current_sequence==ocpip::WRAP){
        effective_cnt=address-(~(b_length-1) & address);
      }
        
      *(((Td*)(rsp->get_data_ptr()))+effective_cnt) = memory[address];

      
      // lock address after ReadEx
      if (read_ex) {
        lock = true;
        locked_address = address;
      }

      // Send response
      //tpP->startOCPResponse(resp);
      time=sc_core::SC_ZERO_TIME;
      phase=tlm::BEGIN_RESP;
      if (read_ex) lock_object->lock_is_understood_by_slave=true; //tell the master we understand his locks
      tlm::tlm_sync_enum retVal=tpP->nb_transport_bw(*rsp,phase, time);
      switch(retVal){
        case tlm::TLM_ACCEPTED: ongoing_rsp=true; break;
        case tlm::TLM_UPDATED:
          if (phase!=tlm::END_RESP) {
            std::cerr<<"Error in "<<name()<<" got unexpected phase update "<<phase<<" in response to BEGIN_RESP"<<std::endl;
            exit(1);
          }
          break;
        case tlm::TLM_COMPLETED: break; //we don't check that it was really the last resp. we just trust the master...
      }

#ifdef DEBUG_G1
      std::cout << "Slave sent response " << rsp->get_response_status()
	   << " time " << sc_core::sc_time_stamp().to_seconds()
	   << " data " << (*(((Td*)(rsp->get_data_ptr()))+reqcnt-1)) << std::endl;
#endif
//      if (read_ex) lock_object->atomic_txn_completed();
      last_request = tlm::TLM_IGNORE_COMMAND;
      rsp=NULL;
    }
  } // end if (last_request)

  // If a previous read has been served, we check for new requests
  if (last_request == tlm::TLM_IGNORE_COMMAND ) {
    if (req) {
      last_request = req->get_command();

      unsigned int MByteEn=0;
      byte_en_helper::setBE(MByteEn, req, sizeof(Td), reqcnt);
    
      // Byte enable to bitmask
      if (MByteEn & 0x1)
        bytemask = 0xFF; 
      if  (MByteEn & 0x2)
        bytemask = 0xFF00 | bytemask; 
      if  (MByteEn & 0x4)
        bytemask = 0xFF0000 | bytemask; 
      if  (MByteEn & 0x8)
        bytemask = 0xFF000000 | bytemask; 

      // Burst addressing
      if (burststart) {
        ocpip::burst_length* b_len;
        if (tpP.get_extension<ocpip::burst_length>(b_len, *req)) b_length=b_len->value;
        else b_length=1;
        ocpip::burst_sequence* b_seq;
        if (tpP.get_extension<ocpip::burst_sequence>(b_seq, *req)){
          current_sequence=b_seq->value.sequence;
          if (current_sequence==ocpip::WRAP)
            address=b_seq->value.xor_wrap_address/4;
          else
            address = req->get_address()/4; // First of burst
        }
        else{
          current_sequence=ocpip::INCR;
          address = req->get_address()/4; // First of burst
        }
        reqcnt=0;
        ocpip::lock* lck;
        read_ex=tpP.get_extension<ocpip::lock>(lck, *req) && (req->get_command()==tlm::TLM_READ_COMMAND);
        unlock_write=tpP.get_extension<ocpip::lock>(lck, *req) && (req->get_command()==tlm::TLM_WRITE_COMMAND);
        if (read_ex) {
          assert(lck->value->number_of_txns==2);
          lock_object=lck->value;
        }
        if (unlock_write) unlock_object=lck->value;
      }
      else {
        if (current_sequence==ocpip::INCR)
          address = address+1; //incrementing address
        else //wrap
        if (current_sequence==ocpip::WRAP)
          if ((address+1) < ((address & (~(b_length-1)))+ b_length)) // wrapping address
            address = address+1; 
          else
            address = address & (~(b_length-1));
        else
          assert(0);
      }
      reqcnt++;
      
      if (b_length==reqcnt) // can handle only precise bursts
        burststart = true; // end of burst, next can start
      else
        burststart = false; // burst continues
        
      if (last_request == tlm::TLM_READ_COMMAND) {
        rsp=req; 
#ifdef DEBUG_G1
        std::cout << "Slave got read request "
         << "  time  = " << sc_core::sc_time_stamp().to_seconds()
         << "  address  = " << rsp->get_address()<< std::endl;
#endif
      }
      else {
        req_data=req;      
        req_data->acquire(); //we will use the data only in the next cycle, but the master could release after END_REQ, so we acquire
#ifdef DEBUG_G1
        std::cout << "Slave got write request "
         << "  time  = " << sc_core::sc_time_stamp().to_seconds()
         << "  reqcnt = "<<reqcnt
         << "  data  = " << (*(((Td*)(req_data->get_data_ptr()))+reqcnt-1)) << std::endl;
#endif

      }
      
      time=sc_core::SC_ZERO_TIME;
      phase=tlm::END_REQ;
      tlm::tlm_sync_enum retVal=tpP->nb_transport_bw(*req,phase, time);
      switch(retVal){
        case tlm::TLM_ACCEPTED: 
          if (unlock_write) unlock_object->atomic_txn_completed();
          break;
        case tlm::TLM_UPDATED:
          std::cerr<<"Error in "<<name()<<" got unexpected phase update "<<phase<<" in response to END_REQ"<<std::endl;
          exit(1);
          break;
        case tlm::TLM_COMPLETED:;
      }
      req=NULL;
    }
  }
} // end of method

tlm::tlm_sync_enum Slave::nb_transport(tlm::tlm_generic_payload& txn, tlm::tlm_phase& ph, sc_core::sc_time& tim){
  switch(ph){
    case tlm::BEGIN_REQ: req=&txn; break;
    case tlm::END_RESP: ongoing_rsp=false; break;
    default:
      if (ph==ocpip::BEGIN_RESET) {
        //since we use synch protection we can do reset here (as we will have done our job for this cycle already)
        lock=false;
        address = 0;
        burststart = true;
        last_request = tlm::TLM_IGNORE_COMMAND;
        unlock_object=lock_object=NULL;
        bytemask = 0;
        b_length=1;
        reqcnt=0;       
        reset=true;
      }
      else
      if (ph==ocpip::END_RESET) reset=false;
      else{
        std::cerr<<"Error in "<<name()<<" got unexpected phase "<<phase<<" in nb_transport_fw"<<std::endl;
        exit(1);        
      }
  }
  return tlm::TLM_ACCEPTED;
}


