///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// (c) Copyright OCP-IP 2008
// OCP-IP Confidential and Proprietary
//
//
//============================================================================
//      Project : OCP SLD WG
//       Author : James Aldis (TI France)
//                Robert Guenzel (from TU of Braunschweig) for Greensocs Ltd.
//
//          $Id:
//
//  Description :  Simple slave for example.
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#include "simpleSlave.h"

// -------------------------------------------------------------------
// constructor
// -------------------------------------------------------------------
template<unsigned int BUSWIDTH>
Slave<BUSWIDTH>::Slave(
    sc_core::sc_module_name n,
    int            id,
    Ta             memory_byte_size,
    std::ostream*       debug_os_ptr
) : sc_core::sc_module(n),
    tpP("tpPort", this, &Slave::setOCPTL1MasterTiming),
    clk("clkPort"),
    m_ID(id),
    m_MemoryByteSize(memory_byte_size),
    m_Memory(NULL),
    m_debug_os_ptr(debug_os_ptr),
    m_curSThreadBusy(0),
    m_threads(1),
    m_datahandshake(false),
    m_writeresp_enable(false),
    m_sthreadbusy(false),
    m_sthreadbusy_exact(false),
    m_mthreadbusy(false),
    m_cmdaccept(true),
    m_limitreq_max(4),
    m_Latency(3),
    m_req_txn(NULL),
    m_mtb(0)

{
    // Note: member variables that depend on values of
    // configuration parameters are constructed when those
    // values are known - at the end of elaboration.

    // setup a SystemC thread process, which uses dynamic sensitive
    SC_THREAD(requestThreadProcess);
    sensitive<<clk.pos();

    // setup a SystemC thread process, which uses dynamic sensitive
    SC_THREAD(responseThreadProcess);
    sensitive<<clk.pos();

    assert(m_Latency);

    //assuming default timing the slave will expect mthreadbusy to be stable 1 ps after the clock edge
    m_mthreadbusy_sample_time=sc_core::sc_time(1, sc_core::SC_PS);
    tpP.register_configuration_listener_callback(this, &Slave::set_configuration);
    tpP.make_generic();
    tpP.register_nb_transport_fw(this, &Slave::nb_transport);
    tpP.activate_synchronization_protection();

}

template<unsigned int BUSWIDTH>
Slave<BUSWIDTH>::~Slave()
{
    delete m_Memory;
}

template<unsigned int BUSWIDTH>
void Slave<BUSWIDTH>::set_configuration(const ocpip::ocp_parameters& passedConfig, const std::string& channelName){
    if (passedConfig.config_state==ocpip::ocp_generic) return; //no change yet
    m_parameters=passedConfig;
    // Set the number of threads
    m_threads=passedConfig.threads;

    if (m_threads > 1) {
        std::cout << "Warning: Singled threaded reference Slave "
            << name() << " attached to multi-threaded OCP." << std::endl;
        std::cout << "Only commands sent on thread 0 will be processed."
            << std::endl;
    }

    // Does the channel use data handshaking?
    m_datahandshake=passedConfig.datahandshake;
    // Is so, quit as this Slave does not handle data handshake.
    assert(!m_datahandshake);

    // Do writes get reponses?
    m_writeresp_enable=passedConfig.writeresp_enable;

    // is SThreadBusy part of the channel?
    m_sthreadbusy=passedConfig.sthreadbusy;

    // is this slave expected to follow the threadbusy exact protocol?
    m_sthreadbusy_exact=passedConfig.sthreadbusy_exact;

    // is MThreadBusy part of the channel?
    m_mthreadbusy=passedConfig.mthreadbusy;

    // is SCmdAccept part of the channel?
    m_cmdaccept=passedConfig.cmdaccept;

    /////////////
    //
    // Process Slave Parameters
    //
    /////////////

    // For Debugging
    if (m_debug_os_ptr) {
        (*m_debug_os_ptr) << "DB (" << name() << "): "
            << "Configuring Slave." << std::endl;
    }

    /////////////
    //
    // Initialize the Slave with New Parameters
    //
    /////////////


    // Create the memory:

    if (m_Memory) {
        // Just in case we are called multiple times.
        delete m_Memory;
    }
    char id_buff[10];
    sprintf(id_buff,"%d",m_ID);
    std::string my_id(id_buff);
    m_Memory =
       new MemoryCl<ocpip::ocp_data_class_unsigned<BUSWIDTH,32> >(my_id,passedConfig.addr_wdth,sizeof(Td));

}

template<unsigned int BUSWIDTH>
void Slave<BUSWIDTH>::setModuleConfiguration(map_string_type& passedMap){
    if (!(getIntOCPConfigValue("", "limitreq_max",
            m_limitreq_max, passedMap,"")) ) {
        // Could not find the parameter so we must set it to a default
#ifdef DEBUG
        cout << "Warning: paramter \"" << "limitreq_max"
               << "\" was not found in the module parameter map." << endl;
        cout << "         setting missing parameter to "<<m_limitreq_max<<"." << endl;
#endif
    }

    if (!(getIntOCPConfigValue("", "latency0",
            m_Latency, passedMap,"")) ) {
        // Could not find the parameter so we must set it to a default
#ifdef DEBUG
        cout << "Warning: paramter \"" << "latency0"
               << "\" was not found in the module parameter map." << endl;
        cout << "         setting missing parameter to "<<m_Latency<<"." << endl;
#endif
    }
}

// -------------------------------------------------------------------
//  SystemC Method Slave::end_of_elaboration()
// -------------------------------------------------------------------
//
//  At this point, everything has been built and connected.
//  We are now free to get our OCP parameters and to set up our
//  own variables that depend on them.
//
template<unsigned int BUSWIDTH>
void Slave<BUSWIDTH>::end_of_elaboration()
{
    sc_core::sc_module::end_of_elaboration();
    // Clear the response queue
    m_ResponseQueue.resetState();

    //get clock period information
    if (dynamic_cast<sc_core::sc_clock*>(clk.get_interface())){
      m_clkPeriod=(dynamic_cast<sc_core::sc_clock*>(clk.get_interface()))->period();
    }
    else{
      std::cout<<"An sc_clock has to be connected to the clock port of "<<name()<<std::endl;
      assert(dynamic_cast<sc_core::sc_clock*>(clk.get_interface()));
    }

    if (m_mthreadbusy){
      ocpip::ocp_tl1_slave_timing myTiming;
      //sthreadbusy gets deasserted after mthreadbusy got read (see responseThreadProcess)
      myTiming.SThreadBusyStartTime=m_mthreadbusy_sample_time;
      tpP.set_slave_timing(myTiming);

    }

    if (m_sthreadbusy | m_sthreadbusy_exact){
      tb_txn=tpP.get_tb_transaction();
      tb=tpP.template get_extension<ocpip::cmd_thread_busy>(*tb_txn);
      tb_ph=ocpip::CMD_THREAD_BUSY_CHANGE;
    }
}

template<unsigned int BUSWIDTH>
void Slave<BUSWIDTH>::setOCPTL1MasterTiming(ocpip::ocp_tl1_master_timing master_timing){
  if (master_timing.MThreadBusyStartTime+sc_core::sc_time(2, sc_core::SC_PS) > m_mthreadbusy_sample_time){
    ocpip::ocp_tl1_slave_timing myTiming;
    m_mthreadbusy_sample_time=master_timing.MThreadBusyStartTime+sc_core::sc_time(2, sc_core::SC_PS);  //since we use DCP we add 1 for DCP and one for sampling
    myTiming.SThreadBusyStartTime=m_mthreadbusy_sample_time;
    tpP.set_slave_timing(myTiming);
  }
}

template<unsigned int BUSWIDTH>
void Slave<BUSWIDTH>::requestThreadProcess()
{

    //The response to the new request
    ocpip_legacy::OCPResponseGrp<Td>   resp;

    tlm::tlm_phase phase;

    // Time after which the response can be sent or this
    // request can be cleared from incoming queue.
    sc_core::sc_time          send_time;
    sc_core::sc_time          txn_time;

    // We are in the initialization call.
    // Wait for the first simulation cycle.
    sc_core::wait();

    // main loop
    while (true) {
        // ----------------------------------------------------------
        // (1) Get the next request
        // ----------------------------------------------------------
        //tpP->getOCPRequestBlocking(req,false);
        while(!m_req_txn) sc_core::wait(req_event);
        tlm::tlm_generic_payload* txn=m_req_txn;
        resp.SDataPtr=(Td*)(txn); //we use the SDataPtr to hold the ref to our txn

        m_req_txn=NULL;
        // ----------------------------------------------------------
        // (2) process the new request and generate a response.
        // ----------------------------------------------------------

        // compute the word address
        if (txn->get_address() >= m_MemoryByteSize) {
            txn->set_address(txn->get_address() - m_MemoryByteSize);
        }

        unsigned int MByteEn=-1;


        if (txn->get_byte_enable_ptr()){
          MByteEn=0;
          unsigned int mask=0x1;
          for (unsigned int i=0;
                   i<((BUSWIDTH+7)/8);
                   i++)
            if (txn->get_byte_enable_ptr()[i % txn->get_byte_enable_length()]==0xFF)
              MByteEn|=mask;
            mask<<=1;
        }

        // write to or read from the memory
        switch (txn->get_command()) {
            case tlm::TLM_WRITE_COMMAND:
                assert(tpP.template get_extension<ocpip::posted>(*txn)); //we do not support nonposted writes
                // posted write to memory
                m_Memory->write(txn->get_address(), *((Td*)txn->get_data_ptr()),MByteEn);
                if (m_writeresp_enable){
                  resp.SResp = ocpip_legacy::OCP_SRESP_DVA;
                }
                else{
                  // note that posted writes do not have responses.
                  // However, they do have a processing delay that can
                  // contribute to a max request limit back up.
                  // To solve this problem, requests that have no
                  // response to generate a dummy respose with
                  // SRESP=NULL which is defined as "No response".
                  // Dummy responses are never sent out on the channel.
                  resp.SResp = ocpip_legacy::OCP_SRESP_NULL;
                }
                break;

            case tlm::TLM_READ_COMMAND:
                // NOTE that for a single threaded slave,
                // Read-EX works just like Read
                // read from memory
                m_Memory->read(txn->get_address(), *((Td*)txn->get_data_ptr()),MByteEn);
                // setup a read response
                resp.SResp = ocpip_legacy::OCP_SRESP_DVA;
                break;

            default:
                std::cout << "MCmd #" << txn->get_command() << " not supported yet."
                         << std::endl;
                sc_core::sc_stop();
                break;
        }

        // ----------------------------------------------------------
        // (3) generate a completion time stamp and add the response
        //     to the queue
        // ----------------------------------------------------------

        // compute pipelined response delay
        send_time = sc_core::sc_time_stamp() + m_clkPeriod*m_Latency - sc_core::sc_get_time_resolution();
        // purge the queue of any posted write place holder responses
        // that have reached their send times
        m_ResponseQueue.purgePlaceholders();

        m_ResponseQueue.enqueueBlocking(resp, send_time);
        // ----------------------------------------------------------
        // (4) if our queue is full, generate back pressure halt
        //     the flow of requests. Otherwise, accept the request
        //     and move on.
        // ----------------------------------------------------------

        // Do we need to set SThreadBusy??
        if (m_sthreadbusy && (m_ResponseQueue.length() >= m_limitreq_max)) {
            sc_core::wait(); //wait until next cycle to set busy
            //m_curSThreadBusy = 1;
            tb->value=1;
            txn_time=sc_core::SC_ZERO_TIME;
            //tpP->putSThreadBusy(m_curSThreadBusy);
            tpP->nb_transport_bw(*tb_txn, tb_ph, txn_time);
        }

        // Should we accept this command?
        if ( m_cmdaccept ) {
            // if queue is full, delay accepting request
            while (m_ResponseQueue.length() >= m_limitreq_max) {
                // Our queue is full. Wait for this to change.
                sc_core::wait();
            }
            // now it is okay to accept the request
            //tpP->putSCmdAccept();
            phase=tlm::END_REQ;
            txn_time=sc_core::SC_ZERO_TIME;
            tlm::tlm_sync_enum retVal=tpP->nb_transport_bw(*txn, phase, txn_time);
            switch(retVal){
              case tlm::TLM_ACCEPTED: break;
              case tlm::TLM_UPDATED:
                std::cerr<<"TLM_UPDATED is not allowed in response to END_REQ"<<std::endl;
                exit(1);
              case tlm::TLM_COMPLETED:
                std::cerr<<"COMPLETION is not allowed for OCP TL1!"<<std::endl;
                exit(1);
            }
        }

    }
}

template<unsigned int BUSWIDTH>
void Slave<BUSWIDTH>::responseThreadProcess()
{
    ocpip_legacy::OCPResponseGrp<Td>   resp;
    sc_core::sc_time          send_time;
    tlm::tlm_generic_payload* txn;
    tlm::tlm_phase phase;
    sc_core::sc_time time;

    sc_core::wait();

    // main loop
    while (true) {

        // -------------------------------------------------
        // (1) Find a response to place on the channel
        // -------------------------------------------------

        // Get to next response (wait for one, if necessary).

        // First, clear any stale write latency waits
        m_ResponseQueue.purgePlaceholders();

        // Get the next request off of the queue
        m_ResponseQueue.dequeueBlocking(resp,send_time);
        //resp.SThreadID = selectedThread;
        txn=reinterpret_cast<tlm::tlm_generic_payload*>(resp.SDataPtr);
        // check if we still need to wait
        while (send_time > sc_core::sc_time_stamp()) {
            sc_core::wait();
        }
        if (m_debug_os_ptr) {
            (*m_debug_os_ptr) << "DB (" << name() << "): "
                << "slave wait time = "
                    << send_time.value() << std::endl;
        }
        send_time=sc_core::sc_time_stamp();

        // The response could be a place holder response
        // used to implement write latency. If this is the case,
        // skip the rest of the steps.

        if (resp.SResp == ocpip_legacy::OCP_SRESP_NULL) {
            if (m_debug_os_ptr) {
                (*m_debug_os_ptr) << "DB (" << name() << "): "
                    << "finished Write Latency waiting." << std::endl;
            }
            if (!m_cmdaccept) {
              txn->release(); //we release now if we acquired previously
              if (m_debug_os_ptr) {
                  (*m_debug_os_ptr) << "DB (" << name() << "): "
                      << "release." <<txn<< std::endl;
              }
            }
        } else {

            txn->set_response_status(tlm::TLM_OK_RESPONSE);
            // ----------------------------------
            // (2) is MThreadBusy?
            // ----------------------------------

            if (m_mthreadbusy) {
                sc_core::wait(m_mthreadbusy_sample_time);
                //mthreadbusy = tpP->getMThreadBusy(); set m_mtb in nb_transport
                while (m_mtb & 1) {
                    sc_core::wait();
                    sc_core::wait(m_mthreadbusy_sample_time);
                    //mthreadbusy = tpP->getMThreadBusy();
                }
            }

            // ----------------------------------
            // (3) return a response
            // ----------------------------------

            if (m_debug_os_ptr) {
                (*m_debug_os_ptr) << "DB (" << name() << "): "
                    << "send response." << std::endl;
                (*m_debug_os_ptr) << "DB (" << name() << "): "
                    << "    t = " << sc_core::sc_time_stamp() << std::endl;
                (*m_debug_os_ptr) << "DB (" << name() << "): "
                    << "    SResp: " << resp.SResp << std::endl;
                (*m_debug_os_ptr) << "DB (" << name() << "): "
                    << "    SData: " << (*((Td*)txn->get_data_ptr())) << std::endl;
            }

            // Send out the response
            //tpP->startOCPResponseBlocking(resp);
            phase=tlm::BEGIN_RESP;
            time=sc_core::SC_ZERO_TIME;
            tlm::tlm_sync_enum retVal=tpP->nb_transport_bw(*txn, phase, time);
            switch(retVal){
              case tlm::TLM_UPDATED:
                if (phase!=tlm::END_RESP){
                  std::cerr<<"Unexpected phase "<<phase<<" on return path in "<<name()<<" at "<<sc_core::sc_time_stamp()<<std::endl;
                  exit(1);
                }
                break;
              case tlm::TLM_ACCEPTED:
                sc_core::wait(resp_ack_event);
                break;
              case tlm::TLM_COMPLETED:;
            }

        }

        // We must be able to clear ThreadBusy now as we just sent a
        // response (or cleared a write latency)
        if ( m_sthreadbusy && (tb->value==1) &&
                (m_ResponseQueue.length() < m_limitreq_max) ) {
            // Our queue has been shortened. Clear threadBusy.
            tb->value = 0;
            //tpP->putSThreadBusy(m_curSThreadBusy);
            time=sc_core::SC_ZERO_TIME;
            tpP->nb_transport_bw(*tb_txn, tb_ph, time);
        }

        //if (sc_core::sc_time_stamp()==send_time)
        sc_core::wait(); //advance to next cycle if there was an immediate accept

    }
}

template <unsigned int BUSWIDTH>
tlm::tlm_sync_enum Slave<BUSWIDTH>::nb_transport(tlm::tlm_generic_payload& gp, tlm::tlm_phase& ph, sc_core::sc_time& t){
  switch (ph){
    case tlm::BEGIN_REQ:
      m_req_txn=&gp;
      req_event.notify();
      if (!m_cmdaccept){ //there is no 'real' accept signal within the OCP config, so we have to return UPDATED + END
        if (m_sthreadbusy_exact) assert(tb->value==0); //we should not be busy
        if (gp.get_command()==tlm::TLM_WRITE_COMMAND && !m_writeresp_enable) {
          gp.acquire(); //the master might release after end_req so we acquire
            if (m_debug_os_ptr) {
                (*m_debug_os_ptr) << "DB (" << name() << "): "
                    << "acquire." <<&gp<< std::endl;
            }
        }
        ph=tlm::END_REQ;
        return tlm::TLM_UPDATED;
      }
      break;
    case tlm::END_RESP:
      resp_ack_event.notify();
      break;
    default:
      if (ph==ocpip::RESP_THREAD_BUSY_CHANGE){
        m_mtb=tpP.template get_extension<ocpip::resp_thread_busy>(gp)->value;
      }
      else {
        std::cerr<<"I only expect BEGIN_REQ or END_RESP on the forward path, but got "<<ph<<" In "<<name()<<" at "<<sc_core::sc_time_stamp()<<std::endl;
        exit(1);
      }
  }
  return tlm::TLM_ACCEPTED;
}

// ---------------------------------------------------
// explicit instantiation of the Slave template class
// ---------------------------------------------------
template class Slave< 32 >;
