// 
//  (c) Copyright OCP-IP 2005
//  OCP-IP Confidential and Proprietary
//
// ============================================================================
//      Project : OCP SLD WG
//       Author : Tim Kogel, CoWare, Inc
//          $Id:
//
//  Description : Defintion of datatypes and policies for the 
//		  chronologically ordered queue defined in ChronoQueue.h
//
// ============================================================================

#ifndef CHRONO_QUEUE_DATA_TYPES_H
#define CHRONO_QUEUE_DATA_TYPES_H

#include "systemc.h"

template< typename T > class ChronoQueueElement;
template< typename T > class MemoryPolicyNew;
template< typename T > class MemoryPolicyPool;
template< typename T > class QueueingPolicyFifo;
template< typename T > class QueueingPolicyOutOfOrder;
class NotifyPolicyNone;
class NotifyPolicySCEvent;

//------------------------------------------------------------------------
/** \brief   This class is the basic element of a  chronologically
    ordered queue.

    Each object contains an item, an sc_time type of tag and a next
    pointer to create single linked lists
*/
template< typename T >
class ChronoQueueElement {
public:
  // Constructor
  ChronoQueueElement():
    m_SendTime(SC_ZERO_TIME),
    m_Next(NULL)
  {};

  typedef ChronoQueueElement<T> this_type;

  /** due tag, i.e. the absolute time stamp at which this element 
      be removed from the queue */
  sc_time m_SendTime;
  /** the item */
  T m_Item;
  /** next pointer to create single linked lists */
  this_type* m_Next;

private:
  // Disable copy constructors
  ChronoQueueElement( const ChronoQueueElement & );          
  ChronoQueueElement & operator= ( const ChronoQueueElement & );
};

//------------------------------------------------------------------------
/** \short This class defines a simple memory policy, which simply
    maps to the new and delete operators 
*/
template< typename T >
class MemoryPolicyNew {
public:
  /** \return generated element */
  ChronoQueueElement<T>* create() { return new ChronoQueueElement<T>(); }
  /** \param to be deleted element */
  void destroy(ChronoQueueElement<T>* elem) { delete elem; }
protected:
  /** \param not used in this policy */
  MemoryPolicyNew(unsigned int size) {};
};

//------------------------------------------------------------------------
/** \short This class defines a more elaborated memory policy, which
    manages a pool of objects. In case the initial numner of elements is not
    sufficient, new elements are created on the fly.
*/
template< typename T >
class MemoryPolicyPool {
public:
  /** \return element taken from the pool */
  ChronoQueueElement<T>* create();
  /** \param element to be re-inserted into the pool */
  void destroy(ChronoQueueElement<T>* elem);
protected:
  MemoryPolicyPool(unsigned int size);
  ~MemoryPolicyPool();
private:
  /** initial size of the pool */
  const unsigned int m_SizeOfPool;
  /** actual number of elements in the pool */
  unsigned int m_NumFree;
  /** counts how many elements had to be created on the fly, only for
      statistical purposes */
  unsigned int m_CountExcess;

  ChronoQueueElement<T>*  m_HeadOfPool;
  ChronoQueueElement<T>*  m_TailOfPool;  
};

//------------------------------------------------------------------------
/** \short This class defines a simple FIFO based queueing policy.
 */
template< typename T >
class QueueingPolicyFifo {
  friend class QueueingPolicyOutOfOrder<T>;
public:
  /** \return number of elements pending in the FIFO queue */
  unsigned int numPending() const { return m_NumEntries; }
protected:
  /** \param element appended at the end of the queue
      \return true, if head-of-line tag has changed */
  virtual bool put(ChronoQueueElement<T>* elem);
  /** non-destructive read
      \return first element of the queue */
  ChronoQueueElement<T>* peek() const;
  /** destructive read
      \return first element of the queue */
  ChronoQueueElement<T>* get();

  QueueingPolicyFifo();
  virtual ~QueueingPolicyFifo();
private:
  unsigned int m_NumEntries;
  ChronoQueueElement<T>*  m_HeadOfQueue;
  ChronoQueueElement<T>*  m_TailOfQueue;
};

//------------------------------------------------------------------------
/** \short This class defines a queueing policy, where all elements
    are strictly ordered according to their respective due time
 */
template< typename T >
class QueueingPolicyOutOfOrder: public QueueingPolicyFifo<T> {
public:
protected:
  /** \param element chronologically inserted into the queue
      \return true, if head-of-line tag has changed */
  virtual bool put(ChronoQueueElement<T>* elem);
  QueueingPolicyOutOfOrder():QueueingPolicyFifo<T>() {}; 
};


//------------------------------------------------------------------------
/** \short This class defines an empty notification policy
 */
class NotifyPolicyNone {
protected:
  void cancel() {};
  void notify() {};
  void notify( const sc_time& t ) {};

  NotifyPolicyNone() {};
  ~NotifyPolicyNone() {};
private:
};


//------------------------------------------------------------------------
/** \short This class defines an sc_event based notification
    policy. It's essentially a wrapper around sc_event
 */
class NotifyPolicySCEvent {
public:
  const sc_event& getEvent() const { return m_event; }
protected:
  inline void cancel() { m_event.cancel(); }
  inline void notify() { m_event.notify(); }
  inline void notify( const sc_time& t ) { m_event.notify(t); }
  
  NotifyPolicySCEvent() {};
  ~NotifyPolicySCEvent() {};
private:
  sc_event m_event;
};

//------------------------------------------------------------------------
// inline implementations

template< typename T >
MemoryPolicyPool<T>::MemoryPolicyPool(unsigned int size) 
  : m_SizeOfPool(size)
  , m_NumFree(size)
  , m_CountExcess(0)
  , m_HeadOfPool(NULL)
  , m_TailOfPool(NULL)
{
  // we pre-construct all objects to improve run-time performance
  m_HeadOfPool = new ChronoQueueElement<T>;
  ChronoQueueElement<T>* currentEntry = m_HeadOfPool;
  for (unsigned int i = 1; i < size; i++) {
    currentEntry->m_Next = new ChronoQueueElement<T>;
    currentEntry  = currentEntry->m_Next;
  }
  m_TailOfPool = currentEntry;
}

template< typename T >
MemoryPolicyPool<T>::~MemoryPolicyPool() 
{
  // removes all entries in the free list
  unsigned int countDeleted=0;
  ChronoQueueElement<T>* currentEntry;
  ChronoQueueElement<T>* toBeDeleted = m_HeadOfPool;
  while (toBeDeleted) {
    currentEntry = toBeDeleted->m_Next;
    delete toBeDeleted;
    toBeDeleted = currentEntry;
    countDeleted++;
  }
#ifdef DEBUG_ChronoQueue
  cout << "MemoryPolicyPool statistics: \n";
  cout << "original size : " << m_SizeOfPool << endl;
  cout << "excess  : " << m_CountExcess << endl;
  cout << "free  : " << m_NumFree << endl;
  cout << "deleted  : " << countDeleted << endl;
#endif
}

template< typename T >
inline
ChronoQueueElement<T>* 
MemoryPolicyPool<T>::create()
{
  if (m_NumFree) {
    m_NumFree--;
#ifdef DEBUG_ChronoQueue
    cout << "getNew() : updated m_NumFree=" << m_NumFree << endl;
#endif
    ChronoQueueElement<T>* newItem = m_HeadOfPool;
    m_HeadOfPool = m_HeadOfPool->m_Next;
    newItem->m_Next = NULL; // just for savety
    return newItem;
  } else {
    // pool empty, create additional element
    m_CountExcess++;
    return new ChronoQueueElement<T>();
  }
}

template< typename T >
inline
void
MemoryPolicyPool<T>::destroy(ChronoQueueElement<T>* elem)
{
  // re-insert element into the pool
  if (!m_HeadOfPool) {
    m_HeadOfPool = elem;
  } else {
    m_TailOfPool->m_Next = elem;
  }
  elem->m_Next = NULL;
  m_TailOfPool = elem;
  m_NumFree++;
#ifdef DEBUG_ChronoQueue
  cout << "free: updated m_NumFree=" << m_NumFree << endl;
#endif
}

template< typename T >
QueueingPolicyFifo<T>::QueueingPolicyFifo()
  : m_NumEntries(0)
  , m_HeadOfQueue(NULL)
  , m_TailOfQueue(NULL)
{
}

template< typename T >
QueueingPolicyFifo<T>::~QueueingPolicyFifo()
{
  // delete all pending elements
  unsigned int countDeleted=0;
  ChronoQueueElement<T>* currentEntry;
  ChronoQueueElement<T>* toBeDeleted = m_HeadOfQueue;
  while (toBeDeleted) {
    currentEntry = toBeDeleted->m_Next;
    delete toBeDeleted;
    toBeDeleted = currentEntry;
    countDeleted++;
  }
#ifdef DEBUG_ChronoQueue
  cout << "QueueingPolicyFifo statistics: \n";
  cout << "deleted  : " << countDeleted << endl;
#endif
}

template< typename T >
inline
bool
QueueingPolicyFifo<T>::put(ChronoQueueElement<T>* elem)
{
  bool head_of_line_tag_changed = false;
  if (!m_NumEntries) {
    // empty queue
    m_HeadOfQueue = elem;
    m_TailOfQueue = elem;
    head_of_line_tag_changed = true;
  } else {
#ifdef DEBUG_ChronoQueue
    if (elem->m_SendTime < m_TailOfQueue->m_SendTime) {
      cerr << "\n\nERROR, elem->m_SendTime " << elem->m_SendTime << " is smaller"
	   << " than m_TailOfQueue->m_SendTime " << m_TailOfQueue->m_SendTime 
	   << ", use QueueingPolicyOutOfOrder\n";
    }
#endif
    // append new element at the end
    m_TailOfQueue->m_Next = elem;
    m_TailOfQueue = elem;
  }
  elem->m_Next = NULL;
  m_NumEntries++;
#ifdef DEBUG_ChronoQueue
  cout << "QueueingPolicyFifo put: updated m_NumEntries=" << m_NumEntries << endl;
#endif

  // return true, if head-of-line tag has changed
  return head_of_line_tag_changed;
}

template< typename T >
inline
ChronoQueueElement<T>* 
QueueingPolicyFifo<T>::peek() const
{
#ifdef DEBUG_ChronoQueue
  if (!m_NumEntries) {
    cout << "WARNING : QueueingPolicyFifo<T>::peek(), peek on empty queue\n";
  }
#endif
  return m_HeadOfQueue;
}

template< typename T >
inline
ChronoQueueElement<T>* 
QueueingPolicyFifo<T>::get()
{
  if (!m_NumEntries) {
#ifdef DEBUG_ChronoQueue
    cout << "WARNING : QueueingPolicyFifo<T>::get(), get on empty queue\n";
#endif
    return NULL;
  }

  ChronoQueueElement<T>* toBeReturned = m_HeadOfQueue;
  m_HeadOfQueue = m_HeadOfQueue->m_Next;
  toBeReturned->m_Next =  NULL;
  m_NumEntries--;
#ifdef DEBUG_ChronoQueue
  cout << "QueueingPolicyFifo get: updated m_NumEntries=" << m_NumEntries << endl;
#endif
  return toBeReturned;
}

template< typename T >
inline
bool
QueueingPolicyOutOfOrder<T>::put(ChronoQueueElement<T>* elem)
{
  // insert in chronological order
  bool head_of_line_tag_changed = false;
  if (!m_NumEntries) {
    // empty queue
    m_HeadOfQueue = elem;
    m_TailOfQueue = elem;
    elem->m_Next = NULL;
    head_of_line_tag_changed = true;
  } else if (m_TailOfQueue->m_SendTime <= elem->m_SendTime){
    // later than last element -> append new element at the end
    m_TailOfQueue->m_Next = elem;
    m_TailOfQueue = elem;
    elem->m_Next = NULL;
  }
  else if (m_HeadOfQueue->m_SendTime > elem->m_SendTime ){
    // earlier than first element -> prepend new element at the head
    elem->m_Next = m_HeadOfQueue;
    m_HeadOfQueue = elem;
    head_of_line_tag_changed = true;
  } else {
    // iterate to insert position
    ChronoQueueElement<T>* it = m_HeadOfQueue;
    while (it->m_Next->m_SendTime <= elem->m_SendTime) {
      it = it->m_Next;
    }
    // insert
    elem->m_Next = it->m_Next;
    it->m_Next = elem;
  }
  m_NumEntries++;
#ifdef DEBUG_ChronoQueue
  cout << "QueueingPolicyOutOfOrder, put: updated m_NumEntries=" << m_NumEntries << endl;
#endif
  return head_of_line_tag_changed;
}


#endif /* CHRONO_QUEUE_DATA_TYPES_H */
