// 
//  (c) Copyright OCP-IP 2005
//  OCP-IP Confidential and Proprietary
//
// ============================================================================
//      Project : OCP SLD WG
//       Author : Tim Kogel, CoWare, Inc
//          $Id:
//
//  Description : chronologically ordered queue
//
// ============================================================================

#ifndef CHRONO_QUEUE_H
#define CHRONO_QUEUE_H

//#define DEBUG_ChronoQueue
#include "ChronoQueueDataTypes.h"

/** \short chronologically ordered queue

    \param T used defined data type of the items managed by this queue
    
    \param MemoryPolicy defines the memory management policy for list
    elements.

    \param QueueingPolicy defines the way pending elements are
    stored. The QueueingPolicy is responsible to ensure the
    chronological order.

    \param NotifyPolicy defines what should happen when the head of
    line element in the pending queue becomes due

    This is a pretty generic class for modeling delayed delivery of
    data. The class \ref ChronoQueueElement serves as a basic list
    element. Apart from the data item itself ChronoQueueElement holds
    the due time stamp of the data item. 

    Intended usage:
    The user should use the create() method defined by the MemoryPolicy
    to get hold of a valid ChronoQueueElement, fill in his data item
    and due tag and then use enqueue to insert the element into the
    pending queue. 

    The NotifyPolicy ChronoQueue will ring as soon as the due
    tag of the head-of-line element is reached. The user can always
    use peekNextItem and peekNextSendTime to access the head-of-line
    data and due tag respectively in a non-destructive way. 

    Eventually (i.e. when the processing of the head-of-line item is
    finished) the user should call the destructive dequeue method to
    remove the head-of-line element from the pending queue. In case
    futher elements are pending, the NotifyPolicy is automaticcally
    armed for the next due tag. Note, that dequeue might return NULL
    when it's called on an empty queue.

    The user itself is responsible to hand back unused elements to the
    MemoryPolicy by using the destroy method.

    Convenience functions on member pointer of newly created element:
    createNew(), getNewItem(), getNewSendTime() and enqueueNew()
    together with dequeueAndDestroy() hide the internal queue element
    data structure. 
*/
template
<
  typename T,
  template <typename> class MemoryPolicy,
  template <typename> class QueueingPolicy,
		      class NotifyPolicy = NotifyPolicyNone
>
class ChronoQueue
  : public MemoryPolicy<T>
  , public QueueingPolicy<T>
  , public NotifyPolicy
{
public:
  typedef MemoryPolicy<T>	  MP;
  typedef QueueingPolicy<T>	  QP;
  typedef NotifyPolicy		  NP;
  typedef ChronoQueueElement<T>   elem_type;

  /** \param size constructor argument for the MemoryPolicy */
  ChronoQueue(int size=0) :
    MP(size),
    QP(),
    NP(),
    m_newElement(NULL),
    m_zeroTime(SC_ZERO_TIME),
    m_zeroItem()
  {}

    // convenience functions operating on member pointer of newly created element
  /** call the create method of the memory policy and assigns the
      result to the internal pointer memeber */
  void createNew()  { m_newElement = MP::create(); }
  /** \return the item of the newly created queue element. The user
      is allowed to modify the item */
  T& getNewItem();
  /** \return the send-time of the newly created queue element. The user
      is allowed to modify the send-time */
  sc_time& getNewSendTime();
  
  /** enqueues the newly created element into the queue and
      invalidates the internal pointer memeber
      \return true if the head-of-line element has changed */
  const bool enqueueNew();
  
  /** \return true if the tag of the head-of-line element is smaller
      than the current simulation time */
  const bool nextItemDue() const;
  /** non-destructive read of the head-of-line item in the pending
      queue */
  const T& peekNextItem() const;
  /** non-destructive read of the head-of-line time tag in the pending
      queue */
  const sc_time& peekNextSendTime() const;
  /** inserts an element into the pending queue
      \return true if the head-of-line time tag has changed 
  */
  const bool enqueue(elem_type* elem);
  /** destructive read of the head-of-line element in the pending
      queue. 
      \return true if the head-of-line element has changed
  */
  ChronoQueueElement<T>* dequeue();
  
  /** combines dequeue method on destroy method from memory policy */
  void dequeueAndDestroy();
private:
  // newly created/temporary element
  ChronoQueueElement<T>* m_newElement;
  ChronoQueueElement<T>* m_tmpElement; 

  // dummy member to serve as empty return references
  sc_time m_zeroTime;
  T m_zeroItem;
  const sc_time m_constZeroTime;
  const T m_constZeroItem;

  // Disable
  ChronoQueue( const ChronoQueue & );          
  ChronoQueue & operator= ( const ChronoQueue & );
};

template
<
  typename T,
  template <typename> class MP,
  template <typename> class QP,
		      class NP
>
inline 
T&
ChronoQueue<T,MP,QP,NP>::getNewItem()
{
  if (m_newElement) {
    return m_newElement->m_Item;
  } else {
    cerr << "ChronoQueue::getNewItem(), no new element, call createNew first\n";
  }
  return m_zeroItem;
}

template
<
  typename T,
  template <typename> class MP,
  template <typename> class QP,
		      class NP
>
inline 
sc_time&
ChronoQueue<T,MP,QP,NP>::getNewSendTime()
{
  if (m_newElement) {
    return m_newElement->m_SendTime;
  } else {
    cerr << "ChronoQueue::getNewSendTime, no new element, call createNew first\n";
  }
  return m_zeroTime;
}

template
<
  typename T,
  template <typename> class MP,
  template <typename> class QP,
		      class NP
>
inline 
const bool
ChronoQueue<T,MP,QP,NP>::enqueueNew()
{
  if (!m_newElement) {
    cerr << "ChronoQueue::enqueueNew(), no new element, call createNew first\n";
    return false;
  } 
  m_tmpElement = m_newElement;
  m_newElement = NULL;
  return enqueue(m_tmpElement);
}

template
<
  typename T,
  template <typename> class MP,
  template <typename> class QP,
		      class NP
>
inline 
void
ChronoQueue<T,MP,QP,NP>::dequeueAndDestroy()
{
  m_tmpElement = dequeue();
  destroy(m_tmpElement);
}


template
<
  typename T,
  template <typename> class MP,
  template <typename> class QP,
		      class NP
>
inline 
const bool
ChronoQueue<T,MP,QP,NP>::nextItemDue() const
{
  if (!QP::numPending()) {
    return false;
  }
  return (QP::peek()->m_SendTime <= sc_time_stamp());
}

template
<
  typename T,
  template <typename> class MP,
  template <typename> class QP,
		      class NP
>
inline 
const T&
ChronoQueue<T,MP,QP,NP>::peekNextItem() const
{
  if (!QP::numPending()) {
    cerr << "ChronoQueue::peekNextItem on empty queue\n";
    return m_constZeroItem;
  }
  return QP::peek()->m_Item;
}

template
<
  typename T,
  template <typename> class MP,
  template <typename> class QP,
		      class NP
>
inline 
const sc_time&
ChronoQueue<T,MP,QP,NP>::peekNextSendTime() const
{
  if (!QP::numPending()) {
    cerr << "ChronoQueue::peekNextSendTime on empty queue\n";
    return m_constZeroTime;
  }
  return QP::peek()->m_SendTime;
}

template
<
  typename T,
  template <typename> class MP,
  template <typename> class QP,
		      class NP
>
inline 
const bool
ChronoQueue<T,MP,QP,NP>::enqueue(elem_type* elem)
{
  assert( elem->m_SendTime >= sc_time_stamp() );
  const bool hol_tag_changed = QP::put(elem);
  if ( hol_tag_changed ) {
    // head of line due date has changed -> notify event
    NP::cancel();
    NP::notify(QP::peek()->m_SendTime - sc_time_stamp());
  }
  return hol_tag_changed;
}

template
<
  typename T,
  template <typename> class MP,
  template <typename> class QP,
		      class NP
>
inline 
ChronoQueueElement<T>*
ChronoQueue<T,MP,QP,NP>::dequeue()
{
  ChronoQueueElement<T>* elem = QP::get();

  if (QP::numPending()) {
    // if there are pending responses left in the queue, notify the internal
    // event at the start time of the next response
    // (or immediately, if the start time of next reponse has already passed)
      if (QP::peek()->m_SendTime <= sc_time_stamp()) {
      // next item is already due
      NP::notify();
    } else {
      NP::notify(QP::peek()->m_SendTime - sc_time_stamp());
    }
  }
  return elem;
}




#endif /* CHRONO_QUEUE_H */
