// 
//  Copyright 2004 OCP-IP
//
// ============================================================================
//      Project : OCP SLD WG
//       Author : Anssi Haverinen, Nokia Inc.
//         $Id: main.cc,v 1.4 2006/02/03 18:02:44 tkogel Exp $
//
//  Description : OCP API - TL-1 Constructor examples
//
// ============================================================================

// Turn off OCP protocol checker, if installed
#define NDEBUG

// Include SystemC
#include "systemc.h"

// Include OCP files
#include "ocp_tl1_data_cl.h"
#include "ocp_tl_param_cl.h"
#include "ocp_tl1_channel.h"
#include "ocp_tl1_channel_clocked.h"
#include "ocp_tl1_trace_monitor.h"

// Include user-defined headers
#include "channel_types.h"
#include "command_line.h"

// Include submodules
#include "master.h"
#include "slave.h"
#include "master_nb.h"
#include "slave_nb.h"

// Create a toggling signal 
class my_clk : public sc_module
{
public:  
  
  sc_out<bool>  clk;
  SC_HAS_PROCESS(my_clk);

  // constructor
  my_clk(sc_module_name name_, double p=1, sc_time_unit t=SC_NS)  {
    m_p = p;
    m_t = t;
    SC_THREAD(c);
  }

  void c() {
    bool clk_t = 1;
    while(1) {
      clk = clk_t;
      wait(m_p/2,m_t);
      clk_t = !clk_t;
    }
  }
private:
  double m_p; 
  sc_time_unit m_t;

};

// Instantiates a few different channels
class top : public sc_module
{
public:  
  typedef OCP_TL1_DataCl<OCPCHANNELBit32, OCPCHANNELBit32> data_type;
  typedef OCP_TL1_Channel< data_type > channel_type;
  typedef OCP_TL1_Channel_Clocked< data_type > clocked_channel_type;
  typedef OCP_TL1_Trace_Monitor< data_type > trace_monitor_type;
  
  sc_in_clk  clk;

  // constructor
  top(sc_module_name name_, 
      MapStringType  ocpParamMap,
      bool use_event,
      bool use_default_event,
      std::string monFileName
      ) 
  {
    // default constructor
    ch2 = new channel_type ("ocp2");

    // full constructor
    ch3 = new channel_type ("ocp3",use_event,use_default_event);

    // clocked channel with clock constructor argument
    ch4 = new channel_type("ocp4", (sc_in_clk *) &clk);

    // clocked channel with input port as clock
    ch5 = new clocked_channel_type("ocp5");
    ch5->p_clk(clk);

    // instantiate trace monitors 
    mon2 = new trace_monitor_type("mon2");
    mon3 = new trace_monitor_type("mon3");
    mon4 = new trace_monitor_type("mon4");
    mon5 = new trace_monitor_type("mon5");
    mon2->p_clk(clk);
    mon3->p_clk(clk);
    mon4->p_clk(clk);
    mon5->p_clk(clk);
    mon2->p_mon(*ch2);
    mon3->p_mon(*ch3);
    mon4->p_mon(*ch4);
    mon5->p_mon(*ch5);

    sl2 = new SlaveNb<data_type>("sl2");
    ms2 = new MasterNb<data_type>("ms2");
    sl3 = new SlaveNb<data_type>("sl3");
    ms3 = new MasterNb<data_type>("ms3");
    sl4 = new Slave<data_type>("sl4");
    ms4 = new Master<data_type>("ms4");
    sl5 = new Slave<data_type>("sl5");
    ms5 = new Master<data_type>("ms5");

    ch2->setConfiguration(ocpParamMap);
    ch3->setConfiguration(ocpParamMap);
    ch4->setConfiguration(ocpParamMap);
    ch5->setConfiguration(ocpParamMap);

    ms2->ipP(*ch2);
    ms2->clk(clk);
    sl2->tpP(*ch2);
    sl2->clk(clk);
    ms3->ipP(*ch3);
    ms3->clk(clk);
    sl3->tpP(*ch3);
    sl3->clk(clk);
    ms4->ipP(*ch4);
    sl4->tpP(*ch4);
    ms5->ipP(*ch5);
    sl5->tpP(*ch5);
  };

  channel_type *ch2;
  channel_type *ch3;
  channel_type *ch4;
  clocked_channel_type *ch5;

  trace_monitor_type *mon2;
  trace_monitor_type *mon3;
  trace_monitor_type *mon4;
  trace_monitor_type *mon5;

  SlaveNb<data_type> *sl2;
  MasterNb<data_type> *ms2;
  SlaveNb<data_type> *sl3;
  MasterNb<data_type> *ms3;
  Slave<data_type> *sl4;
  Master<data_type> *ms4;
  Slave<data_type> *sl5;
  Master<data_type> *ms5;
};

int sc_main(int argc, char* argv[]) {

  typedef top::data_type data_type;
  typedef top::channel_type channel_type;
  typedef top::trace_monitor_type trace_monitor_type;

  // OCP parameters
  MapStringType  ocpParamMap;
  string ocpParamFileName;

  // Command line parameters
  bool use_event;
  bool use_default_event;

  std::string monFileName0;
  std::string monFileName1;
  std::string monFileName2;

  sc_clock clk("clk", 1, SC_NS);

  // Read command line parameters
  process_command_line(argc,argv,
		       ocpParamFileName,
		       use_event,
		       use_default_event,
		       monFileName0,
		       monFileName1,
		       monFileName2);

  // Read OCP parameters from file
  if ( ! ocpParamFileName.empty() ) {
    readMapFromFile(ocpParamFileName, 
		    ocpParamMap
		    );
  }


  sc_signal<bool> clk1;

  // clocked channel with clock generator as clock
  channel_type ch0("ocp0",(sc_clock *) &clk);
  trace_monitor_type mon0("mon0",monFileName0);
  mon0.p_clk(clk1);
  mon0.p_mon(ch0);

  ch0.setConfiguration(ocpParamMap);

  // clocked channel with signal as clock
  channel_type ch1("ocp1",(sc_signal<bool> *) &clk1);
  trace_monitor_type mon1("mon0",monFileName1);
  mon1.p_clk(clk1);
  mon1.p_mon(ch0);

  ch1.setConfiguration(ocpParamMap);

  Slave<data_type> sl0("sl0");
  Master<data_type> ms0("ms0");
  Slave<data_type> sl1("sl1");
  Master<data_type> ms1("ms1");

  my_clk clk2("clk2", 1, SC_NS);

  ms0.ipP(ch0);
  sl0.tpP(ch0);
  ms1.ipP(ch1);
  sl1.tpP(ch1);
  clk2.clk(clk1);

  sc_clock *clk3;
  clk3 = &clk;

  // Add some hierarchy
  top t("top", 
	ocpParamMap,
	use_event,
	use_default_event,
	monFileName2
	); 

  t.clk(clk);

  // Simulator
  sc_start(50, SC_NS);
  
  return(0);
}

