// 
//  (c) Copyright OCP-IP 2005
//  OCP-IP Confidential and Proprietary
//
// ============================================================================
//      Project : OCP SLD WG
//       Author : Tim Kogel, CoWare, Inc
//          $Id:
//
//  Description :OSCI TLM based implementation of the TL3 master interface
//
// ============================================================================

#ifndef _OCP_TL3_MASTER_PROTOCOL_H
#define _OCP_TL3_MASTER_PROTOCOL_H


#include "tlm.h"

#include "ChronoQueue.h"

#include "ocp_tl3_master_if.h"
#include "ocp_tl3_monitor_if.h"

template 
<
  typename REQ,
  typename RESP
>
class OCP_TL3_Master_Protocol :
  public sc_module,
  public OCP_TL3_MasterIF<REQ,RESP>
{
  typedef ChronoQueue<REQ,
		      MemoryPolicyPool,
		      QueueingPolicyOutOfOrder, 
		      NotifyPolicySCEvent> request_queue_type;

  SC_HAS_PROCESS(OCP_TL3_Master_Protocol);

public:
  typedef tlm::tlm_nonblocking_put_if<REQ> put_interface_type ; 
  typedef tlm::tlm_nonblocking_get_peek_if<RESP> get_interface_type ; 
  typedef sc_port<put_interface_type> put_port_type;
  typedef sc_port<get_interface_type> get_port_type;

  typedef OCP_TL3_MonitorIF<REQ,RESP>			      monitor_if_type;
  typedef typename OCP_TL3_MonitorIF<REQ,RESP>::observer_type observer_type;

  put_port_type m_requestPort;
  get_port_type m_responsePort;

  //---------------------------------------------------------------
  // constructor
  //---------------------------------------------------------------
  OCP_TL3_Master_Protocol(sc_module_name name, monitor_if_type* channel) :
    sc_module(name),
    m_requestPort("requestP"), 
    m_responsePort("responseP"),
    m_responseUnread(true),
    m_request_queue(10),
    m_clkPeriod(1,SC_NS),
    m_channel(channel)
  { 
    SC_METHOD(accept_method);
    sensitive << m_acceptEvent;
    dont_initialize();

    SC_METHOD(request_method);
    sensitive << m_request_queue.getEvent();
    dont_initialize();
    
    SC_METHOD(notify_request_start_observer);
    sensitive << m_requestStartObserver;
    dont_initialize();
  }

  //---------------------------------------------------------------
  // destructor
  //---------------------------------------------------------------
  virtual ~OCP_TL3_Master_Protocol()
  { 
  }

  void end_of_elaboration() 
  {
    SC_METHOD(end_of_request_method);
    sensitive << RequestEndEvent();
    dont_initialize();  
  }

public:

  //////////////////////////////////////////////////////////////
  // Implement OCP TL3 Master Interface Methods
  //////////////////////////////////////////////////////////////

  //////////////////////////////////////////////////////////////
  // OCP TL3 master request methods

  bool sendRequest(const REQ& req)
  { return sendRequest(req,SC_ZERO_TIME); }

  bool sendRequest(const REQ& req, const sc_time& time)
  {
    m_request_queue.createNew();
    m_request_queue.getNewItem() = req;
    m_request_queue.getNewSendTime() = time + sc_time_stamp();
    m_request_queue.enqueueNew();
    return true;
  }

  bool sendRequest(const REQ& req, const int cycles)
  { return sendRequest(req, cycles * m_clkPeriod); }
  
  bool sendRequestBlocking(const REQ& req)
  {  
    if (requestInProgress()) {
      wait(RequestEndEvent());
      if (!sendRequest(req))
	{
	  return false;
	}
    }
    wait(RequestEndEvent());
    return true;
  }
  
  bool requestInProgress(void)  const 
  { return !m_requestPort->nb_can_put(); }

  const sc_event& RequestStartEvent(void) const  
  { return m_requestStartEvent; }
  const sc_event& RequestEndEvent(void) const    
  { return m_requestPort->ok_to_put(); }


  //////////////////////////////////////////////////////////////
  // OCP TL3 master response methods

  bool getResponse(RESP& resp)
  { 
    // Check to see if there is already a transaction in progress
    // that we haven't read yet
    if (m_responseUnread && m_responsePort->nb_peek(resp))
      {
	// There is a transaction to get
	// Mark it as read
	m_responseUnread = false;
	return true;
      }
    return false;
  }

  bool getResponseBlocking(RESP& resp)
  { 
    if (!getResponse(resp)) {
      wait(m_responsePort->ok_to_peek());
      return getResponse(resp);
    }
    return true;
  }

  bool acceptResponse()
  { return acceptResponse(SC_ZERO_TIME); }

  bool acceptResponse(const sc_time& accept_time)
  {
    if ( responseInProgress() ) {
      m_acceptEvent.notify(accept_time);
      return true;
    }
    return false;
  }

  bool acceptResponse(const int cycles)	     
  { 
    if ( responseInProgress()) {
      if (cycles <= 0) {
	// Accept Right now
	return acceptResponse(SC_ZERO_TIME);
      } else {
	// wait for "cycles" clock periods and accept then
	return acceptResponse(cycles * m_clkPeriod);
      }
    }
    return false;
  }


  bool responseInProgress(void) const 
  { return m_responsePort->nb_can_peek(); }
  const sc_event& ResponseStartEvent(void) const 
  { return  m_responsePort->ok_to_peek(); }
  const sc_event& ResponseEndEvent(void) const   
  { return m_responseEndEvent; }



  //////////////////////////////////////////////////////////////
  // clock  methods

  void setPeriod( const sc_time& newPeriod )
  {
    m_clkPeriod = newPeriod;
  }

  const sc_time& getPeriod(void) const 
  {
    return m_clkPeriod;
  }

  void RegisterRequestStart(observer_type* monitor)
  { m_request_start_observer.push_back(monitor); }
  void RegisterResponseEnd(observer_type* monitor)
  { m_response_end_observer.push_back(monitor); }

private:
  void request_method()
  {
#ifndef NDEBUG
    if ( !m_request_queue.nextItemDue() ) {
      cerr << name() << ":\n\nERROR request_method! next send time " 
	   <<  m_request_queue.peekNextSendTime() << " not yet due" << endl;
      return;
    }
    
    if ( requestInProgress() ) {
      // this should never happen! the activation scheme is designed
      // so that when this method notified, the response channel should always be free.
      cerr << name() << ":\n\nERROR request_method! requestInProgress should never be true!";
      return;
    }
    //  cout << sc_time_stamp() << ", " << name() << ": send request\n";
#endif
    
    if ( m_requestPort->nb_can_put() )
      {
	// Put the transaction on the channel
	m_requestPort->nb_put(m_request_queue.peekNextItem());

	// Trigger the event
	m_requestStartEvent.notify();
	m_requestStartObserver.notify(SC_ZERO_TIME);
      } else {
	// Already a transaction in progress. Failed.
	cerr << name() << ":\n\nERROR request_method! put request failed" << endl;
      }
  }

  void end_of_request_method()
  {
    //    cout << sc_time_stamp() << ", " << name() << ": request ends\n";
    m_request_queue.dequeueAndDestroy();
  }
  
  void accept_method(void)
  {
    if ( responseInProgress() ) {
      m_responseUnread = true;
      m_responseEndEvent.notify();
      m_responsePort->nb_get(m_resp);
      notify_response_end_observer();
    } else {
      cerr << name() << ":\n\nERROR accept_method! no responseInProgress" << endl;
    }

  }

  void notify_request_start_observer()
  {
    for( observer_iterator_type
	   it  = m_request_start_observer.begin(),
	   end = m_request_start_observer.end();
	 it != end;
	 ++it) {
      (*it)->NotifyRequestStart( m_channel);
    }
  }
  void notify_response_end_observer()
  {
    for( observer_iterator_type
	   it  = m_response_end_observer.begin(),
	   end = m_response_end_observer.end();
	 it != end;
	 ++it) {
      (*it)->NotifyResponseEnd( m_channel);
    }
  }

  //----------------------------------------------------------------------
  // member data
  //----------------------------------------------------------------------
  bool m_responseUnread;
  sc_event m_requestStartEvent;
  sc_event m_responseEndEvent;
  RESP m_resp;
  request_queue_type m_request_queue;
  sc_time m_clkPeriod;
  sc_event m_acceptEvent;
  sc_event m_requestStartObserver;

  typedef ::std::vector< observer_type* > observer_container_type;
  typedef typename observer_container_type::iterator observer_iterator_type;
  observer_container_type m_request_start_observer;
  observer_container_type m_response_end_observer;
  monitor_if_type* m_channel;
};

#endif  // _OCP_TL3_MASTER_PROTOCOL_H
