// 
//  Copyright 2003 OCP-IP
//  OCP-IP Confidential & Proprietary
//
// ============================================================================
//      Project : OCP SLD WG
//      Authors : Yann Bajot, PROSILOG
//                Alan Kamas, Sonics Inc.
//         $Id: SlaveSysC.cpp,v 1.2 2005/07/09 04:09:16 Anssi Exp $
//
//  Description:  TL2 Slave
//
//                This TL2 slave emulates a '3 threads' OCP slave. It uses two
//                SystemC threads, one for requests and one for responses.  The
//                request SC_THREAD catches every request, computes the
//                response and stores it in one of the three response queues,
//                depending on the ThreadID of the request. Then, the response
//                SC_THREAD issues responses to the master.  
//
//  Parameters:
//                The slave accepts the following parameters:
//                  * latencyX
//                  * limitreq_enable
//                  * limitreq_max 
//
//                These parameters are described in section 6.1.3 of
//                the OCP API documentation. Note that for TL2, delays are not
//                expressed in terms of clock cycles but as absolute timings
//                (unit is SC_NS in the slave).
//
// ============================================================================


#include <stdio.h>
#include "SlaveSysC.h"
#include "ocp_globals.h"

#include "SlaveResponseQueue.h"

// ----------------------------------------------------------------------------
// constructor
// ----------------------------------------------------------------------------
SlaveSysC::SlaveSysC(
    sc_module_name n,
    Ta             memory_byte_size,
    ostream*       debug_os_ptr
) : sc_module(n),
    tpP("tpPort"),
    m_MemoryByteSize(memory_byte_size),
    m_RespQueuePtr(NULL),
    m_Memory(NULL),
    m_debug_os_ptr(debug_os_ptr),
    m_OCPParamP(NULL),
    m_threads(1),
    m_datahandshake(false),
    m_dataaccept(true),
    m_writeresp_enable(false),
    m_sthreadbusy(false),
    m_sthreadbusy_exact(false),
    m_mthreadbusy(false),
    m_cmdaccept(true),
    m_burstseq(false),
    m_addrIncr(1),
    m_limitreq_enable(1),
    m_limitreq_max(4),
    m_Latency(NULL)
{
    // Note: member variables that depend on values of configuration parameters
    //       are constructed when those values are known - at the end of elaboration.

    // setup a SystemC thread process, which uses dynamic sensitive
    SC_THREAD(requestThreadProcess);

    // setup a SystemC thread process, which uses dynamic sensitive
    SC_THREAD(responseThreadProcess);
}

// --------------------------------------------------------------------------
// destructor
// --------------------------------------------------------------------------
SlaveSysC::~SlaveSysC()
{
    delete m_Memory;
    delete m_RespQueuePtr;
    delete[] m_Latency;
}

// --------------------------------------------------------------------------
//  SystemC Method SlaveSysC::end_of_elaboration()
// --------------------------------------------------------------------------
// 
//  At this point, everything has been built and connected.
//  We are now free to get our OCP parameters and to set up our
//  own variables that depend on them.
//
void SlaveSysC::end_of_elaboration()
{
    sc_module::end_of_elaboration();

    /////////////
    //
    // Process OCP Parameters from the port
    //
    /////////////

    m_OCPParamP = tpP->GetParamCl();

    // Set the number of threads
    m_threads = m_OCPParamP->threads;

    // Use the Threads value to set up our state that depends on it
    if (m_Latency) {
        delete[] m_Latency;
    }
    m_Latency = new int[m_threads];

    if (m_RespQueuePtr) {
        delete m_RespQueuePtr;
    }
    m_RespQueuePtr = new ThreadedRespQ<Td,Ta >(m_threads);

    // Does the channel use data handshaking?
    m_datahandshake = m_OCPParamP->datahandshake;

    // Does the channel use data handshake accept?
    m_dataaccept = m_OCPParamP->dataaccept;

    // Do writes get reponses?
    m_writeresp_enable = m_OCPParamP->writeresp_enable;

    // is SThreadBusy part of the channel?
    m_sthreadbusy = m_OCPParamP->sthreadbusy;

    // is this slave expected to follow the threadbusy exact protocol?
    m_sthreadbusy_exact = m_OCPParamP->sthreadbusy_exact;

    // is MThreadBusy part of the channel?
    m_mthreadbusy = m_OCPParamP->mthreadbusy;

    // is SCmdAccept part of the channel?
    m_cmdaccept = m_OCPParamP->cmdaccept;

    // Data must be byte sized
    assert( (m_OCPParamP->data_wdth)%8 == 0 );
    // Amount to increment address for each incrementing burst
    m_addrIncr = (m_OCPParamP->data_wdth) / 8;

    // We only support precise bursts with specific lengths
    if (m_OCPParamP->burstseq) {
        assert(m_OCPParamP->burstprecise);
        assert(m_OCPParamP->burstlength);
    }
    m_burstseq = m_OCPParamP->burstseq;

    /////////////
    //
    // Verify that this channel configuration is supported by this Slave
    //
    /////////////

    assert( !(m_datahandshake==true) );
    assert( !(m_mthreadbusy) );
    assert( !(m_sthreadbusy) );

    /////////////
    //
    // Process Slave Parameters
    //
    /////////////

    // For Debugging
    if (m_debug_os_ptr) {
        (*m_debug_os_ptr) << "DB (" << name() << "): "
            << "Configuring Slave." << endl;
        (*m_debug_os_ptr) << "DB (" << name() << "): was passed the following configuration map:" << endl;
        MapStringType::iterator map_it;
        for (map_it = m_ParamMap.begin(); map_it != m_ParamMap.end(); ++map_it) {
            (*m_debug_os_ptr) << "map[" << map_it->first << "] = " << map_it->second << endl;
        }
        cout << endl;
    }
    
    // Currently, myPrefix is not used. Retained for compatibility
    // with various parameter map formats.
    string myPrefix = "";
    string paramName = "undefined";
    char buffer[256];

    // latency(0), latency(1), ... , latency(N)
    // or latency0, latency1, latency2, latencyN
    for (int i=0; i<m_threads; i++) {
        // First try the old style:
        sprintf(buffer,"latency(%d)",i);
        paramName = buffer;
        if (!(OCPParameters ::getIntOCPConfigValue(myPrefix, paramName, m_Latency[i], m_ParamMap)) ) {
            // Could not find it with parenthesis. Try without
            sprintf(buffer,"latency%d",i);
            paramName = buffer;
            if (!(OCPParameters::getIntOCPConfigValue(myPrefix, paramName, m_Latency[i], m_ParamMap)) ) {
                // Could not find the parameter so we must set it to a default
#ifdef DEBUG 
                cout << "Warning: paramter \"" << paramName << "\" for Slave \"" << name() << "\" was not found in the parameter map." << endl;
                cout << "         setting missing parameter to 3." << endl;
#endif
                m_Latency[i] = 3;
            }
        }
    }

    // limitreq_enable
    paramName = "limitreq_enable";
    if (!(OCPParameters::getBoolOCPConfigValue(myPrefix, paramName, m_limitreq_enable, m_ParamMap)) ) {
        // Could not find the parameter so we must set it to a default
#ifdef DEBUG 
        cout << "Warning: paramter \"" << paramName << "\" for Slave \"" << name() << "\" was not found in the parameter map." << endl;
        cout << "         setting missing parameter to false." << endl;
#endif
        m_limitreq_enable = false;
    }

    // limitreq_max
    paramName = "limitreq_max";
    if (!(OCPParameters::getIntOCPConfigValue(myPrefix, paramName, m_limitreq_max, m_ParamMap)) ) {
        // Could not find the parameter so we must set it to a default
#ifdef DEBUG 
        cout << "Warning: paramter \"" << paramName << "\" for Slave \"" << name() << "\" was not found in the parameter map." << endl;
        cout << "         setting missing parameter to 4." << endl;
#endif
        m_limitreq_max = 4;
    }

    // Still have a limit on requests even if limitreq_enable is false to avoid
    // running out of memory.
    if (!m_limitreq_enable) {
        m_limitreq_max = 255;
    }

    /////////////
    //
    // Initialize the Slave with the New Parameters
    //
    /////////////

    // Create the memory:
    if (m_Memory) {
        // Just in case we are called multiple times.
        delete m_Memory;
    }
    char id_buff[10];
    sprintf(id_buff,"%d",m_ID);
    string my_id(id_buff);
    m_Memory = new MemoryCl<Td,Ta >(my_id,m_OCPParamP->addr_wdth,sizeof(Td));
    
}

// --------------------------------------------------------------------------
// SetConfiguration
// --------------------------------------------------------------------------
void SlaveSysC::setConfiguration( MapStringType& passedMap )
{
    // would be nice to process the Map immediately to avoid copying it and
    // storing it, but some of the parameters, such as latency(x) are based
    // on the number of threads and that number is based on an OCP parameter.
    //
    // As long as Slave parameters are functions of OCP parameters, they will
    // have to be processed at the end of elaboration when the OCP parameters
    // are available.
    //
    // Save the Map until then
    m_ParamMap = passedMap;
}


// --------------------------------------------------------------------------
// SystemC Thread Process
// --------------------------------------------------------------------------
//

void SlaveSysC::requestThreadProcess()
{
    // The new request we have just received
    OCPRequestGrp<Td,Ta> req;

    // Time after which the response can be sent or this request can be cleared from outgoing queue
    // so that the items behind it can be sent out.
    sc_time          send_time;

    // Set the initial value of theadbusy:
    tpP->putSThreadBusy (0);


    // We are in the initialization call. Wait for the first simulation cycle.
    wait(10,SC_NS);

    // New item to queue for request processing
    ReqQElement newRequest;

    // ChunkLen and ChunkLast TL2 variables
    unsigned int ChunkLen;
    bool ChunkLast;

    // main loop
    while (true) {
        // ----------------------------------------------------------
        // (1) Get the next request
        // ----------------------------------------------------------

        // Gets the request, but does not release it immediatly
        tpP->getOCPRequestBlocking(req,false,ChunkLen,ChunkLast);

        if (m_debug_os_ptr) {
            (*m_debug_os_ptr) << "DB (" << name() << "): " << " got new request." << endl;
            (*m_debug_os_ptr) << "DB (" << name() << "): " << "    t = " << sc_simulation_time() << " MCmd: " << req.MCmd << " MThreadID: " << req.MThreadID << endl;
            (*m_debug_os_ptr) << "DB (" << name() << "): " << "    MAddr = " << req.MAddr << endl;
        }

        // ----------------------------------------------------------
        // (2) if our queue is full, generate back pressure halt
        //     the flow of requests. Otherwise, accept the request
        //     and move on.
        // ----------------------------------------------------------

        // Do we need to set SThreadBusy??
        if (m_sthreadbusy && (int (m_RespQueuePtr->length(req.MThreadID) ) >= m_limitreq_max)) {

            if (m_debug_os_ptr) {
                (*m_debug_os_ptr) << "DB (" << name () << "): " << " for Thread#" << req.MThreadID << " setting sthreadbusy" << " (queue size is " << int(m_RespQueuePtr->length(req.MThreadID)) << ")" << endl;
            }
            tpP->putSThreadBusyBit (true,req.MThreadID);
        }

        // Should we accept this command?
        if ( m_cmdaccept ) {
            // if queue is full, delay accepting request
            while ( int(m_RespQueuePtr->length(req.MThreadID)) >= m_limitreq_max) {
                // Our queue is full. Wait for this to change.

                if (m_debug_os_ptr) {
                    (*m_debug_os_ptr) << "DB (" << name () << "): "
                        << " request queue " << req.MThreadID << " full at time " << sc_simulation_time() << ": waiting for it to lower." << endl;
                    (*m_debug_os_ptr) << "DB (" << name () << "): " << " m_RespQueuePtr->length("<<req.MThreadID<<") = "<< m_RespQueuePtr->length(req.MThreadID) << endl;
                }

                // Waits some amount of time before testing if queue is not
                // full anymore
                wait(10,SC_NS);
            }

            // now it is okay to accept the request
            if (m_debug_os_ptr) {
                (*m_debug_os_ptr) << "DB (" << name () << "): accepts request at time " << sc_simulation_time() << endl;
            }
            tpP->putSCmdAccept();
        }

        // ----------------------------------------------------------
        // (3) send the new request over to the processing queue
        // ----------------------------------------------------------

        // compute pipelined response delay
        // NOTE for TL2: to mimic the TL1 temporal behaviour, response delay
        // for a response chunk equals the sum of the TL1 response delay for
        // each response of this chunk

        send_time = sc_time_stamp() + sc_time(ChunkLen * m_Latency[req.MThreadID], SC_NS); 

        // If write with response is on, send a response for every WRITE opcode
        if (m_writeresp_enable) {
            // Is the last one - do send response
            if ((req.MCmd == OCP_MCMD_WR) || (req.MCmd == OCP_MCMD_WRC) || (req.MCmd == OCP_MCMD_BCST) ) {
                req.MCmd = OCP_MCMD_WRNP;
            }
        }

        // No data handshake. All requests should already have data and can be processed immediately.
        processRequest(req, ChunkLen, ChunkLast, send_time);
    }

}

void SlaveSysC::processRequest( OCPRequestGrp<Td,Ta>& req, unsigned int chunklen, bool chunklast, sc_time& send_time)
{
    // The response to the new request
    OCPResponseGrp<Td>   resp;

    // ----------------------------------------------------------
    // (1) process the Request
    // ----------------------------------------------------------

    // compute the word address

    // Simple memory wrapping
    if (req.MAddr >= m_MemoryByteSize) { 
        req.MAddr = req.MAddr - m_MemoryByteSize;
    }

    // ----------------------------------------------------------
    // (2) Write data to memory or read data from memory 
    // ----------------------------------------------------------

    // This slave model only supports INCR Burst sequence; if the OCP
    // channel does not support bursts, implicit TL2 burst sequence is
    // INCR
    if(m_burstseq == true) {
        assert(req.MBurstSeq == OCP_MBURSTSEQ_INCR);
    } 

    // Start Address
    unsigned int cur_addr = req.MThreadID * 100 + req.MAddr;


    // write to or read from the memory
    switch (req.MCmd) {
        case OCP_MCMD_WR:
        case OCP_MCMD_BCST:
            // posted burst write to memory
            
            // Address sequence is 'INCR'

            for(unsigned int i=0; i< chunklen; i++) {

                // Simple memory wrapping to avoid memory overflow
                if (cur_addr >= m_MemoryByteSize) {
                    cur_addr = cur_addr - m_MemoryByteSize;
                }

                m_Memory->write(cur_addr,req.MDataPtr[i],req.MByteEn);

                cur_addr += m_addrIncr;
            }

            // note that posted writes do not have responses. However,
            // they do have a processing delay that can contribute to
            // a max request limit back up.
            // To solve this problem, requests that have no response
            // generate a dummy respose with SRESP=NULL which is defined
            // as "No response".
            // Dummy responses are never sent out on the channel.
            resp.SResp = OCP_SRESP_NULL;
            resp.SThreadID = req.MThreadID;
            resp.SDataPtr = NULL;
            break;

        case OCP_MCMD_RD:
        case OCP_MCMD_RDEX:
            // NOTE that for a single threaded slave, Read-EX works just like Read 
            // read from memory

            // We must allocate memory for the SDataPtr pointer. This memory
            // will be freed by the response thread, when all the SData
            // cells have been sent.
            if(resp.SDataPtr != NULL) {
                delete [] resp.SDataPtr;
                resp.SDataPtr = NULL;
            }
            resp.SDataPtr = new unsigned int [chunklen];

            // Address sequence is 'INCR'

            for(unsigned int i=0; i< chunklen; i++) {

                // Simple memory wrapping to avoid memory overflow
                if (cur_addr >= m_MemoryByteSize) {
                    cur_addr = cur_addr - m_MemoryByteSize;
                }

                m_Memory->read(cur_addr,resp.SDataPtr[i],req.MByteEn);

                cur_addr += m_addrIncr;
            }

            // setup a read response
            resp.SResp = OCP_SRESP_DVA;
            resp.SThreadID = req.MThreadID;
            break;

        case OCP_MCMD_WRNP:
            // Non-posted write to memory
            
            // Address sequence is 'INCR'

            // Start address
            cur_addr = req.MAddr;

            for(unsigned int i=0; i< chunklen; i++) {

                // Simple memory wrapping to avoid memory overflow
                if (cur_addr >= m_MemoryByteSize) {
                    cur_addr = cur_addr - m_MemoryByteSize;
                }

                m_Memory->write(cur_addr,req.MDataPtr[i],req.MByteEn);

                cur_addr += m_addrIncr;
            }

            // Generate an acknowledgement response
            resp.SResp = OCP_SRESP_DVA;
            resp.SThreadID = req.MThreadID;
            resp.SDataPtr = NULL;
            break;

        default:
            cout << "ERROR: SlaveSysC - MCmd #" << req.MCmd << " not supported yet." << endl;
            sc_stop();
            break;
    }

    // ----------------------------------------------------------
    // (3) Add response to the response queue
    // ----------------------------------------------------------

    // purge the queue of any posted write place holder responses
    // that have reached their send times
    m_RespQueuePtr->purgePlaceholders();
    freeSThreadBusy();

    m_RespQueuePtr->enqueueBlocking(resp, chunklen, chunklast, send_time);
}


void SlaveSysC::freeSThreadBusy(void)
{
    // Checks to see if SThreadBusy can be freed, and then
    // releases it if it can

    // No point to this rountine if SThreadBusy no part of the channel.
    if (!m_sthreadbusy) {
        return;
    }

    for (int i=0; i<m_threads; i++) {
        if ( int(m_RespQueuePtr->length(i) ) < m_limitreq_max ) {
            // This thread's queue has been shortened. Clear threadBusy.
            tpP->putSThreadBusyBit(false,(unsigned int) i);

            if (m_debug_os_ptr) {
	        (*m_debug_os_ptr) << "DB (" << name () << "): " << " freeing Thread#" << i << " by resetting sthreadbusy" << endl;

            }
        }
    }
}

void SlaveSysC::responseThreadProcess()
{
    OCPResponseGrp<Td>   resp;
    sc_time          send_time;
    sc_time          CurTime;
    unsigned int     chunklen;
    bool             chunklast;

    // Initial wait before the response thread starts
    wait(10,SC_NS);

    // main loop
    while (true) {

        // -------------------------------------------------
        // (1) Find a response to place on the channel
        // -------------------------------------------------

        // Get to next response (wait for one, if necessary).
        
        // First, clear any stale write latency waits
        m_RespQueuePtr->purgePlaceholders();

        // If our queues have dropped, clear threadbusy bits
        freeSThreadBusy();

        if (m_debug_os_ptr) {
            (*m_debug_os_ptr) << "DB (" << name() << "): "
                << "Waiting for response to send." << endl;
        }

        // Get the next response to send. Aribtration is handled by
        // the ResponseQueue
        m_RespQueuePtr->dequeueBlocking(resp,chunklen, chunklast, send_time);

        if (m_debug_os_ptr) {
            (*m_debug_os_ptr) << "DB (" << name() << "): "
                << "Can send response. SResp = " << resp.SResp << " Send time = " << send_time << endl;
        }

        // check if we still need to wait
        CurTime = sc_time_stamp();
        if (send_time > CurTime) {
            wait(send_time-CurTime);
        }

        if (m_debug_os_ptr) {
            (*m_debug_os_ptr) << "DB (" << name() << "): "
                << "slave wait time = "
                << send_time.value() << endl;
        }

        // The response could be a place holder response 
        // used to implement write latency. If this is the case,
        // skip the rest of the steps.

        if (resp.SResp == OCP_SRESP_NULL) {
            if (m_debug_os_ptr) {
                (*m_debug_os_ptr) << "DB (" << name() << "): "
                    << "finished Write Latency waiting." << endl;
            }
        } else {

            // ----------------------------------
            // (2) is MThreadBusy?
            // ----------------------------------

            if (m_mthreadbusy) {
                while (tpP->getMThreadBusyBit(resp.SThreadID)) {
                    if (m_debug_os_ptr) {
                        (*m_debug_os_ptr) << "DB (" << name() << "): " <<
                            "MThread " << resp.SThreadID <<
                            " is busy. Waiting for MThreadBusy to change..."
                            << endl;
                    }
                    wait(tpP->MThreadBusyEvent());
                }
            }

            // ----------------------------------
            // (3) return a response
            // ----------------------------------

            if (m_debug_os_ptr) {
                (*m_debug_os_ptr) << "DB (" << name() << "): "
                    << "send response." << endl;
                (*m_debug_os_ptr) << "DB (" << name() << "): "
                    << "    t = " << sc_simulation_time() << endl;
                (*m_debug_os_ptr) << "DB (" << name() << "): "
                    << "    SThreadID: " << resp.SThreadID << endl;
                (*m_debug_os_ptr) << "DB (" << name() << "): "
                    << "    SResp: " << resp.SResp << endl;
                (*m_debug_os_ptr) << "DB (" << name() << "): "
                    << "    SData: " ;
                
                if(resp.SDataPtr != NULL) {
                    // This is a READ response
                    for(unsigned int i=0;i<chunklen;i++) {
                        (*m_debug_os_ptr) << resp.SDataPtr[i] << " ";
                    }
                    (*m_debug_os_ptr) << endl; 
                }

            }

            // Send out the response
            tpP->sendOCPResponseBlocking(resp,chunklen,chunklast);

            // If some memory wre allocated for a READ response, we must free it now:
            if(resp.SDataPtr != NULL) {
                delete [] resp.SDataPtr;
            }
        }

        // We must be able to clear ThreadBusy now as we just sent a request (or cleared a write latency)
        freeSThreadBusy();
    }
}
        
// --------------------------------------------------------------------------
//  Method
// --------------------------------------------------------------------------

bool SlaveSysC::MputDirect(
    int MasterID, bool IsWrite, Td *Data, Ta Address, int NumWords)
{
    return (true);
}

