//
//
// Copyright (c) 2004 Sonics, Inc.
//
// Confidential and Proprietary Information of Sonics, Inc.
// Use, disclosure, or reproduction is prohibited without
// written permission from Sonics, Inc.
//
// $Id: IStl.h,v 1.1 2007/01/25 22:09:07 halexan Exp $
//
//

#ifndef _IOcp2Stl_h
#define _IOcp2Stl_h

#include "ocp_globals.h"
#include "OcpBundleDefs.h"
#include "IntTypes.h"
#include <inttypes.h>
#include <deque>
#include <string>

#if __GNUC__ >= 3 && __GNUC_MINOR__ >= 2
#include <iostream>
#endif

class OCPParameters;
namespace OcpIp {
    
enum StlCommandType {
    STL_INVALID=0,
    STL_TRANSFER,
    STL_WAIT,
    STL_SIGNAL,
    STL_RESET,
    STL_CONTROLSTATUS
};

template<typename Td, typename Ta>
struct StlTransferCommand {
    StlTransferCommand() :
        idleCycles(0) {}
    const uint32_t                     idleCycles;
    const OCPTL2RequestGrp<Td, Ta>     request;

    // Expect masks.
    const deque<Td>                    dataMask;
    const deque<snx_uint64_t>          dataInfoMask;
};

struct StlWaitCommand {
    StlWaitCommand() : andor( false ) {}
    struct Waiter {
        bool         m_tbflags;
        bool         m_response;
        SignalName   m_signal;
        snx_uint64_t m_value;
        snx_uint64_t m_mask;
        void clear() {
            m_tbflags = m_response = false;
            m_value = m_mask = static_cast<snx_uint64_t>(-1);
        }
    };
    bool isAnd() const { return andor == false; }
    bool isOr () const { return andor == true; }
    const bool          andor;
    const deque<Waiter> waiters;
};

struct StlSignalCommand {
    StlSignalCommand() :
        signal ( S_MReset_n ),
        tbflags( false ),
        value  ( static_cast<snx_uint64_t>(-1) ),
        mask   ( static_cast<snx_uint64_t>(-1) ) {}
    virtual SignalName         getSignal( bool& tbflags ) const;
    virtual bool               drivenValue() const;
    virtual snx_uint64_t       drivenValue( snx_uint64_t& mask ) const;
    const SignalName           signal;
    const bool                 tbflags;
    const snx_uint64_t         value;
    const snx_uint64_t         mask;
};

class StlResetCommand {
  public:
    const uint32_t resetCycles;

    // Coverity needs constructor
    StlResetCommand(uint32_t _resetCycles = 0) : resetCycles(_resetCycles) {}
};

class StlControlStatusCommand {
  public:
    const bool         readWrite;
    const bool         controlStatus;
    const snx_uint64_t value;
    const snx_uint64_t mask;
    bool isRead () const { return readWrite == false; }
    bool isWrite() const { return readWrite == true ; }
    bool isControl() const { return controlStatus == false; }
    bool isStatus () const { return controlStatus == true ; }

    // Coverity needs constructor
    StlControlStatusCommand(bool         _readWrite=0,
                            bool         _controlStatus=0,
                            snx_uint64_t _value=0,
                            snx_uint64_t _mask=0) : readWrite(_readWrite),
                                                    controlStatus(_controlStatus),
                                                    value(_value),
                                                    mask(_mask) {}
};

class IStlReader {
  public:
    virtual void next( bool& eof ) = 0;
    virtual void setErrStream( ostream& err=cerr ) = 0;
    virtual void setDefaultThread( uint32_t ) = 0;
    virtual uint32_t       getCommandDelay() const = 0;
    virtual StlCommandType getCommandType()  const = 0;
    virtual string         getFileName()     const = 0;
    virtual uint32_t       getLineNum()      const = 0;

    // retrieve STL command structs
    template<typename Td, typename Ta>
      void                          setConversionTypes();
    template<typename Td, typename Ta>
      StlTransferCommand<Td,Ta>     getTransferCommand()      const;
    virtual StlWaitCommand          getWaitCommand()          const = 0;
    virtual StlSignalCommand        getSignalCommand()        const = 0;
    virtual StlResetCommand         getResetCommand()         const = 0;
    virtual StlControlStatusCommand getControlStatusCommand() const = 0;

  protected:
    IStlReader() {};
    IStlReader( const IStlReader& other );

  public:
    // Factory functions
    static deque<IStlReader*> findStlPrograms(  const char*          instanceName,
                                                bool                 coreSystem,
                                                const OCPParameters& intParams,
                                                const string& directory="" );
    static IStlReader* create( const char*          instanceName,
                               bool                 coreSystem,
                               const OCPParameters& intParams,
                               const string&        fileName );
};
}

#endif /* _IOcp2Stl_h_ */
