#include "simpleSlave.h"

// -------------------------------------------------------------------
// constructor
// -------------------------------------------------------------------
template<typename TdataCl>
Slave<TdataCl>::Slave(
    sc_module_name n,
    int            id,
    Ta             memory_byte_size,
    ostream*       debug_os_ptr
) : sc_module(n),
    tpP("tpPort"),
    clk("clkPort"),
    m_ID(id),
    m_MemoryByteSize(memory_byte_size),
    m_Memory(NULL),
    m_debug_os_ptr(debug_os_ptr),
    m_curSThreadBusy(0),
    m_threads(1),
    m_datahandshake(false),
    m_writeresp_enable(false),
    m_sthreadbusy(false),
    m_sthreadbusy_exact(false),
    m_mthreadbusy(false),
    m_cmdaccept(true),
    m_limitreq_max(4),
    m_Latency(3)
{
    // Note: member variables that depend on values of 
    // configuration parameters are constructed when those 
    // values are known - at the end of elaboration.

    // setup a SystemC thread process, which uses dynamic sensitive
    SC_THREAD(requestThreadProcess);
    sensitive<<clk.pos();

    // setup a SystemC thread process, which uses dynamic sensitive
    SC_THREAD(responseThreadProcess);
    sensitive<<clk.pos();

    // setup a SystemC thread process to check and 
    // set sideband signals
    SC_THREAD(exerciseSidebandThreadProcess);
    sensitive<<clk.pos();
    
    assert(m_Latency);
    
    //assuming default timing the slave will expect mthreadbusy to be stable 1 ps after the clock edge
    m_mthreadbusy_sample_time=sc_time(1, SC_PS);

}

template<typename TdataCl>
Slave<TdataCl>::~Slave()
{
    delete m_Memory;
}

template<typename TdataCl>
void Slave<TdataCl>::set_configuration(OCPParameters& passedConfig, std::string channelName){
    m_parameters=passedConfig;
    // Set the number of threads
    m_threads=passedConfig.threads;

    if (m_threads > 1) {
        cout << "Warning: Singled threaded reference Slave " 
            << name() << " attached to multi-threaded OCP." << endl;
        cout << "Only commands sent on thread 0 will be processed." 
            << endl;
    }
        
    // Does the channel use data handshaking?
    m_datahandshake=passedConfig.datahandshake;
    // Is so, quit as this Slave does not handle data handshake.
    assert(!m_datahandshake);

    // Do writes get reponses?
    m_writeresp_enable=passedConfig.writeresp_enable;

    // is SThreadBusy part of the channel?
    m_sthreadbusy=passedConfig.sthreadbusy;

    // is this slave expected to follow the threadbusy exact protocol?
    m_sthreadbusy_exact=passedConfig.sthreadbusy_exact;

    // is MThreadBusy part of the channel?
    m_mthreadbusy=passedConfig.mthreadbusy;
    
    // is SCmdAccept part of the channel?
    m_cmdaccept=passedConfig.cmdaccept;

    /////////////
    //
    // Process Slave Parameters
    //
    /////////////

    // For Debugging
    if (m_debug_os_ptr) {
        (*m_debug_os_ptr) << "DB (" << name() << "): "
            << "Configuring Slave." << endl;
        (*m_debug_os_ptr) << "DB (" 
            << name() 
               << "): was passed the following configuration map:" << endl;
        MapStringType::iterator map_it;
        for (map_it = passedConfig.Map.begin(); 
                map_it != passedConfig.Map.end(); ++map_it) {
            (*m_debug_os_ptr) << "map[" << map_it->first << "] = " 
                    << map_it->second << endl;
        }
        cout << endl;
    }
    
    /////////////
    //
    // Initialize the Slave with New Parameters
    //
    /////////////


    // Create the memory:

    if (m_Memory) {
        // Just in case we are called multiple times.
        delete m_Memory;
    }
    char id_buff[10];
    sprintf(id_buff,"%d",m_ID);
    string my_id(id_buff);
    m_Memory = 
       new MemoryCl<TdataCl>(my_id,passedConfig.addr_wdth,sizeof(Td));

}

template<typename TdataCl>
void Slave<TdataCl>::setModuleConfiguration(MapStringType& passedMap){
    if (!(OCPParameters::getIntOCPConfigValue("", "limitreq_max", 
            m_limitreq_max, passedMap)) ) {
        // Could not find the parameter so we must set it to a default
#ifdef DEBUG 
        cout << "Warning: paramter \"" << "limitreq_max" 
               << "\" was not found in the module parameter map." << endl;
        cout << "         setting missing parameter to "<<m_limitreq_max<<"." << endl;
#endif
    }

    if (!(OCPParameters::getIntOCPConfigValue("", "latency0", 
            m_Latency, passedMap)) ) {
        // Could not find the parameter so we must set it to a default
#ifdef DEBUG 
        cout << "Warning: paramter \"" << "latency0" 
               << "\" was not found in the module parameter map." << endl;
        cout << "         setting missing parameter to "<<m_Latency<<"." << endl;
#endif
    }
}

// -------------------------------------------------------------------
//  SystemC Method Slave::end_of_elaboration()
// -------------------------------------------------------------------
// 
//  At this point, everything has been built and connected.
//  We are now free to get our OCP parameters and to set up our
//  own variables that depend on them.
//
template<typename TdataCl>
void Slave<TdataCl>::end_of_elaboration()
{
    sc_module::end_of_elaboration();
    // Clear the response queue
    m_ResponseQueue.resetState();

    //get clock period information
    if (dynamic_cast<sc_clock*>(clk.get_interface())){
      m_clkPeriod=(dynamic_cast<sc_clock*>(clk.get_interface()))->period();
    }
    else{
      cout<<"An sc_clock has to be connected to the clock port of "<<name()<<endl<<flush;
      assert(dynamic_cast<sc_clock*>(clk.get_interface()));
    }
    tpP->addOCPConfigurationListener(*this);
    
    if (m_mthreadbusy){
      OCP_TL1_Slave_TimingCl myTiming;
      //sthreadbusy gets deasserted after mthreadbusy got read (see responseThreadProcess)
      myTiming.SThreadBusyStartTime=m_mthreadbusy_sample_time;
      tpP->registerTimingSensitiveOCPTL1Slave((OCP_TL1_Master_TimingIF*) this);
      tpP->setOCPTL1SlaveTiming(myTiming);
    }
}

template<typename TdataCl>
void Slave<TdataCl>::setOCPTL1MasterTiming(OCP_TL1_Master_TimingCl master_timing){
  if (master_timing.MThreadBusyStartTime+sc_time(1, SC_PS) > m_mthreadbusy_sample_time){
    OCP_TL1_Slave_TimingCl myTiming;
    m_mthreadbusy_sample_time=master_timing.MThreadBusyStartTime+sc_time(1, SC_PS);
    myTiming.SThreadBusyStartTime=m_mthreadbusy_sample_time;
    tpP->setOCPTL1SlaveTiming(myTiming);
  }
}

template<typename TdataCl>
void Slave<TdataCl>::requestThreadProcess()
{
    // The new request we have just received
    OCPRequestGrp<Td,Ta> req;

    // The response to the new request
    OCPResponseGrp<Td>   resp;

    // Time after which the response can be sent or this 
    // request can be cleared from incoming queue.
    sc_time          send_time;

    // We are in the initialization call. 
    // Wait for the first simulation cycle.
    wait();

    // main loop
    while (true) {
        // ----------------------------------------------------------
        // (1) Get the next request
        // ----------------------------------------------------------
        tpP->getOCPRequestBlocking(req,false);
        // ----------------------------------------------------------
        // (2) process the new request and generate a response.
        // ----------------------------------------------------------

        // compute the word address
        if (req.MAddr >= m_MemoryByteSize) {
            req.MAddr = req.MAddr - m_MemoryByteSize;
        }

        // send a response for writes if channel requires it.
        if ( m_writeresp_enable && (req.MCmd == OCP_MCMD_WR) ) {
            req.MCmd = OCP_MCMD_WRNP;
        }
            
        // write to or read from the memory
        switch (req.MCmd) {
            case OCP_MCMD_WR:
                // posted write to memory
                m_Memory->write(req.MAddr,req.MData,req.MByteEn);

                // note that posted writes do not have responses.
                // However, they do have a processing delay that can 
                // contribute to a max request limit back up.
                // To solve this problem, requests that have no 
                // response to generate a dummy respose with 
                // SRESP=NULL which is defined as "No response".
                // Dummy responses are never sent out on the channel.
                resp.SResp = OCP_SRESP_NULL;
                resp.SThreadID = req.MThreadID;
                break;

            case OCP_MCMD_RD:
            case OCP_MCMD_RDEX:
                // NOTE that for a single threaded slave, 
                // Read-EX works just like Read 
                // read from memory
                m_Memory->read(req.MAddr,resp.SData,req.MByteEn);
                // setup a read response
                resp.SResp = OCP_SRESP_DVA;
                resp.SThreadID = req.MThreadID;
                break;

            case OCP_MCMD_WRNP:
                // Generate an acknowledgement response
                resp.SResp = OCP_SRESP_DVA;
                resp.SThreadID = req.MThreadID;
                resp.SData = 0;
                break;

            default:
                cout << "MCmd #" << req.MCmd << " not supported yet." 
                         << endl;
                sc_stop();
                break;
        }

        // ----------------------------------------------------------
        // (3) generate a completion time stamp and add the response
        //     to the queue
        // ----------------------------------------------------------

        // compute pipelined response delay
        send_time = sc_time_stamp() + m_clkPeriod*m_Latency;

        // purge the queue of any posted write place holder responses
        // that have reached their send times
        m_ResponseQueue.purgePlaceholders();

        m_ResponseQueue.enqueueBlocking(resp, send_time);

        // ----------------------------------------------------------
        // (4) if our queue is full, generate back pressure halt
        //     the flow of requests. Otherwise, accept the request
        //     and move on.
        // ----------------------------------------------------------

        // Do we need to set SThreadBusy??
        if (m_sthreadbusy && (m_ResponseQueue.length() >= m_limitreq_max)) {
            wait(); //wait until next cycle to set busy
            m_curSThreadBusy = 1;
            tpP->putSThreadBusy(m_curSThreadBusy); 
        }

        // Should we accept this command?
        if ( m_cmdaccept ) {
            // if queue is full, delay accepting request
            while (m_ResponseQueue.length() >= m_limitreq_max) {
                // Our queue is full. Wait for this to change.
                wait();
            }
            // now it is okay to accept the request
            tpP->putSCmdAccept();
        }

    }
}

template<typename TdataCl>
void Slave<TdataCl>::responseThreadProcess()
{
    OCPResponseGrp<Td>   resp;
    sc_time          send_time;
    unsigned int     mthreadbusy;

    wait();

    // main loop
    while (true) {

        // -------------------------------------------------
        // (1) Find a response to place on the channel
        // -------------------------------------------------

        // We are single threaded - always choose thread zero:
        int selectedThread = 0;

        // Get to next response (wait for one, if necessary).
        
        // First, clear any stale write latency waits
        m_ResponseQueue.purgePlaceholders();

        // Get the next request off of the queue
        m_ResponseQueue.dequeueBlocking(resp,send_time);
        resp.SThreadID = selectedThread;

        // check if we still need to wait
        while (send_time > sc_time_stamp()) {
            wait();
        }
        if (m_debug_os_ptr) {
            (*m_debug_os_ptr) << "DB (" << name() << "): "
                << "slave wait time = "
                    << send_time.value() << endl;
        }

        // The response could be a place holder response 
        // used to implement write latency. If this is the case,
        // skip the rest of the steps.

        if (resp.SResp == OCP_SRESP_NULL) {
            if (m_debug_os_ptr) {
                (*m_debug_os_ptr) << "DB (" << name() << "): "
                    << "finished Write Latency waiting." << endl;
            }
        } else {

            // ----------------------------------
            // (2) is MThreadBusy?
            // ----------------------------------

            if (m_mthreadbusy) {
                wait(m_mthreadbusy_sample_time);
                mthreadbusy = tpP->getMThreadBusy();
                while (mthreadbusy & 1) {
                    wait();
                    wait(m_mthreadbusy_sample_time);
                    mthreadbusy = tpP->getMThreadBusy();
                }
            }

            // ----------------------------------
            // (3) return a response
            // ----------------------------------

            if (m_debug_os_ptr) {
                (*m_debug_os_ptr) << "DB (" << name() << "): "
                    << "send response." << endl;
                (*m_debug_os_ptr) << "DB (" << name() << "): "
                    << "    t = " << sc_simulation_time() << endl;
                (*m_debug_os_ptr) << "DB (" << name() << "): "
                    << "    SResp: " << resp.SResp << endl;
                (*m_debug_os_ptr) << "DB (" << name() << "): "
                    << "    SData: " << resp.SData << endl;
            }

            // Send out the response
            tpP->startOCPResponseBlocking(resp);
        }

        // We must be able to clear ThreadBusy now as we just sent a 
        // request (or cleared a write latency)
        if ( m_sthreadbusy && (m_curSThreadBusy==1) && 
                (m_ResponseQueue.length() < m_limitreq_max) ) {
            // Our queue has been shortened. Clear threadBusy.
            m_curSThreadBusy = 0;
            tpP->putSThreadBusy(m_curSThreadBusy);
        }
        
        // wait until next cycle to send out the next response (if any)
        wait();

    }
}

// Exercises the sideband signals by setting them with a recurring pattern
// Also loops back error signal from the Master if both Master and Slave 
// versions (MError and SError) are configured into the channel 
template<typename TdataCl>
void Slave<TdataCl>::exerciseSidebandThreadProcess()
{
    // Systematically send out sideband signals on any signals that are attached to us.
    for (int i=0; i<10;i++) wait();
    int tweakCounter =0;
    bool hasMError=m_parameters.merror;
    bool hasSError=m_parameters.serror;
    bool nextSError = false;
    bool hasSInterrupt=m_parameters.interrupt;
    bool nextSInterrupt = false;
    bool hasSFlag=m_parameters.sflag;
    int numSFlag=m_parameters.sflag_wdth;
    unsigned int nextSFlag = 0;
    unsigned int maxSFlag = (1 << numSFlag) -1; 

    // main loop
    while (true) {
        // wait 10 cycles
        for (int i=0; i<10;i++) wait();

        // Now count through my sideband changes
        tweakCounter++;

        // Drive SError every time we are called
        if (hasSError) {
            if (hasMError) {
                // loop MError back through SError
                nextSError=tpP->SgetMError();
                tpP->SputSError(nextSError);
            } else {
                // Toggle SError
                nextSError = !nextSError;
                tpP->SputSError(nextSError);
            }
        }

        // Drive SInterrupt
        if (hasSInterrupt) {
            // Drive every other time we are called
            if (tweakCounter%2 == 0) {
                // Toggle SInterrupt
                nextSInterrupt = !nextSInterrupt;
                tpP->SputSInterrupt(nextSInterrupt);
            }
        }

        // Drive SFlag
        if (hasSFlag) {
            // Drive every fourth time we are called
            if (tweakCounter%4 == 0) {
                nextSFlag += 1;
                if (nextSFlag > maxSFlag) {
                    nextSFlag = 0;
                }
                tpP->SputSFlag(nextSFlag);
            }
        }
    } // end while
}

// ---------------------------------------------------
// explicit instantiation of the Slave template class
// ---------------------------------------------------
template class Slave< OCP_TL1_DataCl<OCPCHANNELBit32, OCPCHANNELBit32> >;
