///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// (c) Copyright OCP-IP 2009-2011
// OCP-IP Confidential and Proprietary
//
//
//============================================================================
//      Project : OCP SLD WG
//       Author : James Aldis, Texas Instruments
//
//          $Id:
//
//  Description :  Testbench for TL1/TL0 adapters: precise aligned SRMD
//
// TL1 and TL0 initiators and targets are created with identical behaviour for
// the different abstraction levels.
// Six simulations are run, 4 with adapters and 2 without, for 0, 1 and 2
//   cascaded adapters.
// All sets of results should be the same, allowing for a few ps of timing
// delay in some cases.
//
//                                                                           //
///////////////////////////////////////////////////////////////////////////////


#include "ocpip_adapters.h"
#include <iostream>
#include <queue>


using namespace std;
using namespace ocpip;
using namespace sc_core;
using namespace tlm;


// all OCPs in this testbench have the same configuration
ocp_parameters gen_ocp_config() {
  ocp_parameters p;
  // aligned INCR bursts length in {1,2,4,8}
  p.burst_aligned = true;
  p.burstlength = true;
  p.burstlength_wdth = 4;
  p.burstsinglereq = true;
  p.datahandshake = true;
  p.dataaccept = true;
  p.datalast = true;
  p.resplast = true;
  // base config
  p.writeresp_enable = true;
  p.addr_wdth = 32;
  p.data_wdth = 32;
  p.mreset = true;
  return p;
}


// traits class for TL0 signal types
class SRMDBURST_TYPES: public tl1_tl0::POD_DEFAULT_TL0_TYPES<> {
public:
  typedef unsigned MBURSTLENGTH_T;
  typedef bool MBURSTSINGLEREQ_T;
  typedef bool MDATAVALID_T;
  typedef bool SDATAACCEPT_T;
  typedef bool MDATALAST_T;
  typedef bool SRESPLAST_T;
};


// simulation duration determined by these parameters - leave space for
// final burst to complete
#define MAX_ADDR 8000
#define MEM_SIZE 8064
#include "tb_common.h"


class ocp_tl0_initiator: public sc_module {
SC_HAS_PROCESS(ocp_tl0_initiator);
public:
  ocp_tl0_initiator(sc_module_name n):
    sc_module(n),
    Clk("Clk"),
    MAddr("MAddr"),
    MCmd("MCmd"),
    MData("MData"),
    SData("SData"),
    SCmdAccept("SCmdAccept"),
    SResp("SResp"),
    MReset_n("MReset_n"),
    MBurstLength("MBurstLength"),
    MBurstSingleReq("MBurstSingleReq"),
    MDataValid("MDataValid"),
    MDataLast("MDataLast"),
    SDataAccept("SDataAccept"),
    SRespLast("SRespLast"),
    cycle(0),
    next_req_cycle(1),
    next_data_cycle(1),
    resps_rxd(0),
    resps_wanted(0),
    curr_addr(0),
    curr_burst_len(1),
    req_active(false),
    data_active(false),
    curr_is_read(true),
    os(sc_module::name())
  {
    SC_METHOD(on_Clk_rising);
    dont_initialize();
    sensitive << Clk.pos();

    MReset_n.initialize(true);
    MCmd.initialize(0);
    MDataValid.initialize(false);

    fill_initiator_memory(memory);
  }

  sc_in<bool> Clk;
  sc_out<unsigned> MAddr;
  sc_out<unsigned> MCmd;
  sc_out<unsigned> MData;
  sc_in<unsigned> SData;
  sc_in<bool> SCmdAccept;
  sc_in<unsigned> SResp;
  sc_out<bool> MReset_n;

  sc_out<unsigned> MBurstLength;
  sc_out<bool> MBurstSingleReq;
  sc_out<bool> MDataValid;
  sc_out<bool> MDataLast;
  sc_in<bool> SDataAccept;
  sc_in<bool> SRespLast;

private:
  void on_Clk_rising() {
    if((curr_addr >= MAX_ADDR) && (resps_rxd == resps_wanted)) {
      stop_sim();
      return;
    }

    // request side
    if(req_active && SCmdAccept)
      os << cycle << " I: req accepted " << resps_wanted << endl;

    if(SCmdAccept || !req_active) {
      if((cycle >= next_req_cycle) && (curr_addr < MAX_ADDR)) {
        // SRMD bursts
        if(curr_burst_len == 1) {
          if((curr_addr & 31) == 0) curr_burst_len = 8;
        } else curr_burst_len /= 2;

        // alternate WR and RD
        curr_is_read = !curr_is_read;

        MBurstLength = curr_burst_len;
        MAddr = curr_addr;
        MBurstSingleReq = true;
        if(curr_is_read) {
          MCmd = 2;
          rd_addr.push(curr_addr);
          resps_wanted += curr_burst_len;
          os << cycle << " I: R req sent @ " << curr_addr << endl;
        } else {
          MCmd = 1;
          rd_addr.push(-1);
          wr_addr.push(curr_addr);
          wr_len.push(curr_burst_len);
          resps_wanted += 1;
          os << cycle << " I: W req sent @ " << curr_addr << endl;
        }
        os << cycle << " I: SRMD req length " << curr_burst_len << endl;

        curr_addr += curr_burst_len * 4;
        req_active = true;

        // wait 1-65 cycles between requests roughly randomly
        unsigned sel = (cycle & 7) ^ ((cycle >> 3) & 7) ^ ((cycle >> 6) & 7);
        next_req_cycle = cycle + 1 + (1 << sel) / 2;
      } else {
        MCmd = 0;
        req_active = false;
      }
    }

    // data side
    if(data_active && SDataAccept)
      os << cycle << " I: data accepted " << resps_wanted << endl;

    if(SDataAccept || !data_active) {
      if((wr_addr.size() > 0) && (cycle >= next_data_cycle)) {
        MDataValid = true;
        data_active = true;
        int &addr(wr_addr.front());
        int &len(wr_len.front());
        unsigned d = *reinterpret_cast<unsigned *>(memory + addr);
        MData = d;
        os << cycle << " I: W data: " << d << endl;
        os << cycle << " I: W data sent @ " << addr << endl;
        os << cycle << " I: W length " << len << endl;
        if(1 == len) {
          wr_addr.pop();
          wr_len.pop();
          MDataLast = true;
        } else {
          addr += 4;
          len--;
          MDataLast = false;
        }
        // wait 1,1,2,1,1,2,1,1,2,1,... cycles between data
        next_data_cycle = cycle + 1 + ((cycle / 2) & 1);
      } else {
        MDataValid = false;
        data_active = false;
      }
    }

    // response side
    if(SResp != 0) {
      resps_rxd++;
      int &addr(rd_addr.front());
      if(addr >= 0) {
        *reinterpret_cast<unsigned *>(memory + addr) = SData;
        os << cycle << " I: R data: " <<  SData << endl;
        os << cycle << " I: R resp received @ " << addr << endl;
      } else {
        os << cycle << " I: W resp received"<< endl;
      }
      if(SRespLast) {
        rd_addr.pop();
        os << cycle << " I: resp last"<< endl;
      } else {
        addr += 4;
      }
    }

    cycle++;
  }

  char memory[MEM_SIZE];
  unsigned cycle;
  unsigned next_req_cycle, next_data_cycle, resps_rxd, resps_wanted;
  unsigned curr_addr, curr_burst_len;
  bool req_active, data_active, curr_is_read;
  queue<int> rd_addr, wr_addr, wr_len;
  ofstream os;
};


class ocp_tl1_initiator: public sc_module {
SC_HAS_PROCESS(ocp_tl1_initiator);
public:
  ocp_tl1_initiator(sc_module_name n):
    sc_module(n),
    Clk("Clk"),
    ocp("ocp", this, &ocp_tl1_initiator::ocp_timing_update),
    cycle(0),
    next_req_cycle(1),
    next_data_cycle(1),
    resps_rxd(0),
    resps_wanted(0),
    curr_addr(0),
    curr_burst_len(1),
    wr_len(0),
    rd_len(0),
    req_active(false),
    data_active(false),
    curr_is_read(true),
    scmdaccept(false),
    sdataaccept(false),
    sresp(0),
    os(sc_module::name()),
    srespT(SC_ZERO_TIME)
  {
    SC_METHOD(on_Clk_rising);
    dont_initialize();
    sensitive << Clk.pos();

    ocp.set_ocp_config(gen_ocp_config());
    ocp.activate_synchronization_protection();
    ocp.register_nb_transport_bw(this, &ocp_tl1_initiator::nb_transport_bw);

    fill_initiator_memory(memory);
  }

  sc_in<bool> Clk;
  ocp_master_socket_tl1<32> ocp;

  void end_of_elaboration() {set_master_timing();}

private:
  void on_Clk_rising() {
    if((curr_addr >= MAX_ADDR) && (resps_rxd == resps_wanted)) {
      stop_sim();
      return;
    }

    // request side
    if(req_active && scmdaccept)
      os << cycle << " I: req accepted " << resps_wanted << endl;

    if(scmdaccept || !req_active) {
      if((cycle >= next_req_cycle) && (curr_addr < MAX_ADDR)) {
        // SRMD bursts
        if(curr_burst_len == 1) {
          if((curr_addr & 31) == 0) curr_burst_len = 8;
        } else curr_burst_len /= 2;

        // alternate WR and RD
        curr_is_read = !curr_is_read;

        tlm_generic_payload *curr = ocp.get_transaction();
        leak_test(*curr);
        extension_api::validate_extension<srmd>(*curr);
        curr->set_byte_enable_ptr(0);
        curr->set_response_status(TLM_INCOMPLETE_RESPONSE);
        curr->set_address(curr_addr);
        curr->set_data_ptr((unsigned char*)memory + curr_addr);
        curr->set_data_length(4 * curr_burst_len);
        curr->set_streaming_width(4 * curr_burst_len);
        if(curr_is_read) {
          curr->set_read();
          rd_addr.push(curr_addr);
          resps_wanted += curr_burst_len;
          os << cycle << " I: R req sent @ " << curr_addr << endl;
        } else {
          curr->set_write();
          ocp.validate_extension<posted>(*curr);
          wr_addr.push(curr);
          resps_wanted += 1;
          os << cycle << " I: W req sent @ " << curr_addr << endl;
        }
        os << cycle << " I: SRMD req length " << curr_burst_len << endl;

        tlm_phase ph = BEGIN_REQ;
        sc_time t = SC_ZERO_TIME;
        if(ocp->nb_transport_fw(*curr, ph, t) == TLM_UPDATED) {
          sc_assert(ph == END_REQ);
          scmdaccept = true;
        } else {
          scmdaccept = false;
        }

        curr_addr += curr_burst_len * 4;
        req_active = true;

        // wait 1-65 cycles between requests roughly randomly
        unsigned sel = (cycle & 7) ^ ((cycle >> 3) & 7) ^ ((cycle >> 6) & 7);
        next_req_cycle = cycle + 1 + (1 << sel) / 2;
      } else {
        req_active = false;
      }
    }

    // data side
    if(data_active && sdataaccept)
      os << cycle << " I: data accepted " << resps_wanted << endl;

    if(sdataaccept || !data_active) {
      if((wr_addr.size() > 0) && (cycle >= next_data_cycle)) {
        tlm_generic_payload *curr = wr_addr.front();

        int addr = curr->get_address() +  4 * wr_len;
        unsigned len = curr->get_data_length() / 4;
        unsigned d = *reinterpret_cast<unsigned *>(memory + addr);
        os << cycle << " I: W data: " << d << endl;
        os << cycle << " I: W data sent @ " << addr << endl;
        os << cycle << " I: W length " << len - wr_len << endl;

        tlm_phase ph = BEGIN_DATA;
        sc_time t = SC_ZERO_TIME;
        if(ocp->nb_transport_fw(*curr, ph, t) == TLM_UPDATED) {
          sc_assert(ph == END_DATA);
          sdataaccept = true;
        } else {
          sdataaccept = false;
        }
        data_active = true;

        if(len == ++wr_len) {
          wr_addr.pop();
          wr_len = 0;
        }

        // wait 1,1,2,1,1,2,1,1,2,1,... cycles between data
        next_data_cycle = cycle + 1 + ((cycle / 2) & 1);
      } else {
        data_active = false;
      }
    }

    // response side
    if(sresp != 0) {
      resps_rxd++;
      if(sresp->is_read()) {
        int addr = rd_addr.front() + 4 * rd_len;
        unsigned d = *reinterpret_cast<unsigned *>(memory + addr);
        os << cycle << " I: R data: " << d << endl;
        os << cycle << " I: R resp received @ " << addr << endl;
        if(4 * (++rd_len) == sresp->get_data_length()) {
          rd_len = 0;
          rd_addr.pop();
          ocp.release_transaction(sresp);
          os << cycle << " I: resp last" << endl;
        }
      } else {
        os << cycle << " I: W resp received" << endl;
        os << cycle << " I: resp last" << endl;
        ocp.release_transaction(sresp);
      }
      sresp = 0;
    }

    cycle++;
  }

  tlm_sync_enum nb_transport_bw(
    tlm_generic_payload &pl, tlm_phase &ph, sc_time &ta)
  {
    if(ph == END_REQ) {
      scmdaccept = true;
      return TLM_ACCEPTED;
    }
    if(ph == END_DATA) {
      sdataaccept = true;
      return TLM_ACCEPTED;
    }
    sc_assert(ph == BEGIN_RESP);
    sresp = &pl;
    ph = END_RESP;
    return TLM_UPDATED;
  }

  char memory[MEM_SIZE];
  unsigned cycle;
  unsigned next_req_cycle, next_data_cycle, resps_rxd, resps_wanted;
  unsigned curr_addr, curr_burst_len, wr_len, rd_len;
  bool req_active, data_active, curr_is_read;
  bool scmdaccept, sdataaccept;
  tlm_generic_payload *sresp;
  queue<int> rd_addr;
  queue<tlm_generic_payload *> wr_addr;
  ofstream os;
  sc_time srespT;

  // not sure if this is redundant when there is no resp-accept in the OCP
  // configuration.  It still exists in the TLM2
  void ocp_timing_update(ocp_tl1_slave_timing times) {
    if(srespT != times.ResponseGrpStartTime) {
      srespT = times.ResponseGrpStartTime;
      set_master_timing();
    }
  }

  void set_master_timing() {
    ocp_tl1_master_timing mytimes;
    mytimes.MRespAcceptStartTime = srespT + sc_get_time_resolution();
    ocp.set_master_timing(mytimes);
  }
};


class ocp_tl0_target: public sc_module {
SC_HAS_PROCESS(ocp_tl0_target);
public:
  ocp_tl0_target(sc_module_name n):
    sc_module(n),
    Clk("Clk"),
    MAddr("MAddr"),
    MCmd("MCmd"),
    MData("MData"),
    SData("SData"),
    SCmdAccept("SCmdAccept"),
    SResp("SResp"),
    MReset_n("MReset_n"),
    MBurstLength("MBurstLength"),
    MBurstSingleReq("MBurstSingleReq"),
    MDataValid("MDataValid"),
    MDataLast("MDataLast"),
    SDataAccept("SDataAccept"),
    SRespLast("SRespLast"),
    cycle(0),
    os(sc_module::name())
  {
    SC_METHOD(on_Clk_rising);
    dont_initialize();
    sensitive << Clk.pos() << MReset_n.neg();

    SResp.initialize(0);
    SCmdAccept.initialize(false);
    SDataAccept.initialize(false);

    fill_target_memory(memory);
  }

  sc_in<bool> Clk;
  sc_in<unsigned> MAddr;
  sc_in<unsigned> MCmd;
  sc_in<unsigned> MData;
  sc_out<unsigned> SData;
  sc_out<bool> SCmdAccept;
  sc_out<unsigned> SResp;
  sc_in<bool> MReset_n;

  sc_in<unsigned> MBurstLength;
  sc_in<bool> MBurstSingleReq;
  sc_in<bool> MDataValid;
  sc_in<bool> MDataLast;
  sc_out<bool> SDataAccept;
  sc_out<bool> SRespLast;

private:
  void on_Clk_rising() {
    SResp = 0;
    if(!MReset_n) {
      // reset all state
      while(rd_addr.size() > 0) rd_addr.pop();
      while(rd_len.size() > 0) rd_len.pop();
      while(wr_addr.size() > 0) wr_addr.pop();
      while(rd_cycle.size() > 0) rd_cycle.pop();
      while(wr_cycle.size() > 0) wr_cycle.pop();
    } else {
      // response side
      if(rd_addr.size() > 0) {
        bool is_read = (rd_addr.front() >= 0);
        if(is_read && (rd_cycle.size() > 0) && (cycle >= rd_cycle.front())) {
          // do a read response
          SResp = 1;
          int &addr(rd_addr.front());
          int &len(rd_len.front());
          unsigned d = *reinterpret_cast<unsigned *>(memory + addr);
          SData = d;
          os << cycle << " T: R data: " <<  d << endl;
          os << cycle << " T: R resp sent @ " << addr << endl;
          if(len == 1) {
            SRespLast = true;
            rd_addr.pop();
            rd_len.pop();
          } else {
            SRespLast = false;
            addr += 4;
            len--;
          }
          rd_cycle.pop();
        }
        if((!is_read) && (wr_cycle.size() > 0) && (cycle >= wr_cycle.front())) {
          // do a write response
          SResp = 1;
          os << cycle << " T: W resp sent" << endl;
          SRespLast = true;
          rd_addr.pop();
          wr_cycle.pop();
        }
      }

      // request side
      // accept in cycles 5,10,15,... and impose 4 cycle minimum latency
      if(SCmdAccept) switch(MCmd) {
        case 0:
          break;
        case 1:
          sc_assert(MBurstSingleReq);
          rd_addr.push(-1);
          wr_addr.push(MAddr);
          os << cycle << " T: W req received @ " << MAddr << endl;
          os << cycle << " T: W req length " << MBurstLength << endl;
          break;
        case 2:
          sc_assert(MBurstSingleReq);
          rd_addr.push(MAddr);
          rd_len.push(MBurstLength);
          os << cycle << " T: R req received @ " << MAddr << endl;
          os << cycle << " T: R req length " << MBurstLength << endl;
          for(unsigned i=0; i<MBurstLength; i++)
            rd_cycle.push((i/4)*4 + cycle + 4);
          break;
      }
      bool new_scmdaccept = ((cycle % 5) == 0) || ((cycle % 17) == 0);
      SCmdAccept = new_scmdaccept;

      // data side
      // accept in cycles 3,6,9,12,...
      // but do not accept data before accepting request
      if(SDataAccept && MDataValid) {
        int &addr(wr_addr.front());
        unsigned d = MData;
        os << cycle << " T: W data: " << d << endl;
        *reinterpret_cast<unsigned *>(memory + addr) = d;
        if(MDataLast) {
          wr_addr.pop();
          wr_cycle.push(cycle + 4);
          os << cycle << " T: W data last" << endl;
        } else {
          addr += 4;
        }
      }
      SDataAccept = (((cycle % 3) == 0) || ((cycle % 13) == 0))
        && ((wr_addr.size() > 0) || new_scmdaccept);

      cycle++;
    }
  }

  unsigned cycle;
  char memory[MEM_SIZE];
  queue<int> rd_addr, rd_len, wr_addr;
  queue<unsigned> rd_cycle, wr_cycle;
  ofstream os;
};


class ocp_tl1_target: public sc_module {
SC_HAS_PROCESS(ocp_tl1_target);
public:
  ocp_tl1_target(sc_module_name n):
    sc_module(n),
    Clk("Clk"),
    ocp("ocp", this, &ocp_tl1_target::ocp_timing_update),
    cycle(0),
    in_reset(false),
    scmdaccept(false),
    sdataaccept(false),
    mcmd(0),
    mdatahs(0),
    wr_len(0),
    rd_len(0),
    wr_addr_size(0),
    os(sc_module::name()),
    cmdT(SC_ZERO_TIME),
    dataT(SC_ZERO_TIME)
  {
    SC_METHOD(on_Clk_rising);
    dont_initialize();
    sensitive << Clk.pos();

    ocp.set_ocp_config(gen_ocp_config());
    ocp.activate_synchronization_protection();
    ocp.register_nb_transport_fw(this, &ocp_tl1_target::nb_transport_fw);

    fill_target_memory(memory);
  }

  sc_in<bool> Clk;
  ocp_slave_socket_tl1<32> ocp;

  void end_of_elaboration() {set_slave_timing();}

private:
  void on_Clk_rising() {
    if(in_reset) return;

    // response side
    if(rd_addr.size() > 0) {
      tlm_generic_payload *pl = rd_addr.front();
      if(pl->is_read() && (rd_cycle.size() > 0) && (cycle >= rd_cycle.front()))
      {
        // do a read response
        pl->set_response_status(TLM_OK_RESPONSE);
        int addr = pl->get_address() + 4 * rd_len;
        unsigned *local_addr = reinterpret_cast<unsigned *>(memory + addr);
        unsigned *init_addr =
          reinterpret_cast<unsigned *>(pl->get_data_ptr() + 4 * rd_len);
        *init_addr = *local_addr;
        os << cycle << " T: R data: " <<  *local_addr << endl;
        os << cycle << " T: R resp sent @ " << addr << endl;
        tlm_phase ph = BEGIN_RESP;
        sc_time t = SC_ZERO_TIME;
        ocp->nb_transport_bw(*pl, ph, t);
        if((++rd_len) * 4 == pl->get_data_length()) {
          rd_addr.pop();
          pl->release();
          rd_len = 0;
        }
        rd_cycle.pop();
      }
      if(pl->is_write() && (wr_cycle.size() > 0) && (cycle >= wr_cycle.front()))
      {
        // do a write response
        pl->set_response_status(TLM_OK_RESPONSE);
        os << cycle << " T: W resp sent" << endl;
        tlm_phase ph = BEGIN_RESP;
        sc_time t = SC_ZERO_TIME;
        ocp->nb_transport_bw(*pl, ph, t);
        rd_addr.pop();
        pl->release();
        wr_cycle.pop();
      }
    }

    // request side
    // accept in cycles 5,10,15,... and impose 4 cycle minimum latency
    if(scmdaccept && (mcmd != 0)) {
      sc_assert(extension_api::get_extension<srmd>(*mcmd) != 0);
      unsigned len = mcmd->get_data_length() / 4;
      unsigned addr = mcmd->get_address();
      if(mcmd->is_write()) {
        os << cycle << " T: W req received @ " << addr << endl;
        os << cycle << " T: W req length " << len << endl;
        ++wr_addr_size;
      } else {
        os << cycle << " T: R req received @ " << addr << endl;
        os << cycle << " T: R req length " << len << endl;
        for(unsigned i=0; i<len; i++) rd_cycle.push((i/4)*4 + cycle + 4);
      }
      mcmd = 0;
    }
    scmdaccept = ((cycle % 5) == 0) || ((cycle % 17) == 0);
    if(scmdaccept && (mcmd != 0)) {
      tlm_phase ph = END_REQ;
      sc_time t = SC_ZERO_TIME;
      ocp->nb_transport_bw(*mcmd, ph, t);
    }

    // data side
    // accept in cycles 3,6,9,12,...
    // but do not accept data before accepting request
    if(sdataaccept && (mdatahs != 0)) {
      int addr = mdatahs->get_address() + wr_len * 4;
      unsigned *local_addr = reinterpret_cast<unsigned *>(memory + addr);
      unsigned *init_addr =
        reinterpret_cast<unsigned *>(mdatahs->get_data_ptr() + 4 * wr_len);
      *local_addr = *init_addr;
      os << cycle << " T: W data: " << *init_addr << endl;
      if(4 * (++wr_len) == mdatahs->get_data_length()) {
        wr_cycle.push(cycle + 4);
        wr_len = 0;
        --wr_addr_size;
        os << cycle << " T: W data last" << endl;
      }
      mdatahs = 0;
    }
    sdataaccept = (((cycle % 3) == 0) || ((cycle % 13) == 0))
        && ((wr_addr_size > 0) || scmdaccept);
    if(sdataaccept && (mdatahs != 0)) {
      tlm_phase ph = END_DATA;
      sc_time t = SC_ZERO_TIME;
      ocp->nb_transport_bw(*mdatahs, ph, t);
    }

    cycle++;
  }

  tlm_sync_enum nb_transport_fw(
    tlm_generic_payload &pl, tlm_phase &ph, sc_time &ta)
  {
    if(ph == END_RESET) {
      in_reset = false;
      while(rd_addr.size() > 0) {
        rd_addr.front()->release();
        rd_addr.pop();
      }
      return TLM_ACCEPTED;
    }
    if(ph == BEGIN_RESET) {
      in_reset = true;
      mcmd = 0;
      ocp.reset();
      while(rd_cycle.size() > 0) rd_cycle.pop();
      while(wr_cycle.size() > 0) wr_cycle.pop();
      return TLM_ACCEPTED;
    }
    if(ph == BEGIN_REQ) {
      leak_test(pl);
      mcmd = &pl;
      rd_addr.push(&pl);
      pl.acquire();
      if(scmdaccept) {
        ph = END_REQ;
        return TLM_UPDATED;
      } else {
        return TLM_ACCEPTED;
      }
    }
    if(ph == BEGIN_DATA) {
      mdatahs = &pl;
      if(sdataaccept) {
        ph = END_DATA;
        return TLM_UPDATED;
      } else {
        return TLM_ACCEPTED;
      }
    }
    sc_assert(ph == END_RESP);
    return TLM_ACCEPTED;
  }

  unsigned cycle;
  bool in_reset;
  bool scmdaccept, sdataaccept;
  tlm_generic_payload *mcmd, *mdatahs;
  unsigned wr_len, rd_len, wr_addr_size;
  char memory[MEM_SIZE];
  queue<tlm_generic_payload *> rd_addr;
  queue<unsigned> rd_cycle, wr_cycle;
  ofstream os;
  sc_time cmdT, dataT;

  void ocp_timing_update(ocp_tl1_master_timing times) {
    if((cmdT != times.RequestGrpStartTime) ||
       (dataT != times.DataHSGrpStartTime)) {
      cmdT = times.RequestGrpStartTime;
      dataT = times.DataHSGrpStartTime;
      set_slave_timing();
    }
  }

  void set_slave_timing() {
    ocp_tl1_slave_timing mytimes;
    mytimes.SCmdAcceptStartTime = cmdT + sc_get_time_resolution();
    mytimes.SDataAcceptStartTime = dataT + sc_get_time_resolution();
    ocp.set_slave_timing(mytimes);
  }
};


class testbench00: public sc_module {
public:
  testbench00(sc_module_name n):
    sc_module(n),
    Clk("Clk"),
    initiator("initiator"),
    target("target")
  {
    // clock
    initiator.Clk(Clk);
    target.Clk(Clk);

    // TL0 interface
    initiator.MAddr(MAddr);
    initiator.MCmd(MCmd);
    initiator.MData(MData);
    initiator.SData(SData);
    initiator.SCmdAccept(SCmdAccept);
    initiator.SResp(SResp);
    initiator.MReset_n(MReset_n);
    target.MAddr(MAddr);
    target.MCmd(MCmd);
    target.MData(MData);
    target.SData(SData);
    target.SCmdAccept(SCmdAccept);
    target.SResp(SResp);
    target.MReset_n(MReset_n);

    initiator.MBurstLength(MBurstLength);
    initiator.MBurstSingleReq(MBurstSingleReq);
    initiator.MDataValid(MDataValid);
    initiator.MDataLast(MDataLast);
    initiator.SDataAccept(SDataAccept);
    initiator.SRespLast(SRespLast);
    target.MBurstLength(MBurstLength);
    target.MBurstSingleReq(MBurstSingleReq);
    target.MDataValid(MDataValid);
    target.MDataLast(MDataLast);
    target.SDataAccept(SDataAccept);
    target.SRespLast(SRespLast);
  }

  sc_in<bool> Clk;

private:
  ocp_tl0_initiator initiator;
  ocp_tl0_target target;

  sc_signal<unsigned> MAddr;
  sc_signal<unsigned> MCmd;
  sc_signal<unsigned> MData;
  sc_signal<unsigned> SData;
  sc_signal<bool> SCmdAccept;
  sc_signal<unsigned> SResp;
  sc_signal<bool> MReset_n;

  sc_signal<unsigned> MBurstLength;
  sc_signal<bool> MBurstSingleReq;
  sc_signal<bool> MDataValid;
  sc_signal<bool> MDataLast;
  sc_signal<bool> SDataAccept;
  sc_signal<bool> SRespLast;
};


class testbench01: public sc_module {
public:
  testbench01(sc_module_name n):
    sc_module(n),
    Clk("Clk"),
    initiator("initiator"),
    target("target"),
    adapter("adapter")
  {
    // OCP configuration
    adapter.set_ocp_config(gen_ocp_config());

    // clock
    initiator.Clk(Clk);
    target.Clk(Clk);
    adapter.Clk(Clk);

    // TL0 interface
    initiator.MAddr(MAddr);
    initiator.MCmd(MCmd);
    initiator.MData(MData);
    initiator.SData(SData);
    initiator.SCmdAccept(SCmdAccept);
    initiator.SResp(SResp);
    initiator.MReset_n(MReset_n);
    adapter.MAddr(MAddr);
    adapter.MCmd(MCmd);
    adapter.MData(MData);
    adapter.SData(SData);
    adapter.SCmdAccept(SCmdAccept);
    adapter.SResp(SResp);
    adapter.MReset_n(MReset_n);

    initiator.MBurstLength(MBurstLength);
    initiator.MBurstSingleReq(MBurstSingleReq);
    initiator.MDataValid(MDataValid);
    initiator.MDataLast(MDataLast);
    initiator.SDataAccept(SDataAccept);
    initiator.SRespLast(SRespLast);
    adapter.MBurstLength(MBurstLength);
    adapter.MBurstSingleReq(MBurstSingleReq);
    adapter.MDataValid(MDataValid);
    adapter.MDataLast(MDataLast);
    adapter.SDataAccept(SDataAccept);
    adapter.SRespLast(SRespLast);

    // TL1 interface
    adapter.ocpTL1(target.ocp);
  }

  sc_in<bool> Clk;

private:
  ocp_tl0_initiator initiator;
  ocp_tl1_target target;
  tl0_initiator_to_tl1_target<32, SRMDBURST_TYPES> adapter;

  sc_signal<unsigned> MAddr;
  sc_signal<unsigned> MCmd;
  sc_signal<unsigned> MData;
  sc_signal<unsigned> SData;
  sc_signal<bool> SCmdAccept;
  sc_signal<unsigned> SResp;
  sc_signal<bool> MReset_n;

  sc_signal<unsigned> MBurstLength;
  sc_signal<bool> MBurstSingleReq;
  sc_signal<bool> MDataValid;
  sc_signal<bool> MDataLast;
  sc_signal<bool> SDataAccept;
  sc_signal<bool> SRespLast;
};


class testbench10: public sc_module {
public:
  testbench10(sc_module_name n):
    sc_module(n),
    Clk("Clk"),
    initiator("initiator"),
    target("target"),
    adapter("adapter")
  {
    // OCP configuration
    adapter.set_ocp_config(gen_ocp_config());

    // clock
    initiator.Clk(Clk);
    target.Clk(Clk);
    adapter.Clk(Clk);

    // TL1 interface
    initiator.ocp(adapter.ocpTL1);

    // TL0 interface
    adapter.MAddr(MAddr);
    adapter.MCmd(MCmd);
    adapter.MData(MData);
    adapter.SData(SData);
    adapter.SCmdAccept(SCmdAccept);
    adapter.SResp(SResp);
    adapter.MReset_n(MReset_n);
    target.MAddr(MAddr);
    target.MCmd(MCmd);
    target.MData(MData);
    target.SData(SData);
    target.SCmdAccept(SCmdAccept);
    target.SResp(SResp);
    target.MReset_n(MReset_n);

    adapter.MBurstLength(MBurstLength);
    adapter.MBurstSingleReq(MBurstSingleReq);
    adapter.MDataValid(MDataValid);
    adapter.MDataLast(MDataLast);
    adapter.SDataAccept(SDataAccept);
    adapter.SRespLast(SRespLast);
    target.MBurstLength(MBurstLength);
    target.MBurstSingleReq(MBurstSingleReq);
    target.MDataValid(MDataValid);
    target.MDataLast(MDataLast);
    target.SDataAccept(SDataAccept);
    target.SRespLast(SRespLast);
  }

  sc_in<bool> Clk;

private:
  ocp_tl1_initiator initiator;
  ocp_tl0_target target;
  tl1_initiator_to_tl0_target<32, SRMDBURST_TYPES> adapter;

  sc_signal<unsigned> MAddr;
  sc_signal<unsigned> MCmd;
  sc_signal<unsigned> MData;
  sc_signal<unsigned> SData;
  sc_signal<bool> SCmdAccept;
  sc_signal<unsigned> SResp;
  sc_signal<bool> MReset_n;

  sc_signal<unsigned> MBurstLength;
  sc_signal<bool> MBurstSingleReq;
  sc_signal<bool> MDataValid;
  sc_signal<bool> MDataLast;
  sc_signal<bool> SDataAccept;
  sc_signal<bool> SRespLast;
};


class testbench11: public sc_module {
public:
  testbench11(sc_module_name n):
    sc_module(n),
    Clk("Clk"),
    initiator("initiator"),
    target("target")
  {
    // clock
    initiator.Clk(Clk);
    target.Clk(Clk);

    // TL1 interface
    initiator.ocp(target.ocp);
  }

  sc_in<bool> Clk;

private:
  ocp_tl1_initiator initiator;
  ocp_tl1_target target;
};


class testbench101: public sc_module {
public:
  testbench101(sc_module_name n):
    sc_module(n),
    Clk("Clk"),
    initiator("initiator"),
    target("target"),
    t_print(&cout),
    adapter10("adapter10", &t_print),
    adapter01("adapter01", &t_print)
  {
    // OCP configuration
    adapter10.set_ocp_config(gen_ocp_config());
    adapter01.set_ocp_config(gen_ocp_config());

    // clock
    initiator.Clk(Clk);
    target.Clk(Clk);
    adapter10.Clk(Clk);
    adapter01.Clk(Clk);

    // TL1 interface
    initiator.ocp(adapter10.ocpTL1);
    adapter01.ocpTL1(target.ocp);

    // TL0 interface
    adapter10.MAddr(MAddr);
    adapter10.MCmd(MCmd);
    adapter10.MData(MData);
    adapter10.SData(SData);
    adapter10.SCmdAccept(SCmdAccept);
    adapter10.SResp(SResp);
    adapter10.MReset_n(MReset_n);
    adapter01.MAddr(MAddr);
    adapter01.MCmd(MCmd);
    adapter01.MData(MData);
    adapter01.SData(SData);
    adapter01.SCmdAccept(SCmdAccept);
    adapter01.SResp(SResp);
    adapter01.MReset_n(MReset_n);

    adapter10.MBurstLength(MBurstLength);
    adapter10.MBurstSingleReq(MBurstSingleReq);
    adapter10.MDataValid(MDataValid);
    adapter10.MDataLast(MDataLast);
    adapter10.SDataAccept(SDataAccept);
    adapter10.SRespLast(SRespLast);
    adapter01.MBurstLength(MBurstLength);
    adapter01.MBurstSingleReq(MBurstSingleReq);
    adapter01.MDataValid(MDataValid);
    adapter01.MDataLast(MDataLast);
    adapter01.SDataAccept(SDataAccept);
    adapter01.SRespLast(SRespLast);

    // timing:  TL1 initiator and target produce transport calls at +0 ps
    //   but they aren't seen by adapters until +1 ps because of synch protection
    //   and only at this time are TL0 signals written
    adapter01.set_sample_times(
      sc_time(2.0, SC_PS),
      // adapter10 writes MCmd always during TLM-transport-from-initiator at +1 ps
      sc_time(3.0, SC_PS)
      // same for MDataValid but sampling is delayed to be after MCmd sampling,
      // which would be automatic but shown here for clarity
    );
    adapter10.set_sample_times(
      sc_time(2.0, SC_PS),
      // adapter01 writes SResp always during TLM-transport-from-target at +1 ps
      sc_time(5.0, SC_PS),
      // TL1 target may signal SCmdAccept during its clocked process with an
      // explicit transport call.  This would be seen at +1 ps by adapter01 and
      // the TL0 SCmdAccept signal would be stable for sampling at +2 ps.  But
      // TL1 target might return SCmdAccept to a fw transport call, which it
      // would see at the MCmd sampling time of adapter01 plus 1, ie 3 ps.  The
      // socket PEQ will convert this into a separate backward transport call
      // seen by adapter01 at 4 ps.  Thus SCmdAccept should be sampled at 5 ps.
      sc_time(6.0, SC_PS)
      // data trails command by 1 ps
    );
  }

  sc_in<bool> Clk;

private:
  ocp_tl1_initiator initiator;
  ocp_tl1_target target;

  tl1_tl0::get_TL0_timing_example<ostream> t_print;
  tl1_initiator_to_tl0_target<32, SRMDBURST_TYPES> adapter10;
  tl0_initiator_to_tl1_target<32, SRMDBURST_TYPES> adapter01;

  sc_signal<unsigned> MAddr;
  sc_signal<unsigned> MCmd;
  sc_signal<unsigned> MData;
  sc_signal<unsigned> SData;
  sc_signal<bool> SCmdAccept;
  sc_signal<unsigned> SResp;
  sc_signal<bool> MReset_n;

  sc_signal<unsigned> MBurstLength;
  sc_signal<bool> MBurstSingleReq;
  sc_signal<bool> MDataValid;
  sc_signal<bool> MDataLast;
  sc_signal<bool> SDataAccept;
  sc_signal<bool> SRespLast;
};


class testbench010: public sc_module {
public:
  testbench010(sc_module_name n):
    sc_module(n),
    Clk("Clk"),
    initiator("initiator"),
    target("target"),
    adapter01("adapter01"),
    adapter10("adapter10")
  {
    // OCP configuration
    adapter01.set_ocp_config(gen_ocp_config());
    adapter10.set_ocp_config(gen_ocp_config());

    // clock
    initiator.Clk(Clk);
    target.Clk(Clk);
    adapter01.Clk(Clk);
    adapter10.Clk(Clk);

    // TL1 interface
    adapter01.ocpTL1(adapter10.ocpTL1);

    // TL0 interfaces
    initiator.MAddr(MAddr[0]);
    initiator.MCmd(MCmd[0]);
    initiator.MData(MData[0]);
    initiator.SData(SData[0]);
    initiator.SCmdAccept(SCmdAccept[0]);
    initiator.SResp(SResp[0]);
    initiator.MReset_n(MReset_n[0]);
    adapter01.MAddr(MAddr[0]);
    adapter01.MCmd(MCmd[0]);
    adapter01.MData(MData[0]);
    adapter01.SData(SData[0]);
    adapter01.SCmdAccept(SCmdAccept[0]);
    adapter01.SResp(SResp[0]);
    adapter01.MReset_n(MReset_n[0]);

    initiator.MBurstLength(MBurstLength[0]);
    initiator.MBurstSingleReq(MBurstSingleReq[0]);
    initiator.MDataValid(MDataValid[0]);
    initiator.MDataLast(MDataLast[0]);
    initiator.SDataAccept(SDataAccept[0]);
    initiator.SRespLast(SRespLast[0]);
    adapter01.MBurstLength(MBurstLength[0]);
    adapter01.MBurstSingleReq(MBurstSingleReq[0]);
    adapter01.MDataValid(MDataValid[0]);
    adapter01.MDataLast(MDataLast[0]);
    adapter01.SDataAccept(SDataAccept[0]);
    adapter01.SRespLast(SRespLast[0]);

    adapter10.MAddr(MAddr[1]);
    adapter10.MCmd(MCmd[1]);
    adapter10.MData(MData[1]);
    adapter10.SData(SData[1]);
    adapter10.SCmdAccept(SCmdAccept[1]);
    adapter10.SResp(SResp[1]);
    adapter10.MReset_n(MReset_n[1]);
    target.MAddr(MAddr[1]);
    target.MCmd(MCmd[1]);
    target.MData(MData[1]);
    target.SData(SData[1]);
    target.SCmdAccept(SCmdAccept[1]);
    target.SResp(SResp[1]);
    target.MReset_n(MReset_n[1]);

    adapter10.MBurstLength(MBurstLength[1]);
    adapter10.MBurstSingleReq(MBurstSingleReq[1]);
    adapter10.MDataValid(MDataValid[1]);
    adapter10.MDataLast(MDataLast[1]);
    adapter10.SDataAccept(SDataAccept[1]);
    adapter10.SRespLast(SRespLast[1]);
    target.MBurstLength(MBurstLength[1]);
    target.MBurstSingleReq(MBurstSingleReq[1]);
    target.MDataValid(MDataValid[1]);
    target.MDataLast(MDataLast[1]);
    target.SDataAccept(SDataAccept[1]);
    target.SRespLast(SRespLast[1]);

    // timing: nothing needed as adapters' TL0 interfaces are
    //  directly connected to TL0-native initiator and target
  }

  sc_in<bool> Clk;

private:
  ocp_tl0_initiator initiator;
  ocp_tl0_target target;
  tl0_initiator_to_tl1_target<32, SRMDBURST_TYPES> adapter01;
  tl1_initiator_to_tl0_target<32, SRMDBURST_TYPES> adapter10;

  sc_signal<unsigned> MAddr[2];
  sc_signal<unsigned> MCmd[2];
  sc_signal<unsigned> MData[2];
  sc_signal<unsigned> SData[2];
  sc_signal<bool> SCmdAccept[2];
  sc_signal<unsigned> SResp[2];
  sc_signal<bool> MReset_n[2];

  sc_signal<unsigned> MBurstLength[2];
  sc_signal<bool> MBurstSingleReq[2];
  sc_signal<bool> MDataValid[2];
  sc_signal<bool> MDataLast[2];
  sc_signal<bool> SDataAccept[2];
  sc_signal<bool> SRespLast[2];
};


class testbench: public sc_module {
public:
  testbench(sc_module_name n, bool run_01 = true, bool run_10 = true):
    sc_module(n),
    tb00("00"),
    tb01("01"),
    tb10("10"),
    tb11("11"),
    tb101("101"),
    tb010("010"),
    Clk("Clk", sc_time(10.0, SC_NS))
  {
    tb00.Clk(Clk);
    tb11.Clk(Clk);

    if(run_01) tb01.Clk(Clk);
    else tb01.Clk(tieoff);

    if(run_10) tb10.Clk(Clk);
    else tb10.Clk(tieoff);

    if(run_01 && run_10) {
      tb010.Clk(Clk);
      tb101.Clk(Clk);
    } else {
      tb101.Clk(tieoff);
      tb010.Clk(tieoff);
    }
  }
private:
  testbench00 tb00;
  testbench01 tb01;
  testbench10 tb10;
  testbench11 tb11;
  testbench101 tb101;
  testbench010 tb010;
  sc_clock Clk;
  sc_signal<bool> tieoff;
};


#ifndef OMIT_SC_MAIN
int sc_main(int argc, char **argv) {
  sc_report_handler::set_actions(SC_ERROR, SC_DISPLAY | SC_ABORT);
  bool run_01 = ((argc < 2) || ('A' == *argv[1]) || ('0' == *argv[1]));
  bool run_10 = ((argc < 2) || ('A' == *argv[1]) || ('1' == *argv[1]));
  testbench theTestbench("testbench", run_01, run_10);
  sc_start();
  return 0;
}
#endif

