///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// (c) Copyright OCP-IP 2009-2011
// OCP-IP Confidential and Proprietary
//
//
//============================================================================
//      Project : OCP SLD WG
//       Author : James Aldis, Texas Instruments
//
//          $Id:
//
//  Description :  Testbench for TL1/TL0 adapters
//
// TL1 and TL0 initiators and targets are created with identical behaviour for
// the different abstraction levels.
// Six simulations are run, 4 with adapters and 2 without, for 0, 1 and 2
//   cascaded adapters.
// All sets of results should be the same, allowing for a few ps of timing
// delay in some cases.
//
//                                                                           //
///////////////////////////////////////////////////////////////////////////////


#include "ocpip_adapters.h"
#include <iostream>
#include <queue>


using namespace std;
using namespace ocpip;
using namespace sc_core;
using namespace tlm;


// all OCPs in this testbench have the same configuration
ocp_parameters gen_ocp_config() {
  ocp_parameters p;
  p.writeresp_enable = true;
  p.addr_wdth = 32;
  p.data_wdth = 32;
  p.mreset = true;
  return p;
}


// simulation duration determined by this parameter
#define MEM_SIZE 400
#include "tb_common.h"


class ocp_tl0_initiator: public sc_module {
SC_HAS_PROCESS(ocp_tl0_initiator);
public:
  ocp_tl0_initiator(sc_module_name n):
    sc_module(n),
    Clk("Clk"),
    MAddr("MAddr"),
    MCmd("MCmd"),
    MData("MData"),
    SData("SData"),
    SCmdAccept("SCmdAccept"),
    SResp("SResp"),
    MReset_n("MReset_n"),
    cycle(0),
    reqs_sent(0),
    next_req_cycle(1),
    resps_rxd(0),
    req_active(false),
    os(sc_module::name())
  {
    SC_METHOD(on_Clk_rising);
    dont_initialize();
    sensitive << Clk.pos();

    MReset_n.initialize(true);
    MCmd.initialize(0);

    fill_initiator_memory(memory);
  }

  sc_in<bool> Clk;
  sc_out<unsigned> MAddr;
  sc_out<unsigned> MCmd;
  sc_out<unsigned> MData;
  sc_in<unsigned> SData;
  sc_in<bool> SCmdAccept;
  sc_in<unsigned> SResp;
  sc_out<bool> MReset_n;

private:
  void on_Clk_rising() {
    if((reqs_sent * 4 + 3 >= MEM_SIZE) && (resps_rxd == reqs_sent)) {
      stop_sim();
      return;
    }

    // request side
    if(req_active && SCmdAccept)
      os << cycle << " I: req accepted " << reqs_sent << endl;

    if(SCmdAccept || !req_active) {
      if((cycle >= next_req_cycle) && (reqs_sent * 4 + 3 < MEM_SIZE)) {
        // incrementing word addresses
        unsigned addr = reqs_sent * 4;
        MAddr = addr;
        // alternate WR and RD
        char dir;
        if(reqs_sent & 1) {
          MCmd = 1;
          unsigned d = *reinterpret_cast<unsigned *>(memory + addr);
          MData = d;
          os << cycle << " I: W data: " <<  d << endl;
          rd_addr.push(-1);
          dir = 'W';
        } else {
          MCmd = 2;
          rd_addr.push(addr);
          dir = 'R';
        }
        reqs_sent++;
        req_active = true;
        os << cycle << " I: " << dir << " req sent @ " << addr << endl;
        // wait 1,2,4,8,16,1,2,4,8,... cycles between requests
        next_req_cycle = cycle + (1 << (reqs_sent & 7));
      } else {
        MCmd = 0;
        req_active = false;
      }
    }

    // response side
    if(SResp != 0) {
      resps_rxd++;
      int addr = rd_addr.front();
      rd_addr.pop();
      char dir;
      if(addr >= 0) {
        *reinterpret_cast<unsigned *>(memory + addr) = SData;
        dir = 'R';
        os << cycle << " I: R data: " <<  SData << endl;
      } else {
        dir = 'W';
      }
      os << cycle << " I: " << dir << " resp received @ " << addr << endl;
    }
    cycle++;
  }

  char memory[MEM_SIZE];
  unsigned cycle;
  unsigned reqs_sent, next_req_cycle, resps_rxd;
  bool req_active;
  queue<int> rd_addr;
  ofstream os;
};


class ocp_tl1_initiator: public sc_module {
SC_HAS_PROCESS(ocp_tl1_initiator);
public:
  ocp_tl1_initiator(sc_module_name n):
    sc_module(n),
    Clk("Clk"),
    ocp("ocp", this, &ocp_tl1_initiator::ocp_timing_update),
    cycle(0),
    reqs_sent(0),
    next_req_cycle(1),
    resps_rxd(0),
    req_active(false),
    scmdaccept(false),
    sresp(0),
    os(sc_module::name()),
    srespT(SC_ZERO_TIME)
  {
    SC_METHOD(on_Clk_rising);
    dont_initialize();
    sensitive << Clk.pos();

    ocp.set_ocp_config(gen_ocp_config());
    ocp.activate_synchronization_protection();
    ocp.register_nb_transport_bw(this, &ocp_tl1_initiator::nb_transport_bw);

    fill_initiator_memory(memory);
  }

  sc_in<bool> Clk;
  ocp_master_socket_tl1<32> ocp;

private:
  void on_Clk_rising() {
    if((reqs_sent * 4 + 3 >= MEM_SIZE) && (resps_rxd == reqs_sent)) {
      stop_sim();
      return;
    }

    // request side
    if(req_active && scmdaccept)
      os << cycle << " I: req accepted " << reqs_sent << endl;

    if(scmdaccept || !req_active) {
      if((cycle >= next_req_cycle) && (reqs_sent * 4 + 3 < MEM_SIZE)) {
        tlm_generic_payload *pl = ocp.get_transaction();
        leak_test(*pl);
        pl->set_data_length(4);
        pl->set_streaming_width(4);
        pl->set_byte_enable_ptr(0);
        pl->set_response_status(TLM_INCOMPLETE_RESPONSE);
        // incrementing word addresses
        unsigned addr = reqs_sent * 4;
        pl->set_address(addr);
        pl->set_data_ptr((unsigned char*)memory + addr);
        // alternate WR and RD
        char dir;
        if(reqs_sent & 1) {
          pl->set_write();
          ocp.validate_extension<posted>(*pl);
          unsigned d = *reinterpret_cast<unsigned *>(memory + addr);
          os << cycle << " I: W data: " <<  d << endl;
          rd_addr.push(-1);
          dir = 'W';
        } else {
          pl->set_read();
          rd_addr.push(addr);
          dir = 'R';
        }
        tlm_phase ph = BEGIN_REQ;
        sc_time t = SC_ZERO_TIME;
        if(ocp->nb_transport_fw(*pl, ph, t) == TLM_UPDATED) {
          sc_assert(ph == END_REQ);
          scmdaccept = true;
        } else {
          scmdaccept = false;
        }
        reqs_sent++;
        req_active = true;
        os << cycle << " I: " << dir << " req sent @ " << addr << endl;
        // wait 1,2,4,8,16,1,2,4,8,... cycles between requests
        next_req_cycle = cycle + (1 << (reqs_sent & 7));
      } else {
        req_active = false;
      }
    }

    // response side
    if(sresp != 0) {
      resps_rxd++;
      int addr = rd_addr.front();
      rd_addr.pop();
      char dir;
      if(addr >= 0) {
        unsigned d = *reinterpret_cast<unsigned *>(memory + addr);
        dir = 'R';
        os << cycle << " I: R data: " << d << endl;
      } else {
        dir = 'W';
      }
      os << cycle << " I: " << dir << " resp received @ " << addr << endl;
      ocp.release_transaction(sresp);
      sresp = 0;
    }
    cycle++;
  }

  tlm_sync_enum nb_transport_bw(
    tlm_generic_payload &pl, tlm_phase &ph, sc_time &ta)
  {
    if(ph == END_REQ) {
      scmdaccept = true;
      return TLM_ACCEPTED;
    }
    sc_assert(ph == BEGIN_RESP);
    sresp = &pl;
    ph = END_RESP;
    return TLM_UPDATED;
  }

  char memory[MEM_SIZE];
  unsigned cycle;
  unsigned reqs_sent, next_req_cycle, resps_rxd;
  bool req_active;
  queue<int> rd_addr;
  bool scmdaccept;
  tlm_generic_payload *sresp;
  ofstream os;
  sc_time srespT;

  // not sure if this is redundant when there is no resp-accept in the OCP
  // configuration.  It still exists in the TLM2
  void ocp_timing_update(ocp_tl1_slave_timing times) {
    if(srespT != times.ResponseGrpStartTime) {
      srespT = times.ResponseGrpStartTime;
      set_master_timing();
    }
  }

  void set_master_timing() {
    ocp_tl1_master_timing mytimes;
    mytimes.MRespAcceptStartTime = srespT + sc_get_time_resolution();
    ocp.set_master_timing(mytimes);
  }
};


class ocp_tl0_target: public sc_module {
SC_HAS_PROCESS(ocp_tl0_target);
public:
  ocp_tl0_target(sc_module_name n):
    sc_module(n),
    Clk("Clk"),
    MAddr("MAddr"),
    MCmd("MCmd"),
    MData("MData"),
    SData("SData"),
    SCmdAccept("SCmdAccept"),
    SResp("SResp"),
    MReset_n("MReset_n"),
    cycle(0),
    os(sc_module::name())
  {
    SC_METHOD(on_Clk_rising);
    dont_initialize();
    sensitive << Clk.pos() << MReset_n.neg();

    SResp.initialize(0);
    SCmdAccept.initialize(false);

    fill_target_memory(memory);
  }

  sc_in<bool> Clk;
  sc_in<unsigned> MAddr;
  sc_in<unsigned> MCmd;
  sc_in<unsigned> MData;
  sc_out<unsigned> SData;
  sc_out<bool> SCmdAccept;
  sc_out<unsigned> SResp;
  sc_in<bool> MReset_n;

private:
  void on_Clk_rising() {
    if(!MReset_n) {
      // reset all state
      while(resp_cycle.size() > 0) {
        resp_cycle.pop();
        rd_addr.pop();
      }
      SResp = 0;
    } else {
      // response side
      if((resp_cycle.size() > 0) && (cycle >= resp_cycle.front())) {
        resp_cycle.pop();
        SResp = 1;
        int addr = rd_addr.front();
        rd_addr.pop();
        char dir;
        if(addr < 0) {
          dir = 'W';
        } else {
          dir = 'R';
          unsigned d = *reinterpret_cast<unsigned *>(memory + addr);
          SData = d;
          os << cycle << " T: R data: " <<  d << endl;
        }
        os << cycle << " T: " << dir << " resp sent @ " << addr << endl;
      } else {
        SResp = 0;
      }

      // request side
      // accept in cycles 3,6,9,12,15,... and impose 4 cycle minimum latency
      if(SCmdAccept) {
        switch(MCmd) {
        case 0:
          break;
        case 1:
          rd_addr.push(-1);
          *reinterpret_cast<unsigned *>(memory + MAddr) = unsigned(MData);
          os << cycle << " T: W data: " <<  unsigned(MData) << endl;
          os << cycle << " T: W req received @ " << MAddr << endl;
          resp_cycle.push(cycle + 4);
          break;
        case 2:
          rd_addr.push(MAddr);
          os << cycle << " T: R req received @ " << MAddr << endl;
          resp_cycle.push(cycle + 4);
          break;
        }
      }
      SCmdAccept = ((cycle % 3) == 0);

      cycle++;
    }
  }

  unsigned cycle;
  char memory[MEM_SIZE];
  queue<int> rd_addr;
  queue<unsigned> resp_cycle;
  ofstream os;
};


class ocp_tl1_target: public sc_module {
SC_HAS_PROCESS(ocp_tl1_target);
public:
  ocp_tl1_target(sc_module_name n):
    sc_module(n),
    Clk("Clk"),
    ocp("ocp", this, &ocp_tl1_target::ocp_timing_update),
    cycle(0),
    in_reset(false),
    scmdaccept(false),
    mcmd(0),
    os(sc_module::name()),
    cmdT(SC_ZERO_TIME),
    dataT(SC_ZERO_TIME)
  {
    SC_METHOD(on_Clk_rising);
    dont_initialize();
    sensitive << Clk.pos();

    ocp.set_ocp_config(gen_ocp_config());
    ocp.activate_synchronization_protection();
    ocp.register_nb_transport_fw(this, &ocp_tl1_target::nb_transport_fw);

    fill_target_memory(memory);
  }

  sc_in<bool> Clk;
  ocp_slave_socket_tl1<32> ocp;

private:
  void on_Clk_rising() {
    if(in_reset) return;

    // response side
    if((resp_cycle.size() > 0) && (cycle >= resp_cycle.front())) {
      resp_cycle.pop();
      tlm_generic_payload *pl = rd_addr.front();
      rd_addr.pop();
      pl->set_response_status(TLM_OK_RESPONSE);
      int addr = pl->get_address();
      char dir;
      unsigned *local_addr = reinterpret_cast<unsigned *>(memory + addr);
      unsigned *init_addr = reinterpret_cast<unsigned *>(pl->get_data_ptr());
      if(pl->is_write()) {
        dir = 'W';
        addr = -1;
      } else {
        dir = 'R';
        *init_addr = *local_addr;
        os << cycle << " T: R data: " << *local_addr << endl;
      }
      tlm_phase ph = BEGIN_RESP;
      sc_time t = SC_ZERO_TIME;
      ocp->nb_transport_bw(*pl, ph, t);
      pl->release();
      os << cycle << " T: " << dir << " resp sent @ " << addr << endl;
    }

    // request side
    // accept in cycles 3,6,9,12,15,... and impose 4 cycle minimum latency
    if(scmdaccept) {
      if(mcmd != 0) {
        tlm_generic_payload *pl = mcmd;
        int addr = pl->get_address();
        if(pl->is_write()) {
          unsigned *local_addr = reinterpret_cast<unsigned *>(memory + addr);
          unsigned *init_addr = reinterpret_cast<unsigned *>(pl->get_data_ptr());
          *local_addr = *init_addr;
          os << cycle << " T: W data: " << *local_addr << endl;
          os << cycle << " T: W req received @ " << addr << endl;
        } else {
          os << cycle << " T: R req received @ " << addr << endl;
        }
        resp_cycle.push(cycle + 4);
        mcmd = 0;
      }
    }
    scmdaccept = ((cycle % 3) == 0);
    if(scmdaccept && (mcmd != 0)) {
      tlm_phase ph = END_REQ;
      sc_time t = SC_ZERO_TIME;
      ocp->nb_transport_bw(*mcmd, ph, t);
    }

    cycle++;
  }

  tlm_sync_enum nb_transport_fw(
    tlm_generic_payload &pl, tlm_phase &ph, sc_time &ta)
  {
    if(ph == END_RESET) {
      in_reset = false;
      while(rd_addr.size() > 0) {
        rd_addr.front()->release();
        rd_addr.pop();
      }
      return TLM_ACCEPTED;
    }
    if(ph == BEGIN_RESET) {
      in_reset = true;
      mcmd = 0;
      ocp.reset();
      while(resp_cycle.size() > 0) resp_cycle.pop();
      return TLM_ACCEPTED;
    }
    if(ph == BEGIN_REQ) {
      mcmd = &pl;
      rd_addr.push(&pl);
      pl.acquire();
      leak_test(pl);
      if(scmdaccept) {
        ph = END_REQ;
        return TLM_UPDATED;
      } else {
        return TLM_ACCEPTED;
      }
    }
    sc_assert(ph == END_RESP);
    return TLM_ACCEPTED;
  }

  unsigned cycle;
  bool in_reset;
  bool scmdaccept;
  tlm_generic_payload *mcmd;
  char memory[MEM_SIZE];
  queue<tlm_generic_payload *> rd_addr;
  queue<unsigned> resp_cycle;
  ofstream os;
  sc_time cmdT, dataT;

  void ocp_timing_update(ocp_tl1_master_timing times) {
    if((cmdT != times.RequestGrpStartTime) ||
       (dataT != times.DataHSGrpStartTime)) {
      cmdT = times.RequestGrpStartTime;
      dataT = times.DataHSGrpStartTime;
      set_slave_timing();
    }
  }

  void set_slave_timing() {
    ocp_tl1_slave_timing mytimes;
    mytimes.SCmdAcceptStartTime = cmdT + sc_get_time_resolution();
    mytimes.SDataAcceptStartTime = dataT + sc_get_time_resolution();
    ocp.set_slave_timing(mytimes);
  }
};


class testbench00: public sc_module {
public:
  testbench00(sc_module_name n):
    sc_module(n),
    Clk("Clk"),
    initiator("initiator"),
    target("target")
  {
    // clock
    initiator.Clk(Clk);
    target.Clk(Clk);

    // TL0 interface
    initiator.MAddr(MAddr);
    initiator.MCmd(MCmd);
    initiator.MData(MData);
    initiator.SData(SData);
    initiator.SCmdAccept(SCmdAccept);
    initiator.SResp(SResp);
    initiator.MReset_n(MReset_n);
    target.MAddr(MAddr);
    target.MCmd(MCmd);
    target.MData(MData);
    target.SData(SData);
    target.SCmdAccept(SCmdAccept);
    target.SResp(SResp);
    target.MReset_n(MReset_n);
  }

  sc_in<bool> Clk;

private:
  ocp_tl0_initiator initiator;
  ocp_tl0_target target;

  sc_signal<unsigned> MAddr;
  sc_signal<unsigned> MCmd;
  sc_signal<unsigned> MData;
  sc_signal<unsigned> SData;
  sc_signal<bool> SCmdAccept;
  sc_signal<unsigned> SResp;
  sc_signal<bool> MReset_n;
};


class testbench01: public sc_module {
public:
  testbench01(sc_module_name n):
    sc_module(n),
    Clk("Clk"),
    initiator("initiator"),
    target("target"),
    adapter("adapter")
  {
    // OCP configuration
    adapter.set_ocp_config(gen_ocp_config());

    // clock
    initiator.Clk(Clk);
    target.Clk(Clk);
    adapter.Clk(Clk);

    // TL0 interface
    initiator.MAddr(MAddr);
    initiator.MCmd(MCmd);
    initiator.MData(MData);
    initiator.SData(SData);
    initiator.SCmdAccept(SCmdAccept);
    initiator.SResp(SResp);
    initiator.MReset_n(MReset_n);
    adapter.MAddr(MAddr);
    adapter.MCmd(MCmd);
    adapter.MData(MData);
    adapter.SData(SData);
    adapter.SCmdAccept(SCmdAccept);
    adapter.SResp(SResp);
    adapter.MReset_n(MReset_n);

    // TL1 interface
    adapter.ocpTL1(target.ocp);
  }

  sc_in<bool> Clk;

private:
  ocp_tl0_initiator initiator;
  ocp_tl1_target target;
  tl0_initiator_to_tl1_target<32> adapter;

  sc_signal<unsigned> MAddr;
  sc_signal<unsigned> MCmd;
  sc_signal<unsigned> MData;
  sc_signal<unsigned> SData;
  sc_signal<bool> SCmdAccept;
  sc_signal<unsigned> SResp;
  sc_signal<bool> MReset_n;
};


class testbench10: public sc_module {
public:
  testbench10(sc_module_name n):
    sc_module(n),
    Clk("Clk"),
    initiator("initiator"),
    target("target"),
    adapter("adapter")
  {
    // OCP configuration
    adapter.set_ocp_config(gen_ocp_config());

    // clock
    initiator.Clk(Clk);
    target.Clk(Clk);
    adapter.Clk(Clk);

    // TL1 interface
    initiator.ocp(adapter.ocpTL1);

    // TL0 interface
    adapter.MAddr(MAddr);
    adapter.MCmd(MCmd);
    adapter.MData(MData);
    adapter.SData(SData);
    adapter.SCmdAccept(SCmdAccept);
    adapter.SResp(SResp);
    adapter.MReset_n(MReset_n);
    target.MAddr(MAddr);
    target.MCmd(MCmd);
    target.MData(MData);
    target.SData(SData);
    target.SCmdAccept(SCmdAccept);
    target.SResp(SResp);
    target.MReset_n(MReset_n);
  }

  sc_in<bool> Clk;

private:
  ocp_tl1_initiator initiator;
  ocp_tl0_target target;
  tl1_initiator_to_tl0_target<32> adapter;

  sc_signal<unsigned> MAddr;
  sc_signal<unsigned> MCmd;
  sc_signal<unsigned> MData;
  sc_signal<unsigned> SData;
  sc_signal<bool> SCmdAccept;
  sc_signal<unsigned> SResp;
  sc_signal<bool> MReset_n;
};


class testbench11: public sc_module {
public:
  testbench11(sc_module_name n):
    sc_module(n),
    Clk("Clk"),
    initiator("initiator"),
    target("target")
  {
    // clock
    initiator.Clk(Clk);
    target.Clk(Clk);

    // TL1 interface
    initiator.ocp(target.ocp);
  }

  sc_in<bool> Clk;

private:
  ocp_tl1_initiator initiator;
  ocp_tl1_target target;
};


class testbench101: public sc_module {
public:
  testbench101(sc_module_name n):
    sc_module(n),
    Clk("Clk"),
    initiator("initiator"),
    target("target"),
    t_print(&cout),
    adapter10("adapter10", &t_print),
    adapter01("adapter01", &t_print)
  {
    // OCP configuration
    adapter10.set_ocp_config(gen_ocp_config());
    adapter01.set_ocp_config(gen_ocp_config());

    // clock
    initiator.Clk(Clk);
    target.Clk(Clk);
    adapter10.Clk(Clk);
    adapter01.Clk(Clk);

    // TL1 interface
    initiator.ocp(adapter10.ocpTL1);
    adapter01.ocpTL1(target.ocp);

    // TL0 interface
    adapter10.MAddr(MAddr);
    adapter10.MCmd(MCmd);
    adapter10.MData(MData);
    adapter10.SData(SData);
    adapter10.SCmdAccept(SCmdAccept);
    adapter10.SResp(SResp);
    adapter10.MReset_n(MReset_n);
    adapter01.MAddr(MAddr);
    adapter01.MCmd(MCmd);
    adapter01.MData(MData);
    adapter01.SData(SData);
    adapter01.SCmdAccept(SCmdAccept);
    adapter01.SResp(SResp);
    adapter01.MReset_n(MReset_n);

    // timing:  TL1 initiator and target produce transport calls at +0 ps
    //   but they aren't seen by adapters until +1 ps because of synch protection
    //   and only at this time are TL0 signals written
    adapter01.set_sample_times(
      sc_time(2.0, SC_PS),
      // adapter10 writes MCmd always during TLM-transport-from-initiator at +1 ps
      sc_time(3.0, SC_PS)
      // same for MDataValid but sampling is delayed to be after MCmd sampling,
      // which would be automatic but shown here for clarity
    );
    adapter10.set_sample_times(
      sc_time(2.0, SC_PS),
      // adapter01 writes SResp always during TLM-transport-from-target at +1 ps
      sc_time(5.0, SC_PS),
      // TL1 target may signal SCmdAccept during its clocked process with an
      // explicit transport call.  This would be seen at +1 ps by adapter01 and
      // the TL0 SCmdAccept signal would be stable for sampling at +2 ps.  But
      // TL1 target might return SCmdAccept to a fw transport call, which it
      // would see at the MCmd sampling time of adapter01 plus 1, ie 3 ps.  The
      // socket PEQ will convert this into a separate backward transport call
      // seen by adapter01 at 4 ps.  Thus SCmdAccept should be sampled at 5 ps.
      sc_time(6.0, SC_PS)
      // data trails command by 1 ps
    );
  }

  sc_in<bool> Clk;

private:
  ocp_tl1_initiator initiator;
  ocp_tl1_target target;

  tl1_tl0::get_TL0_timing_example<ostream> t_print;
  tl1_initiator_to_tl0_target<32> adapter10;
  tl0_initiator_to_tl1_target<32> adapter01;

  sc_signal<unsigned> MAddr;
  sc_signal<unsigned> MCmd;
  sc_signal<unsigned> MData;
  sc_signal<unsigned> SData;
  sc_signal<bool> SCmdAccept;
  sc_signal<unsigned> SResp;
  sc_signal<bool> MReset_n;
};


class testbench010: public sc_module {
public:
  testbench010(sc_module_name n):
    sc_module(n),
    Clk("Clk"),
    initiator("initiator"),
    target("target"),
    adapter01("adapter01"),
    adapter10("adapter10")
  {
    // OCP configuration
    adapter01.set_ocp_config(gen_ocp_config());
    adapter10.set_ocp_config(gen_ocp_config());

    // clock
    initiator.Clk(Clk);
    target.Clk(Clk);
    adapter01.Clk(Clk);
    adapter10.Clk(Clk);

    // TL1 interface
    adapter01.ocpTL1(adapter10.ocpTL1);

    // TL0 interfaces
    initiator.MAddr(MAddr[0]);
    initiator.MCmd(MCmd[0]);
    initiator.MData(MData[0]);
    initiator.SData(SData[0]);
    initiator.SCmdAccept(SCmdAccept[0]);
    initiator.SResp(SResp[0]);
    initiator.MReset_n(MReset_n[0]);
    adapter01.MAddr(MAddr[0]);
    adapter01.MCmd(MCmd[0]);
    adapter01.MData(MData[0]);
    adapter01.SData(SData[0]);
    adapter01.SCmdAccept(SCmdAccept[0]);
    adapter01.SResp(SResp[0]);
    adapter01.MReset_n(MReset_n[0]);

    adapter10.MAddr(MAddr[1]);
    adapter10.MCmd(MCmd[1]);
    adapter10.MData(MData[1]);
    adapter10.SData(SData[1]);
    adapter10.SCmdAccept(SCmdAccept[1]);
    adapter10.SResp(SResp[1]);
    adapter10.MReset_n(MReset_n[1]);
    target.MAddr(MAddr[1]);
    target.MCmd(MCmd[1]);
    target.MData(MData[1]);
    target.SData(SData[1]);
    target.SCmdAccept(SCmdAccept[1]);
    target.SResp(SResp[1]);
    target.MReset_n(MReset_n[1]);

    // timing: nothing needed as adapters' TL0 interfaces are
    //  directly connected to TL0-native initiator and target
  }

  sc_in<bool> Clk;

private:
  ocp_tl0_initiator initiator;
  ocp_tl0_target target;
  tl0_initiator_to_tl1_target<32> adapter01;
  tl1_initiator_to_tl0_target<32> adapter10;

  sc_signal<unsigned> MAddr[2];
  sc_signal<unsigned> MCmd[2];
  sc_signal<unsigned> MData[2];
  sc_signal<unsigned> SData[2];
  sc_signal<bool> SCmdAccept[2];
  sc_signal<unsigned> SResp[2];
  sc_signal<bool> MReset_n[2];
};


class testbench: public sc_module {
public:
  testbench(sc_module_name n, bool run_01 = true, bool run_10 = true):
    sc_module(n),
    tb00("00"),
    tb01("01"),
    tb10("10"),
    tb11("11"),
    tb101("101"),
    tb010("010"),
    Clk("Clk", sc_time(10.0, SC_NS))
  {
    tb00.Clk(Clk);
    tb11.Clk(Clk);

    if(run_01) tb01.Clk(Clk);
    else tb01.Clk(tieoff);

    if(run_10) tb10.Clk(Clk);
    else tb10.Clk(tieoff);

    if(run_01 && run_10) {
      tb010.Clk(Clk);
      tb101.Clk(Clk);
    } else {
      tb101.Clk(tieoff);
      tb010.Clk(tieoff);
    }
  }
private:
  testbench00 tb00;
  testbench01 tb01;
  testbench10 tb10;
  testbench11 tb11;
  testbench101 tb101;
  testbench010 tb010;
  sc_clock Clk;
  sc_signal<bool> tieoff;
};


#ifndef OMIT_SC_MAIN
int sc_main(int argc, char **argv) {
  sc_report_handler::set_actions(SC_ERROR, SC_DISPLAY | SC_ABORT);
  bool run_01 = ((argc < 2) || ('A' == *argv[1]) || ('0' == *argv[1]));
  bool run_10 = ((argc < 2) || ('A' == *argv[1]) || ('1' == *argv[1]));
  testbench theTestbench("testbench", run_01, run_10);
  sc_start();
  return 0;
}
#endif

