///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// (c) Copyright OCP-IP 2009-2011
// OCP-IP Confidential and Proprietary
//
//
//============================================================================
//      Project : OCP SLD WG
//       Author : James Aldis, Texas Instruments
//
//          $Id:
//
//  Description :  Testbench for TL1/TL0 adapters
//
// TL1 and TL0 initiators and targets are created with identical behaviour for
// the different abstraction levels.
// Six simulations are run, 4 with adapters and 2 without, for 0, 1 and 2
//   cascaded adapters.
// All sets of results should be the same, allowing for a few ps of timing
// delay in some cases.
//
//                                                                           //
///////////////////////////////////////////////////////////////////////////////


#include "ocpip_adapters.h"
#include <iostream>
#include <queue>


using namespace std;
using namespace ocpip;
using namespace sc_core;
using namespace tlm;


// all OCPs in this testbench have the same configuration
ocp_parameters gen_ocp_config() {
  ocp_parameters p;
  p.writeresp_enable = false;
  p.addr_wdth = 32;
  p.data_wdth = 32;
  p.mreset = true;
  p.writenonpost_enable = true;
  p.rdlwrc_enable = true;
  p.readex_enable = true;
  return p;
}


// simulation duration determined by this parameter
#define MEM_SIZE 4001
#include "tb_common.h"


class ocp_tl0_initiator: public sc_module {
SC_HAS_PROCESS(ocp_tl0_initiator);
public:
  ocp_tl0_initiator(sc_module_name n):
    sc_module(n),
    Clk("Clk"),
    MAddr("MAddr"),
    MCmd("MCmd"),
    MData("MData"),
    SData("SData"),
    SCmdAccept("SCmdAccept"),
    SResp("SResp"),
    MReset_n("MReset_n"),
    cycle(0),
    reqs_sent(0),
    rs_reqs_sent(0),
    next_req_cycle(1),
    resps_rxd(0),
    req_active(false),
    last_rdl_addr(0),
    reqs_sent_at_last_sema(0),
    last_cmd(WR),
    os(sc_module::name())
  {
    SC_METHOD(on_Clk_rising);
    dont_initialize();
    sensitive << Clk.pos();

    MReset_n.initialize(true);
    MCmd.initialize(0);

    fill_initiator_memory(memory);
  }

  sc_in<bool> Clk;
  sc_out<unsigned> MAddr;
  sc_out<unsigned> MCmd;
  sc_out<unsigned> MData;
  sc_in<unsigned> SData;
  sc_in<bool> SCmdAccept;
  sc_in<unsigned> SResp;
  sc_out<bool> MReset_n;

private:
  enum cmd {WR = 1, RD = 2, RDEX = 3, RDL = 4, WRNP = 5, WRC = 6};

  void on_Clk_rising() {
    if((reqs_sent * 4 + 3 >= MEM_SIZE) && (resps_rxd == rs_reqs_sent)) {
      stop_sim();
      return;
    }

    // request side
    if(req_active && SCmdAccept)
      os << cycle << " I: req accepted " << reqs_sent << endl;

    if(SCmdAccept || !req_active) {
      if((cycle >= next_req_cycle) && (reqs_sent * 4 + 3 < MEM_SIZE)) {
        // incrementing word addresses
        unsigned addr = reqs_sent * 4;
        char dir;
        bool wr;

        // select commands fairly randomly
        if(last_cmd == RDEX) {
          last_cmd = WR;
          dir = 'W';
          wr = true;
        } else {
          unsigned sel = (cycle ^ (cycle >> 4)) %
            (resps_rxd >= reqs_sent_at_last_sema ? 5 : 4);
            // wait for semaphore response before doing next one
            // (because of RD/WR ordering error in this testbench)
          switch(sel) {
            case 0: last_cmd = RD; dir = 'R'; wr = false; break;
            case 1: last_cmd = WR; dir = 'W'; wr = true; break;
            case 2: last_cmd = RDEX; dir = 'X'; wr = false; break;
            case 3: last_cmd = WRNP; dir = 'N'; wr = true; break;
            case 4:
              reqs_sent_at_last_sema = rs_reqs_sent + 1;
              if(last_rdl_addr == 0) {
                last_cmd = RDL;
                last_rdl_addr = addr;
                dir = 'L';
                wr = false;
              } else {
                last_cmd = WRC;
                addr = last_rdl_addr;
                last_rdl_addr = 0;
                dir = 'C';
                wr = true;
              }
              break;
          }
        }

        MCmd = last_cmd;
        MAddr = addr;

        if(wr) {
          unsigned d = *reinterpret_cast<unsigned *>(memory + addr);
          MData = d;
          os << cycle << " I: W data: " <<  d << endl;
        }
        if(last_cmd != WR) {
          rd_addr.push(wr ? -1 : addr);
          rs_reqs_sent++;
        }
        reqs_sent++;
        req_active = true;
        os << cycle << " I: " << dir << " req sent @ " << addr << endl;
        // wait 1,2,4,8,16,1,2,4,8,... cycles between requests
        next_req_cycle = cycle + (1 << (reqs_sent & 7));
      } else {
        MCmd = 0;
        req_active = false;
      }
    }

    // response side
    unsigned rs = SResp;
    if(rs != 0) {
      char rsc = (rs == 1 ? 'D' : (rs == 2 ? 'F' : 'E'));
      resps_rxd++;
      int addr = rd_addr.front();
      rd_addr.pop();
      char dir = 'W';
      if(addr >= 0) {
        // abandon semaphore if any RD gets an error, including the RDL
        if(rsc != 'D') last_rdl_addr = 0;
        else {
          *reinterpret_cast<unsigned *>(memory + addr) = SData;
          os << cycle << " I: R data: " <<  SData << endl;
        }
        dir = 'R';
      }
      os << cycle << " I: " << dir << " resp " << rsc << " received @ " << addr << endl;
    }
    cycle++;
  }

  char memory[MEM_SIZE];
  unsigned cycle;
  unsigned reqs_sent, rs_reqs_sent, next_req_cycle, resps_rxd;
  bool req_active;
  unsigned last_rdl_addr, reqs_sent_at_last_sema, last_cmd;
  queue<int> rd_addr;
  ofstream os;
};


class ocp_tl1_initiator: public sc_module {
SC_HAS_PROCESS(ocp_tl1_initiator);
public:
  ocp_tl1_initiator(sc_module_name n):
    sc_module(n),
    Clk("Clk"),
    ocp("ocp", this, &ocp_tl1_initiator::ocp_timing_update),
    cycle(0),
    reqs_sent(0),
    rs_reqs_sent(0),
    next_req_cycle(1),
    resps_rxd(0),
    req_active(false),
    last_rdl_addr(0),
    reqs_sent_at_last_sema(0),
    last_cmd_rdex(0),
    scmdaccept(false),
    sresp(0),
    os(sc_module::name()),
    srespT(SC_ZERO_TIME)
  {
    SC_METHOD(on_Clk_rising);
    dont_initialize();
    sensitive << Clk.pos();

    ocp.set_ocp_config(gen_ocp_config());
    ocp.activate_synchronization_protection();
    ocp.register_nb_transport_bw(this, &ocp_tl1_initiator::nb_transport_bw);

    fill_initiator_memory(memory);
  }

  sc_in<bool> Clk;
  ocp_master_socket_tl1<32> ocp;

private:
  void on_Clk_rising() {
    if((reqs_sent * 4 + 3 >= MEM_SIZE) && (resps_rxd == rs_reqs_sent)) {
      stop_sim();
      return;
    }

    // request side
    if(req_active && scmdaccept)
      os << cycle << " I: req accepted " << reqs_sent << endl;

    if(scmdaccept || !req_active) {
      if((cycle >= next_req_cycle) && (reqs_sent * 4 + 3 < MEM_SIZE)) {
        tlm_generic_payload *pl = ocp.get_transaction();
        leak_test(*pl);
        pl->set_data_length(4);
        pl->set_streaming_width(4);
        pl->set_byte_enable_ptr(0);
        pl->set_response_status(TLM_INCOMPLETE_RESPONSE);
        ocp.invalidate_extension<posted>(*pl);
        ocp.invalidate_extension<lock>(*pl);
        ocp.invalidate_extension<semaphore>(*pl);
        // incrementing word addresses
        unsigned addr = reqs_sent * 4;
        char dir;

        // select commands fairly randomly
        if(last_cmd_rdex != 0) {
          pl->set_write();
          ocp.validate_extension<posted>(*pl);
          ocp.validate_extension<lock>(*pl);
          lock *lk;
          ocp.get_extension<lock>(lk, *pl);
          lk->value = last_cmd_rdex;
          dir = 'W';
          last_cmd_rdex = 0;
        } else {
          last_cmd_rdex = 0;
          unsigned sel = (cycle ^ (cycle >> 4)) %
            (resps_rxd >= reqs_sent_at_last_sema ? 5 : 4);
            // wait for semaphore response before doing next one
            // (because of RD/WR ordering error in this testbench)
          switch(sel) {
            case 0:
              pl->set_read();
              dir = 'R';
              break;
            case 1:
              pl->set_write();
              ocp.validate_extension<posted>(*pl);
              dir = 'W';
              break;
            case 2:
              pl->set_read();
              dir = 'X';
              ocp.validate_extension<lock>(*pl);
              lock *lk;
              ocp.get_extension<lock>(lk, *pl);
              lk->value = lock_object::pool();
              lk->value->number_of_txns = 2;
              last_cmd_rdex = lk->value;
              break;
            case 3:
              pl->set_write();
              dir = 'N';
              break;
            case 4:
              reqs_sent_at_last_sema = rs_reqs_sent + 1;
              ocp.validate_extension<semaphore>(*pl);
              if(last_rdl_addr == 0) {
                pl->set_read();
                last_rdl_addr = addr;
                dir = 'L';
              } else {
                semaphore *sf;
                ocp.get_extension<semaphore>(sf, *pl);
                sf->value = false;
                pl->set_write();
                addr = last_rdl_addr;
                last_rdl_addr = 0;
                dir = 'C';
              }
              break;
          }
        }

        pl->set_address(addr);
        pl->set_data_ptr((unsigned char*)memory + addr);

        if(pl->is_write()) {
          unsigned d = *reinterpret_cast<unsigned *>(memory + addr);
          os << cycle << " I: W data: " <<  d << endl;
        }
        if(dir != 'W') {
          rd_addr.push(pl->is_write() ? -1 : addr);
          mcmd.push(dir);
          rs_reqs_sent++;
        }
        os << cycle << " I: " << dir << " req sent @ " << addr << endl;

        tlm_phase ph = BEGIN_REQ;
        sc_time t = SC_ZERO_TIME;
        if(ocp->nb_transport_fw(*pl, ph, t) == TLM_UPDATED) {
          sc_assert(ph == END_REQ);
          scmdaccept = true;
        } else {
          scmdaccept = false;
        }
        if(dir == 'W') ocp.release_transaction(pl);
        reqs_sent++;
        req_active = true;
        // wait 1,2,4,8,16,1,2,4,8,... cycles between requests
        next_req_cycle = cycle + (1 << (reqs_sent & 7));
      } else {
        req_active = false;
      }
    }

    // response side
    if(sresp != 0) {
      unsigned mc = mcmd.front();

      char rsc = 'D';
      if(mc == 'C') {
        semaphore *sf;
        ocp.get_extension<semaphore>(sf, *sresp);
        if(sf->value) rsc = 'F';
      }
      if((rsc == 'D') && (sresp->get_response_status() != TLM_OK_RESPONSE))
        rsc = 'E';

      if(mc == 'X') {
        lock *lk;
        ocp.get_extension<lock>(lk, *sresp);
        sc_assert(lk->value->lock_is_understood_by_slave);
        lk->value->atomic_txn_completed();
      }

      resps_rxd++;
      int addr = rd_addr.front();
      rd_addr.pop();
      mcmd.pop();

      char dir = 'W';
      if(addr >= 0) {
        // abandon semaphore if any RD gets an error, including the RDL
        if(rsc != 'D') last_rdl_addr = 0;
        else {
          unsigned d = *reinterpret_cast<unsigned *>(memory + addr);
          os << cycle << " I: R data: " << d << endl;
        }
        dir = 'R';
      }
      os << cycle << " I: " << dir << " resp " << rsc << " received @ " << addr << endl;
      ocp.release_transaction(sresp);
      sresp = 0;
    }
    cycle++;
  }

  tlm_sync_enum nb_transport_bw(
    tlm_generic_payload &pl, tlm_phase &ph, sc_time &ta)
  {
    if(ph == END_REQ) {
      scmdaccept = true;
      return TLM_ACCEPTED;
    }
    sc_assert(ph == BEGIN_RESP);
    sresp = &pl;
    ph = END_RESP;
    return TLM_UPDATED;
  }

  char memory[MEM_SIZE];
  unsigned cycle;
  unsigned reqs_sent, rs_reqs_sent, next_req_cycle, resps_rxd;
  bool req_active;
  queue<int> rd_addr;
  queue<char> mcmd;
  unsigned last_rdl_addr, reqs_sent_at_last_sema;
  lock_object_base *last_cmd_rdex;
  bool scmdaccept;
  tlm_generic_payload *sresp;
  ofstream os;
  sc_time srespT;

  // not sure if this is redundant when there is no resp-accept in the OCP
  // configuration.  It still exists in the TLM2
  void ocp_timing_update(ocp_tl1_slave_timing times) {
    if(srespT != times.ResponseGrpStartTime) {
      srespT = times.ResponseGrpStartTime;
      set_master_timing();
    }
  }

  void set_master_timing() {
    ocp_tl1_master_timing mytimes;
    mytimes.MRespAcceptStartTime = srespT + sc_get_time_resolution();
    ocp.set_master_timing(mytimes);
  }

  struct lock_object: lock_object_base {
    lock_object(): next(0), count(0) {}
    static lock_object *pool(lock_object *used = 0) {
      static lock_object *pool = 0;
      if(used == 0) {
        if(pool == 0) {pool = new lock_object; pool->next = 0;}
        lock_object *rv = pool;
        pool = pool->next;
        return rv;
      } else {
        used->next = pool;
        pool = used;
        return 0;
      }
    }
    // both target and initiator have to liberate the lock_object
    // before it can be reused
    void atomic_txn_completed() {if(2 == ++count) pool(this);}
    lock_object *next;
    unsigned count;
  };
};


class ocp_tl0_target: public sc_module {
SC_HAS_PROCESS(ocp_tl0_target);
public:
  ocp_tl0_target(sc_module_name n):
    sc_module(n),
    Clk("Clk"),
    MAddr("MAddr"),
    MCmd("MCmd"),
    MData("MData"),
    SData("SData"),
    SCmdAccept("SCmdAccept"),
    SResp("SResp"),
    MReset_n("MReset_n"),
    cycle(0),
    os(sc_module::name())
  {
    SC_METHOD(on_Clk_rising);
    dont_initialize();
    sensitive << Clk.pos() << MReset_n.neg();

    SResp.initialize(0);
    SCmdAccept.initialize(false);

    fill_target_memory(memory);
  }

  sc_in<bool> Clk;
  sc_in<unsigned> MAddr;
  sc_in<unsigned> MCmd;
  sc_in<unsigned> MData;
  sc_out<unsigned> SData;
  sc_out<bool> SCmdAccept;
  sc_out<unsigned> SResp;
  sc_in<bool> MReset_n;

private:
  enum cmd {WR = 1, RD = 2, RDEX = 3, RDL = 4, WRNP = 5, WRC = 6};

  void on_Clk_rising() {
    if(!MReset_n) {
      // reset all state
      while(resp_cycle.size() > 0) {
        resp_cycle.pop();
        rd_addr.pop();
      }
      SResp = 0;
    } else {
      // response side
      if((resp_cycle.size() > 0) && (cycle >= resp_cycle.front())) {
        resp_cycle.pop();
        unsigned rs = resp.front();
        SResp = rs;
        resp.pop();
        int addr = rd_addr.front();
        rd_addr.pop();
        char dir;
        if(addr < 0) {
          dir = 'W';
        } else {
          dir = 'R';
          unsigned d = 0;
          if(rs == 1) d = *reinterpret_cast<unsigned *>(memory + addr);
          SData = d;
          os << cycle << " T: R data: " << d << endl;
        }
        os << cycle << " T: " << dir << " " << rs << " resp sent @ " << addr << endl;
      } else {
        SResp = 0;
      }

      // request side
      // accept in cycles 3,6,9,12,15,... and impose 4 cycle minimum latency
      unsigned mcmd = MCmd;
      if((SCmdAccept) && (mcmd != 0)) {
        unsigned rs = (((cycle >> 5) ^ cycle) & 0xa5 ? 1 : 3); // DVA or ERR
        switch(mcmd) {
        case WRNP:
          if(rs == 1)
            *reinterpret_cast<unsigned *>(memory + MAddr) = unsigned(MData);
          os << cycle << " T: W data: " <<  unsigned(MData) << endl;
          os << cycle << " T: WNP req received @ " << MAddr << endl;
          rd_addr.push(-1);
          resp.push(rs);
          resp_cycle.push(cycle + 4);
          break;
        case WRC:
          if((cycle & 127) < 8) rs = 2;  // sempahore FAIL case
          if(rs == 1)
            *reinterpret_cast<unsigned *>(memory + MAddr) = unsigned(MData);
          os << cycle << " T: W data: " <<  unsigned(MData) << endl;
          os << cycle << " T: WRC req received @ " << MAddr << endl;
          rd_addr.push(-1);
          resp.push(rs);
          resp_cycle.push(cycle + 4);
          break;
        case WR:
          if(rs == 1)
            *reinterpret_cast<unsigned *>(memory + MAddr) = unsigned(MData);
          os << cycle << " T: W data: " <<  unsigned(MData) << endl;
          os << cycle << " T: W req received @ " << MAddr << endl;
          break;
        case RD:
          os << cycle << " T: R req received @ " << MAddr << endl;
          rd_addr.push(MAddr);
          resp.push(rs);
          resp_cycle.push(cycle + 4);
          break;
        case RDEX:
          os << cycle << " T: RX req received @ " << MAddr << endl;
          rd_addr.push(MAddr);
          resp.push(rs);
          resp_cycle.push(cycle + 4);
          break;
        case RDL:
          os << cycle << " T: RL req received @ " << MAddr << endl;
          rd_addr.push(MAddr);
          resp.push(rs);
          resp_cycle.push(cycle + 4);
          break;
        }
      }
      SCmdAccept = ((cycle % 3) == 0);

      cycle++;
    }
  }

  unsigned cycle;
  char memory[MEM_SIZE];
  queue<int> rd_addr;
  queue<unsigned> resp_cycle, resp;
  ofstream os;
};


class ocp_tl1_target: public sc_module {
SC_HAS_PROCESS(ocp_tl1_target);
public:
  ocp_tl1_target(sc_module_name n):
    sc_module(n),
    Clk("Clk"),
    ocp("ocp", this, &ocp_tl1_target::ocp_timing_update),
    cycle(0),
    in_reset(false),
    scmdaccept(false),
    mcmd(0),
    os(sc_module::name()),
    cmdT(SC_ZERO_TIME),
    dataT(SC_ZERO_TIME)
  {
    SC_METHOD(on_Clk_rising);
    dont_initialize();
    sensitive << Clk.pos();

    ocp.set_ocp_config(gen_ocp_config());
    ocp.activate_synchronization_protection();
    ocp.register_nb_transport_fw(this, &ocp_tl1_target::nb_transport_fw);

    fill_target_memory(memory);
  }

  sc_in<bool> Clk;
  ocp_slave_socket_tl1<32> ocp;

private:
  void on_Clk_rising() {
    if(in_reset) return;

    // response side
    if((resp_cycle.size() > 0) && (cycle >= resp_cycle.front())) {
      resp_cycle.pop();
      tlm_generic_payload *pl = rd_addr.front();
      int addr = pl->get_address();
      unsigned rs = (pl->get_response_status() == TLM_OK_RESPONSE ? 1 : 3);
      char dir;
      if(pl->is_write()) {
        dir = 'W';
        addr = -1;
        semaphore *sf;
        if((ocp.get_extension<semaphore>(sf, *pl) != 0) && sf->value) rs = 2;
      } else {
        dir = 'R';
        unsigned *local_addr = reinterpret_cast<unsigned *>(memory + addr);
        unsigned *init_addr = reinterpret_cast<unsigned *>(pl->get_data_ptr());
        if(rs == 1) *init_addr = *local_addr;
        os << cycle << " T: R data: " << (rs == 1 ? *local_addr : 0) << endl;
      }
      tlm_phase ph = BEGIN_RESP;
      sc_time t = SC_ZERO_TIME;
      ocp->nb_transport_bw(*pl, ph, t);
      rd_addr.pop();
      pl->release();
      os << cycle << " T: " << dir << " " << rs << " resp sent @ " << addr << endl;
    }

    // request side
    // accept in cycles 3,6,9,12,15,... and impose 4 cycle minimum latency
    if(scmdaccept) {
      if(mcmd != 0) {
        // DVA or ERR
        mcmd->set_response_status(((cycle >> 5) ^ cycle) & 0xa5 ?
          TLM_OK_RESPONSE : TLM_GENERIC_ERROR_RESPONSE);
        int addr = mcmd->get_address();

        // 6 different commands can be received and we need to detect unlocking WR
        bool has_resp = true;
        if(mcmd->is_write()) {
          unsigned *local_addr = reinterpret_cast<unsigned *>(memory + addr);
          unsigned *init_addr = reinterpret_cast<unsigned *>(mcmd->get_data_ptr());
          os << cycle << " T: W data: " << *init_addr << endl;
          lock *lk;
          if(ocp.get_extension<lock>(lk, *mcmd)) {
            // unlocking WR, may be posted or non-posted in general
            // can assume OCP protocol has been adhered to
            lk->value->atomic_txn_completed();
          }
          semaphore *sf;
          if(ocp.get_extension<semaphore>(sf, *mcmd)) {
            // WRC
            // sempahore FAIL case
            if((cycle & 127) < 8) sf->value = true;
            else if(mcmd->get_response_status() == TLM_OK_RESPONSE)
              *local_addr = *init_addr;
            os << cycle << " T: WRC req received @ " << addr << endl;
          } else {
            if(mcmd->get_response_status() == TLM_OK_RESPONSE)
              *local_addr = *init_addr;
            if(ocp.get_extension<posted>(*mcmd) == 0) {
              // WRNP
              os << cycle << " T: WNP req received @ " << addr << endl;
            } else {
              // WR
              has_resp = false;
              os << cycle << " T: W req received @ " << addr << endl;
            }
          }
        } else {
          // is a read
          lock *lk;
          if(ocp.get_extension<lock>(lk, *mcmd)) {
            // RDEX
            lk->value->lock_is_understood_by_slave = true;
            os << cycle << " T: RX req received @ " << addr << endl;
          } else {
            semaphore *sf;
            if(ocp.get_extension<semaphore>(sf, *mcmd)) {
              // RDL
              os << cycle << " T: RL req received @ " << addr << endl;
            } else {
              // RD
              os << cycle << " T: R req received @ " << addr << endl;
            }
          }
        }
        if(has_resp) {
          rd_addr.push(mcmd);
          resp_cycle.push(cycle + 4);
        } else  mcmd->release();
        mcmd = 0;
      }
    }
    scmdaccept = ((cycle % 3) == 0);
    if(scmdaccept && (mcmd != 0)) {
      tlm_phase ph = END_REQ;
      sc_time t = SC_ZERO_TIME;
      ocp->nb_transport_bw(*mcmd, ph, t);
    }

    cycle++;
  }

  tlm_sync_enum nb_transport_fw(
    tlm_generic_payload &pl, tlm_phase &ph, sc_time &ta)
  {
    if(ph == END_RESET) {
      in_reset = false;
      while(rd_addr.size() > 0) {
        rd_addr.front()->release();
        rd_addr.pop();
      }
      return TLM_ACCEPTED;
    }
    if(ph == BEGIN_RESET) {
      in_reset = true;
      mcmd = 0;
      ocp.reset();
      while(resp_cycle.size() > 0) resp_cycle.pop();
      return TLM_ACCEPTED;
    }
    if(ph == BEGIN_REQ) {
      mcmd = &pl;
      pl.acquire();
      leak_test(pl);
      if(scmdaccept) {
        ph = END_REQ;
        return TLM_UPDATED;
      } else {
        return TLM_ACCEPTED;
      }
    }
    sc_assert(ph == END_RESP);
    return TLM_ACCEPTED;
  }

  unsigned cycle;
  bool in_reset;
  bool scmdaccept;
  tlm_generic_payload *mcmd;
  char memory[MEM_SIZE];
  queue<tlm_generic_payload *> rd_addr;
  queue<unsigned> resp_cycle;
  ofstream os;
  sc_time cmdT, dataT;

  void ocp_timing_update(ocp_tl1_master_timing times) {
    if((cmdT != times.RequestGrpStartTime) ||
       (dataT != times.DataHSGrpStartTime)) {
      cmdT = times.RequestGrpStartTime;
      dataT = times.DataHSGrpStartTime;
      set_slave_timing();
    }
  }

  void set_slave_timing() {
    ocp_tl1_slave_timing mytimes;
    mytimes.SCmdAcceptStartTime = cmdT + sc_get_time_resolution();
    mytimes.SDataAcceptStartTime = dataT + sc_get_time_resolution();
    ocp.set_slave_timing(mytimes);
  }
};


class testbench00: public sc_module {
public:
  testbench00(sc_module_name n):
    sc_module(n),
    Clk("Clk"),
    initiator("initiator"),
    target("target")
  {
    // clock
    initiator.Clk(Clk);
    target.Clk(Clk);

    // TL0 interface
    initiator.MAddr(MAddr);
    initiator.MCmd(MCmd);
    initiator.MData(MData);
    initiator.SData(SData);
    initiator.SCmdAccept(SCmdAccept);
    initiator.SResp(SResp);
    initiator.MReset_n(MReset_n);
    target.MAddr(MAddr);
    target.MCmd(MCmd);
    target.MData(MData);
    target.SData(SData);
    target.SCmdAccept(SCmdAccept);
    target.SResp(SResp);
    target.MReset_n(MReset_n);
  }

  sc_in<bool> Clk;

private:
  ocp_tl0_initiator initiator;
  ocp_tl0_target target;

  sc_signal<unsigned> MAddr;
  sc_signal<unsigned> MCmd;
  sc_signal<unsigned> MData;
  sc_signal<unsigned> SData;
  sc_signal<bool> SCmdAccept;
  sc_signal<unsigned> SResp;
  sc_signal<bool> MReset_n;
};


class testbench01: public sc_module {
public:
  testbench01(sc_module_name n):
    sc_module(n),
    Clk("Clk"),
    initiator("initiator"),
    target("target"),
    adapter("adapter")
  {
    // OCP configuration
    adapter.set_ocp_config(gen_ocp_config());

    // clock
    initiator.Clk(Clk);
    target.Clk(Clk);
    adapter.Clk(Clk);

    // TL0 interface
    initiator.MAddr(MAddr);
    initiator.MCmd(MCmd);
    initiator.MData(MData);
    initiator.SData(SData);
    initiator.SCmdAccept(SCmdAccept);
    initiator.SResp(SResp);
    initiator.MReset_n(MReset_n);
    adapter.MAddr(MAddr);
    adapter.MCmd(MCmd);
    adapter.MData(MData);
    adapter.SData(SData);
    adapter.SCmdAccept(SCmdAccept);
    adapter.SResp(SResp);
    adapter.MReset_n(MReset_n);

    // TL1 interface
    adapter.ocpTL1(target.ocp);
  }

  sc_in<bool> Clk;

private:
  ocp_tl0_initiator initiator;
  ocp_tl1_target target;
  tl0_initiator_to_tl1_target<32> adapter;

  sc_signal<unsigned> MAddr;
  sc_signal<unsigned> MCmd;
  sc_signal<unsigned> MData;
  sc_signal<unsigned> SData;
  sc_signal<bool> SCmdAccept;
  sc_signal<unsigned> SResp;
  sc_signal<bool> MReset_n;
};


class testbench10: public sc_module {
public:
  testbench10(sc_module_name n):
    sc_module(n),
    Clk("Clk"),
    initiator("initiator"),
    target("target"),
    adapter("adapter")
  {
    // OCP configuration
    adapter.set_ocp_config(gen_ocp_config());

    // clock
    initiator.Clk(Clk);
    target.Clk(Clk);
    adapter.Clk(Clk);

    // TL1 interface
    initiator.ocp(adapter.ocpTL1);

    // TL0 interface
    adapter.MAddr(MAddr);
    adapter.MCmd(MCmd);
    adapter.MData(MData);
    adapter.SData(SData);
    adapter.SCmdAccept(SCmdAccept);
    adapter.SResp(SResp);
    adapter.MReset_n(MReset_n);
    target.MAddr(MAddr);
    target.MCmd(MCmd);
    target.MData(MData);
    target.SData(SData);
    target.SCmdAccept(SCmdAccept);
    target.SResp(SResp);
    target.MReset_n(MReset_n);
  }

  sc_in<bool> Clk;

private:
  ocp_tl1_initiator initiator;
  ocp_tl0_target target;
  tl1_initiator_to_tl0_target<32> adapter;

  sc_signal<unsigned> MAddr;
  sc_signal<unsigned> MCmd;
  sc_signal<unsigned> MData;
  sc_signal<unsigned> SData;
  sc_signal<bool> SCmdAccept;
  sc_signal<unsigned> SResp;
  sc_signal<bool> MReset_n;
};


class testbench11: public sc_module {
public:
  testbench11(sc_module_name n):
    sc_module(n),
    Clk("Clk"),
    initiator("initiator"),
    target("target")
  {
    // clock
    initiator.Clk(Clk);
    target.Clk(Clk);

    // TL1 interface
    initiator.ocp(target.ocp);
  }

  sc_in<bool> Clk;

private:
  ocp_tl1_initiator initiator;
  ocp_tl1_target target;
};


class testbench101: public sc_module {
public:
  testbench101(sc_module_name n):
    sc_module(n),
    Clk("Clk"),
    initiator("initiator"),
    target("target"),
    t_print(&cout),
    adapter10("adapter10", &t_print),
    adapter01("adapter01", &t_print)
  {
    // OCP configuration
    adapter10.set_ocp_config(gen_ocp_config());
    adapter01.set_ocp_config(gen_ocp_config());

    // clock
    initiator.Clk(Clk);
    target.Clk(Clk);
    adapter10.Clk(Clk);
    adapter01.Clk(Clk);

    // TL1 interface
    initiator.ocp(adapter10.ocpTL1);
    adapter01.ocpTL1(target.ocp);

    // TL0 interface
    adapter10.MAddr(MAddr);
    adapter10.MCmd(MCmd);
    adapter10.MData(MData);
    adapter10.SData(SData);
    adapter10.SCmdAccept(SCmdAccept);
    adapter10.SResp(SResp);
    adapter10.MReset_n(MReset_n);
    adapter01.MAddr(MAddr);
    adapter01.MCmd(MCmd);
    adapter01.MData(MData);
    adapter01.SData(SData);
    adapter01.SCmdAccept(SCmdAccept);
    adapter01.SResp(SResp);
    adapter01.MReset_n(MReset_n);

    // timing:  TL1 initiator and target produce transport calls at +0 ps
    //   but they aren't seen by adapters until +1 ps because of synch protection
    //   and only at this time are TL0 signals written
    adapter01.set_sample_times(
      sc_time(2.0, SC_PS),
      // adapter10 writes MCmd always during TLM-transport-from-initiator at +1 ps
      sc_time(3.0, SC_PS)
      // same for MDataValid but sampling is delayed to be after MCmd sampling,
      // which would be automatic but shown here for clarity
    );
    adapter10.set_sample_times(
      sc_time(2.0, SC_PS),
      // adapter01 writes SResp always during TLM-transport-from-target at +1 ps
      sc_time(5.0, SC_PS),
      // TL1 target may signal SCmdAccept during its clocked process with an
      // explicit transport call.  This would be seen at +1 ps by adapter01 and
      // the TL0 SCmdAccept signal would be stable for sampling at +2 ps.  But
      // TL1 target might return SCmdAccept to a fw transport call, which it
      // would see at the MCmd sampling time of adapter01 plus 1, ie 3 ps.  The
      // socket PEQ will convert this into a separate backward transport call
      // seen by adapter01 at 4 ps.  Thus SCmdAccept should be sampled at 5 ps.
      sc_time(6.0, SC_PS)
      // data trails command by 1 ps
    );
  }

  sc_in<bool> Clk;

private:
  ocp_tl1_initiator initiator;
  ocp_tl1_target target;

  tl1_tl0::get_TL0_timing_example<ostream> t_print;
  tl1_initiator_to_tl0_target<32> adapter10;
  tl0_initiator_to_tl1_target<32> adapter01;

  sc_signal<unsigned> MAddr;
  sc_signal<unsigned> MCmd;
  sc_signal<unsigned> MData;
  sc_signal<unsigned> SData;
  sc_signal<bool> SCmdAccept;
  sc_signal<unsigned> SResp;
  sc_signal<bool> MReset_n;
};


class testbench010: public sc_module {
public:
  testbench010(sc_module_name n):
    sc_module(n),
    Clk("Clk"),
    initiator("initiator"),
    target("target"),
    adapter01("adapter01"),
    adapter10("adapter10")
  {
    // OCP configuration
    adapter01.set_ocp_config(gen_ocp_config());
    adapter10.set_ocp_config(gen_ocp_config());

    // clock
    initiator.Clk(Clk);
    target.Clk(Clk);
    adapter01.Clk(Clk);
    adapter10.Clk(Clk);

    // TL1 interface
    adapter01.ocpTL1(adapter10.ocpTL1);

    // TL0 interfaces
    initiator.MAddr(MAddr[0]);
    initiator.MCmd(MCmd[0]);
    initiator.MData(MData[0]);
    initiator.SData(SData[0]);
    initiator.SCmdAccept(SCmdAccept[0]);
    initiator.SResp(SResp[0]);
    initiator.MReset_n(MReset_n[0]);
    adapter01.MAddr(MAddr[0]);
    adapter01.MCmd(MCmd[0]);
    adapter01.MData(MData[0]);
    adapter01.SData(SData[0]);
    adapter01.SCmdAccept(SCmdAccept[0]);
    adapter01.SResp(SResp[0]);
    adapter01.MReset_n(MReset_n[0]);

    adapter10.MAddr(MAddr[1]);
    adapter10.MCmd(MCmd[1]);
    adapter10.MData(MData[1]);
    adapter10.SData(SData[1]);
    adapter10.SCmdAccept(SCmdAccept[1]);
    adapter10.SResp(SResp[1]);
    adapter10.MReset_n(MReset_n[1]);
    target.MAddr(MAddr[1]);
    target.MCmd(MCmd[1]);
    target.MData(MData[1]);
    target.SData(SData[1]);
    target.SCmdAccept(SCmdAccept[1]);
    target.SResp(SResp[1]);
    target.MReset_n(MReset_n[1]);

    // timing: nothing needed as adapters' TL0 interfaces are
    //  directly connected to TL0-native initiator and target
  }

  sc_in<bool> Clk;

private:
  ocp_tl0_initiator initiator;
  ocp_tl0_target target;
  tl0_initiator_to_tl1_target<32> adapter01;
  tl1_initiator_to_tl0_target<32> adapter10;

  sc_signal<unsigned> MAddr[2];
  sc_signal<unsigned> MCmd[2];
  sc_signal<unsigned> MData[2];
  sc_signal<unsigned> SData[2];
  sc_signal<bool> SCmdAccept[2];
  sc_signal<unsigned> SResp[2];
  sc_signal<bool> MReset_n[2];
};


class testbench: public sc_module {
public:
  testbench(sc_module_name n, bool run_01 = true, bool run_10 = true):
    sc_module(n),
    tb00("00"),
    tb01("01"),
    tb10("10"),
    tb11("11"),
    tb101("101"),
    tb010("010"),
    Clk("Clk", sc_time(10.0, SC_NS))
  {
    tb00.Clk(Clk);
    tb11.Clk(Clk);

    if(run_01) tb01.Clk(Clk);
    else tb01.Clk(tieoff);

    if(run_10) tb10.Clk(Clk);
    else tb10.Clk(tieoff);

    if(run_01 && run_10) {
      tb010.Clk(Clk);
      tb101.Clk(Clk);
    } else {
      tb101.Clk(tieoff);
      tb010.Clk(tieoff);
    }
  }
private:
  testbench00 tb00;
  testbench01 tb01;
  testbench10 tb10;
  testbench11 tb11;
  testbench101 tb101;
  testbench010 tb010;
  sc_clock Clk;
  sc_signal<bool> tieoff;
};


#ifndef OMIT_SC_MAIN
int sc_main(int argc, char **argv) {
  sc_report_handler::set_actions(SC_ERROR, SC_DISPLAY | SC_ABORT);
  bool run_01 = ((argc < 2) || ('A' == *argv[1]) || ('0' == *argv[1]));
  bool run_10 = ((argc < 2) || ('A' == *argv[1]) || ('1' == *argv[1]));
  testbench theTestbench("testbench", run_01, run_10);
  sc_start();
  return 0;
}
#endif

