///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// (c) Copyright OCP-IP 2009-2010
// OCP-IP Confidential and Proprietary
//
//
//============================================================================
//      Project : OCP SLD WG
//       Author : Herve Alexanian - Sonics, inc.
//
//          $Id:
//
//  Description :  This file defines utility classes designed to be used when
//                 writing adapters for TL1 or TL2 sockets
//
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

template<typename item_t, template <typename item_t> class algo_pol_t>
OCPIP_VERSION::simple_arb<item_t, algo_pol_t>::~simple_arb() {}

template<typename item_t, template <typename item_t> class algo_pol_t>
void
OCPIP_VERSION::simple_arb<item_t, algo_pol_t>::set_range( const item_t& min, const item_t& max ) {
    m_min = min;
    m_max = max;
    m_algo.set_range( m_min, m_max );
}

template<typename item_t, template <typename item_t> class algo_pol_t>
bool
OCPIP_VERSION::simple_arb<item_t, algo_pol_t>::operator()( const std::set<item_t> candidates, item_t& winner ) {
    typename std::set<item_t>::const_iterator it;
    for ( it = candidates.begin(); it != candidates.end(); ++it ) {
	assert( *it >= m_min && *it <= m_max );
    }
    std::set<item_t> filter_candidates = candidates;
    typename std::deque<arbiter_filter_if<item_t>*>::const_iterator fit;
    for ( fit = m_filters.begin(); fit != m_filters.end(); ++fit ) {
	(*fit)->operator()( filter_candidates );
    }
    return m_algo( filter_candidates, winner ); // let caller call winner()
}

template<typename item_t, template <typename item_t> class algo_pol_t>
void
OCPIP_VERSION::simple_arb<item_t, algo_pol_t>::winner( const item_t& item, bool end_of_block ) {
    m_algo.winner( item, end_of_block );
}

template<typename item_t>
void
OCPIP_VERSION::simple_rr_arb_algo<item_t>::set_range( const item_t& min, const item_t& max ) {
    for ( item_t i=min; i <= max; ++i ) {
	m_rr_order.push_back( i );
    }
}

template<typename item_t>
bool
OCPIP_VERSION::simple_rr_arb_algo<item_t>::operator()( const std::set<item_t>& candidates, item_t& result ) {
    typename std::vector<item_t>::iterator rr_it;
    typename std::set<item_t>::const_iterator cit;
    for ( rr_it = m_rr_order.begin(); rr_it != m_rr_order.end(); ++rr_it ) {
	for ( cit = candidates.begin(); cit != candidates.end(); ++cit ) {
	    if (*cit == *rr_it) {
		result = *cit;
		return true;
	    }
	}
    }
    return false;
}

template<typename item_t>
void
OCPIP_VERSION::simple_rr_arb_algo<item_t>::winner( const item_t& item, bool end_of_block ) {
    typename std::vector<item_t>::iterator found = std::find( m_rr_order.begin(),
							      m_rr_order.end(), item );
    assert( found != m_rr_order.end() && "could not find winner iterator in Round Robin" );
    m_rr_order.erase( found );
    m_rr_order.push_back( item );
}


template <typename item_t, typename busy_t>
OCPIP_VERSION::arbiter_threadbusy_filter<item_t, busy_t>::arbiter_threadbusy_filter( const busy_t& word )
{
    set_busy_word( word );
}

template <typename item_t, typename busy_t>
OCPIP_VERSION::arbiter_threadbusy_filter<item_t, busy_t>::arbiter_threadbusy_filter() :
    m_p_busy_word( NULL )
{}

template <typename item_t, typename busy_t>
void
OCPIP_VERSION::arbiter_threadbusy_filter<item_t, busy_t>::set_busy_word( const busy_t& word )
{
    m_p_busy_word = &word;
}

template <typename item_t, typename busy_t>
bool
OCPIP_VERSION::arbiter_threadbusy_filter<item_t, busy_t>::is_busy( const item_t& thread ) const
{
    assert( m_p_busy_word );
    return ( *m_p_busy_word >> thread ) & 0x1;
}

template <typename item_t, typename busy_t>
void
OCPIP_VERSION::arbiter_threadbusy_filter<item_t, busy_t>::operator()( std::set<item_t>& candidates )
{
    assert( m_p_busy_word );
    typename std::set<item_t>::const_iterator it;
    int index = 0;
    for ( it = candidates.begin(); it != candidates.end(); ++it ) {
	if ( is_busy( index ) )
	    candidates.erase( it );
	++index;
    }
}

template<typename item_t>
OCPIP_VERSION::arbiter_interleave_filter<item_t>::arbiter_interleave_filter( uint32_t size, bool optional )
    : m_size( size )
    , m_num_interleaves( 0 )
    , m_optional ( optional )
    , m_locked   ( false )
    , m_locked_to( 0 )
{}

template<typename item_t>
void
OCPIP_VERSION::arbiter_interleave_filter<item_t>::operator()( std::set<item_t>& candidates )
{
    if ( m_locked ) {
	bool found = ( candidates.find( m_locked_to ) != candidates.end() );
	if ( m_optional ) {
	    if ( found ) {
		candidates.clear();
		candidates.insert( m_locked_to );
	    }
	} else {
	    candidates.clear();
	    if ( found ) {
		candidates.insert( m_locked_to );
	    }
	}
    }
}

template<typename item_t>
void
OCPIP_VERSION::arbiter_interleave_filter<item_t>::winner( const item_t& index, bool release )
{
    m_locked    = true;
    m_locked_to = index;
    if ( ++m_num_interleaves == m_size || release ) {
	m_num_interleaves = 0;
	m_locked = false;
    }
}

template <typename item_t,template <typename item_t, typename alloc_t> class container_t>
OCPIP_VERSION::simple_phase_queue<item_t, container_t>::simple_phase_queue( size_t size_ ) :
    m_max_size( size_ ),
    m_insert( *this )
{}

template <typename item_t,template <typename item_t, typename alloc_t> class container_t>
bool
OCPIP_VERSION::simple_phase_queue<item_t, container_t>::nb_put( const item_t& item )
{
    if ( this->empty() ) {
	m_not_empty_ev.notify( sc_core::SC_ZERO_TIME );
    }
    if ( nb_can_put() ) {
	*m_insert = item;
	return true;
    }
    return false;
}

template <typename item_t,template <typename item_t, typename alloc_t> class container_t>
bool
OCPIP_VERSION::simple_phase_queue<item_t, container_t>::nb_can_put( tlm::tlm_tag<item_t>* ) const
{
    return ( m_max_size == 0 || this->size() < m_max_size );
}

template <typename item_t,template <typename item_t, typename alloc_t> class container_t>
const sc_core::sc_event&
OCPIP_VERSION::simple_phase_queue<item_t, container_t>::ok_to_put( tlm::tlm_tag<item_t>* ) const
{
    return m_not_full_ev;
}

template <typename item_t,template <typename item_t, typename alloc_t> class container_t>
bool
OCPIP_VERSION::simple_phase_queue<item_t, container_t>::nb_get( item_t& item )
{
    if ( !nb_can_get() )
	return false;
	
    bool success = nb_peek( item );
    assert( success );
    if ( !nb_can_put() ) {
	m_not_full_ev.notify( sc_core::SC_ZERO_TIME );
    }
    this->erase( this->begin() );
    return success;
}

template <typename item_t,template <typename item_t, typename alloc_t> class container_t>
bool
OCPIP_VERSION::simple_phase_queue<item_t, container_t>::nb_can_get( tlm::tlm_tag<item_t>* ) const
{
    return !this->empty();
}

template <typename item_t,template <typename item_t, typename alloc_t> class container_t>
const sc_core::sc_event&
OCPIP_VERSION::simple_phase_queue<item_t, container_t>::ok_to_get( tlm::tlm_tag<item_t>* ) const
{
    return m_not_empty_ev;
}

template <typename item_t,template <typename item_t, typename alloc_t> class container_t>
bool
OCPIP_VERSION::simple_phase_queue<item_t, container_t>::nb_peek( item_t& item ) const
{
    if ( !nb_can_peek() )
	return false;
    item = *( this->begin() );
    return true;
}

template <typename item_t,template <typename item_t, typename alloc_t> class container_t>
bool
OCPIP_VERSION::simple_phase_queue<item_t, container_t>::nb_can_peek( tlm::tlm_tag<item_t>* ) const
{
    return !this->empty();
}

template <typename item_t,template <typename item_t, typename alloc_t> class container_t>
const sc_core::sc_event&
OCPIP_VERSION::simple_phase_queue<item_t, container_t>::ok_to_peek( tlm::tlm_tag<item_t>* ) const
{
    return m_not_empty_ev;
}

template <typename payload_t, typename delay_t>
OCPIP_VERSION::ocp_cycle_based_peq<payload_t, delay_t>::ocp_cycle_based_peq( sc_core::sc_module_name nm )
    : sc_core::sc_module( nm )
    , m_clk( "clk" )
    , m_cycle_count( 0 )
{
    SC_METHOD( process );
    sensitive << m_clk.pos();
}

template <typename payload_t, typename delay_t>
void
OCPIP_VERSION::ocp_cycle_based_peq<payload_t, delay_t>::notify( payload_t& payload, delay_t delay ) {
    if ( m_scheduled.empty() ) {
	m_new_scheduling_ev.notify( sc_core::SC_ZERO_TIME );
    }
    m_scheduled.insert( std::make_pair( m_cycle_count + delay, &payload ) );
    if ( delay == 0 ) {
	m_event.notify();
    }
}

template <typename payload_t, typename delay_t>
payload_t*
OCPIP_VERSION::ocp_cycle_based_peq<payload_t, delay_t>::get_next_transaction()
{
    if ( m_scheduled.empty() ) {
	return 0;
    }
    delay_t first_scheduled_time = m_scheduled.begin()->first;
    if ( first_scheduled_time <= m_cycle_count ) {
	payload_t* p = m_scheduled.begin()->second;
	m_scheduled.erase( m_scheduled.begin() );
	return p;
    }
    return 0;
}

template <typename payload_t, typename delay_t>
void
OCPIP_VERSION::ocp_cycle_based_peq<payload_t, delay_t>::process()
{
    if ( m_scheduled.empty() ) {
	m_cycle_count = 0;
	next_trigger( m_new_scheduling_ev );
	return;
    }

    if ( m_clk.posedge() ) {
	++m_cycle_count;
	// find delays scheduled to mature in this cycle
	// loop should exit quickly because the multi-map is sorted ascendingly
	typename std::multimap<delay_t, payload_t*>::iterator it;
	delay_t first_scheduled_time = m_scheduled.begin()->first;
	if ( first_scheduled_time == m_cycle_count ) {
	    m_event.notify(); // immediate
	} else {
	    // entry was earlier than now, the user must not have called
	    // get_transaction. The entry is stale until he does so, do nothing
	}
    }
}

template <typename module_t, typename delay_t, typename txn_t, template <typename payload_t, typename delay_t> class peq_t>
OCPIP_VERSION::phase_delayed_accept<module_t, delay_t, txn_t, peq_t>::phase_delayed_accept( sc_core::sc_module_name nm )
    : sc_core::sc_module( nm )
    , m_clk  ( "clk" )
    , m_peq  ( "peq" )
    , m_cb   ( NULL  )
{
    m_peq.m_clk( m_clk );
    SC_METHOD( on_peq );
    sensitive << m_peq.get_event();    
}

template <typename module_t, typename delay_t, typename txn_t, template <typename payload_t, typename delay_t> class peq_t>
void
OCPIP_VERSION::phase_delayed_accept<module_t, delay_t, txn_t, peq_t>::accept( txn_t& txn, delay_t delay )
{
    m_peq.notify( txn, delay );
}

template <typename module_t, typename delay_t, typename txn_t, template <typename payload_t, typename delay_t> class peq_t>
void
OCPIP_VERSION::phase_delayed_accept<module_t, delay_t, txn_t, peq_t>::on_peq()
{
    typename peq_type::transaction_type* p_txn;
    while ( ( p_txn = m_peq.get_next_transaction() ) != NULL ) {
	assert( m_p_mod != NULL );
	assert( m_cb    != NULL );
	(m_p_mod->*m_cb)( *p_txn );
    }
}
