///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// (c) Copyright OCP-IP 2009-2011
// OCP-IP Confidential and Proprietary
//
//
//============================================================================
//      Project : OCP SLD WG
//       Author : James Aldis, Texas Instruments
//
//          $Id:
//
//  Description :  TL1/TL0 OCP Master Adapter
//
// SystemC Module Adapting between a TLM2-based OCP-TL1 slave interface
// and a TL0 (signal-level) OCP interface
// Typical use is between a TL0 OCP initiator and a TL1 OCP target
//
// Module is a class template.  Template parameters are the TLM2 bus width
// and a traits class defining the types of the TL0 signals
//
// Timing:
//   Module is insensitive to timing on TL1 side
//   User needs to tell Module when to sample on TL0 side
//   Module will publish its TL1 timing using normal Tl1 mechanism
//   Module's timing on TL0 side depends on user-supplied sample times
//   and TL1 timing and can be obtained using the show_TL0_timing(x) API.
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#ifndef OCPIP_VERSION
  #error tl1_initiator_to_tl0_target.h may not be included directly. Use #include "ocpip_adapters.h" or #include "ocpip_adapters_X_X_X.h" (where desired ocp version is X.X.X)
#endif


// namespace
namespace OCPIP_VERSION { namespace tl1_tl0 {


// forward declaration
namespace support_01 {struct lock_object;}


// non-template base class containing actual functionality
// implementation in cpp file
class tl0_to_tl1_base: public sc_core::sc_module {
public:
  virtual void end_of_elaboration();
  typedef void (tl0_to_tl1_base::*callable)();

protected:
  sc_core::sc_in<bool> clk_port;

  // constructor/destructor only callable from the derived adapter template
  tl0_to_tl1_base(sc_core::sc_module_name n);
  virtual ~tl0_to_tl1_base();

  // methods for communicating with the derived adapter template
  //   functions called by this base class are pure virtual
  //   functions called by the derived adapter template are implemented

  // TL1 TLM2 transport
  virtual tlm::tlm_sync_enum nb_transport_fw(
    tlm::tlm_generic_payload &pl, tlm::tlm_phase &ph, sc_core::sc_time &ta) = 0;
  tlm::tlm_sync_enum nb_transport_bw(
    tlm::tlm_generic_payload &pl, tlm::tlm_phase &ph, sc_core::sc_time &ta);

public:
  // TL0 port access; getters and setters
  // these are public to allow burst trackers to access the ports

  // clk-enable
  virtual bool enableclk() = 0;

  // basic
  virtual sc_dt::uint64 maddr() = 0;
  virtual unsigned mcmd() = 0;
  virtual unsigned char mdata(unsigned) = 0;
  virtual bool mdatavalid() = 0;
  virtual bool mrespaccept() = 0;
  virtual void sdata(unsigned char, unsigned) = 0;
  virtual void sdata() = 0;
  virtual void scmdaccept(bool) = 0;  bool new_scmdaccept;  bool scmdaccept_wr;
  virtual void sdataaccept(bool) = 0;  bool new_sdataaccept;  bool sdataaccept_wr;
  virtual void sresp(unsigned) = 0;  unsigned new_sresp;  bool sresp_wr;

  // simple
  virtual unsigned maddrspace() = 0;
  virtual sc_dt::uint64 mbyteen() = 0;
  virtual sc_dt::uint64 mdatabyteen() = 0;
  virtual void mdatainfo(tlm::tlm_generic_payload *, unsigned) = 0;
  virtual void mreqinfo(tlm::tlm_generic_payload *) = 0;
  virtual void sdatainfo(tlm::tlm_generic_payload *, unsigned) = 0;
  virtual void srespinfo(tlm::tlm_generic_payload *) = 0;

  // burst
  virtual unsigned matomiclength() = 0;
  virtual unsigned mblockheight() = 0;
  virtual sc_dt::uint64 mblockstride() = 0;
  virtual unsigned mburstlength() = 0;
  virtual bool mburstprecise() = 0;
  virtual unsigned mburstseq() = 0;
  virtual bool mburstsinglereq() = 0;
  virtual bool mdatalast() = 0;
  virtual bool mdatarowlast() = 0;
  virtual bool mreqlast() = 0;
  virtual bool mreqrowlast() = 0;
  virtual void sresplast(bool) = 0;
  virtual void sresprowlast(bool) = 0;

  // tag
  virtual unsigned mdatatagid() = 0;
  virtual unsigned mtagid() = 0;
  virtual bool mtaginorder() = 0;
  virtual void stagid(unsigned) = 0;
  virtual void staginorder(bool) = 0;

  // thread
  virtual unsigned mconnid() = 0;
  virtual unsigned mdatathreadid() = 0;
  virtual unsigned mthreadbusy() = 0;
  virtual unsigned mthreadid() = 0;
  virtual void sdatathreadbusy(unsigned) = 0;
  virtual void sthreadbusy(unsigned) = 0;
  virtual void sthreadid(unsigned) = 0;

  // sideband
  virtual bool merror() = 0;
  virtual const sc_core::sc_event &merror_event() = 0;
  virtual void mflag(tlm::tlm_generic_payload *) = 0;
  virtual const sc_core::sc_event &mflag_event() = 0;
  virtual bool mreset_n() = 0;
  virtual const sc_core::sc_event &mreset_event() = 0;
  virtual void serror(bool) = 0;
  virtual void sflag(tlm::tlm_generic_payload *) = 0;
  virtual void sinterrupt(bool) = 0;
  virtual void sreset_n(bool) = 0;

  ocp_parameters ocp_config;
  unsigned byte_wdth, max_burst_bytes;
  support_01::lock_object *newest_lock;

protected:
  // generic payload memory management
  virtual tlm::tlm_generic_payload *get_tb_transaction() = 0;
  virtual tlm::tlm_generic_payload *get_flag_transaction() = 0;
  virtual tlm::tlm_generic_payload *get_error_transaction() = 0;
  virtual tlm::tlm_generic_payload *get_reset_transaction() = 0;
  virtual tlm::tlm_generic_payload *get_transaction() = 0;
  virtual void release_transaction(tlm::tlm_generic_payload *pl) = 0;
  void remove_from_list(tlm::tlm_generic_payload &pl);
  tlm::tlm_generic_payload &add_transaction(unsigned cmd);
  virtual void ocp_socket_reset() = 0;

  sc_core::sc_time request_t;
  sc_core::sc_time data_t;
  sc_core::sc_time respaccept_t;
  sc_core::sc_time threadbusy_t;
  sc_core::sc_time resp_t;

private:
  tlm::tlm_generic_payload *oldest_txn, *newest_txn;

  sc_core::sc_event request_e;
  sc_core::sc_event data_e;
  sc_core::sc_event respaccept_e;
  sc_core::sc_event threadbusy_e;

  void clk_rising_edge();
  void request_sample();
  void data_sample();
  void respaccept_sample();
  void threadbusy_sample();
  void handle_mreset();
  void handle_mflag();
  void handle_merror();
  void exit_reset();

  // specifically NON-delta-protected variables for inter-process communication
  // clock edge process drives to false and is known to run before sampling and
  // TLM transport processes
  bool respaccept_state;
  tlm::tlm_generic_payload *curr_tl0_cmd, *curr_tl0_data, *curr_tl0_resp;
  unsigned curr_mthreadbusy;

  // lower-level functionality
  void end_req(tlm::tlm_generic_payload &pl);
  void end_data(tlm::tlm_generic_payload &pl);
  void end_resp(tlm::tlm_generic_payload &pl);

  // process for writing critical signals from more than one process
  void writer_process();
  sc_core::sc_event writer_trigger;
};


// tl1_tl0 namespace
}


// template for user-instantiable adapter module
template<
  int BUSWIDTH,
  class TL0_TYPES = tl1_tl0::POD_DEFAULT_TL0_TYPES<>,
  int SRESET_N_INITIAL = 1
>
class tl0_initiator_to_tl1_target: public tl1_tl0::tl0_to_tl1_base {
typedef tl1_tl0::tl0_to_tl1_base base;
public:
  // constructor
  tl0_initiator_to_tl1_target(
      sc_core::sc_module_name n, tl1_tl0::get_TL0_timing_base *gtb = 0):
    base(n),
    Clk(clk_port),
    ocpTL1("OCP_TL1", this, &tl0_initiator_to_tl1_target::slave_timing_update),
    EnableClk("EnableClk"),
    MAddr("MAddr"),
    MCmd("MCmd"),
    MData("MData"),
    MDataValid("MDataValid"),
    MRespAccept("MRespAccept"),
    SData("SData"),
    SCmdAccept("SCmdAccept"),
    SDataAccept("SDataAccept"),
    SResp("SResp"),
    MAddrSpace("MAddrSpace"),
    MByteEn("MByteEn"),
    MDataByteEn("MDataByteEn"),
    MDataInfo("MDataInfo"),
    MReqInfo("MReqInfo"),
    SDataInfo("SDataInfo"),
    SRespInfo("SRespInfo"),
    MAtomicLength("MAtomicLength"),
    MBlockHeight("MBlockHeight"),
    MBlockStride("MBlockStride"),
    MBurstLength("MBurstLength"),
    MBurstPrecise("MBurstPrecise"),
    MBurstSeq("MBurstSeq"),
    MBurstSingleReq("MBurstSingleReq"),
    MDataLast("MDataLast"),
    MDataRowLast("MDataRowLast"),
    MReqLast("MReqLast"),
    MReqRowLast("MReqRowLast"),
    SRespLast("SRespLast"),
    SRespRowLast("SRespRowLast"),
    MDataTagID("MDataTagID"),
    MTagID("MTagID"),
    MTagInOrder("MTagInOrder"),
    STagID("STagID"),
    STagInOrder("STagInOrder"),
    MConnID("MConnID"),
    MDataThreadID("MDataThreadID"),
    MThreadBusy("MThreadBusy"),
    MThreadID("MThreadID"),
    SDataThreadBusy("SDataThreadBusy"),
    SThreadBusy("SThreadBusy"),
    SThreadID("SThreadID"),
    MError("MError"),
    MFlag("MFlag"),
    MReset_n("MReset_n"),
    SError("SError"),
    SFlag("SFlag"),
    SInterrupt("SInterrupt"),
    SReset_n("SReset_n"),
    eoe_called(false),
    timing_received(false),
    get_TL0_timing(gtb)
  {
    // backward TLM2 transport calls go into the base class
    ocpTL1.register_nb_transport_bw(this, &base::nb_transport_bw);
    ocpTL1.activate_synchronization_protection();

    SCmdAccept.initialize(typename TL0_TYPES::SCMDACCEPT_T(false));
    SDataAccept.initialize(typename TL0_TYPES::SDATAACCEPT_T(false));
    SResp.initialize(typename TL0_TYPES::SRESP_T(tl1_tl0::RESP_NULL));
    SReset_n.initialize(typename TL0_TYPES::SRESET_N_T(SRESET_N_INITIAL));
  }

  // setup functions called by instantiator eg wrapper around TL0 module

  void set_max_burst_bytes(unsigned b) {max_burst_bytes = b;}

  void set_sample_times(
    const sc_core::sc_time &request,
    const sc_core::sc_time &data = sc_core::SC_ZERO_TIME,
    const sc_core::sc_time &respaccept = sc_core::SC_ZERO_TIME,
    const sc_core::sc_time &threadbusy = sc_core::SC_ZERO_TIME)
  {
    // set sample times in base class - inputs equal to 0 are ignored
    if(request > sc_core::SC_ZERO_TIME) base::request_t = request;
    if(data > sc_core::SC_ZERO_TIME) base::data_t = data;
    if(respaccept > sc_core::SC_ZERO_TIME) base::respaccept_t = respaccept;
    if(threadbusy > sc_core::SC_ZERO_TIME) base::threadbusy_t = threadbusy;
    // keep data and request in-order on TL1
    if(base::request_t >= base::data_t)
      base::data_t =
        base::request_t + sc_core::sc_get_time_resolution();
  }

  void show_TL0_timing(ocp_tl1_slave_timing times) {
    if(get_TL0_timing != 0) {
      sc_core::sc_time unit = sc_core::sc_get_time_resolution();
      times.ResponseGrpStartTime += unit;
      times.SCmdAcceptStartTime += unit;
      times.SDataAcceptStartTime += unit;
      times.SThreadBusyStartTime += unit;
      times.SDataThreadBusyStartTime += unit;
      get_TL0_timing->slave(this, times);
    }
  }

  sc_core::sc_time calc_respacc_t() {
    return sc_dt::sc_max(
      base::resp_t + sc_core::sc_get_time_resolution(),
      base::respaccept_t);
  }

  void slave_timing_update(ocp_tl1_slave_timing times) {
    // called from socket to provide TL1 timing of connected slave
    show_TL0_timing(times);
    timing_received = true;
    // Response start time may affect MRespAcceptStartTime
    sc_core::sc_time old_tl1_respaccept_t = calc_respacc_t();
    base::resp_t = times.ResponseGrpStartTime;
    sc_core::sc_time new_tl1_respaccept_t = calc_respacc_t();
    if((old_tl1_respaccept_t != new_tl1_respaccept_t) && eoe_called)
      set_master_timing();
  }

  void end_of_elaboration() {
    base::end_of_elaboration();
    eoe_called = true;
    set_master_timing();
    if(!timing_received) show_TL0_timing(ocp_tl1_slave_timing());
  }

  void set_master_timing() {
    // publish timing on TL1: the adapter does not have "default timing"
    ocp_tl1_master_timing times;
    times.RequestGrpStartTime = base::request_t;
    times.DataHSGrpStartTime = base::data_t;
    times.MThreadBusyStartTime = base::threadbusy_t;
    times.MRespAcceptStartTime = calc_respacc_t();
    ocpTL1.set_master_timing(times);
  }

  void set_ocp_config(const ocp_parameters &P) {
    // OCP socket is not generic because attached to TL0 OCP interface which
    //   must have a known configuration
    ocpTL1.set_ocp_config(P);
    base::ocp_config = P;
  }

  // SystemC communications
  // clock (reference to actual clock object in base class)
  sc_core::sc_in<bool> &Clk;

  // TL1 socket
  ocp_master_socket_tl1<BUSWIDTH> ocpTL1;

  // TL0 ports - those whose type is TIEOFF<> effectively do not exist
  //  and definitely do not need to be bound

  // clk-enable
  sc_core::sc_in<typename TL0_TYPES::ENABLE_CLK_T> EnableClk;

  // basic
  sc_core::sc_in<typename TL0_TYPES::MADDR_T> MAddr;
  sc_core::sc_in<typename TL0_TYPES::MCMD_T> MCmd;
  sc_core::sc_in<typename TL0_TYPES::DATA_T> MData;
  sc_core::sc_in<typename TL0_TYPES::MDATAVALID_T> MDataValid;
  sc_core::sc_in<typename TL0_TYPES::MRESPACCEPT_T> MRespAccept;
  sc_core::sc_out<typename TL0_TYPES::DATA_T> SData;
  sc_core::sc_out<typename TL0_TYPES::SCMDACCEPT_T> SCmdAccept;
  sc_core::sc_out<typename TL0_TYPES::SDATAACCEPT_T> SDataAccept;
  sc_core::sc_out<typename TL0_TYPES::SRESP_T> SResp;

  // simple
  sc_core::sc_in<typename TL0_TYPES::MADDRSPACE_T> MAddrSpace;
  sc_core::sc_in<typename TL0_TYPES::MBYTEEN_T> MByteEn;
  sc_core::sc_in<typename TL0_TYPES::MDATABYTEEN_T> MDataByteEn;
  sc_core::sc_in<typename TL0_TYPES::MDATAINFO_T> MDataInfo;
  sc_core::sc_in<typename TL0_TYPES::MREQINFO_T> MReqInfo;
  sc_core::sc_out<typename TL0_TYPES::SDATAINFO_T> SDataInfo;
  sc_core::sc_out<typename TL0_TYPES::SRESPINFO_T> SRespInfo;

  // burst
  sc_core::sc_in<typename TL0_TYPES::MATOMICLENGTH_T> MAtomicLength;
  sc_core::sc_in<typename TL0_TYPES::MBLOCKHEIGHT_T> MBlockHeight;
  sc_core::sc_in<typename TL0_TYPES::MBLOCKSTRIDE_T> MBlockStride;
  sc_core::sc_in<typename TL0_TYPES::MBURSTLENGTH_T> MBurstLength;
  sc_core::sc_in<typename TL0_TYPES::MBURSTPRECISE_T> MBurstPrecise;
  sc_core::sc_in<typename TL0_TYPES::MBURSTSEQ_T> MBurstSeq;
  sc_core::sc_in<typename TL0_TYPES::MBURSTSINGLEREQ_T> MBurstSingleReq;
  sc_core::sc_in<typename TL0_TYPES::MDATALAST_T> MDataLast;
  sc_core::sc_in<typename TL0_TYPES::MDATAROWLAST_T> MDataRowLast;
  sc_core::sc_in<typename TL0_TYPES::MREQLAST_T> MReqLast;
  sc_core::sc_in<typename TL0_TYPES::MREQROWLAST_T> MReqRowLast;
  sc_core::sc_out<typename TL0_TYPES::SRESPLAST_T> SRespLast;
  sc_core::sc_out<typename TL0_TYPES::SRESPROWLAST_T> SRespRowLast;

  // tag
  sc_core::sc_in<typename TL0_TYPES::MDATATAGID_T> MDataTagID;
  sc_core::sc_in<typename TL0_TYPES::MTAGID_T> MTagID;
  sc_core::sc_in<typename TL0_TYPES::MTAGINORDER_T> MTagInOrder;
  sc_core::sc_out<typename TL0_TYPES::STAGID_T> STagID;
  sc_core::sc_out<typename TL0_TYPES::STAGINORDER_T> STagInOrder;

  // thread
  sc_core::sc_in<typename TL0_TYPES::MCONNID_T> MConnID;
  sc_core::sc_in<typename TL0_TYPES::MDATATHREADID_T> MDataThreadID;
  sc_core::sc_in<typename TL0_TYPES::MTHREADBUSY_T> MThreadBusy;
  sc_core::sc_in<typename TL0_TYPES::MTHREADID_T> MThreadID;
  sc_core::sc_out<typename TL0_TYPES::SDATATHREADBUSY_T> SDataThreadBusy;
  sc_core::sc_out<typename TL0_TYPES::STHREADBUSY_T> SThreadBusy;
  sc_core::sc_out<typename TL0_TYPES::STHREADID_T> SThreadID;

  // sideband
  sc_core::sc_in<typename TL0_TYPES::MERROR_T> MError;
  sc_core::sc_in<typename TL0_TYPES::MFLAG_T> MFlag;
  sc_core::sc_in<typename TL0_TYPES::MRESET_N_T> MReset_n;
  sc_core::sc_out<typename TL0_TYPES::SERROR_T> SError;
  sc_core::sc_out<typename TL0_TYPES::SFLAG_T> SFlag;
  sc_core::sc_out<typename TL0_TYPES::SINTERRUPT_T> SInterrupt;
  sc_core::sc_out<typename TL0_TYPES::SRESET_N_T> SReset_n;

private:
  // calls from base class

  // forward TLM2 transport calls go to the OCP socket
  tlm::tlm_sync_enum nb_transport_fw(
    tlm::tlm_generic_payload &pl, tlm::tlm_phase &ph, sc_core::sc_time &ta) {
    return ocpTL1->nb_transport_fw(pl, ph, ta);
  }

  // TL0 port access

  // clk-enable
  bool enableclk() {return tl1_tl0::port_to_bool(EnableClk);}

  // basic
  sc_dt::uint64 maddr() {return tl1_tl0::port_to_uint64(MAddr);}
  unsigned mcmd() {return tl1_tl0::port_to_unsigned(MCmd);}
  unsigned char mdata(unsigned byte_lane) {
    return TL0_TYPES::data_to_uchar(MData->read(), byte_lane);
  }
  bool mdatavalid() {return tl1_tl0::port_to_bool(MDataValid);}
  bool mrespaccept() {return tl1_tl0::port_to_bool(MRespAccept);}
  typename TL0_TYPES::DATA_T curr_mdata;
  void sdata(unsigned char d, unsigned byte_lane) {
    TL0_TYPES::data_from_uchar(d, byte_lane, curr_mdata);
  }
  void sdata() {SData = curr_mdata;}
  void scmdaccept(bool d) {SCmdAccept = typename TL0_TYPES::SCMDACCEPT_T(d);}
  void sdataaccept(bool d) {SDataAccept = typename TL0_TYPES::SDATAACCEPT_T(d);}
  void sresp(unsigned d) {SResp = typename TL0_TYPES::SRESP_T(d);}

  // simple
  unsigned maddrspace() {return tl1_tl0::port_to_unsigned(MAddrSpace);}
  sc_dt::uint64 mbyteen() {return tl1_tl0::port_to_uint64(MByteEn);}
  sc_dt::uint64 mdatabyteen() {return tl1_tl0::port_to_uint64(MDataByteEn);}
  void mdatainfo(tlm::tlm_generic_payload *txn, unsigned beat) {
    typename TL0_TYPES::MDATAINFO_T info = MDataInfo;
    TL0_TYPES::conv_mdatainfo(info, *txn, beat);
  }
  void mreqinfo(tlm::tlm_generic_payload *txn) {
    typename TL0_TYPES::MREQINFO_T info = MReqInfo;
    TL0_TYPES::conv_mreqinfo(info, *txn);
  }
  void sdatainfo(tlm::tlm_generic_payload *txn, unsigned beat) {
    typename TL0_TYPES::SDATAINFO_T info;
    TL0_TYPES::conv_sdatainfo(info, *txn, beat);
    SDataInfo = info;
  }
  void srespinfo(tlm::tlm_generic_payload *txn) {
    typename TL0_TYPES::SRESPINFO_T info;
    TL0_TYPES::conv_srespinfo(info, *txn);
    SRespInfo = info;
  }

  // burst
  unsigned matomiclength() {return tl1_tl0::port_to_unsigned(MAtomicLength);}
  unsigned mblockheight() {return tl1_tl0::port_to_unsigned(MBlockHeight);}
  sc_dt::uint64 mblockstride() {return tl1_tl0::port_to_uint64(MBlockStride);}
  unsigned mburstlength() {return tl1_tl0::port_to_unsigned(MBurstLength);}
  bool mburstprecise() {return tl1_tl0::port_to_bool(MBurstPrecise);}
  unsigned mburstseq() {return tl1_tl0::port_to_unsigned(MBurstSeq);}
  bool mburstsinglereq() {return tl1_tl0::port_to_bool(MBurstSingleReq);}
  bool mdatalast() {return tl1_tl0::port_to_bool(MDataLast);}
  bool mdatarowlast() {return tl1_tl0::port_to_bool(MDataRowLast);}
  bool mreqlast() {return tl1_tl0::port_to_bool(MReqLast);}
  bool mreqrowlast() {return tl1_tl0::port_to_bool(MReqRowLast);}
  void sresplast(bool d) {SRespLast = typename TL0_TYPES::SRESPLAST_T(d);}
  void sresprowlast(bool d) {SRespRowLast = typename TL0_TYPES::SRESPROWLAST_T(d);}

  // tag
  unsigned mdatatagid() {return tl1_tl0::port_to_unsigned(MDataTagID);}
  unsigned mtagid() {return tl1_tl0::port_to_unsigned(MTagID);}
  bool mtaginorder() {return tl1_tl0::port_to_bool(MTagInOrder);}
  void stagid(unsigned d) {STagID = typename TL0_TYPES::STAGID_T(d);}
  void staginorder(bool d) {STagInOrder = typename TL0_TYPES::STAGINORDER_T(d);}

  // thread
  unsigned mconnid() {return tl1_tl0::port_to_unsigned(MConnID);}
  unsigned mdatathreadid() {return tl1_tl0::port_to_unsigned(MDataThreadID);}
  unsigned mthreadbusy() {return tl1_tl0::port_to_unsigned(MThreadBusy);}
  unsigned mthreadid() {return tl1_tl0::port_to_unsigned(MThreadID);}
  void sdatathreadbusy(unsigned d) {SDataThreadBusy = typename TL0_TYPES::SDATATHREADBUSY_T(d);}
  void sthreadbusy(unsigned d) {SThreadBusy = typename TL0_TYPES::STHREADBUSY_T(d);}
  void sthreadid(unsigned d) {SThreadID = typename TL0_TYPES::STHREADID_T(d);}

  // sideband
  bool merror() {return tl1_tl0::port_to_bool(MError);}
  const sc_core::sc_event &merror_event() {return MError->default_event();}
  void mflag(tlm::tlm_generic_payload *txn) {
    typename TL0_TYPES::MFLAG_T flag = MFlag;
    TL0_TYPES::conv_mflag(flag, *txn);
  }
  const sc_core::sc_event &mflag_event() {return MFlag->default_event();}
  bool mreset_n() {return tl1_tl0::port_to_bool(MReset_n);}
  const sc_core::sc_event &mreset_event() {return MReset_n->default_event();}
  void serror(bool d) {SError = typename TL0_TYPES::SERROR_T(d);}
  void sflag(tlm::tlm_generic_payload *txn) {
    typename TL0_TYPES::SFLAG_T flag;
    TL0_TYPES::conv_sflag(flag, *txn);
    SFlag = flag;
  }
  void sinterrupt(bool d) {SInterrupt = typename TL0_TYPES::SINTERRUPT_T(d);}
  void sreset_n(bool d) {SReset_n = typename TL0_TYPES::SRESET_N_T(d);}

  // generic payload memory management
  tlm::tlm_generic_payload *get_tb_transaction() {return ocpTL1.get_tb_transaction();}
  tlm::tlm_generic_payload *get_flag_transaction() {return ocpTL1.get_flag_transaction();}
  tlm::tlm_generic_payload *get_error_transaction() {return ocpTL1.get_error_transaction();}
  tlm::tlm_generic_payload *get_reset_transaction() {return ocpTL1.get_reset_transaction();}
  tlm::tlm_generic_payload *get_transaction() {return ocpTL1.get_transaction();}
  void release_transaction(tlm::tlm_generic_payload *pl) {ocpTL1.release_transaction(pl);}
  void ocp_socket_reset() {ocpTL1.reset();}

  bool eoe_called, timing_received;
  tl1_tl0::get_TL0_timing_base *get_TL0_timing;
};


// end ocpip namespace
}

