// Copyright (c) 2005 Sonics, Inc.
//
// Confidential and Proprietary Information of Sonics, Inc.
// Use, disclosure, or reproduction is prohibited without
// written permission from Sonics, Inc.
//
// $Id: ThreadedQueue.h,v 1.1 2007/01/25 21:51:22 halexan Exp $
//
// It's common for us to store things in parallel queues per
// protocol thread
//

#ifndef _SonicsThreadedQueue_h_
#define _SonicsThreadedQueue_h_

#include <deque>
#include <vector>
#include <inttypes.h>

// Note this does not include systemc.h. May consider a future sc_fifo
// based specialization of the class
namespace Sonics
{

template <typename DataT>
class ThreadedQueueIf {
  public:
    typedef uint32_t Thread;
    virtual void         setNumThreads( Thread )               = 0;
    //virtual void         enqueue      ( const DataT& )         = 0;
    virtual void         enqueue      ( Thread, const DataT& ) = 0;
    virtual const DataT& lastEnqueued ( Thread ) const         = 0;
    virtual       DataT& lastEnqueued ( Thread )               = 0;
    virtual const DataT& peek         ( Thread ) const         = 0;
    virtual       DataT& peek         ( Thread )               = 0;
    virtual void         dequeue      ( Thread )               = 0;
    virtual size_t       size         ( Thread ) const         = 0;
};

template <typename DataT, typename StorageT=std::deque<DataT> >
class ThreadedQueue : public ThreadedQueueIf<DataT>
{
    typedef typename ThreadedQueueIf<DataT>::Thread Thread;

  public:
    virtual ~ThreadedQueue() {
        m_queue.clear();
    }
    virtual void setNumThreads( Thread threads ) {
        // only once
        assert( m_queue.empty() );
        m_queue.resize( threads );
    }

    virtual void enqueue( Thread thread, const DataT& data ) {
        assert( thread < m_queue.size() );
        m_queue[ thread ].push_back( data );
    }

    virtual const DataT& lastEnqueued( Thread thread ) const {
        assert( thread < m_queue.size() );
        assert( size( thread ) > 0 );
        return m_queue[ thread ].back();
    }
    virtual DataT& lastEnqueued( Thread thread ) {
        assert( thread < m_queue.size() );
        assert( size( thread ) > 0 );
        return m_queue[ thread ].back();
    }

/*     virtual void enqueue( const DataT& data ) { */
/*         enqueue( Ocp2::getThread( data ), data ); */
/*     } */

    virtual const DataT& peek( Thread thread ) const {
        assert( thread < m_queue.size() );
        assert( size( thread ) > 0 );
        return m_queue[ thread ].front();
    }
    virtual DataT& peek( Thread thread ) {
        assert( thread < m_queue.size() );
        assert( size( thread ) > 0 );
        return m_queue[ thread ].front();
    }

    virtual void dequeue( Thread thread ) {
        m_queue[ thread ].pop_front();
    }

    virtual size_t size( Thread thread ) const {
        assert( thread < m_queue.size() );
        return m_queue[ thread ].size();
    }

  private:
    vector<StorageT> m_queue;
};
}
#endif /* _SonicsThreadedQueue_h */
