//////////////////////////////////////////////////////////////////////
//
//  (c) Copyright OCP-IP 2006
//  OCP-IP Confidential and Proprietary
//
// Author: Herve Alexanian, Sonics, inc.
//
// $Id:
//
// Slave Transaction Level (TL2) model
//
//////////////////////////////////////////////////////////////////////
#ifndef _OcpIpSlaveTL2_h_
#define _OcpIpSlaveTL2_h_

#include <iostream>
#include <map>
#include <deque>

// OCP-IP Channel header files
#include "ocp_globals.h"
#include "ocp_tl2_slave_port.h"
#include "ocp_param.h"
#include "ocp_utils.h"
#include "ThreadArbiter.h"

class IChipDMCore_1;

using std::vector;
using std::deque;
using std::multimap;
namespace sc_extension {
    class ScSemaphore;
};

namespace OcpIp {
template <class Td, class Ta> class MemoryCl;

struct SlaveTl2Config {

    // default values
    SlaveTl2Config() :
        request_buffers     ( -1        ),
        readdata_buffers    ( -1        ),
        writedata_buffers   ( -1        ),
        reqaccept_latency   ( 0         ),
        dataaccept_latency  ( 0         ),
        req2resp_latency    ( 3         ),
        postedwrite_latency ( 0         ),
        meminit             ( "unknown" ),
        meminit_fixeddata   ( 0         )
        {}

    int    request_buffers;
    int    readdata_buffers;
    int    writedata_buffers;
    int    reqaccept_latency;
    int    dataaccept_latency;
    int    req2resp_latency;
    int    postedwrite_latency;
    string meminit;
    int    meminit_fixeddata;
};

template <class Td, class Ta>
class SlaveTL2 : public sc_module
{
  public:
    typedef OCPTL2RequestGrp<Td, Ta> Request;
    typedef OCPTL2ResponseGrp<Td>    Response;

    // constructor (signature used when deriving a module)
    SlaveTL2( sc_module_name );
    ~SlaveTL2();
    static void enableAutoSReset();
    OCP_TL2_SlavePort<Td,Ta> tpP;
    SlaveTl2Config            m_config;

  private:
    // this nested class is there because we need a response method per thread
    friend struct Responder;
    struct Responder : public sc_module {
        SC_HAS_PROCESS( Responder );
        Responder( sc_module_name, SlaveTL2& slave, int thread );
        virtual ~Responder() {}
        void sendMethod();
        SlaveTL2& m_slave;
        int       m_thread;
    };
    vector<Responder*> m_responders;

    // methods
    SC_HAS_PROCESS( SlaveTL2 );
    void getRequestMethod();
    void acceptRequestMethod();
    void SResetInitMethod();
    void prepareResponseMethod();
    void acceptedResponseMethod();

    virtual void end_of_elaboration();
    const OCPParameters* m_pOcpParams;
    sc_time m_clkPeriod;

    // Response Latency (default = 1 (response 1 cycle after received request)
    int m_responseLatency;
    int m_postedWriteLatency;

    // -------------------------------------------------------------
    // Internal variables
    // -------------------------------------------------------------
    sc_event m_acceptRequestEvent;
    // Used to flag that back pressure is being applied to requests

    bool m_sResetAsserted;
    MTimingGrp              m_mTimes;
    STimingGrp              m_sTimes;

    // Data flow
    Request                 m_currentRequest;
    typedef typename multimap<sc_time, Request>::iterator QueueIterator;
    multimap<sc_time, Request>  m_requestQueue;
    // the deleted entries don't really have a reason to persist other than
    // to avoid a local variable in prepareResponseMethod (and to get some safe
    // guarding)
    deque<QueueIterator>                   m_deletedEntries;
    vector<pair<QueueIterator, Response> > m_responseCandidate;
    sc_event                               m_processEvent;
    vector<sc_extension::ScSemaphore*>     m_responseSemaphore;
    int                                    m_activeResponseThread;
    
    // Flow control buffer model (crude)
    int                                    m_maxActiveRequests;
    int                                    m_maxActiveReadData;
    int                                    m_maxActiveWriteData;
    vector<unsigned int>                   m_numActiveRequests;
    vector<unsigned int>                   m_numActiveReadData;
    vector<unsigned int>                   m_numActiveWriteData;
    bool canAcceptRequest( const Request& )            const;
    bool canAcceptRequest( OCPMCmdType, unsigned int ) const;

    // Processing
    void scheduleProcessing();
    bool processRequest ( const Request&, Response& );
    void updateActiveCounts( const Request& );
    void releaseRequestEntry( QueueIterator );
    const Ta m_memSizeInBytes;
    MemoryCl<Td,Ta>* m_pMemory;

    vector<OcpIp::BurstSequence<Ta> > m_burstSequence;
    vector<OcpIp::BurstCounter >      m_burstCounter;

    // Thread arbitration
    Sonics::ThreadBusyProxy*    m_pThreadBusyAccess;
    Sonics::ThreadArbiter       m_threadArbiter;
};
}

#endif // _OcpIpSlaveTL2_h_
