////////////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2005 Sonics, Inc.
//
// Confidential and Proprietary Information of Sonics, Inc.
// Use, disclosure, or reproduction is prohibited without
// written permission from Sonics, Inc.
//
// $Id:
//
// Class to model the common pattern of matching an expected sequence
//
////////////////////////////////////////////////////////////////////////////////
#ifndef _SonicsExpectQueue_h
#define _SonicsExpectQueue_h

#include <deque>
#include "boost/function.hpp"

using std::deque;
namespace Sonics {
template <typename ExpectT, typename MatchT>
class ExpectQueue
{
  public:
    typedef boost::function< bool ( ExpectT&, const MatchT& ) > MatchFunc;

    ExpectQueue () :
        m_numMatched( 0 ),
        m_expectFinished( false )            
        {}
    void expect( const ExpectT&, MatchFunc, bool lastExpect=false );
    bool match ( const MatchT & );

    size_t  size()       const {
        return m_queue.size();
    }
    size_t  numMatched() const {
        return m_numMatched;
    }
    bool    finished()   const {
        return m_expectFinished && ( m_numMatched == m_queue.size() );
    }
        
  private:
    deque<std::pair<ExpectT, MatchFunc> > m_queue;
    size_t m_numMatched;
    bool m_expectFinished;
};

template <typename ExpectT, typename MatchT>
void
ExpectQueue<ExpectT, MatchT>::expect( const ExpectT& exp,
                                     MatchFunc      func,
                                     bool lastExpect )
{
    m_queue.push_back( make_pair( exp, func ) );
    m_expectFinished = lastExpect;
}

template <typename ExpectT, typename MatchT>
bool
ExpectQueue<ExpectT, MatchT>::match( const MatchT& mat )
{
    if ( m_queue.empty() ) {
        return false;
    }
    typename deque<std::pair<ExpectT, MatchFunc> >::iterator matchIt
        = m_queue.begin() + m_numMatched;
    ExpectT&  exp  = matchIt->first;
    MatchFunc func = matchIt->second;
    bool result = func( exp, mat );
    if ( result )
        ++m_numMatched;
    return result;
}
}

#endif /* _SonicsExpectQueue_h */
