//----------------------------------------------------------------------------
//   Copyright 2007-2010 Mentor Graphics Corporation
//   Copyright 2007-2011 Cadence Design Systems, Inc. 
//   Copyright 2010 Synopsys, Inc.
//   All Rights Reserved Worldwide
//
//   Licensed under the Apache License, Version 2.0 (the
//   "License"); you may not use this file except in
//   compliance with the License.  You may obtain a copy of
//   the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in
//   writing, software distributed under the License is
//   distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
//   CONDITIONS OF ANY KIND, either express or implied.  See
//   the License for the specific language governing
//   permissions and limitations under the License.
//----------------------------------------------------------------------------

#ifndef UVM_REPORT_SERVER_H_
#define UVM_REPORT_SERVER_H_

#include "uvmsc/base/uvm_object.h"
#include "uvmsc/report/uvm_report_object.h"

namespace uvm {

// forward class declarations
class uvm_report_global_server;

//----------------------------------------------------------------------------
// CLASS: uvm_report_server
//
//! The class uvm_report_server is a global server that processes all of the reports
//! generated by an uvm_report_handler. None of its methods are intended to be
//! called by normal testbench code, although in some circumstances the virtual
//! methods process_report and/or compose_uvm_info may be overloaded in a
//! subclass.
//----------------------------------------------------------------------------

class uvm_report_server : public uvm_object
{
 public:
  friend class uvm_report_handler;

  //--------------------------------------------------------------------------
  // UVM Standard LRM API below
  //--------------------------------------------------------------------------

  uvm_report_server();

  static void set_server( uvm_report_server* server );

  static uvm_report_server* get_server();

  void set_max_quit_count( int count, bool overridable = true );

  int get_max_quit_count() const;


  void set_quit_count( int quit_count );

  int get_quit_count() const;

  void incr_quit_count();

  void reset_quit_count();


  bool is_quit_count_reached();


  void set_severity_count( uvm_severity severity, int count );

  int get_severity_count( uvm_severity severity ) const;

  void incr_severity_count( uvm_severity severity );

  void reset_severity_counts();


  void set_id_count( const std::string& id, int count );

  int get_id_count( const std::string& id ) const;

  void incr_id_count( const std::string& id );


  virtual void process_report( uvm_severity severity,
                               const std::string& name,
                               const std::string& id,
                               const std::string& message,
                               uvm_action action,
                               UVM_FILE file,
                               const std::string& filename,
                               int line,
                               const std::string& composed_message,
                               int verbosity_level,
                               uvm_report_object* client );

  virtual std::string compose_message( uvm_severity severity,
                                       const std::string& name,
                                       const std::string& id,
                                       const std::string& message,
                                       const std::string& filename,
                                       int line ) const;

  virtual void summarize( UVM_FILE file = 0 ) const;

  void dump_server_state() const;

  /////////////////////////////////////////////////////
  /////////////////////////////////////////////////////
  // Implementation-defined member functions below,
  // not part of UVM Class reference / LRM
  /////////////////////////////////////////////////////
  /////////////////////////////////////////////////////

  static uvm_report_server* m_global_report_server;

 private:
  void f_display( UVM_FILE file, const std::string& str ) const;

  virtual void report( uvm_severity severity,
                       const std::string& name,
                       const std::string& id,
                       const std::string& message,
                       int verbosity_level,
                       const std::string& filename,
                       int line,
                       uvm_report_object* client );

  void copy_severity_counts( uvm_report_server* dst );

  void copy_id_counts( uvm_report_server* dst );

 private:
  // Needed for callbacks
  std::string get_type_name();

  // local data members

  int max_quit_count;
  int quit_count;

  std::map<uvm_severity, int> severity_count;
  typedef std::map<uvm_severity, int>::iterator severity_count_itt;
  typedef std::map<uvm_severity, int>::const_iterator severity_count_citt;

  bool m_max_quit_overridable;

  bool enable_report_id_count_summary;

  std::map<std::string, int> id_count;
  typedef std::map<std::string, int>::iterator id_count_itt;
  typedef std::map<std::string, int>::const_iterator id_count_citt;

}; // class uvm_report_server


//----------------------------------------------------------------------
// CLASS: uvm_report_global_server
//
//! Singleton object that maintains a single global report server
//----------------------------------------------------------------------

class uvm_report_global_server
{
 public:

  uvm_report_global_server()
  {
    get_server();
  }

  //----------------------------------------------------------------------
  // member function: get_server
  //
  // Returns a handle to the central report server.
  //----------------------------------------------------------------------

  uvm_report_server* get_server()
  {
    return uvm_report_server::get_server();
  }

  //----------------------------------------------------------------------
  // member function- set_server (deprecated)
  //----------------------------------------------------------------------

  void set_server( uvm_report_server* server)
  {
    uvm_report_server::set_server(server);
  }

}; // class uvm_report_global_server

} // namespace uvm

#endif // UVM_REPORT_SERVER_H_

