// Scoreboard of the frequency adapter UVCs (freq_generator, freq_detector and registers)
// There will always be error message to display the comparision results between expected output frequency (frequency_generator) and actual frequency (frequency_detector.

class freq_adpt_scoreboard extends uvm_scoreboard;
  typedef enum bit {COV_ENABLE, COV_DISABLE} cover_e;
  cover_e coverage_control = COV_ENABLE;//COV_ENABLE;

  // component utils macro
  `uvm_component_utils_begin(freq_adpt_scoreboard)
    `uvm_field_enum(cover_e, coverage_control, UVM_ALL_ON)
  `uvm_component_utils_end

  // define TLM port imp object
  `uvm_analysis_imp_decl(_i2c)
  `uvm_analysis_imp_decl(_osc_gen)
  `uvm_analysis_imp_decl(_osc_det)

  uvm_analysis_imp_i2c #(i2c_packet, freq_adpt_scoreboard) sb_i2c_in;
  uvm_analysis_imp_osc_gen #(osc_transaction, freq_adpt_scoreboard) sb_osc_gen;
  uvm_analysis_imp_osc_det #(osc_transaction, freq_adpt_scoreboard) sb_osc_det;


  // scoreboard statistics for I2C UVC
  int i2c_packets_in, i2c_in_drop;
  int i2c_compare_in, i2c_miscompare;
  
  // for freq_generator UVC
  int freq_generator_packets_in, freq_generator_in_drop;
  int freq_generator_compare_in, freq_generator_miscompare;
  
  // for freq_detector UVC
  int freq_detector_packets_in, freq_detector_in_drop;
  int freq_detector_compare_in, freq_detector_miscompare;

  // compare
  int num_match = 0;
  int total_match_check = 0;
  int in_dropped = 0;
  int num_mismatch = 0;
  
  // variables for freq_generator coverage check
  real freq;
  
  // variables for freq_detector coverage check and compare
  real freq_out_p, freq_out_n;
  real freq_tol;
  
  // variables for internal caculation of expected DUT outputs
  bit [7:0] INT_REG [2:0];
  bit [7:0] FX2_REG, MUX_REG, DRV_REG;
  // according to DUT outputs:
  bit en_mux;
  bit [1:0] sel_mux;
  
  // Scoreboard storeage
  real freq_in_reg;// a queue to store the input freq in write()
    
  // variables for compare function
  bit pktcompare;
  real actualresult_checker_p;
  real expected_result_pn;
 
  covergroup input_sig_cg;
    option.per_instance = 1;
    SEL_MUX_cov: coverpoint sel_mux ;
    freq_cov : coverpoint freq {
      bins DC         = {[0    :0.9  ]};
      bins Hz         = {[0.9  :10e3 ]};
      bins KHz        = {[10e3 :10e6 ]};
      bins MHz        = {[10e6 :10e7 ]};
      bins TenMhz     = {[10e7 :10e8 ]};
      bins HundredMhz = {[10e8 :10e9 ]};
      bins GHz        = {[10e9 :10e10]};
      `ifdef XRUN
      bins Max        = {[10e10:$    ]};
      `endif
      `ifdef QUESTA
      bins Max        = {[10e10:1.7E308]};
      `endif
      `ifdef VCS
      bins Max        = {[10e10:$    ]};
      `endif
    }
    //ALL_INPUT_COMB_cov: cross freq_cov, SEL_MUX_cov;
  endgroup: input_sig_cg  

// constructor
  function new(string name="", uvm_component parent=null);
    super.new(name, parent);
    sb_i2c_in = new("sb_i2c_in", this);
    sb_osc_gen = new("sb_osc_gen", this);
    sb_osc_det = new("sb_osc_det", this);
    if (coverage_control == COV_ENABLE) begin
     `uvm_info(get_type_name(),"COVERAGE CREATED" , UVM_LOW)
      input_sig_cg = new();
      input_sig_cg.set_inst_name({get_full_name(), ".scoreboard_input_sig_coverage"});
    end
  endfunction
  
    function real abs(input real A); 
      abs = (A<0)? -A:A; 
    endfunction

  // compare function 
  function bit comp_equal (
      input real input_freq, 
      input real actual_result_p,
      logic [1:0] sel
    );
    case(sel)
      2'b00:  expected_result_pn = input_freq/4.0; 
      2'b01:  expected_result_pn = input_freq/2.0;
      2'b10:  expected_result_pn = input_freq;
      2'b11:  expected_result_pn = input_freq*2.0;
    endcase
    
    if ((abs(expected_result_pn - actual_result_p) >= 10e6) && (expected_result_pn != 0)) begin
      `uvm_error("PKT_COMPARE",$sformatf("Frequency MISMATCH freq_generator %f freq_detector %f, when sel_mux = %d", expected_result_pn, actual_result_p, sel))
      num_mismatch++;
      return(0);
    end
    else if ((abs(expected_result_pn - actual_result_p) < 10e6) && (expected_result_pn != 0) ) begin
      `uvm_info("PKT_COMPARE",$sformatf("Frequency MATCH freq_generator %f freq_detector %f, when sel_mux = %d", expected_result_pn, actual_result_p, sel), UVM_LOW)
      return(1);
    end
    else if(expected_result_pn == 0) begin
      `uvm_error("PKT_COMPARE",$sformatf("Frequency ZERO in freq_generator %f ", expected_result_pn))
      in_dropped++;
      return(0);
    end
    
    return(0);
  endfunction

  //write() function for I2C registers block
  virtual function void write_i2c(i2c_packet packet);
    i2c_packet sb_packet;
    //Make a copy for storing in the scoreboard
    $cast(sb_packet, packet.clone()); // Clone returns uvm_object type
    i2c_packets_in++;

    if (sb_packet.tg_addr != sb_packet.tg_id || sb_packet.reg_addr > 2)
      i2c_in_drop++;
    else begin
      if(sb_packet.rw_ == 0)
        INT_REG[sb_packet.reg_addr] = sb_packet.data;
      MUX_REG = INT_REG[1];
      en_mux = MUX_REG[2];
      sel_mux = MUX_REG[1:0];
    end
  endfunction

  // write() function for generator
  virtual function void write_osc_gen(osc_transaction packet);
    osc_transaction sb_packet;
    $cast(sb_packet, packet.clone());  // Clone returns uvm_object type
    // separate the case of diff_sel
    freq_generator_packets_in++;
    freq_in_reg = sb_packet.freq;
  endfunction: write_osc_gen
  
  // write() function for detector
  virtual function void write_osc_det(osc_transaction packet);
    osc_transaction sb_packet;
    $cast(sb_packet, packet.clone());  // Clone returns uvm_object type
      if(en_mux) begin
        // Compare output freq with expected result calculated from input freq    
        actualresult_checker_p = sb_packet.freq;
        
        // if the freq detector monitor finds that the output frequency become unstable after being stable for 8 cycles
        if(actualresult_checker_p == -1) begin 
          num_mismatch++;
          num_match--;
          `uvm_error("PKT_COMPARE",$sformatf("Matched Frequency becomes UNSTABLE at input frequency =  %f", expected_result_pn))
        end
        else begin
          pktcompare =  comp_equal(
            freq_in_reg,
            actualresult_checker_p,
            sel_mux
          );
          
          if( pktcompare ) begin
            `uvm_info(get_type_name(), $sformatf("Scoreboard Compare Match: FREQUENCY Packet\n%s", packet.sprint()), UVM_HIGH)
            num_match++;
          end
          total_match_check++;
          
          if (coverage_control == COV_ENABLE) begin
            `uvm_info(get_type_name(),"FREQ_GENERATOR + REGISTERS SCOREBOARD COVERAGE SAMPLE" , UVM_FULL)
            input_sig_cg.sample();
          end
        end
      end
  endfunction: write_osc_det
  
  // UVM report() phase
  function void report_phase(uvm_phase phase);
    `uvm_info(get_type_name(), $sformatf("Report:\n\tScoreboard: OUTPUT FREQUENCY Check Statistics \n\t\tPackets In: %0d\t Match: %0d\t Mismatch: %0d\t Dropped packet: %0d\n\n", total_match_check, num_match, num_mismatch, in_dropped), UVM_LOW)
    `uvm_info(get_type_name(), $sformatf("Report:\n\tScoreboard: FREQ_GENERATOR Packet Statistics \n\t\tPackets In: %0d\t Packets Dropped: %0d\n\n", freq_generator_packets_in, freq_generator_in_drop), UVM_LOW)
    if ((i2c_miscompare + num_mismatch) > 0)
      `uvm_error(get_type_name(),"Status:\n\nSimulation FAILED\n")
    else begin
      `uvm_info(get_type_name(),"Status:\n\nSimulation PASSED\n", UVM_NONE)
      $display("**  Overall Coverage = %f                                 **", input_sig_cg.get_inst_coverage());
    end
  endfunction : report_phase

endclass: freq_adpt_scoreboard

//------------------------------------------------------------------------------
//
// ------- Mixed Signal version of Scoreboard --------------
//
//------------------------------------------------------------------------------

class freq_adpt_ms_scoreboard extends freq_adpt_scoreboard;
  `uvm_component_utils(freq_adpt_ms_scoreboard)

// variables of freq_generator 
  real ampl, bias, freq;

  // variables for freq_detector coverage check and compare
  real freq_out_p, freq_out_n;
  real freq_tol;
  bit pktcompare;

  // osc_detector queue when change in signal is observed due to a change in the 
  // control registers. The signal change is reported before the corresponding I2C
  // command that was issued
  osc_ms_transaction osc_det_q[$];
  
  // flags to gate the frequency comparisons until the latest information has arrived
  bit osc_gen_change_observed, osc_det_change_observed;

  //DEBUG CHECKERS
  bit [1:0] sel_checker;
  real inputfreq_checker;
  real inputampl_checker;
  real inputbias_checker;
  real actualresult_checker_p, actualresult_checker_n;
  real actualresult_checker_bias, actualresult_checker_ampl;
  real expected_freq;
  event check;
   
  // constructor
  function new(string name, uvm_component parent);
    super.new(name, parent);
  endfunction
   
  function real abs(input real A); 
    abs = (A<0)? -A:A; 
  endfunction

  // compare function 
  function bit comp_equal (
      input bit [1:0] sel,
      input real freq_in,
      input real ampl_in,
      input real bias_in,
      input real actual_freq,
      input real actual_bias,
      input real actual_ampl
    );

    case(sel)
      2'b00:  expected_freq = freq_in/4.0; 
      2'b01:  expected_freq = freq_in/2.0;
      2'b10:  expected_freq = freq_in;
      2'b11:  expected_freq = freq_in*2.0;
    endcase
    if ((abs(expected_freq - actual_freq) >= 10e6) && (expected_freq != 0)) begin
      `uvm_error("PKT_COMPARE",$sformatf("\nFrequency MISMATCH freq_generator %f freq_detector %f, when sel_mux = %d", expected_freq, actual_freq, sel))
      num_mismatch++;
      return(0);
    end
    else if ((abs(expected_freq - actual_freq) < 10e6) && (expected_freq != 0) ) begin
      `uvm_info("PKT_COMPARE",$sformatf("\nFrequency MATCH freq_generator %f freq_detector %f, when sel_mux = %d", expected_freq, actual_freq, sel), UVM_NONE)
      return(1);
    end
    else if(expected_freq == 0) begin
      `uvm_error("PKT_COMPARE",$sformatf("\nFrequency ZERO in freq_generator %f ", expected_freq))
      in_dropped++;
      return(0);
    end
    
    return(0);
  endfunction
  
  //write() function for I2C registers block
  virtual function void write_i2c(i2c_packet packet);
    i2c_packet sb_packet;
    osc_ms_transaction det_packet;
    //Make a copy for storing in the scoreboard
    $cast(sb_packet, packet.clone()); // Clone returns uvm_object type
    i2c_packets_in++;

    if (sb_packet.tg_addr != sb_packet.tg_id || sb_packet.reg_addr > 2)
      i2c_in_drop++;
    else begin
      if(sb_packet.rw_ == 0)
        INT_REG[sb_packet.reg_addr] = sb_packet.data;
      MUX_REG = INT_REG[1];
      en_mux = MUX_REG[2];
      sel_mux = MUX_REG[1:0];
    end

    if(osc_det_change_observed) begin
      det_packet = osc_det_q.pop_front();
      if(en_mux) begin
        // Compare output freq with expected result calculated from input freq
        sel_checker = sel_mux;
        inputfreq_checker = freq;
        inputampl_checker = ampl;
        inputbias_checker = bias;
        actualresult_checker_p = det_packet.freq;
        actualresult_checker_bias = det_packet.bias;
        actualresult_checker_ampl = det_packet.ampl;
        
        pktcompare =  comp_equal(
          sel_checker,
          inputfreq_checker,
          inputampl_checker,
          inputbias_checker,
          actualresult_checker_p,
          actualresult_checker_bias,
          actualresult_checker_ampl
        );
        
        if( pktcompare ) begin
          `uvm_info(get_type_name(), $sformatf("Scoreboard Compare Match: FREQUENCY Packet\n%s", packet.sprint()), UVM_HIGH)
          num_match++;
        end
        total_match_check++;
      end
      osc_det_change_observed = 0;
    end
  endfunction

  // write() function for frequency generator
  virtual function void write_osc_gen(osc_transaction packet);
    osc_ms_transaction sb_packet;
    $cast(sb_packet, packet.clone());  // Clone returns uvm_object type
    freq_generator_packets_in++;
    freq = sb_packet.freq;
    ampl = sb_packet.ampl;
    bias = sb_packet.bias;
    `uvm_info("WRITE_OSC_GEN",$sformatf("\nFreq = %f\tAmpl = %f \t Bias = %f",freq,ampl,bias),UVM_LOW)
    osc_gen_change_observed = 1;
  endfunction: write_osc_gen
  
  // write() function for detector
  virtual function void write_osc_det(osc_transaction packet);
    osc_ms_transaction sb_packet;
    $cast(sb_packet, packet.clone());  // Clone returns uvm_object type
    freq_detector_packets_in++;
    // Compare immediately if the report from the detector monitor was due to a change from the 
    // generator
    if(osc_gen_change_observed) begin
      if(en_mux) begin
        // Compare output freq with expected result calculated from input freq
        sel_checker = sel_mux;
        inputfreq_checker = freq;
        inputampl_checker = ampl;
        inputbias_checker = bias;
        actualresult_checker_p = sb_packet.freq;
        actualresult_checker_bias = sb_packet.bias;
        actualresult_checker_ampl = sb_packet.ampl;
        
        pktcompare =  comp_equal(
          sel_checker,
          inputfreq_checker,
          inputampl_checker,
          inputbias_checker,
          actualresult_checker_p,
          actualresult_checker_bias,
          actualresult_checker_ampl
        );
        
        if( pktcompare ) begin
          `uvm_info(get_type_name(), $sformatf("Scoreboard Compare Match: FREQUENCY Packet\n%s", packet.sprint()), UVM_HIGH)
          num_match++;
        end
        total_match_check++;
      end
      osc_gen_change_observed = 0;
    end
    // Else push to the queue and allow the comparison to happen in the I2C write function
    else begin
      osc_det_q.push_back(sb_packet);
      osc_det_change_observed = 1;
    end


  endfunction: write_osc_det
  

  // UVM report() phase
  function void report_phase(uvm_phase phase);
    `uvm_info(get_type_name(), $sformatf("Report:\n\tScoreboard: OUTPUT FREQUENCY Check Statistics \n\t\tPackets In: %0d\t Match: %0d\t Mismatch: %0d\t Dropped packet: %0d\n\n", total_match_check, num_match, num_mismatch, in_dropped), UVM_LOW)
    `uvm_info(get_type_name(), $sformatf("Report:\n\tScoreboard: FREQ_GENERATOR Packet Statistics \n\t\tPackets In: %0d\t Packets Dropped: %0d\t Miscompare: %0d\n\n", freq_generator_packets_in, freq_generator_in_drop, freq_generator_miscompare), UVM_LOW)
    if ((i2c_miscompare + num_mismatch) > 0)
      `uvm_error(get_type_name(),"Status:\n\nSimulation FAILED\n")
    else begin
      `uvm_info(get_type_name(),"Status:\n\nSimulation PASSED\n", UVM_NONE)
    end
  endfunction : report_phase
 
  
endclass: freq_adpt_ms_scoreboard



