//Slew limiting block
//SystemVerilog equivalent of Verilog-AMS slew() filter
//Units for slewp and slewn : V/s

`timescale 1ns/1fs
`ifdef XRUN
import cds_rnm_pkg::*;
`endif
`ifdef VCS
import snps_msv_nettype_pkg::*;
`endif

module slew(input real in, output real out, input real slewp, input real slewn);

	real out_int;
	
	event startClk, stopClk;
	
	real samplePeriod = 1; //Sample period in s - will use 1s to convert to timescale units
	//Sample period can't be less than 1fs, because of SV limitations. Therefore if samplePeriod ends up as <1fs
	//It will get set to 1fs.
	real samplePeriod1s;
	
	parameter real slew_steps = 10;
	
	bit clk; //Internal clock for slew rate limiting
	

	always begin
		if(slewp < 1e20 && slewn < 1e20 && (slewp != 0) && (slewn != 0)) begin
			if(slewp >= -slewn) samplePeriod = 1/(slewp*slew_steps);
			else samplePeriod = -1/(slewn * slew_steps);
			//Is samplePeriod < 2fs , then samplePeriod = 2fs
			if(samplePeriod < 2e-15) samplePeriod = 2e-15; 
		end
		samplePeriod1s = samplePeriod*1s; // In timescale units
		@(slewp, slewn);
		->stopClk;
		#(1step) ->startClk;
	end

  initial begin
		out_int = in;
	end
	
	bit clk_running;
	
	//Triggered on change in in.
	always @(in) begin
		if(!clk_running)
			->startClk;
	end
	
	always
		fork
			begin : clkGen
				clk = 1'b0;
				@startClk clk_running = 1;
				forever begin
					clk = ~clk;   
					#(samplePeriod*1s / 2.0); //Wait for samplePeriod/2 in timescale units					
				end  

			end
			
			begin
				@stopClk;
				disable clkGen;
				clk_running = 0;
			end
		join
   
  real dVmaxp, dVmaxn; 
  
	always @(posedge clk) begin
					
				if((out_int != in)) begin
					// calculate maximum possible step size
					dVmaxp = slewp * samplePeriod; 
					dVmaxn = slewn * samplePeriod;
					//If input is out of range of max step size, just take the max step and then re-evaluate next clk cycle
					if((in > out_int) && ((in - out_int) > dVmaxp))
						out_int += dVmaxp; 
					else if((in < out_int) && ((in - out_int) < dVmaxn))
						out_int += dVmaxn;
					else begin
						out_int = in;
						->stopClk;
					end
				end
				else ->stopClk;
				
	end 

	assign out = out_int;
	
endmodule
