// A module of frequency doubler

`timescale 1ns/1fs

module freq_doubler(
	output logic clk_out,
	input logic clk_in,
	input logic [7:0] PW_adj
);

  parameter real pw_scale = 0.01;   //Scale factor for pulse width adjustment
                                    //1 bit = pw_scale % of output clock duty cycle away from 50%
  
  real Tlast, Tcur;     //internal variables to capture last clock edge time and latest clock edge time
  real clk_outperiod;   //output clock period
  reg clk_in_int, clk_outval;
  real clk_outpw;       //output clock positive half-cycle pulse width
  real final_dutycycle; //output clock duty cycle = 50 + pw_scale*pw_adj
  
  bit clk_in_comfirm;
  
  initial begin
    Tlast = 0;
    Tcur = 0;
    clk_outval = 0;
  end
  
  //Filter clk_in signal to remove invalid drives
  assign clk_in_int = (clk_in === 1'b1) ? 1'b1 : 1'b0;
  
  //Capture clk_in's half-cycle time on every edge of clk_in
  always @(posedge clk_in_int) begin    
    Tlast = Tcur;
    Tcur = $realtime;
    clk_outperiod = 0.5* (Tcur - Tlast);
    clk_outpw = clk_outperiod * (final_dutycycle) / 100.0; // calculate the pulse width of clk_out
    clk_in_comfirm = 1;
  end
  
  // fork to detect if clk_in is oscillating
  always begin
    fork 
      begin
        @(posedge clk_in_int);
        clk_in_comfirm = 1;
        disable TIMEOUT;
      end
      TIMEOUT:begin
        #(2.001ns) clk_in_comfirm = 0;
      end
    join_any
  end  
  
  always begin
    wait (Tlast > 0) begin
      if(!clk_in_comfirm) begin
        clk_outval = 0;
        @(posedge clk_in_int);
      end
      else begin
        clk_outval = 1;  // generate pulse according to the calculated width
        #(clk_outpw);
        clk_outval = 0;
        #(clk_outperiod - clk_outpw);
      end
    end
  end
      
  always begin
    final_dutycycle = $signed(PW_adj)*pw_scale + 50;  
    @(PW_adj);
  end
  
  assign clk_out = clk_outval;
  
	

endmodule

