// This is I2C target model. The target device ID is parameterizable.
// data in SDA: MSB->LSB
/* The process of reading data from SDA
   - Idle      (waiting for start)
   - Start     (reading the 7-bit target device ID)
   - Read RW   (SDA -- 0)
   - ACK       (SDA -- 0)
   - Read addr (8-bit)
   - ACK       (SDA -- 0)
   - Read data (8-bit)
   - ACK       (SDA -- 0)
   - Stop      (send data to "data" output port)
   - Idle

   The process of writing data to SDA
   - Idle           (waiting for start)
   - Start          (reading the 7-bit target device ID)
   - Read RW        (SDA -- 0)
   - ACK            (SDA -- 0)
   - Read addr      (8-bit)
   - ACK            (SDA -- 0)
   - Repeated start (reading the 7-bit target device ID)
   - Read RW        (SDA -- 1)
   - ACK            (SDA -- 0)
   - Write data     (8-bit)
   - NACK           (SDA -- 0)
   - Stop           
   - Idle
*/
`timescale 1ns/1ps
package target_states;
typedef enum bit [3:0] {IDLE, START, READ_RW, READ_REG_ADDR, READ_DATA, WRITE_DATA, STOP, ACK_s, SEND_ACK, SEND_NACK} target_state_t;
endpackage

module I2C_target import target_states::*; (
  input wire SCL,
  inout wire SDA,
  output logic RW,
  output logic reg_en,
  output logic [7:0] addr,
  inout wire [7:0] data
  );
  parameter [6:0] target_id = 7'b1001101;
  parameter [7:0] register_size = 8'd3;
  //Internal registers
  bit [7:0] addr_reg, data_reg;
  bit [6:0] target_id_rx;
  reg NACK_int;
  bit data_1bit;
  
  bit start_detected, rpt_start_detected, stop_detected;
  logic RW_int;
  logic [7:0] reg_addr, reg_data;
  
  target_state_t state = IDLE;
  target_state_t next_state;
  target_state_t state_buff;
  int bit_count;
  
	event start_e, stop_e; // for debug
	bit repeated_start_happened; // flag that indicates the happened repeated start (only on for 1 clock cycle)
	
	bit started, stopping; 
	initial begin
	  reg_addr = 0;
	  reg_data = 0;
	end
	always@(negedge SDA) begin: start_det
	  #1step; // avoid the race condiiton when both SDA and SCL are at falling edge
	  if(SCL && !started) begin // the "started" flag is to avoid the case that the 0->1 transition on SDA is misdetected as "start"
	    -> start_e;
  	  started = 1;
	    stopping = 0;
	    state = START;
      next_state = START; // initially set to make next_state logic within always block work
	    target_id_rx = 0;
	    bit_count = 0;
	    reg_data = 0;
	    repeated_start_happened = 0;
	  end
	end
	
	always@(posedge SDA) begin: stop_det
	  if(SCL & stopping) begin // "stopping" flag is on when the state is STOP
	    ->stop_e;
	    stopping = 0;
	    started = 0;
	    reg_addr = 0;
	    reg_data = 0;
	    bit_count = 0;
	    state = IDLE;
	  end
	end
	
	bit write_reg_en, read_reg_en;
	target_state_t last_state;

  always@(negedge SCL) begin
    if (state != IDLE && next_state != START) begin
      state = next_state;
    end

    NACK_int = (state == SEND_ACK) ? 0 : 1;
    if(state == WRITE_DATA) begin // WRITE_DATA happens at negedge of SCL
        RW_int = 1;
        if(bit_count < 8) begin 
          last_state = WRITE_DATA;
          data_1bit = (data >> (7-bit_count));  
          started = 1;
          NACK_int = 1;
          bit_count = bit_count + 1; 
        end
        else if(bit_count == 8) begin
          bit_count = 0;
          data_1bit = 1; // release SDA
          @(SCL);
          next_state = (SDA) ? STOP : WRITE_DATA;
        end
    end
  end

  always@(posedge SCL) begin

    case(state) 
      IDLE: begin
        NACK_int = 1;
        RW_int = 0;
        last_state = IDLE;
      end
      START: begin
        RW_int = 0;
        if (!repeated_start_happened) 
          repeated_start_happened = (last_state == READ_DATA);
        last_state = START;
        if(bit_count < 6) begin // if a register if not full
          target_id_rx = target_id_rx | (SDA << (6-bit_count)) ; 
          bit_count = bit_count + 1; // increment the bit position of the data stored in each register address
        end
        else if(bit_count == 6) begin
          target_id_rx = target_id_rx | (SDA << (6-bit_count)) ; 
          next_state = READ_RW; // checking in READ_RW
          bit_count = 0;
        end
      end
      READ_RW: begin
        last_state = READ_RW;
        state_buff = (SDA) ? WRITE_DATA : READ_REG_ADDR; // store next state and change it after sending ack
        RW_int = SDA;
        if (target_id == target_id_rx) begin // send ACK if target address is valid
          next_state = SEND_ACK;
        end
        else begin
					next_state = SEND_NACK;
				end
      end
      SEND_ACK: begin
        last_state = SEND_ACK;
        NACK_int = 0;
        next_state = state_buff;
      end
      SEND_NACK: begin
        last_state = SEND_NACK;
        NACK_int = 1;
        next_state = STOP;
      end
      READ_REG_ADDR: begin
        if(bit_count < 8) begin 
          last_state =  READ_REG_ADDR;
          reg_addr = (reg_addr | (SDA << (7-bit_count)));  
          NACK_int = 1;
          bit_count = bit_count + 1; 
        end
        if(bit_count == 8) begin
          bit_count = 0;
          state_buff = READ_DATA;
          if($isunknown(^reg_addr) || !(reg_addr inside {[0:register_size-1]})) begin // Send NACK if there's unvalid bit or reg_addr is not correct
            next_state = SEND_NACK;
            state_buff = STOP;
          end
          else
            next_state = SEND_ACK;
        end
      end
      READ_DATA: begin
        started = (bit_count == 0) ? 0 : 1;
        last_state = READ_DATA;
        if(bit_count < 8) begin 
          reg_data = (reg_data | (SDA << (7-bit_count)));  
          NACK_int = 1;
          bit_count = bit_count + 1;
        end
        if(bit_count == 8) begin
          bit_count = 0;
          if($isunknown(^reg_data)) begin // Send NACK if there's unvalid bit
            next_state = SEND_NACK;
          end
          else begin
            next_state = SEND_ACK;
            read_reg_en = 1; // write data that's been read from SDA
          end
          state_buff = STOP;
        end
      end
      STOP: begin
        NACK_int = 1;
        stopping = 1;
        started = 0;
        RW_int = 0;
        last_state = STOP;
        next_state = IDLE;
      end
    endcase
  end
  
  always@(posedge reg_en) begin
    #3 read_reg_en = 0; // generate a pulse
  end

  assign write_reg_en = RW_int; 
  
  logic SDA_int;
  logic data_dn;
  
  // drive the SDA down to 0 if the current single bit data is 0. Otherwise write 1
  always_comb begin
    if(state == WRITE_DATA) begin
      data_dn = data_1bit ? 1: 0; 
    end
    else data_dn = 1;
  end

  bit reg_en_int;
  always @(read_reg_en, write_reg_en)
    reg_en_int <= read_reg_en | write_reg_en;

  assign SDA_int = (NACK_int && data_dn);
  assign (pull1, strong0) SDA = SDA_int ? 1'bz : 1'b0;  
  assign RW = RW_int | repeated_start_happened;
  assign addr = reg_addr;
  assign data = (RW && write_reg_en) ? 8'bz : ((!RW && read_reg_en) ? reg_data : 8'bz); 
  assign reg_en = reg_en_int;
  
endmodule
